/**
 * @package     libapache2-vncproxy
 * @file        mod_vncproxy.c
 * @author      Nicholas J. Kisseberth <nkissebe@purdue.edu>
 * @copyright   Copyright (c) 2010-2014 HUBzero Foundation, LLC.
 * @license     http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
 *
 * Copyright (c) 2010-2014 HUBzero Foundation, LLC.
 *
 * This file is part of: The HUBzero(R) Platform for Scientific Collaboration
 *
 * The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
 * software: you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * HUBzero is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * HUBzero is a registered trademark of HUBzero Foundation, LLC.
 */

/**
 * @TODO:
 *
 *	- filter shouldn't be restricted to getline mode
 *  - possibly close all open file descriptors
 *	  before exec() so we don't have that as
 *    a requirement for the helpers
 *  - add encoded hostname mode support (vncsession.host.domain:port)
 *  - add host and port query results to availble helper variables
 *  - triple check apache process lifecycle works right when do the clean
 *    shutdown rather than abrupt exit().
 *  - DBD_PARAMS and DBD_DRIVER should not be required when using helper
 *  - Only required configuration (usable) should be vncProxy On
 */

#define CORE_PRIVATE /* access to core_module */

#include "httpd.h"
#include "http_config.h"
#include "http_core.h"
#include "http_log.h"
#include "http_connection.h"

#include "apr_strings.h"

#include <sys/fcntl.h>
#include <unistd.h> /* _exit(), execl() */

#include "mod_proxy.h"
#include "mod_ssl.h"

static int vncproxy_connect(request_rec *r, char *token);

/* We have to peek into the internal apr_socket_t structure
 * to pull out the raw file descriptor so we can pass it
 * to our helper program. I could find no legitimate way
 * to obtain this, there is more to this but I only need socketdes.
 */
struct apr_socket_t
{
	apr_pool_t *pool;
	int socketdes;
};

/*
 * Declare ourselves so the configuration routines can find and know us.
 * We'll fill it in at the end of the module.
 */
module AP_MODULE_DECLARE_DATA vncproxy_module;

/*
 * vncProxy configuration record.  Used for both per-directory and per-server
 * configuration data.
 */

#define CONFIG_MODE_SERVER 1
#define CONFIG_MODE_DIRECTORY 2
#define CONFIG_MODE_COMBO 3     /* Shouldn't ever happen. */

typedef struct vncproxy_cfg
{
	int cmode; /* Environment to which record applies (directory, server, or combination).  */
	const char *vncProxyHelperStderr;
	const char *vncProxyHelper;
	const char *vncProxyDBDriver;
	const char *vncProxyDBDParams;
	int vncProxy;
	int vncProxyShutdownSSLOnExec;
} vncproxy_cfg;

static const char *vncproxy_banner = "mod_vncproxy/1.3.2";

/*
 * Locate our directory configuration record for the current request.
 */
static vncproxy_cfg *our_dconfig(const request_rec *r)
{
	vncproxy_cfg *cfg = NULL;
	
	if (r != NULL) 
		cfg = ap_get_module_config(r->per_dir_config, &vncproxy_module);

	return cfg;
}

static const char *cmd_vncProxy(cmd_parms *cmd, void *mconfig, int flag)
{
	vncproxy_cfg *cfg = (vncproxy_cfg *) mconfig;

	ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, cmd->server, "cmd_vncProxy(%d)", flag);

	cfg->vncProxy = flag;
	return NULL;	
}

static const char *cmd_vncProxyShutdownSSLOnExec(cmd_parms *cmd, void *mconfig, int flag)
{
	vncproxy_cfg *cfg = (vncproxy_cfg *) mconfig;

	ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, cmd->server, "cmd_vncProxyShutdownSSLOnExec(%d)", flag);

	cfg->vncProxyShutdownSSLOnExec = flag;
	return NULL;
}

static const char *cmd_vncProxyDBDriver(cmd_parms *cmd, void *mconfig, const char *param)
{
	vncproxy_cfg *cfg = (vncproxy_cfg *) mconfig;

	ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, cmd->server, "cmd_vncProxyDBDriver(\"%s\")", param);

	if (strlen(param) > 0)
		cfg->vncProxyDBDriver = param;

	return NULL;
}

static const char *cmd_vncProxyDBDParams(cmd_parms *cmd, void *mconfig, const char *param)
{
	vncproxy_cfg *cfg = (vncproxy_cfg *) mconfig;

	ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, cmd->server, "cmd_vncProxyDBDParams(*redacted* length=%ld)", strlen(param));

	if (strlen(param) > 0)
		cfg->vncProxyDBDParams = param;

	return NULL;
}

static const char *cmd_vncProxyHelperStderr(cmd_parms *cmd, void *mconfig, const char *param)
{
	vncproxy_cfg *cfg = (vncproxy_cfg *) mconfig;

	if (param && *param) {
		cfg->vncProxyHelperStderr = param;
	}

	ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, cmd->server, "cmd_vncProxyHelperStderr(\"%s\")", param);

	return NULL;
}

/*
 * vncProxyHelper <program to run>
 *
 * Replace %t with session token
 * Replace %u with uri
 * Replace %r with remote ip address
 *
 * Example: vncProxyHelper /usr/bin/vncproxy %t 
 */

static const char *cmd_vncProxyHelper(cmd_parms *cmd, void *mconfig, const char *param)
{
	vncproxy_cfg *cfg = (vncproxy_cfg *) mconfig;

	ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, cmd->server, "cmd_vncProxyHelper(\"%s\")", param);

	if (strlen(param) > 0) {
		cfg->vncProxyHelper = param;
	}

	return NULL;
}

/* Build ProxyHelperString with substitutions made. Its a wee bit inefficient as written.  */

static const char *buildProxyHelperString(apr_pool_t *pool, const char *param, const char *token,
										  const char *uri, const char *remoteip)
{
	char *p, *s, *r;

	ap_log_perror(APLOG_MARK, APLOG_DEBUG, 0, pool, "buildProxyHelperString(%s): start", param);

	if (strlen(param) <= 0) {
		ap_log_perror(APLOG_MARK, APLOG_DEBUG, 0, pool, "buildProxyHelperString(): end [no helper defined]");
		return NULL;
	}

	p = strchr(param, '%');

	if (p == NULL) {
		ap_log_perror(APLOG_MARK, APLOG_DEBUG, 0, pool, "buildProxyHelperString(): end [no substitutions needed]");
		return param;
	}

	r = apr_pstrndup(pool, param, p - param);

	s = p;

	while (*s) {
		if (s[1] == 't') {
			r = apr_pstrcat(pool, r, token, NULL);
			s += 2;
		}
		else if (s[1] == 'r') {
			if (remoteip == NULL) {
				remoteip = "unknown";
			}
			r = apr_pstrcat(pool, r, remoteip, NULL);
			s += 2;
		}
		else if (s[1] == 'u') {
			r = apr_pstrcat(pool, r, uri, NULL);
			s += 2;
		}
		else {
			r = apr_pstrcat(pool, r, "%", NULL);
			s += 1;
		}

		p = strchr(s, '%');

		if (p == NULL) {
			r = apr_pstrcat(pool, r, s, NULL);
			ap_log_perror(APLOG_MARK, APLOG_DEBUG, 0, pool, "buildProxyHelperString(): end [%s]", r);
			return (r);
		}

		r = apr_pstrcat(pool, r, apr_pstrndup(pool, s, p - s), NULL);

		s = p;
	}

	ap_log_perror(APLOG_MARK, APLOG_DEBUG, 0, pool, "buildProxyHelperString(): end [%s]", r);
	return r;
}

static const command_rec vncproxy_cmds[] = {
	AP_INIT_FLAG("vncProxy", cmd_vncProxy, NULL, RSRC_CONF, "on if VNC proxy requests should be accepted"),
	AP_INIT_FLAG("vncProxyShutdownSSLOnExec", cmd_vncProxyShutdownSSLOnExec, NULL, RSRC_CONF, "on if an existing"
		"SSL session should be shutdown before executing vncProxyHelper"),
	AP_INIT_TAKE1(
				"vncProxyHelperStderr", /* directive name */
				cmd_vncProxyHelperStderr, /* config action routine */
				NULL, /* argument to include in call */
				OR_OPTIONS, /* where available */
				"the filename of the vncproxy log" /* directive description */
				),
	AP_INIT_TAKE1(
				"vncProxyDBDriver", /* directive name */
				cmd_vncProxyDBDriver, /* config action routine */
				NULL, /* argument to include in call */
				OR_OPTIONS, /* where available */
				"DBD Driver Name" /* directive description */
				),
	AP_INIT_TAKE1(
				"vncProxyDBDParams", /* directive name */
				cmd_vncProxyDBDParams, /* config action routine */
				NULL, /* argument to include in call */
				OR_OPTIONS, /* where available */
				"DBD Parameters" /* directive description */
				),
	AP_INIT_TAKE1(
				"vncProxyHelper", /* directive name */
				cmd_vncProxyHelper, /* config action routine */
				NULL, /* argument to include in call */
				OR_OPTIONS, /* where available */
				"vncProxy Helper Executable" /* directive description */
				),
	{ NULL }
};

/*
 * This function gets called to create a per-directory configuration
 * record.  This will be called for the "default" server environment, and for
 * each directory for which the parser finds any of our directives applicable.
 * If a directory doesn't have any of our directives involved (i.e., they
 * aren't in the .htaccess file, or a <Location>, <Directory>, or related
 * block), this routine will *not* be called - the configuration for the
 * closest ancestor is used.
 */
static void *vncproxy_create_dir_config(apr_pool_t *p, char *dirspec)
{
	vncproxy_cfg *cfg;

	cfg = (vncproxy_cfg *) apr_pcalloc(p, sizeof (vncproxy_cfg));

	ap_log_perror(APLOG_MARK, APLOG_DEBUG, 0, p, "vncproxy_create_dir_config(p,\"%s\")", dirspec);

	cfg->cmode = CONFIG_MODE_DIRECTORY;
	cfg->vncProxyHelperStderr = "/var/log/vncproxy/vncproxy.log";
	cfg->vncProxyDBDriver = "mysql";
	cfg->vncProxyDBDParams = "";
	cfg->vncProxyHelper = "/usr/bin/vncproxy";
	cfg->vncProxy = 0;
	cfg->vncProxyShutdownSSLOnExec = 1;

	return (void *) cfg;
}

/*
 * This function gets called to merge two per-directory configuration
 * records.  This is typically done to cope with things like .htaccess files
 * or <Location> directives for directories that are beneath one for which a
 * configuration record was already created.  The routine has the
 * responsibility of creating a new record and merging the contents of the
 * other two into it appropriately.  If the module doesn't declare a merge
 * routine, the record for the closest ancestor location (that has one) is
 * used exclusively.
 */
static void *vncproxy_merge_dir_config(apr_pool_t *p, void *parent_conf, void *newloc_conf)
{
	vncproxy_cfg *cfg = (vncproxy_cfg *) apr_pcalloc(p, sizeof (vncproxy_cfg));
	vncproxy_cfg *pcfg = (vncproxy_cfg *) parent_conf;
	vncproxy_cfg *ncfg = (vncproxy_cfg *) newloc_conf;

	ap_log_perror(APLOG_MARK, APLOG_DEBUG, 0, p, "vncproxy_merge_dir_config()");

	/*
	 * Some things get copied directly from the more-specific record, rather
	 * than getting merged.
	 */
	cfg->vncProxyDBDriver = ncfg->vncProxyDBDriver;
	cfg->vncProxyDBDParams = ncfg->vncProxyDBDParams;
	cfg->vncProxyHelperStderr = ncfg->vncProxyHelperStderr;
	cfg->vncProxyHelper = ncfg->vncProxyHelper;
	cfg->vncProxy = ncfg->vncProxy;
	cfg->vncProxyShutdownSSLOnExec = ncfg->vncProxyShutdownSSLOnExec;

	/*
	 * If we're merging records for two different types of environment (server
	 * and directory), mark the new record appropriately.  Otherwise, inherit
	 * the current value.
	 */
	cfg->cmode = (pcfg->cmode == ncfg->cmode) ? pcfg->cmode : CONFIG_MODE_COMBO;

	return (void *) cfg;
}

/*
 * This function gets called to create a per-server configuration
 * record.  It will always be called for the "default" server.
 */
static void *vncproxy_create_server_config(apr_pool_t *p, server_rec *s)
{
	vncproxy_cfg *cfg;
	cfg = (vncproxy_cfg *) apr_pcalloc(p, sizeof (vncproxy_cfg));

	ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, s, "vncproxy_create_server_config()");

	cfg->cmode = CONFIG_MODE_SERVER;
	cfg->vncProxyDBDriver = "mysql";
	cfg->vncProxyDBDParams = "";
	cfg->vncProxyHelperStderr = "/var/log/vncproxy/vncproxy.log";
	cfg->vncProxyHelper = "/usr/bin/vncproxy";
	cfg->vncProxy = 0;
	cfg->vncProxyShutdownSSLOnExec = 1;

	return (void *) cfg;
}

/*
 * This function gets called to merge two per-server configuration
 * records.  This is typically done to cope with things like virtual hosts and
 * the default server configuration  The routine has the responsibility of
 * creating a new record and merging the contents of the other two into it
 * appropriately.  If the module doesn't declare a merge routine, the more
 * specific existing record is used exclusively.
 */
static void *vncproxy_merge_server_config(apr_pool_t *p, void *base_conf, void *virt_conf)
{
	vncproxy_cfg *cfg = (vncproxy_cfg *) apr_pcalloc(p, sizeof (vncproxy_cfg));
	vncproxy_cfg *base = (vncproxy_cfg *) base_conf;
	vncproxy_cfg *virt = (vncproxy_cfg *) virt_conf;

	ap_log_perror(APLOG_MARK, APLOG_DEBUG, 0, p, "vncproxy_merge_server_config()");
	/*
	 * Our inheritance rules are our own, and part of our module's semantics.
	 * Basically, just note whence we came.
	 */
	cfg->cmode = (base->cmode == virt->cmode) ? base->cmode : CONFIG_MODE_COMBO;
	ap_log_perror(APLOG_MARK, APLOG_DEBUG, 0, p, "base->vncProxyHelper = %s", base->vncProxyHelper);
	ap_log_perror(APLOG_MARK, APLOG_DEBUG, 0, p, "virt->vncProxyHelper = %s", virt->vncProxyHelper);

	cfg->vncProxyDBDParams = (virt->vncProxyDBDParams == NULL) ? base->vncProxyDBDParams : virt->vncProxyDBDParams;
	cfg->vncProxyDBDriver = (virt->vncProxyDBDriver == NULL) ? base->vncProxyDBDriver : virt->vncProxyDBDriver;
	cfg->vncProxyHelperStderr = (virt->vncProxyHelperStderr == NULL) ? base->vncProxyHelperStderr : virt->vncProxyHelperStderr;
	cfg->vncProxyHelper = (virt->vncProxyHelper == NULL) ? base->vncProxyHelper : virt->vncProxyHelper;
	cfg->vncProxy = (virt->vncProxy == 0) ? base->vncProxy : virt->vncProxy;
	cfg->vncProxyShutdownSSLOnExec = (virt->vncProxyShutdownSSLOnExec == 0) ? base->vncProxyShutdownSSLOnExec : virt->vncProxyShutdownSSLOnExec;

	return (void *) cfg;
}

static int vncproxy_create_request(request_rec *r)
{
	vncproxy_cfg *cfg = NULL;

	if (!r->main && !r->prev) {
		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_create_request(pri): start");
		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_create_request(pri): ADD vncproxy_input_filter()");
		ap_add_input_filter("VNCPROXY_INPUT_FILTER", NULL, r, r->connection);
		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_create_request(pri): end [ok]");
	}
	else {
		ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, 0, "vncproxy_create_request(sub): start");
		ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, 0, "vncproxy_create_request(sub): end [ok]");
	}

	return OK;
}

/*
	For each request, checks if the first non empty line is a legacy vnc connect
	method. If it is we fix it to be HTTP/apache compliant. Otherwise we remove
	the filter from the filter list as soon as possible.

	This is currently implemented as a line filter and assume the data we require
	is all in a single brigade, if it gets called in a non AP_MODE_GETLINE mode it
	removes itself from the filter list for the remainder of the request.
 */

static apr_status_t vncproxy_input_filter(ap_filter_t *f, apr_bucket_brigade *bb,
										 ap_input_mode_t mode, apr_read_type_e block,
										 apr_off_t readbytes)
{
	apr_size_t read_count = 0;
	apr_bucket *e;
	apr_status_t rv;
	apr_size_t len = 0;
	request_rec *r = f->r;
	apr_bucket *patmatch_bucket, *nexte;
	apr_size_t patmatch_offset = 0;
	char *pos;
	vncproxy_cfg *cfg = NULL;

	const char *pattern = "CONNECT vncsession:";
	apr_size_t n;
	apr_bucket *sep_bucket = NULL;
	apr_off_t *sep_bucket_offset = 0;

	apr_bucket *port_bucket = NULL;
	const char *str = NULL;

	cfg = our_dconfig(r);

	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_input_filter(): start");

	rv = ap_get_brigade(f->next, bb, AP_MODE_GETLINE, APR_BLOCK_READ, 0);

	if (rv != APR_SUCCESS) {
		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_input_filter(): end [eof]");
		return rv;
	}

	if (!cfg || !cfg->vncProxy) {
		ap_remove_input_filter(f);

		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_input_filter(): end [APR_SUCCESS, vncProxy Off, REMOVE filter]");
		return APR_SUCCESS;
	}

	if (readbytes == 0) {
		if (f && f->c && f->c->base_server) {
			readbytes = f->c->base_server->limit_req_line +2;
		}
		else {
			readbytes = 8192;
		}
	}

	readbytes -= 5; // save room for :4900

	if (mode != AP_MODE_GETLINE) {
		ap_remove_input_filter(f);

		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_input_filter(): end [not in getline mode, REMOVE filter]");
		return APR_SUCCESS;
	}

	if (r == NULL) {
		ap_remove_input_filter(f);

		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_input_filter(): end [not in a request, REMOVE filter]");
		return APR_SUCCESS;
	}

	if (block != APR_BLOCK_READ) {
		ap_remove_input_filter(f);

		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_input_filter(): end [not in blocking mode, REMOVE filter]");
		return APR_SUCCESS;
	}

	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_input_filter(): reading line of input");

	/* insure ap_rgetline allocates memory each time thru the loop
	 * if there are empty lines
	 */

	if (APR_BRIGADE_EMPTY(bb)) {
		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_input_filter(): end [no input]");
		return APR_EGENERAL;
	}

	for (e = APR_BRIGADE_FIRST(bb); e != APR_BRIGADE_SENTINEL(bb); e = APR_BUCKET_NEXT(e)) {
		str = NULL;
		len = 0;
		n = 0;

		if (APR_BUCKET_IS_EOS(e)) {
			ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_input_filter(): end [EOS]");
			return APR_EGENERAL;
		}

		rv = apr_bucket_read(e, &str, &len, APR_BLOCK_READ);

		read_count += len;

		if (rv != APR_SUCCESS) {
			ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_input_filter(): end [read error]");
			return rv;
		}

		while (*pattern && len) {

			if (*pattern != *str) {
				if (read_count > 2) {
					ap_remove_input_filter(f);
					ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_input_filter(): end [no match, REMOVE filter]");
					return APR_SUCCESS;
				}

				ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_input_filter(): end [no match]");
				return APR_SUCCESS;
			}
			pattern++;
			str++;
			len--;
			n++;
		}

		if (*pattern) /* ran out of bucket data, get another bucket */
			continue;

		break;
	}

	if (e == APR_BRIGADE_SENTINEL(bb)) {
		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_input_filter(): end [unexpected end of input]");
		return APR_EGENERAL;
	}

	/* bucket where end of pattern match was found, and offset to exact point */
	patmatch_bucket = e;
	patmatch_offset = n - 1;

	/* split the bucket at match point */
	apr_bucket_split(patmatch_bucket, patmatch_offset);

	/* make a new bucket with a '.' in it */
	sep_bucket = apr_bucket_immortal_create(".", 1, bb->bucket_alloc);

	/* insert '.' after current bucket */
	APR_BUCKET_INSERT_AFTER(e, sep_bucket);

	/* get bucket after '.' bucket */
	e = APR_BUCKET_NEXT(sep_bucket);

	/* split bucket to get just first character */
	apr_bucket_split(e, 1);

	/* get next bucket, because we are about to forget current one */
	nexte = APR_BUCKET_NEXT(e);

	/* remove bucket containing ':' and destroy it */
	APR_BUCKET_REMOVE(e);
	apr_bucket_destroy(e);

	/* rewind to '.' bucket */
	e = sep_bucket;

	/* read buckets until we get to a space */
	for (; e != APR_BRIGADE_SENTINEL(bb); e = APR_BUCKET_NEXT(e)) {
		str = NULL;
		len = 0;
		n = 0;

		if (APR_BUCKET_IS_EOS(e)) {
			ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_input_filter(): end [EOS while fixing method]");
			return APR_EGENERAL;
		}

		rv = apr_bucket_read(e, &str, &len, APR_BLOCK_READ);

		if (rv != APR_SUCCESS) {
			ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_input_filter(): end [bucket read error while fixing method]");
			return rv;
		}

		while (len && *str != ' ') {
			len--;
			str++;
			n++;
		}

		if (*str == ' ')
			break;
	}

	if (e == APR_BRIGADE_SENTINEL(bb)) {
		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_input_filter(): end [unexpected end of data while fixing method]");
		return APR_EGENERAL;
	}

	/* spit bucket at space, insert port bucket */

	apr_bucket_split(e, n);
	port_bucket = apr_bucket_immortal_create(":4900", 5, bb->bucket_alloc);
	APR_BUCKET_INSERT_AFTER(e, port_bucket);

	ap_remove_input_filter(f);

	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_input_filter(): end [fixed method, REMOVE filter]");
	return APR_SUCCESS;
}

static int vncproxy_post_read_request(request_rec *r)
{
	vncproxy_cfg *cfg;

	cfg = our_dconfig(r);

	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_post_read_request(): start");

	if ((!cfg) || (!cfg->vncProxy)) {
	        ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_post_read_request(): end [declined, vncProxy Off]");
	        return DECLINED;
	}

	if (r->method_number == M_CONNECT) {
		if (strncmp(r->parsed_uri.hostname, "vncsession", 10) != 0) {
			ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_post_read_request(): end [declined, not vncsession CONNECT]");
			return DECLINED;
		}
	
		r->proxyreq = PROXYREQ_PROXY;
		r->uri = r->unparsed_uri;
		r->handler = "vncproxy-handler";
		r->hostname = apr_pstrdup( r->pool, r->server->server_hostname );
		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_post_read_request(): servername = %s", r->parsed_uri.hostname);

		if ((cfg && cfg->vncProxyShutdownSSLOnExec) ) {
				apr_table_setn(r->subprocess_env, "ssl-accurate-shutdown", "1");
				ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_post_read_request(): match, enable ssl-accurate-shutdown");
		}

		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_post_read_request(): end [declined, but CONNECT marked for vncproxy-handler]");
		return DECLINED;
	}
	else if (r->method_number == M_GET) {

		if (strncmp(r->the_request,"GET /rfb/token/",15) != 0) {
			ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_post_read_request(%s): end [declined, not rfb GET]",r->the_request);
			return DECLINED;
		}

		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_post_read_request(%s)",r->the_request);

		r->proxyreq = PROXYREQ_PROXY;
		r->uri = r->unparsed_uri;
		r->handler = "vncproxy-handler";

		if ((cfg && cfg->vncProxyShutdownSSLOnExec) ) {
				apr_table_setn(r->subprocess_env, "ssl-accurate-shutdown", "1");
				ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_post_read_request(): match, enable ssl-accurate-shutdown");
		}

		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_post_read_request(): end [declined, but GET marked for vncproxy-handler]");

		return DECLINED;
	}
	else
	{
        ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_post_read_request(): end [declined, not CONNECT or GET method]");
        return DECLINED;
	}
}

static int vncproxy_translate_name(request_rec *r)
{
	vncproxy_cfg *cfg;

	cfg = our_dconfig(r);

	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_translate_name(): start");

	if ((!cfg) || (!cfg->vncProxy)) {
	        ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_translate_name(): end [declined, vncProxy Off]");
	        return DECLINED;
	}

	if (r->proxyreq) { /* someone has already set up the proxy, it was possibly in vncproxy_post_read_request() */
		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_translate_name(): end [ok, vncproxy request detected]");
       	return OK;
    }

	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_translate_name(): end [declined, vncproxy request not detected]");
	return DECLINED;
}

static int vncproxy_handler(request_rec *r)
{
	vncproxy_cfg *cfg;
	apr_status_t rv;
    apr_sockaddr_t *uri_addr;
    apr_uri_t uri;
	char *token;

	cfg = our_dconfig(r);

	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_handler(): start");

	if (!cfg || !cfg->vncProxy) {
		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_handler(): end [declined, vncProxy Off]");
		return DECLINED;
	}

	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_handler(): [%s]", r->the_request);

	if (strcmp(r->handler, "vncproxy-handler") == 0) {

		if (r->method_number == M_CONNECT) {
			/* parse token out of CONNECT string, or GET url, pass to vncproxy_connect */

			if (APR_SUCCESS != apr_uri_parse_hostinfo(r->pool, r->unparsed_uri, &uri)) {
				ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_handler(): end [URI cannot be parsed %s]", r->unparsed_uri);
				return HTTP_BAD_REQUEST;
			}
	
			ap_log_rerror(APLOG_MARK, APLOG_INFO, 0, r, "vncproxy_handler(): connecting to %s:%d", uri.hostname, uri.port);

			if (strncmp(uri.hostname, "vncsession.", 11) != 0) {
				ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_handler(): end [declined]");
				return DECLINED;
			}

			token = uri.hostname + 11;
		}
		else if (r->method_number == M_GET) {
			char *request = apr_pstrdup(r->pool, r->the_request);
			token = strrchr(request,' ');
			if (token)
				*token = 0;
			token = strrchr(request,'/');
			if (token && *token)
				token++;
		}
		else
		{
			ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_handler(): end [declined, not CONNECT or GET]");
			return DECLINED;
		}

		if (*token == '\0' || token == NULL) {
			ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_handler(): end [connect, failed (no token)]");
		}

		rv = vncproxy_connect(r,token);

		if (rv == OK) {
			ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_handler(): end [connect, ok]");
		} else {
			ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_handler(): end [connect, failed (%d)]", rv);
		}

		return rv;
	}

	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_handler(): end [declined, not marked for vncproxy-handler]");
	return DECLINED;
}

static APR_OPTIONAL_FN_TYPE(ssl_is_https) *proxy_is_https = NULL;

static int vncproxy_connect(request_rec *r, char *token)
{
	apr_socket_t *sock;
	apr_bucket_brigade *bb = apr_brigade_create(r->pool, r->connection->bucket_alloc);
	apr_status_t rv;
	apr_size_t nbytes;
	char buffer[HUGE_STRING_LEN];
	apr_socket_t *client_socket = ap_get_module_config(r->connection->conn_config, &core_module);
	vncproxy_cfg *cfg = NULL;
	int status = 0;
	pid_t pid;
	const char *vncProxyHelper;
	char **argv_out;
	apr_bucket *eos, *e;
	int fd = -1;
	int maxfd = 0;

	proxy_is_https = APR_RETRIEVE_OPTIONAL_FN(ssl_is_https);

	cfg = our_dconfig(r);

	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_connect(): start");

	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_connect(): URL [%s]", r->unparsed_uri);

	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_connect(): handle connection");

	nbytes = apr_snprintf(buffer, sizeof (buffer), "HTTP/1.0 200" CRLF);
	ap_xlate_proto_to_ascii(buffer, nbytes);
	ap_fwrite(r->connection->output_filters, bb, buffer, nbytes);
	ap_log_rerror(APLOG_MARK, APLOG_INFO, 0, r, "vncproxy_connect(): HTTP/1.0 200 Connection Established");

	nbytes = apr_snprintf(buffer, sizeof (buffer), "Proxy-agent: %s" CRLF CRLF, vncproxy_banner);
	ap_xlate_proto_to_ascii(buffer, nbytes);
	ap_fwrite(r->connection->output_filters, bb, buffer, nbytes);

	ap_fflush(r->connection->output_filters, bb);

	if (cfg && cfg->vncProxyShutdownSSLOnExec) {
		/* Send end of stream to trigger SSL module to send close-notify alert */
		e = ap_bucket_eoc_create(r->connection->bucket_alloc);
		eos = APR_BRIGADE_LAST(bb);
		while ((APR_BRIGADE_SENTINEL(bb) != eos)
			&& !APR_BUCKET_IS_EOS(eos)) {
			eos = APR_BUCKET_PREV(eos);
		}
		if (eos == APR_BRIGADE_SENTINEL(bb)) {
			APR_BRIGADE_INSERT_TAIL(bb, e);
		}
		else {
			APR_BUCKET_INSERT_BEFORE(eos, e);
		}
		ap_pass_brigade(r->connection->output_filters, bb);
	}

#if (AP_SERVER_MAJORVERSION_NUMBER == 2 && AP_SERVER_MINORVERSION_NUMBER >= 4)
	vncProxyHelper = buildProxyHelperString(r->pool, cfg->vncProxyHelper, token, "", r->useragent_ip);
#else
	vncProxyHelper = buildProxyHelperString(r->pool, cfg->vncProxyHelper, token, "", r->connection->remote_ip);
#endif

	apr_tokenize_to_argv(vncProxyHelper, &argv_out, r->pool);

	if (chdir("/") != 0) {
		ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "vncproxy_connect(): end [unable to change directory to /]");
		return HTTP_INTERNAL_SERVER_ERROR;
	}

	pid = fork();

	if (pid == -1) {
		ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "vncproxy_connect(): end [backgrounding fork() failed]");
		return HTTP_INTERNAL_SERVER_ERROR;
	}

	if (pid == 0) {

		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_connect(): Background process [%d] created", getpid());

		rv = setsid();

		if (rv == -1) {
			ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "vncproxy_connect(): Can't create new process session");
			_exit(1);
		}

		pid = fork();

		if (pid == -1) {
			ap_log_rerror(APLOG_MARK, APLOG_WARNING, 0, r, "vncproxy_connect(): Failed to fork() prior to execution of vncProxyHelper [%s](%d)",
					vncProxyHelper, errno);
			_exit(2);
		}

		if (pid == 0) {
			ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_connect(): Disassociated process [%d] created.", getpid());

			if (cfg && cfg->vncProxyDBDriver)
				setenv("DBD_DRIVER", cfg->vncProxyDBDriver, 1);
			else
				setenv("DBD_DRIVER", "mysql", 1);

			if (cfg && cfg->vncProxyDBDParams)
				setenv("DBD_PARAMS", cfg->vncProxyDBDParams, 1);
			else
				setenv("DBD_PARAMS", "", 1);

			dup2(client_socket->socketdes, 0);
			dup2(client_socket->socketdes, 1);

			if (cfg->vncProxyHelperStderr) {

				fd = open(cfg->vncProxyHelperStderr, O_NOFOLLOW | O_APPEND | O_WRONLY | O_CREAT | O_NONBLOCK, S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP);

				if (fd < 0) 
				{
					ap_log_rerror(APLOG_MARK, APLOG_WARNING, 0, r, "Unable to open logfile [%s]", cfg->vncProxyHelperStderr);
				}
				else {
					dup2(fd, 2);
					close(fd);
				}
			}

			ap_log_rerror(APLOG_MARK, APLOG_INFO, 0, r, "Executing [%s]", vncProxyHelper);

			maxfd = sysconf(_SC_OPEN_MAX);

			for (fd = 3; fd < maxfd; fd++) {
				rv = close(fd);

				if (rv == -1) {
					if (errno == EBADF)
						continue;

					ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "vncproxy_connect(): Failed to close file descriptor [%d]", fd);
				} else if (rv == 0) {
					ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_connect(): Closed file descriptor [%d]", fd);
				}
			}

			execvp(argv_out[0], argv_out);

			ap_log_rerror(APLOG_MARK, APLOG_WARNING, 0, r, "vncproxy_connect(): Failed to execute vncProxyHelper [%s](%d)",
					vncProxyHelper, errno);

			_exit(3);
		}

		ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_connect(): Exiting background process [%d]", getpid());
		_exit(0); // let session leader parent die
	}

	waitpid(pid, &status, 0); // wait for intermediate background process to exit

	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_connect(): Apache parent worker process [%d] continuing", getpid());

	/* abort connection */
	r->input_filters = NULL; // Parent will shutdown SSL session if we don't clear filters and signal abort
	r->output_filters = NULL;
	r->connection->output_filters = NULL;
	r->proto_output_filters = NULL;
	r->connection->input_filters = NULL;
	r->input_filters = NULL;
	r->connection->aborted = 1;
	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "vncproxy_connect(): end [success]");
	return OK;
}

static void vncproxy_register_hooks(apr_pool_t *p)
{
	static const char * const aszSucc[] = {"mod_proxy.c", "mod_proxy_connect.c", "mod_rewrite.c", NULL};
	static const char * const modproxy[] = { "mod_proxy.c" , NULL };
	static const char * const modproxyrewrite[] = { "mod_proxy.c", "mod_rewrite.c" , NULL };

	ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, 0, "vncproxy_register_hooks()");

	ap_hook_create_request(vncproxy_create_request, NULL, aszSucc, APR_HOOK_FIRST);

	ap_register_input_filter("VNCPROXY_INPUT_FILTER", vncproxy_input_filter, NULL, AP_FTYPE_PROTOCOL);

	ap_hook_post_read_request(vncproxy_post_read_request, modproxy, NULL, APR_HOOK_FIRST);
	ap_hook_translate_name(vncproxy_translate_name, NULL, modproxyrewrite, APR_HOOK_FIRST); 
	ap_hook_handler(vncproxy_handler, NULL, aszSucc, APR_HOOK_FIRST);

}

module AP_MODULE_DECLARE_DATA vncproxy_module = {
	STANDARD20_MODULE_STUFF,
	vncproxy_create_dir_config, /* per-directory config creator */
	vncproxy_merge_dir_config, /* dir config merger */
	vncproxy_create_server_config, /* server config creator */
	vncproxy_merge_server_config, /* server config merger */
	vncproxy_cmds, /* command table */
	vncproxy_register_hooks, /* set up other request processing hooks */
};
