/**
 * @package     hubzero-vncproxy2
 * @file		vncproxy2.c
 * @author      Nicholas J. Kisseberth <nkissebe@purdue.edu>
 * @copyright   Copyright (c) 2010-2011 Purdue University. All rights reserved.
 * @license     http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
 *
 * Copyright (c) 2010-2011 Purdue University
 * All rights reserved.
 *
 * This file is part of: The HUBzero(R) Platform for Scientific Collaboration
 *
 * The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
 * software: you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * HUBzero is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * HUBzero is a registered trademark of Purdue University.
 */

/**
 * @TODO:
 *
 * - test ssl input sink
 * - improve efficiency of input ssl sink
 * - add timeout to input ssl sink
 * - replace socat with own tunneling code
 * - use sigaction() instead of old signal() and handle signal based shutdowns cleanly
 * - use SIGCHLD signals to monitor child termination
 * - group some processes together, not everything should be disassociated
 * - process_connection should timeout after some period of time
 * - process_connection should quit after some number of headers
 * - test logmsg for buffer overflow conditions
 * - maybe close file descriptors before exec() of socat to close dbd fd
 * - test how well child processes terminate to prevent zombies
 * - check proper use of return -1, exit() or _exit() after various fork()s
 * - run with valgrind again
 * - seperate main() for vncproxy2 and vncproxy2-helper
 * - if database parameters not set, read /etc/hubzero.conf and look
 *   in default document root for configuration.php and find parameters
 *   there
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <assert.h>
#include <time.h>
#include <netdb.h>
#include <apr_pools.h>
#include <apr_strings.h>
#include <apu.h>
#include <apr_dbd.h>
#include <getopt.h>
#include <syslog.h>
#include <libgen.h>
#include <signal.h>

volatile sig_atomic_t gSignal = 0;
volatile sig_atomic_t gAlarm = 0;

static char progname[PATH_MAX] = {0};
static int log_priority = LOG_INFO;
static int syslog_priority = LOG_WARNING;

#define MLOG_EMERG   __FILE__,__LINE__,LOG_EMERG
#define MLOG_ALERT   __FILE__,__LINE__,LOG_ALERT
#define MLOG_CRIT    __FILE__,__LINE__,LOG_CRIT
#define MLOG_ERR     __FILE__,__LINE__,LOG_ERR
#define MLOG_WARNING __FILE__,__LINE__,LOG_WARNING
#define MLOG_NOTICE  __FILE__,__LINE__,LOG_NOTICE
#define MLOG_INFO    __FILE__,__LINE__,LOG_INFO
#define MLOG_DEBUG   __FILE__,__LINE__,LOG_DEBUG

void
logmsg(char *file, int line, int priority, char *msg, ...)
{
	static char *priorities[8] = {"[emerg]", "[alert]", "[crit]", "[error]", "[warn]", "[notice]", "[info]",
		"[debug]"};
	va_list ap;
	char buffer[1500] = {0, 0};
	char pidstr[8];
	time_t t;
	struct tm tm_result;
	int len;

	if ((priority >= 0) && (priority < sizeof (priorities)) && (priority <= log_priority)) {

		if (time(&t) == -1) {
			t = 0;
		}

		buffer[0] = '[';

		if ((localtime_r(&t, &tm_result) == NULL) || (asctime_r(&tm_result, buffer + 1) == NULL)) {
			strcpy(buffer + 1, "Thu Jan 01 00:00:00 1970");
		}

		va_start(ap,msg);
		len = snprintf(buffer + 25, sizeof (buffer) - 25, "] %s %s(%d): [%d] ", priorities[priority], file, line, getpid());
        	vsnprintf(buffer + 25 + len, sizeof (buffer) - 25 - len, msg, ap);
		va_end(ap);

		strcat(buffer, "\n");
		fprintf(stderr, buffer);
		fflush(stderr);
	}

	if ((syslog_priority >= 0) && (syslog_priority < sizeof (priorities)) && (priority <= syslog_priority)) {
	
		va_start(ap, msg);
		vsyslog(priority, msg, ap);
		va_end(ap);
	}
}

void
do_proxy_vnc_sighandler(int sig)
{
	if (sig == SIGALRM)
		gAlarm = 1;
	else
		gSignal = sig;
}

int
open_database(const apr_dbd_driver_t *driver, apr_pool_t *pool, char *dbd_params, apr_dbd_t **handle)
{

	int delay = 1;
	int tries = 1;

	while (apr_dbd_open(driver, pool, dbd_params, handle) != APR_SUCCESS) {
		if (tries > 6) {
			logmsg(MLOG_WARNING, "Unable to open database connection. Tried %d times. Aborting", tries);
			return -1;
		}

		logmsg(MLOG_WARNING, "Unable to open database connection. Retrying in %ds", delay);

		sleep(delay);

		delay *= 2;

		if (delay > 8) {
			delay = 8;
		}

		tries++;
	}

	return 0;
}

int
sink_until_ssl_handshake()
{
	char buf;
	int rv;

	logmsg(MLOG_DEBUG, "Sinking input until possible SSL handshake arrives");

	while (1) {
		rv = read(STDIN_FILENO, &buf, 1);

		if (rv == 0) {
			return -1;
		}

		if (buf == 16) {
			write(STDOUT_FILENO, &buf, 1);
			logmsg(MLOG_DEBUG, "Possible start of SSL handshake found");
			return 0;
		}

		continue;
	}
}

int
process_vnc_connect_method(apr_pool_t *pool, int sink, char *remoteip, char *token, int portbase,
	const apr_dbd_driver_t *driver, char *dbd_params, int send_response)
{
	int delay = 1;
	int maxdelay = 3600;
	struct timeval tv;
	double starttime = 0.0, endtime = 0.0;
	char query[1024];
	int nrows;
	apr_dbd_results_t *res = NULL;
	apr_dbd_row_t *row = NULL;
	apr_status_t rv;
	int viewid = 0;
	pid_t pid = -1;
	int interval;
	char *c;
	int dispnum;
	int sessnum;
	int timeout;
	int id;
	apr_dbd_t *handle;
	int status;
	char hostname[41];
	char username[33];
	int port;
	int tries = 1;
	const char *etoken;
	const char *eremoteip;
	const char *value;

	if (open_database(driver, pool, dbd_params, &handle) != 0) {
		logmsg(MLOG_ERR, "Unable to open database to lookup token");
		return -1;
	}

	etoken = apr_dbd_escape(driver, pool, token, handle);

	if (etoken == NULL) {
		logmsg(MLOG_ERR, "Unable to build query safe token string");
		return -1;
	}

	eremoteip = apr_dbd_escape(driver, pool, remoteip, handle);

	if (eremoteip == NULL) {
		logmsg(MLOG_ERR, "Unable to build query safe remoteip string");
		return -1;
	}

	rv = snprintf(query, sizeof (query), "SELECT viewperm.sessnum,viewuser,display.hostname,session.dispnum,"
		" session.timeout,portbase"
		" FROM viewperm"
		" JOIN display ON viewperm.sessnum = display.sessnum"
		" JOIN session ON session.sessnum = display.sessnum"
		" JOIN host ON display.hostname = host.hostname"
		" WHERE viewperm.viewtoken='%s' LIMIT 1;", etoken);

	if (rv < 0 || rv >= sizeof (query)) {
		logmsg(MLOG_ERR, "Failed to build query string");
		return -1;
	}

	rv = apr_dbd_select(driver, pool, handle, &res, query, 0);

	logmsg(MLOG_DEBUG, query);

	row = NULL;

	if ((rv != APR_SUCCESS) || (apr_dbd_get_row(driver, pool, res, &row, -1) != 0)) {
		logmsg(MLOG_WARNING, "No database entry found for token [%s]", token);

		if (send_response) {
			printf("HTTP/1.1 404 Not Found\n\n");
			fflush(stdout);
			logmsg(MLOG_INFO, "Response: HTTP/1.1 404 Not Found");
		}

		rv = apr_dbd_close(driver, handle);

		if (rv != APR_SUCCESS) {
			logmsg(MLOG_ERR, "Unable to close database connection");
			return -1;
		}

		return 0;
	}

	value = apr_dbd_get_entry(driver, row, 0);

	if (value == NULL) {
		logmsg(MLOG_ERR, "Unable to read sessnum column from result row");
		return -1;
	}

	sessnum = atoi(value);

	if (sessnum <= 0) {
		logmsg(MLOG_ERR, "Invalid sessnum column read from result row");
		return -1;
	}

	value = apr_dbd_get_entry(driver, row, 1);

	if (value == NULL) {
		logmsg(MLOG_ERR, "Unable to read viewuser column from result row");
		return -1;
	}

	strncpy(username, value, sizeof (username));

	value = apr_dbd_get_entry(driver, row, 2);

	if (value == NULL) {
		logmsg(MLOG_ERR, "Unable to read hostname column from result row");
		return -1;
	}

	strncpy(hostname, value, sizeof (hostname));

	value = apr_dbd_get_entry(driver, row, 3);

	if (value == NULL) {
		logmsg(MLOG_ERR, "Unable to read dispnum column from result row");
		return -1;
	}

	dispnum = atoi(value);

	if (dispnum <= 0) {
		logmsg(MLOG_ERR, "Invalid dispnum column read from result row");
		return -1;
	}

	value = apr_dbd_get_entry(driver, row, 4);

	if (value == NULL) {
		logmsg(MLOG_ERR, "Unable to read timeout column from result row");
		return -1;
	}

	timeout = atoi(value);

	if (timeout <= 0) {
		logmsg(MLOG_ERR, "Invalid timeout column read from result row");
		return -1;
	}

	port = dispnum + portbase;

	logmsg(MLOG_NOTICE, "Map %s@%s for %s to %s:%d", username, remoteip, token, hostname, port);

	rv = apr_dbd_get_row(driver, pool, res, &row, -1); // clears connection

	if (rv != -1) {
		logmsg(MLOG_ERR, "Database transaction not finished when expected.");
		return -1;
	}

	if (send_response) {
		printf("HTTP/1.0 200 Connection Established\n");
		printf("Proxy-agent: HUBzero connection redirector\n\n");
		fflush(stdout);
		logmsg(MLOG_INFO, "Response: HTTP/1.0 200 Connection Established");
	}

	pid = fork();

	if (pid == -1) {
		logmsg(MLOG_ERR, "Unable to fork() for exec of socat");
		return -1;
	}

	if (pid == 0) {
		logmsg(MLOG_DEBUG, "Starting new process [%d] to exec() socat tunnel in", getpid());

		/* dbd may be open, but closing it with apr_dbd_close here corrupts parent */
		/* maybe should close file descriptors here */

		rv = snprintf(query, sizeof (query), "tcp4:%s:%d", hostname, port);

		if (rv < 0 || rv >= sizeof (query)) {
			logmsg(MLOG_ERR, "Failed to build argument string");
			exit(2);
		} else {
			logmsg(MLOG_DEBUG, "Executing: socat - %s.", query);
			closelog();
			execlp("socat", "socat", "-", query, NULL);
			logmsg(MLOG_ERR, "Failed to exec() socat");
		}

		exit(3);
	}

	if (sink) {
		rv = sink_until_ssl_handshake();

		if (rv == -1) {
			logmsg(MLOG_ERR, "Sinking until SSL handshake detected failed");
			return -1;
		}
	}

	if (freopen("/dev/null", "r", stdin) == NULL) {
		logmsg(MLOG_ERR, "Failed to redirect /dev/null to stdin");
		return -1;
	}

	if (freopen("/dev/null", "w", stdout) == NULL) {
		logmsg(MLOG_ERR, "Failed to redirect stdout to /dev/null");
		return -1;
	}

	rv = gettimeofday(&tv, NULL);

	if (rv == 0) {
		starttime = tv.tv_sec + (tv.tv_usec / 1000000.0);
	} else {
		starttime = 0;
	}

	rv = snprintf(query, sizeof (query), "UPDATE session SET accesstime=NOW() WHERE sessnum = '%d';", sessnum);

	if (rv < 0 || rv >= sizeof (query)) {
		logmsg(MLOG_ERR, "Failed to build query string");
		return -1;
	}

	rv = apr_dbd_query(driver, handle, &nrows, query);

	logmsg(MLOG_DEBUG, "%s", query);

	if (rv != APR_SUCCESS) {
		logmsg(MLOG_ERR, "Failed to update session accesstime");
		return -1;
	}

	rv = snprintf(query, sizeof (query), "INSERT INTO view(sessnum,username,remoteip,start,heartbeat) "
		" VALUE (%d,'%s','%s',now(),now());", sessnum, username, eremoteip);

	if (rv < 0 || rv >= sizeof (query)) {
		logmsg(MLOG_ERR, "Failed to build query string");
		return -1;
	}

	rv = apr_dbd_query(driver, handle, &nrows, query);

	logmsg(MLOG_DEBUG, "%s", query);

	if (rv != APR_SUCCESS) {
		logmsg(MLOG_ERR, "Failed to insert view record");
		return -1;
	}

	rv = snprintf(query, sizeof (query), "SELECT LAST_INSERT_ID();");

	if (rv < 0 || rv >= sizeof (query)) {
		logmsg(MLOG_ERR, "Failed to build query string");
		return -1;
	}

	rv = apr_dbd_select(driver, pool, handle, &res, query, 0);

	logmsg(MLOG_DEBUG, "%s", query);

	if (rv != APR_SUCCESS) {
		logmsg(MLOG_ERR, "Failed to query last insert id");
		return -1;
	}

	row = NULL;

	if (apr_dbd_get_row(driver, pool, res, &row, -1) != APR_SUCCESS) {
		logmsg(MLOG_ERR, "Fail to retrieve last insert id");
		return -1;
	}

	value = apr_dbd_get_entry(driver, row, 0);

	if (value == NULL) {
		logmsg(MLOG_ERR, "Unable to read insert id column from result row");
		return -1;
	}

	viewid = atoi(value);

	if (viewid <= 0) {
		logmsg(MLOG_ERR, "Invalid insert id column read from result row");
		return -1;
	}

	logmsg(MLOG_NOTICE, "View %d (%s@%s) started", viewid, username, remoteip);

	rv = apr_dbd_get_row(driver, pool, res, &row, -1); // clears connection

	if (rv != -1) {
		logmsg(MLOG_ERR, "Database transaction not finished when expected");
		return -1;
	}

	rv = apr_dbd_close(driver, handle);

	if (rv != APR_SUCCESS) {
		logmsg(MLOG_ERR, "Unable to close database connection");
		return -1;
	}

	interval = (int) (timeout * 0.40);

	if (signal(SIGALRM, do_proxy_vnc_sighandler) == SIG_ERR) {
		logmsg(MLOG_ERR, "Failed to assign signal handler to SIGALRM");
		return -1;
	}

	if (signal(SIGCHLD, do_proxy_vnc_sighandler) == SIG_ERR) {
		logmsg(MLOG_ERR, "Failed to assign signal handler to SIGCHLD");
		return -1;
	}

	if (signal(SIGHUP, do_proxy_vnc_sighandler) == SIG_ERR) {
		logmsg(MLOG_ERR, "Failed to assign signal handler to SIGHUP");
		return -1;
	}

	if (signal(SIGTERM, do_proxy_vnc_sighandler) == SIG_ERR) {
		logmsg(MLOG_ERR, "Failed to assign signal handler to SIGTERM");
		return -1;
	}

	if (signal(SIGINT, do_proxy_vnc_sighandler) == SIG_ERR) {
		logmsg(MLOG_ERR, "Failed to assign signal handler to SIGINT");
		return -1;
	}

	alarm(interval);

	while (1) {
		pause();

		if (gSignal) {
			while (waitpid(-1, &status, WNOHANG) > 0)
				; // reap child processes

			rv = gettimeofday(&tv, NULL);

			if (rv == 0) {
				endtime = tv.tv_sec + (tv.tv_usec / 1000000.0);
			} else {
				endtime = 0;
			}

			if (starttime == 0) {
				starttime = endtime;
			}

			if (endtime < starttime) {
				endtime = starttime;
			}

			logmsg(MLOG_NOTICE, "View %d (%s@%s) ended after %lf seconds",
				viewid, username, remoteip, endtime - starttime);

			if (open_database(driver, pool, dbd_params, &handle) != 0) {
				logmsg(MLOG_ERR, "Unable to open database to record end of view");
				return -1;
			}

			rv = snprintf(query, sizeof (query), "INSERT INTO viewlog(sessnum,username,remoteip,time,duration) "
				"SELECT sessnum, username, remoteip, start, %lf "
				"FROM view WHERE viewid='%d'", (double) (endtime - starttime), viewid);

			if (rv < 0 || rv >= sizeof (query)) {
				logmsg(MLOG_ERR, "Failed to build viewlog insert query string");
				return -1;
			}

			logmsg(MLOG_DEBUG, "%s", query);

			rv = apr_dbd_query(driver, handle, &nrows, query);

			if (rv != APR_SUCCESS) {
				logmsg(MLOG_ERR, "Failed to insert viewlog record");
				return -1;
			}

			snprintf(query, sizeof (query), "UPDATE session JOIN view ON session.sessnum=view.sessnum "
				"SET session.accesstime=now() "
				"WHERE viewid='%d'", viewid);

			if (rv < 0 || rv >= sizeof (query)) {
				logmsg(MLOG_ERR, "Failed to build session update accesstime query string");
				return -1;
			}

			logmsg(MLOG_DEBUG, "%s", query);

			rv = apr_dbd_query(driver, handle, &nrows, query);

			if (rv != APR_SUCCESS) {
				logmsg(MLOG_ERR, "Failed to update session accesstime record");
				return -1;
			}

			rv = snprintf(query, sizeof (query), "DELETE FROM view WHERE viewid='%d'", viewid);

			if (rv < 0 || rv >= sizeof (query)) {
				logmsg(MLOG_ERR, "Failed to build view deletion query string");
				return -1;
			}

			logmsg(MLOG_DEBUG, "%s", query);

			rv = apr_dbd_query(driver, handle, &nrows, query);

			if (rv != APR_SUCCESS) {
				logmsg(MLOG_ERR, "Failed to delete view record");
				return -1;
			}

			rv = apr_dbd_close(driver, handle);

			if (rv != APR_SUCCESS) {
				logmsg(MLOG_ERR, "Unable to close database connection");
				return -1;
			}

			break;
		}

		if (gAlarm == 1) {
			logmsg(MLOG_INFO, "Refreshing heartbeat for view %d", viewid);

			rv = snprintf(query, sizeof (query), "UPDATE view SET heartbeat=NOW() WHERE viewid='%d' LIMIT 1;", viewid);

			if (rv < 0 || rv >= sizeof (query)) {
				logmsg(MLOG_ERR, "Failed to build view heartbeat update query string");
				return -1;
			}

			if (open_database(driver, pool, dbd_params, &handle) != 0) {
				logmsg(MLOG_ERR, "Unable to open database to update heartbeat of view [%d]", viewid);
				return -1;
			}

			logmsg(MLOG_DEBUG, "%s", query);

			rv = apr_dbd_query(driver, handle, &nrows, query);

			if (rv != APR_SUCCESS) {
				logmsg(MLOG_ERR, "Failed to update view heartbeat");
				return -1;
			}

			apr_dbd_close(driver, handle);

			if (rv != APR_SUCCESS) {
				logmsg(MLOG_ERR, "Unable to close database connection after heartbeat update");
				return -1;
			}

			interval = (int) (timeout * 0.40);

			gAlarm = 0;

			alarm(interval);
		}
	}

	return 0;
}

int
process_connection(apr_pool_t *pool, int sink, char *remoteip, int portbase, const apr_dbd_driver_t *driver,
	char *dbd_params)
{
	char *result = NULL;
	char *connect_uri;
	char *token = NULL;
	char *c;
	int rv;
	int state = 0; // 0 = skipping blank lines, 1 = getting request, 2 = getting headers
	int portnum;
	char line[8191];
	char request[8191];

	request[0] = 0;

	while (1) {
		result = fgets(line, 8191, stdin);

		if ((result == NULL) && (feof(stdin))) {
			logmsg(MLOG_NOTICE, "EOF reached");
			break;
		}

		if (result == NULL) {
			logmsg(MLOG_NOTICE, "Error reading input.");
			break;
		}

		if (state == 0) // skipping blank lines
		{
			if ((line[0] == '\r' && line[1] == '\n') || (line[0] == '\n')) {
				logmsg(MLOG_NOTICE, "Request: []");
				continue;
			}

			state = 1;
			// fallthrough
		}

		if (state == 1) // getting request
		{
			int length = strlen(line);

			if (length > 1) {
				if (line[length - 2] == '\r' && line[length - 1] == '\n')
					line[length - 2] = 0;
			} else if (length > 0) {
				if (line[length - 1] == '\n')
					line[length - 1] = 0;
			}

			strncpy(request, line, 8191);
			logmsg(MLOG_NOTICE, "Request: [%s]", request);

			state = 2;
			continue;
		}

		if (state == 2) // getting headers
		{
			int length = strlen(line);

			if (length > 1) {
				if (line[length - 2] == '\r' && line[length - 1] == '\n')
					line[length - 2] = 0;
			} else if (length > 0) {
				if (line[length - 1] == '\n')
					line[length - 1] = 0;
			}

			logmsg(MLOG_INFO, "Header: [%s]", line);

			// if ((line[0] == '\r' && line[1] == '\n' ) || (line[0] == '\n'))
			if (line[0] == 0)
				break;
		}
	}

	if (strncmp(request, "GET / ", 6) == 0) {
		logmsg(MLOG_NOTICE, "Response: HTTP/1.1 200 OK");

		printf("HTTP/1.1 200 OK\n");
		printf("Content-Type: text/html\n\n");
		printf("<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Strict//EN\" "
			"\"http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd\">\n"
			"<html xmlns=\"http://www.w3.org/1999/xhtml\" lang=\"en\" xml:lang=\"en\">\n"
			"\t<head>\n"
			"\t\t<title></title>\n"
			"\t</head>\n"
			"\t<body>\n"
			"\t\tConnection to proxy succeeded.\n"
			"\t</body>\n"
			"</html>\n");
		printf("\n\n");
	} else if (strncmp(request, "GET ", 4) == 0) {
		printf("HTTP/1.1 404 Not Found\n\n");
		logmsg(MLOG_NOTICE, "Response: HTTP/1.1 404 Not Found");
	} else if (strncmp(request, "CONNECT ", 8) != 0) {
		printf("HTTP/1.1 501 Not Implemented\n\n");
		logmsg(MLOG_NOTICE, "Response: HTTP/1.1 501 Not Implemented");
	} else if (strncmp(request, "CONNECT vncsession", 18) != 0) {
		printf("HTTP/1.1 404 Not Found\n\n");
		logmsg(MLOG_NOTICE, "Response: HTTP/1.1 404 Not Found");
	} else if (request[18] != '.' && request[18] != ':') {
		printf("HTTP/1.1 404 Not Found\n\n");
		logmsg(MLOG_NOTICE, "Response: HTTP/1.1 404 Not Found");
	} else {
		connect_uri = request + 19;
		c = connect_uri;

		while (*c) {
			if ((*c == ' ') || (*c == '\r') || (*c == '\n') || (*c == ':'))
				break;

			c++;
		}

		if (*c) {
			*c++ = 0;
			portnum = atoi(c);
		}

		token = connect_uri;
	}

	fflush(stdout);

	if (token != NULL) {
		return process_vnc_connect_method(pool, sink, remoteip, token, portbase, driver, dbd_params, 1);
	}

	return 0;
}

void
server_sighandler(int sig)
{
	size_t len = 0;
	char *msg, *c;

	if (sig == SIGHUP)
		msg = "Caught signal SIGHUP. Exiting.\n";
	else if (sig == SIGINT)
		msg = "Caught signal SIGINT. Exiting.\n";
	else if (sig == SIGQUIT)
		msg = "Caught signal SIGQUIT. Exiting.\n";
	else if (sig == SIGTERM)
		msg = "Caught signal SIGTERM. Exiting.\n";
	else if (sig == SIGSEGV)
		msg = "Caught signal SIGSEGV. Exiting.\n";
	else
		msg = "Caught unexpected signal. Exiting.\n";

	for (c = msg; *c; c++)
		len++;

	write(2, msg, len);
	_exit(1);
}

int
server(apr_pool_t *pool, int sink, int background, int disassociate, int listenp, int portbase, char *pidfile,
	const apr_dbd_driver_t *driver, char *dbd_params)
{
	int l;
	int s;
	int on = 1;
	struct sockaddr_in saddr;
	struct sockaddr_in caddr;
	socklen_t caddr_len = sizeof (caddr);
	socklen_t saddr_len = sizeof (saddr);
	int status;
	char host[NI_MAXHOST];
	char service[NI_MAXSERV];
	pid_t pid;
	int rv;

	if (freopen("/dev/null", "r", stdin) == NULL) {
		logmsg(MLOG_ERR, "Failed to redirect /dev/null to stdin");
		return -1;
	}

	if (signal(SIGHUP, SIG_IGN) == SIG_ERR) {
		logmsg(MLOG_ERR, "Failed to assign signal handler to SIGHUP");
		return -1;
	}

	if (signal(SIGINT, server_sighandler) == SIG_ERR) {
		logmsg(MLOG_ERR, "Failed to assign signal handler to SIGINT");
		return -1;
	}

	if (signal(SIGTERM, server_sighandler) == SIG_ERR) {
		logmsg(MLOG_ERR, "Failed to assign signal handler to SIGTERM");
		return -1;
	}

	if (signal(SIGSEGV, server_sighandler) == SIG_ERR) {
		logmsg(MLOG_ERR, "Failed to assign signal handler to SIGSEGV");
		return -1;
	}

	l = socket(AF_INET, SOCK_STREAM, 0);

	if (l == -1) {
		logmsg(MLOG_ERR, "Failed to create server socket file descriptor");
		return -1;
	}

	rv = setsockopt(l, SOL_SOCKET, SO_REUSEADDR, &on, sizeof (on));

	if (rv != 0) {
		logmsg(MLOG_ERR, "Unable to set SO_REUSEADDR on server socket");
		return -1;
	}

	bzero(&saddr, sizeof (saddr));
	saddr.sin_family = AF_INET;
	saddr.sin_addr.s_addr = htonl(INADDR_ANY);
	saddr.sin_port = htons(listenp);

	if (bind(l, (struct sockaddr *) &saddr, saddr_len) < 0) {
		logmsg(MLOG_ERR, "Unable to start server, bind() failed");
		return -1;
	}

	if (listen(l, 10) < 0) {
		logmsg(MLOG_ERR, "Unable to start server, listen() failed");
		return -1;
	}

	if (pidfile != NULL) {
		FILE *f;

		f = fopen(pidfile, "w");

		if (f == NULL) {
			logmsg(MLOG_ERR, "Unable to open pidfile [%s]", pidfile);
			return -1;
		} else {
			if (!fprintf(f, "%d\n", getpid())) {
				logmsg(MLOG_ERR, "Unable to write pid [%d] to [%s]", getpid(), pidfile);
				return -1;
			}

			rv = fclose(f);

			if (rv == -1) {
				logmsg(MLOG_ERR, "Unable to close pidfile [%s]", pidfile);
				return -1;
			}
		}

		logmsg(MLOG_DEBUG, "Wrote pid [%d] to [%s]", getpid(), pidfile);
	}

	logmsg(MLOG_NOTICE, "Server is ready");

	while (1) {
		logmsg(MLOG_DEBUG, "Server is waiting for a connection");

		s = accept(l, (struct sockaddr *) &caddr, &caddr_len);

		if (s < 0) {
			if (errno == EINTR) {
				logmsg(MLOG_DEBUG, "Accept loop interrupted");
				continue;
			}

			logmsg(MLOG_ERR, "Unable to continue server, accept() failed [%d]", errno);
			break;
		}

		logmsg(MLOG_DEBUG, "Got connection");

		if (background || disassociate) {
			pid = fork();

			if (pid == -1) {
				logmsg(MLOG_ERR, "Client handler backgrounding fork() failed");
				return -1;
			}

			if (pid) {
				rv = close(s);

				if (rv == -1) {
					logmsg(MLOG_ERR, "Failed to close serve after handoff");
					return -1;
				}

				if (disassociate) {
					logmsg(MLOG_DEBUG, "waiting for [%d] to exit", pid);
					waitpid(pid,&status,0); // wait for intermediate background process to exit if disassociating
				}

				continue;
			}

			logmsg(MLOG_DEBUG, "Client handler background process [%d] created", getpid());
		}

		if (disassociate && pid == 0) {

			rv = setsid();

			if (rv == -1) {
				logmsg(MLOG_ERR, "Can'tcreate new client handling process session");
				_exit(1);
			}

			pid = fork();

			if (pid == -1) {
				logmsg(MLOG_ERR, "Disassociating client handler fork() failed");
				return -1;
			}

			if (pid) {
				logmsg(MLOG_DEBUG, "Parent client handler process [%d] exiting [disassociating]", getpid());
				exit(0);
			}

			logmsg(MLOG_DEBUG, "Client handler disassociated process [%d] created", getpid());
		}

		if (background) {
			rv = close(l);

			if (rv == -1) {
				logmsg(MLOG_ERR, "Failed to close server socket in client handler");
				_exit(1);
			}
		}

		rv = getnameinfo((struct sockaddr *) &caddr, caddr_len, host, NI_MAXHOST, service, NI_MAXSERV, NI_NUMERICHOST);

		if (rv != 0) {
			logmsg(MLOG_ERR, "Unable to resolve client address info");
			_exit(1);
		}

		logmsg(MLOG_NOTICE, "Accepted connection from %s", host);

		rv = dup2(s, 0);

		if (rv == -1) {
			logmsg(MLOG_ERR, "Unable to dup() client socket to stdin");
			_exit(1);
		}

		rv = dup2(s, 1);

		if (rv == -1) {
			logmsg(MLOG_ERR, "Unable to dup() client socket to stdout");
			_exit(1);
		}

		rv = process_connection(pool, sink, host, portbase, driver, dbd_params);

		if (background) {
			return rv;
		}

		if (rv == 0) {
			logmsg(MLOG_DEBUG, "Successfully processed connection");
		} else {
			logmsg(MLOG_DEBUG, "Error while processing connection");
		}

		if (close(s) != 0) {
			logmsg(MLOG_ERR, "Failed close client socket [%d]", errno);
			break;
		}

		if (freopen("/dev/null", "r", stdin) == NULL) {
			logmsg(MLOG_ERR, "Failed to redirect /dev/null to stdin");
			return -1;
		}

		if (freopen("/dev/null", "w", stdout) == NULL) {
			logmsg(MLOG_ERR, "Failed to redirect stdout to /dev/null");
			return -1;
		}
	}

	rv = close(l);

	if (rv == -1) {
		logmsg(MLOG_ERR, "Failed to close listening socket");
	}

	return -1;
}

int
main(int argc, const char * const argv[])
{
	int listenp = -1; /* --port=# 		*/
	char *logfile = NULL; /* --logfile=file 	*/
	int portbase = -1; /* --portbase=# 	*/
	char *pidfile = NULL; /* --pidfile=file 	*/
	char *token = NULL; /* --token=string 	*/
	char *remoteip = NULL; /* --remoteip=string 	*/
	char *dbd_driver = NULL; /* DBD_DRIVER		*/
	char *dbd_params = NULL; /* DBD_PARAMS		*/
	int background = 0; /* -b                   */
	int disassociate = 0; /* -d                   */
	int sink = -1; /* -s                   */

	apr_status_t rv;
	const apr_dbd_driver_t *driver;
	apr_pool_t *pool = NULL;
	long maxfd = 0;
	pid_t pid;
	int status;
	int fd;
	int c;

	strncpy(progname, basename((char *) argv[0]), sizeof (progname));

	opterr = 0;

	while (1) {
		static struct option long_options[] = {
			{ "portbase", required_argument, 0, 'P'},
			{ "port", required_argument, 0, 'p'},
			{ "logfile", required_argument, 0, 'l'},
			{ "pidfile", required_argument, 0, 'i'},
			{ "token", required_argument, 0, 't'},
			{ "remoteip", required_argument, 0, 'r'},
			{ "loglevel", required_argument, 0, 'g'},
			{ "sysloglevel", required_argument, 0, 's' },
			{ 0, 0, 0, 0}
		};

		int option_index = 0;

		c = getopt_long(argc, (char **) argv, "wbzdt:l:", long_options, &option_index);

		if (c == -1)
			break;

		switch (c) {
		case 0:
			if (long_options[option_index].flag != 0)
				break;
			printf("option %s", long_options[option_index].name);
			if (optarg)
				printf(" with arg %s", optarg);
			printf("\n");
			break;

		case 'b':
			background = 1;
			break;

		case 'd':
			disassociate = 1;
			break;

		case 'p':
			listenp = atoi(optarg);
			break;

		case 'P':
			portbase = atoi(optarg);
			break;

		case 'l':
			logfile = optarg;
			break;

		case 'i':
			pidfile = optarg;
			break;

		case 't':
			token = optarg;
			break;

		case 'r':
			remoteip = optarg;
			break;

		case 'g':
			if (strcasecmp(optarg,"none") == 0)
				log_priority = -1;
			else {
				log_priority = atoi(optarg);
				if ((log_priority < 0) || (log_priority > LOG_DEBUG))
					log_priority = LOG_NOTICE;
			}
			break;

		case 's':
			if (strcasecmp(optarg,"none") == 0)
				syslog_priority = -1;
			else {
				syslog_priority = atoi(optarg);
				if ((syslog_priority < 0) || (syslog_priority > LOG_DEBUG))
					syslog_priority = LOG_WARNING;
			}
			break;

		case 'z':
			sink = 1;
			break;

		case '?':
			break;

		default:
			break;
		}
	}

	/* Print any remaining command line arguments (not options). */
	if (optind < argc) {
		printf("non-option ARGV-elements: ");
		while (optind < argc)
			printf("%s ", argv[optind++]);
		putchar('\n');
	}

	if (log_priority >= 0) {
		char *source = "";

		if (logfile == NULL) {
			if ((background == 1) || (disassociate == 1)) {
				logfile = "/var/log/hubzero/vncproxy2.log";
				source = " (default)";
			}
		}

		if (logfile != NULL) {
			if (freopen(logfile, "a", stderr) == NULL) {
				logmsg(MLOG_ERR, "Unable to open logfile [%s]", logfile);
				logmsg(MLOG_DEBUG, "Logging to [stderr] (failover)");
			}
		}

		if (logfile == NULL) {
			logmsg(MLOG_DEBUG, "Logging to [stderr] (default)");
		} 
		else {
			logmsg(MLOG_DEBUG, "Logging to [%s]%s", logfile, source);
		}
	}
	else {
		logmsg(MLOG_DEBUG, "Logging to [/dev/null] (implied)");
		freopen("/dev/null","a",stderr);
	}

	if (syslog_priority >= 0) {
		openlog(progname, LOG_PID, (token == NULL) ? LOG_DAEMON : LOG_USER);
	}

	logmsg(MLOG_DEBUG, "Started vncproxy2 process [%d]", getpid());

	rv = apr_app_initialize(&argc, &argv, NULL);

	if (rv != APR_SUCCESS) {
		logmsg(MLOG_ERR, "Unable to initialize application");
		exit(1);
	}

	rv = atexit(apr_terminate);

	if (rv != 0) {
		logmsg(MLOG_ERR, "Failed to register apr_terminate() with atexit()");
		apr_terminate();
		exit(2);
	}

	rv = apr_pool_create(&pool, NULL);

	if (rv != APR_SUCCESS) {
		logmsg(MLOG_ERR, "Unable to create memory pool");
		exit(2);
	}

	rv = apr_dbd_init(pool);

	if (rv != APR_SUCCESS) {
		logmsg(MLOG_ERR, "Unable to initialize APR DBD environment");
		exit(3);
	}

	maxfd = sysconf(_SC_OPEN_MAX);

	closelog();

	for (fd = 3; fd < maxfd; fd++) {
		rv = close(fd);

		if (rv == -1) {
			if (errno == EBADF)
				continue;

			logmsg(MLOG_ERR, "Failed to close file descriptor [%d]", fd);
			exit(8);
		} else if (rv == 0) {
			logmsg(MLOG_DEBUG, "Closed file descriptor [%d]", fd);
		}
	}

	if (logfile == NULL) {
		logmsg(MLOG_DEBUG, "Will log to stderr (default)");
	} else {
		logmsg(MLOG_DEBUG, "Will log to [%s]", logfile);
	}

	if (disassociate == 0) {
		logmsg(MLOG_DEBUG, "Will not disassociate from parent process (default)");
	} else {
		logmsg(MLOG_DEBUG, "Will disassociate from parent process");
	}

	if (background == 0) {
		background = disassociate;
		if (disassociate)
			logmsg(MLOG_DEBUG, "Will run in background (implied)");
		else
			logmsg(MLOG_DEBUG, "Will run in foreground (default)");
	} else {
		logmsg(MLOG_DEBUG, "Will run in %s", (!background) ? "foreground" : "background");
	}

	if (sink == -1) {
		sink = 0;
		logmsg(MLOG_DEBUG, "Will not sink input waiting for ssl handshake (default)");
	} else {
		logmsg(MLOG_DEBUG, "Will sink input waitiing for ssl handshake");
	}

	if (log_priority >=0) {
		logmsg(MLOG_DEBUG, "Log level is [%d]", log_priority);
	} else {
		logmsg(MLOG_DEBUG, "Log level is [None]");
	}

	if (syslog_priority >=0) {
		logmsg(MLOG_DEBUG, "Syslog level is [%d]", syslog_priority);
	} else {
		logmsg(MLOG_DEBUG, "Syslog level is [None]");
	}

	if (portbase == -1) {
		portbase = 4000;
		logmsg(MLOG_DEBUG, "Will use portbase of [4000] (default)");
	} else {
		logmsg(MLOG_DEBUG, "Will use portbase of [%d]", portbase);
	}

	if (token == NULL) {
		logmsg(MLOG_DEBUG, "Will run as server (default)");

		if (listenp == -1) {
			listenp = 8080;
			logmsg(MLOG_DEBUG, "Will listen on port [8080] (default)");
		} else {
			logmsg(MLOG_DEBUG, "Will listen on port [%d]", listenp);
		}

		if (pidfile == NULL) {
			pidfile = "/var/run/vncproxy2.pid";
			logmsg(MLOG_DEBUG, "Will record PID in [%s] (default)", pidfile);
		} else {
			logmsg(MLOG_DEBUG, "Will record PID in [%s]", pidfile);
		}
	} else { // Running as helper uses remoteip, token, portbase, dbd_driver, dbd_params

		if (remoteip == NULL) {
			remoteip = "0.0.0.0";
			logmsg(MLOG_DEBUG, "Will record remote ip as [%s] (default)", remoteip);
		} else {
			logmsg(MLOG_DEBUG, "Will record remote ip as [%s]", remoteip);
		}

		logmsg(MLOG_DEBUG, "Will lookup token [%s]", token);
	}

	dbd_driver = getenv("DBD_DRIVER");

	if ((dbd_driver == NULL) || (*dbd_driver == 0)) {
		logmsg(MLOG_DEBUG, "Will use database driver [mysql] (default)");
		dbd_driver = "mysql";
	} else {
		logmsg(MLOG_DEBUG, "Will use database driver [%s]", dbd_driver);
	}

	dbd_driver = apr_pstrdup(pool, dbd_driver);

	if (dbd_driver == NULL) {
		logmsg(MLOG_ERR, "Unable to allocate memory for DBD_DRIVER variable");
		exit(4);
	}

	dbd_params = getenv("DBD_PARAMS");

	if ((dbd_params == NULL) || (*dbd_params == 0)) {
		logmsg(MLOG_DEBUG, "Will use default database parameters (default)");
		dbd_params = "";
	} else {
		logmsg(MLOG_DEBUG, "Will use database parameters [*redacted*]");
	}

	dbd_params = apr_pstrdup(pool, dbd_params);

	if (dbd_params == NULL) {
		logmsg(MLOG_ERR, "Unable to allocate memory for DBD_PARAMS variable");
		exit(5);
	}

	rv = apr_dbd_get_driver(pool, dbd_driver, &driver);

	if (rv != APR_SUCCESS) {
		logmsg(MLOG_ERR, "Unable to load APR_DBD driver [%s]", dbd_driver);
		exit(6);
	}

	if (chdir("/") != 0) {
		logmsg(MLOG_ERR, "Unable to change directory to [/]");
		exit(7);
	}

	umask(0);

	if (background || disassociate) {
		logmsg(MLOG_NOTICE, "Backgrounding process.");

		pid = fork();

		if (pid == -1) {
			logmsg(MLOG_ERR, "Backgrounding fork() failed");
			exit(8);
		}

		if (pid) {
			logmsg(MLOG_DEBUG, "Parent process [%d] exiting [backgrounding]", getpid());
			_exit(0);
		}

		logmsg(MLOG_DEBUG, "Background process [%d] created", getpid());
	}

	if (disassociate) {
		rv = setsid();

		if (rv == -1) {
			logmsg(MLOG_ERR, "Can't create new process session");
			exit(8);
		}

		pid = fork();

		if (pid == -1) {
			logmsg(MLOG_ERR, "Disassociating fork() failed");
			exit(8);
		}

		if (pid) {
			logmsg(MLOG_DEBUG, "Parent process [%d] exiting [disassociating]", getpid());
			_exit(0);
		}

		logmsg(MLOG_DEBUG, "Disassociated process [%d] created", getpid());
	}

	if (token == NULL) {
		logmsg(MLOG_INFO, "Starting vncproxy2 server mode");

		rv = server(pool, sink, background, disassociate, listenp, portbase, pidfile, driver, dbd_params);

		if (rv != 0) {
			logmsg(MLOG_ERR, "The vncproxy2 server quit due to an error");
			exit(9);
		}

		logmsg(MLOG_DEBUG, "The vncproxy2 server quit normally");
	} else {
		logmsg(MLOG_INFO, "Starting vncproxy2 helper mode");

		rv = process_vnc_connect_method(pool, sink, remoteip, token, portbase, driver, dbd_params, 0);

		if (rv != 0) {
			logmsg(MLOG_ERR, "The vncproxy2 helper quit due to an error");
			exit(10);
		}

		logmsg(MLOG_DEBUG, "The vncproxy2 helper quit normally");
	}

	exit(0);
}
