/**
 * @package     hubzero-vncproxy
 * @file        connect.c
 * @author      Nicholas J. Kisseberth <nkissebe@purdue.edu>
 * @copyright   Copyright (c) 2010-2013 HUBzero Foundation, LLC.
 * @license     http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
 *
 * Copyright (c) 2010-2013 HUBzero Foundation, LLC.
 *
 * This file is part of: The HUBzero(R) Platform for Scientific Collaboration
 *
 * The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
 * software: you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * HUBzero is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * HUBzero is a registered trademark of HUBzero Foundation, LLC.
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <assert.h>
#include <time.h>
#include <netdb.h>
#include <apr_pools.h>
#include <apr_strings.h>
#include <apu.h>
#include <apr_dbd.h>
#include <getopt.h>
#include <syslog.h>
#include <libgen.h>
#include <signal.h>
#include <pwd.h>
#include <grp.h>
#include <sys/types.h>

#include "log.h"

volatile sig_atomic_t gAlarm = 0;
volatile sig_atomic_t gSignal = 0;

void
do_proxy_vnc_sighandler(int sig)
{
	if (sig == SIGALRM)
		gAlarm = 1;
	else
		gSignal = sig;
}

int
open_database(const apr_dbd_driver_t *driver, apr_pool_t *pool, char *dbd_params, apr_dbd_t **handle)
{

	int delay = 1;
	int tries = 1;

	while (apr_dbd_open(driver, pool, dbd_params, handle) != APR_SUCCESS) {
		if (tries > 6) {
			logmsg(MLOG_WARNING, "Unable to open database connection. Tried %d times. Aborting", tries);
			return -1;
		}

		logmsg(MLOG_WARNING, "Unable to open database connection. Retrying in %ds", delay);

		sleep(delay);

		delay *= 2;

		if (delay > 8) {
			delay = 8;
		}

		tries++;
	}

	return 0;
}

int
sink_until_ssl_handshake()
{
	char buf;
	int rv;

	logmsg(MLOG_DEBUG, "Sinking input until possible SSL handshake arrives");

	while (1) {
		rv = read(STDIN_FILENO, &buf, 1);

		if (rv == 0) {
			return -1;
		}

		if (buf == 16) {
			write(STDOUT_FILENO, &buf, 1);
			logmsg(MLOG_DEBUG, "Possible start of SSL handshake found");
			return 0;
		}

		continue;
	}
}

int
process_vnc_connect_method(apr_pool_t *pool, int sink, char *remoteip, char *token, int portbase,
	const apr_dbd_driver_t *driver, char *dbd_params, int send_response)
{
	int delay = 1;
	int maxdelay = 3600;
	struct timeval tv;
	double starttime = 0.0, endtime = 0.0;
	char query[1024];
	int nrows;
	apr_dbd_results_t *res = NULL;
	apr_dbd_row_t *row = NULL;
	apr_status_t rv;
	int viewid = 0;
	pid_t pid = -1;
	int interval;
	char *c;
	int dispnum;
	int sessnum;
	int timeout;
	int id;
	apr_dbd_t *handle;
	int status;
	char hostname[41];
	char username[33];
	int port;
	int tries = 1;
	const char *etoken;
	const char *eremoteip;
	const char *value;

	if (open_database(driver, pool, dbd_params, &handle) != 0) {
		logmsg(MLOG_ERR, "Unable to open database to lookup token");
		return -1;
	}

	etoken = apr_dbd_escape(driver, pool, token, handle);

	if (etoken == NULL) {
		logmsg(MLOG_ERR, "Unable to build query safe token string");
		return -1;
	}

	eremoteip = apr_dbd_escape(driver, pool, remoteip, handle);

	if (eremoteip == NULL) {
		logmsg(MLOG_ERR, "Unable to build query safe remoteip string");
		return -1;
	}

	rv = snprintf(query, sizeof (query), "SELECT viewperm.sessnum,viewuser,display.hostname,session.dispnum,"
		" session.timeout,portbase"
		" FROM viewperm"
		" JOIN display ON viewperm.sessnum = display.sessnum"
		" JOIN session ON session.sessnum = display.sessnum"
		" JOIN host ON display.hostname = host.hostname"
		" WHERE viewperm.viewtoken='%s' LIMIT 1;", etoken);

	if (rv < 0 || rv >= sizeof (query)) {
		logmsg(MLOG_ERR, "Failed to build query string");
		return -1;
	}

	rv = apr_dbd_select(driver, pool, handle, &res, query, 0);

	logmsg(MLOG_DEBUG, query);

	row = NULL;

	if ((rv != APR_SUCCESS) || (apr_dbd_get_row(driver, pool, res, &row, -1) != 0)) {
		logmsg(MLOG_WARNING, "No database entry found for token [%s]", token);

		if (send_response) {
			printf("HTTP/1.1 404 Not Found\n\n");
			fflush(stdout);
			logmsg(MLOG_INFO, "Response: HTTP/1.1 404 Not Found");
		}

		rv = apr_dbd_close(driver, handle);

		if (rv != APR_SUCCESS) {
			logmsg(MLOG_ERR, "Unable to close database connection");
			return -1;
		}

		return 0;
	}

	value = apr_dbd_get_entry(driver, row, 0);

	if (value == NULL) {
		logmsg(MLOG_ERR, "Unable to read sessnum column from result row");
		return -1;
	}

	sessnum = atoi(value);

	if (sessnum <= 0) {
		logmsg(MLOG_ERR, "Invalid sessnum column read from result row");
		return -1;
	}

	value = apr_dbd_get_entry(driver, row, 1);

	if (value == NULL) {
		logmsg(MLOG_ERR, "Unable to read viewuser column from result row");
		return -1;
	}

	strncpy(username, value, sizeof (username));

	value = apr_dbd_get_entry(driver, row, 2);

	if (value == NULL) {
		logmsg(MLOG_ERR, "Unable to read hostname column from result row");
		return -1;
	}

	strncpy(hostname, value, sizeof (hostname));

	value = apr_dbd_get_entry(driver, row, 3);

	if (value == NULL) {
		logmsg(MLOG_ERR, "Unable to read dispnum column from result row");
		return -1;
	}

	dispnum = atoi(value);

	if (dispnum <= 0) {
		logmsg(MLOG_ERR, "Invalid dispnum column read from result row");
		return -1;
	}

	value = apr_dbd_get_entry(driver, row, 4);

	if (value == NULL) {
		logmsg(MLOG_ERR, "Unable to read timeout column from result row");
		return -1;
	}

	timeout = atoi(value);

	if (timeout <= 0) {
		logmsg(MLOG_ERR, "Invalid timeout column read from result row");
		return -1;
	}

	port = dispnum + portbase;

	logmsg(MLOG_NOTICE, "Map %s@%s for %s to %s:%d", username, remoteip, token, hostname, port);

	rv = apr_dbd_get_row(driver, pool, res, &row, -1); // clears connection

	if (rv != -1) {
		logmsg(MLOG_ERR, "Database transaction not finished when expected.");
		return -1;
	}

	if (send_response) {
		printf("HTTP/1.0 200 Connection Established\n");
		printf("Proxy-agent: HUBzero connection redirector\n\n");
		fflush(stdout);
		logmsg(MLOG_INFO, "Response: HTTP/1.0 200 Connection Established");
	}

	pid = fork();

	if (pid == -1) {
		logmsg(MLOG_ERR, "Unable to fork() for exec of socat");
		return -1;
	}

	if (pid == 0) {
		logmsg(MLOG_DEBUG, "Starting new process [%d] to exec() socat tunnel in", getpid());

		/* dbd may be open, but closing it with apr_dbd_close here corrupts parent */
		/* maybe should close file descriptors here */

		rv = snprintf(query, sizeof (query), "tcp4:%s:%d", hostname, port);

		if (rv < 0 || rv >= sizeof (query)) {
			logmsg(MLOG_ERR, "Failed to build argument string");
			exit(2);
		} else {
			logmsg(MLOG_DEBUG, "Executing: socat - %s.", query);
			closelog();
			execlp("socat", "socat", "-", query, NULL);
			logmsg(MLOG_ERR, "Failed to exec() socat");
		}

		exit(3);
	}

	if (sink) {
		rv = sink_until_ssl_handshake();

		if (rv == -1) {
			logmsg(MLOG_ERR, "Sinking until SSL handshake detected failed");
			return -1;
		}
	}

	if (freopen("/dev/null", "r", stdin) == NULL) {
		logmsg(MLOG_ERR, "Failed to redirect /dev/null to stdin");
		return -1;
	}

	if (freopen("/dev/null", "w", stdout) == NULL) {
		logmsg(MLOG_ERR, "Failed to redirect stdout to /dev/null");
		return -1;
	}

	rv = gettimeofday(&tv, NULL);

	if (rv == 0) {
		starttime = tv.tv_sec + (tv.tv_usec / 1000000.0);
	} else {
		starttime = 0;
	}

	rv = snprintf(query, sizeof (query), "UPDATE session SET accesstime=NOW() WHERE sessnum = '%d';", sessnum);

	if (rv < 0 || rv >= sizeof (query)) {
		logmsg(MLOG_ERR, "Failed to build query string");
		return -1;
	}

	rv = apr_dbd_query(driver, handle, &nrows, query);

	logmsg(MLOG_DEBUG, "%s", query);

	if (rv != APR_SUCCESS) {
		logmsg(MLOG_ERR, "Failed to update session accesstime");
		return -1;
	}

	rv = snprintf(query, sizeof (query), "INSERT INTO view(sessnum,username,remoteip,start,heartbeat) "
		" VALUE (%d,'%s','%s',now(),now());", sessnum, username, eremoteip);

	if (rv < 0 || rv >= sizeof (query)) {
		logmsg(MLOG_ERR, "Failed to build query string");
		return -1;
	}

	rv = apr_dbd_query(driver, handle, &nrows, query);

	logmsg(MLOG_DEBUG, "%s", query);

	if (rv != APR_SUCCESS) {
		logmsg(MLOG_ERR, "Failed to insert view record");
		return -1;
	}

	rv = snprintf(query, sizeof (query), "SELECT LAST_INSERT_ID();");

	if (rv < 0 || rv >= sizeof (query)) {
		logmsg(MLOG_ERR, "Failed to build query string");
		return -1;
	}

	rv = apr_dbd_select(driver, pool, handle, &res, query, 0);

	logmsg(MLOG_DEBUG, "%s", query);

	if (rv != APR_SUCCESS) {
		logmsg(MLOG_ERR, "Failed to query last insert id");
		return -1;
	}

	row = NULL;

	if (apr_dbd_get_row(driver, pool, res, &row, -1) != APR_SUCCESS) {
		logmsg(MLOG_ERR, "Fail to retrieve last insert id");
		return -1;
	}

	value = apr_dbd_get_entry(driver, row, 0);

	if (value == NULL) {
		logmsg(MLOG_ERR, "Unable to read insert id column from result row");
		return -1;
	}

	viewid = atoi(value);

	if (viewid <= 0) {
		logmsg(MLOG_ERR, "Invalid insert id column read from result row");
		return -1;
	}

	logmsg(MLOG_NOTICE, "View %d (%s@%s) started", viewid, username, remoteip);

	rv = apr_dbd_get_row(driver, pool, res, &row, -1); // clears connection

	if (rv != -1) {
		logmsg(MLOG_ERR, "Database transaction not finished when expected");
		return -1;
	}

	rv = apr_dbd_close(driver, handle);

	if (rv != APR_SUCCESS) {
		logmsg(MLOG_ERR, "Unable to close database connection");
		return -1;
	}

	interval = (int) (timeout * 0.40);

	if (signal(SIGALRM, do_proxy_vnc_sighandler) == SIG_ERR) {
		logmsg(MLOG_ERR, "Failed to assign signal handler to SIGALRM");
		return -1;
	}

	if (signal(SIGCHLD, do_proxy_vnc_sighandler) == SIG_ERR) {
		logmsg(MLOG_ERR, "Failed to assign signal handler to SIGCHLD");
		return -1;
	}

	if (signal(SIGHUP, do_proxy_vnc_sighandler) == SIG_ERR) {
		logmsg(MLOG_ERR, "Failed to assign signal handler to SIGHUP");
		return -1;
	}

	if (signal(SIGTERM, do_proxy_vnc_sighandler) == SIG_ERR) {
		logmsg(MLOG_ERR, "Failed to assign signal handler to SIGTERM");
		return -1;
	}

	if (signal(SIGINT, do_proxy_vnc_sighandler) == SIG_ERR) {
		logmsg(MLOG_ERR, "Failed to assign signal handler to SIGINT");
		return -1;
	}

	alarm(interval);

	while (1) {
		pause();

		if (gSignal) {
			while (waitpid(-1, &status, WNOHANG) > 0)
				; // reap child processes

			rv = gettimeofday(&tv, NULL);

			if (rv == 0) {
				endtime = tv.tv_sec + (tv.tv_usec / 1000000.0);
			} else {
				endtime = 0;
			}

			if (starttime == 0) {
				starttime = endtime;
			}

			if (endtime < starttime) {
				endtime = starttime;
			}

			logmsg(MLOG_NOTICE, "View %d (%s@%s) ended after %lf seconds",
				viewid, username, remoteip, endtime - starttime);

			if (open_database(driver, pool, dbd_params, &handle) != 0) {
				logmsg(MLOG_ERR, "Unable to open database to record end of view");
				return -1;
			}

			rv = snprintf(query, sizeof (query), "INSERT INTO viewlog(sessnum,username,remoteip,time,duration) "
				"SELECT sessnum, username, remoteip, start, %lf "
				"FROM view WHERE viewid='%d'", (double) (endtime - starttime), viewid);

			if (rv < 0 || rv >= sizeof (query)) {
				logmsg(MLOG_ERR, "Failed to build viewlog insert query string");
				return -1;
			}

			logmsg(MLOG_DEBUG, "%s", query);

			rv = apr_dbd_query(driver, handle, &nrows, query);

			if (rv != APR_SUCCESS) {
				logmsg(MLOG_ERR, "Failed to insert viewlog record");
				return -1;
			}

			snprintf(query, sizeof (query), "UPDATE session JOIN view ON session.sessnum=view.sessnum "
				"SET session.accesstime=now() "
				"WHERE viewid='%d'", viewid);

			if (rv < 0 || rv >= sizeof (query)) {
				logmsg(MLOG_ERR, "Failed to build session update accesstime query string");
				return -1;
			}

			logmsg(MLOG_DEBUG, "%s", query);

			rv = apr_dbd_query(driver, handle, &nrows, query);

			if (rv != APR_SUCCESS) {
				logmsg(MLOG_ERR, "Failed to update session accesstime record");
				return -1;
			}

			rv = snprintf(query, sizeof (query), "DELETE FROM view WHERE viewid='%d'", viewid);

			if (rv < 0 || rv >= sizeof (query)) {
				logmsg(MLOG_ERR, "Failed to build view deletion query string");
				return -1;
			}

			logmsg(MLOG_DEBUG, "%s", query);

			rv = apr_dbd_query(driver, handle, &nrows, query);

			if (rv != APR_SUCCESS) {
				logmsg(MLOG_ERR, "Failed to delete view record");
				return -1;
			}

			rv = apr_dbd_close(driver, handle);

			if (rv != APR_SUCCESS) {
				logmsg(MLOG_ERR, "Unable to close database connection");
				return -1;
			}

			break;
		}

		if (gAlarm == 1) {
			logmsg(MLOG_INFO, "Refreshing heartbeat for view %d", viewid);

			rv = snprintf(query, sizeof (query), "UPDATE view SET heartbeat=NOW() WHERE viewid='%d' LIMIT 1;", viewid);

			if (rv < 0 || rv >= sizeof (query)) {
				logmsg(MLOG_ERR, "Failed to build view heartbeat update query string");
				return -1;
			}

			if (open_database(driver, pool, dbd_params, &handle) != 0) {
				logmsg(MLOG_ERR, "Unable to open database to update heartbeat of view [%d]", viewid);
				return -1;
			}

			logmsg(MLOG_DEBUG, "%s", query);

			rv = apr_dbd_query(driver, handle, &nrows, query);

			if (rv != APR_SUCCESS) {
				logmsg(MLOG_ERR, "Failed to update view heartbeat");
				return -1;
			}

			apr_dbd_close(driver, handle);

			if (rv != APR_SUCCESS) {
				logmsg(MLOG_ERR, "Unable to close database connection after heartbeat update");
				return -1;
			}

			interval = (int) (timeout * 0.40);

			gAlarm = 0;

			alarm(interval);
		}
	}

	return 0;
}

