/**
 * @package     hubzero-vncproxy
 * @file		connect.c
 * @author      Nicholas J. Kisseberth <nkissebe@purdue.edu>
 * @copyright   Copyright (c) 2010-2011 Purdue University. All rights reserved.
 * @license     http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
 *
 * Copyright (c) 2010-2011 Purdue University
 * All rights reserved.
 *
 * This file is part of: The HUBzero(R) Platform for Scientific Collaboration
 *
 * The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
 * software: you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * HUBzero is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * HUBzero is a registered trademark of Purdue University.
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <time.h>
#include <netdb.h>
#include <apr_pools.h>
#include <apr_strings.h>
#include <apu.h>
#include <apr_dbd.h>
#include <syslog.h>
#include <libgen.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/fcntl.h>

#include "log.h"

volatile sig_atomic_t gAlarm = 0;
volatile sig_atomic_t gSignal = 0;

void
do_proxy_vnc_sighandler(int sig)
{
	if (sig == SIGALRM)
		gAlarm = 1;
	else
		gSignal = sig;
}

int
open_database(const apr_dbd_driver_t *driver, apr_pool_t *pool, char *dbd_params, apr_dbd_t **handle)
{

	int delay = 1;
	int tries = 1;

	while (apr_dbd_open(driver, pool, dbd_params, handle) != APR_SUCCESS) {
		if (tries > 6) {
			logmsg(MLOG_WARNING, "Unable to open database connection. Tried %d times. Aborting", tries);
			return -1;
		}

		logmsg(MLOG_WARNING, "Unable to open database connection. Retrying in %ds", delay);

		sleep(delay);

		delay *= 2;

		if (delay > 8) {
			delay = 8;
		}

		tries++;
	}

	return 0;
}

int
sink_until_ssl_handshake()
{
	char buf;
	int rv;

	logmsg(MLOG_DEBUG, "Sinking input until possible SSL handshake arrives");

	while (1) {
		rv = read(STDIN_FILENO, &buf, 1);

		if (rv == 0) {
			return -1;
		}

		if (buf == 16) {
			write(STDOUT_FILENO, &buf, 1);
			logmsg(MLOG_DEBUG, "Possible start of SSL handshake found");
			return 0;
		}

		continue;
	}
}

int
socat(int fd1, char *address, int port, int bufsize)
{
	char *buffer1 = malloc(bufsize);
	char *buffer2 = malloc(bufsize);
	char *buf1 = buffer1;
    int  len1 = 0;
    int  left1 = bufsize;
    char *buf2 = buffer2;
    int  len2 = 0;
    int  left2 = bufsize;
	fd_set rfds, wfds, efds;
    int nfds = fd1;
	int rv;
	char c;
	int count = 0;
	int fd2;
	int fd1_flags = 0;
	int fd2_flags = 0;
    struct addrinfo hints;
	struct addrinfo *tmpaddr;
	struct addrinfo *hostaddr = NULL;

	logmsg(MLOG_INFO, "socat(%d,%s,%d,%d)", fd1,address,port,bufsize);

    if (buffer1 == NULL)
		return -1;

    if (buffer2 == NULL)
		return -1;

	memset(&hints, 0, sizeof(hints));

	hints.ai_family = PF_UNSPEC;
	hints.ai_socktype = SOCK_STREAM;
	hints.ai_flags |= AI_CANONNAME;

	rv = getaddrinfo(address, NULL, &hints, &hostaddr);

	for (tmpaddr = hostaddr; tmpaddr; tmpaddr = tmpaddr->ai_next) {
		if (tmpaddr->ai_family == AF_UNIX)
			continue;

    	if ((fd2 = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
			logmsg(MLOG_ERR,"socat(): unable to create socket [%d]", errno);
			free(buffer1);
			free(buffer2);
			freeaddrinfo(hostaddr);
        	return -1;
    	}
		
		((struct sockaddr_in *) tmpaddr->ai_addr)->sin_port = htons(port);

		do {
			rv = connect(fd2, tmpaddr->ai_addr, tmpaddr->ai_addrlen);
		}
		while (rv != 0 && errno == EINTR);

		if (rv != 0) {
			logmsg(MLOG_ERR,"socat(): connect failed [%d]", errno);
        	shutdown(fd2, SHUT_RDWR);
        	close(fd2);
			free(buffer1);
			free(buffer2);
			freeaddrinfo(hostaddr);
			return -1;
    	}
	}

	freeaddrinfo(hostaddr);

	if (fd2 > nfds)
		nfds = fd2;

	fd1_flags = fcntl(fd1, F_GETFD);
	fd2_flags = fcntl(fd2, F_GETFD);
	
	fd1_flags &= ~(O_RDONLY | O_WRONLY | O_APPEND);
	fd2_flags &= ~(O_RDONLY | O_WRONLY | O_APPEND);

	fd1_flags |= O_RDWR | O_NONBLOCK;
	fd2_flags |= O_RDWR | O_NONBLOCK;

	fcntl(fd1, F_SETFD, fd1_flags);
	fcntl(fd2, F_SETFD, fd2_flags);

	nfds++;

	while(1) {
		if (gSignal)
			break;

		FD_ZERO(&rfds);
		FD_ZERO(&wfds);
		FD_ZERO(&efds);
		
		if (left1 > 0) { // room in read buffer for fd1, ready to read fd1
			logmsg(MLOG_DEBUG, "socat(): ready to read %d from fd1=%d", left1,fd1);
			FD_SET(fd1, &rfds);
		}

		if (left2 > 0) { // room in read buffer for fd2, ready to read fd2
			logmsg(MLOG_DEBUG, "socat(): ready to read %d from fd2=%d", left2, fd2);
			FD_SET(fd2, &rfds);
		}

		if (len1 > 0) { // data in read buffer for fd1, ready to write to fd2
			logmsg(MLOG_DEBUG, "socat(): ready to write %d to fd2=%d", len1,fd2);
			FD_SET(fd2, &wfds);
		}

		if (len2 > 0) { // data in read buffer for fd2, ready to write to fd1
			logmsg(MLOG_DEBUG, "socat(): ready to write %d to fd1=%d", len2,fd1);
			FD_SET(fd1, &wfds);
		}

		// OOB data is not in our intended use case, we include the implementation
		// from select_tut(2) for completeness. Untested and seems incomplete to me.

		FD_SET(fd1, &efds); // always on look out for OOB data
		FD_SET(fd2, &efds); // always on look out for OOB data

		logmsg(MLOG_DEBUG, "socat(): select() waiting");

		rv = select(nfds, &rfds, &wfds, &efds, NULL);

		if (rv == -1 && errno == EINTR)
			continue;

		if (rv < 0) {
			logmsg(MLOG_ERR,"socat(): select() failed: %d", errno);
			break;
		}

		if (FD_ISSET(fd1, &efds)) { // we pass OOB data from fd1 to fd2 one byte at a time
			count = recv(fd1, &c, 1, MSG_OOB);

			logmsg(MLOG_DEBUG,"socat(): OOB recv(fd1=%d) = %d/%x",fd1,count,c);

			if (count > 0) {
				do {
					count = send(fd2, &c, 1, MSG_OOB);
				}
				while (count == -1 && (errno == EINTR) || (errno == EAGAIN));

				logmsg(MLOG_DEBUG,"socat(): OOB send(fd2=%d,%x) = %d",fd2,c,count);
			}
			else if (count < 0) {
				if ((errno != EAGAIN) && (errno != EINTR))
					break;
			}
			else if (count == 0) {
				break;
			}
		}

		if (FD_ISSET(fd2, &efds)) { // we pass OOB data from fd2 to fd1 one byte at a time
			count = recv(fd2, &c, 1, MSG_OOB);

			logmsg(MLOG_DEBUG,"socat(): OOB recv(fd2=%d) = %d/%x",fd2,count,c);

			if (count > 0) {
				do {
					count = send(fd1, &c, 1, MSG_OOB);
				}
				while (count == -1 && (errno == EINTR) || (errno == EAGAIN));

				logmsg(MLOG_DEBUG,"socat(): OOB send(fd1=%d,%x) = %d",fd1,c,count);
			}
			else if (count < 0) {
				if ((errno != EAGAIN) && (errno != EINTR))
					break;
			}
			else if (count == 0) {
				break;
			}
		}

		if (FD_ISSET(fd1, &rfds)) { // read available data from fd1 into remaining space in buffer1

			if (left1 > 0) {
				count = read(fd1, buf1+len1, left1);

				logmsg(MLOG_DEBUG,"socat(): read(fd1=%d,offset=%d,left1=%d) = %d",fd1,len1,left1,count);

				if (count > 0) {
					left1 -= count;
					len1 += count;
				}
				else if (count == -1) {
					if ((errno != EAGAIN) && (errno != EINTR))
						break;
				}
				else if (count == 0)
					break;
			}
		}

		if (FD_ISSET(fd2, &wfds)) { // write as much data as possible from buffer1 to fd2

			if (len1 >0) {
				count = write(fd2, buf1, len1);

				logmsg(MLOG_DEBUG,"socat(): write(fd2=%d,offset=%d,len1=%d) = %d",fd2,buf1-buffer1,len1,count);

				if (count > 0) {
					buf1 += count;
					len1 -= count;
				}
				else if (count == -1) {
					if ((errno != EAGAIN) && (errno != EINTR))
						break;
				}

				if (len1 == 0) {
					buf1 = buffer1;
					left1 = bufsize;
				}
			}
		}

		if (FD_ISSET(fd2, &rfds)) { // read available data from fd2 into remaining space in buffer2

			if (left2 > 0) {
				count = read(fd2, buf2+len2, left2);

				logmsg(MLOG_DEBUG,"socat(): read(fd2=%d,offset=%d,left2=%d) = %d",fd2,len2,left2,count);

				if (count > 0) {
					left2 -= count;
					len2 += count;
				}
				else if (count == -1) {
					if ((errno != EAGAIN) && (errno != EINTR))
						break;
				}
				else if (count == 0)
					break;
			}
		}

		if (FD_ISSET(fd1, &wfds)) { // write as much data as possible from buffer2 to fd1
			if (len2 >0) {

				count = write(fd1, buf2, len2);

				logmsg(MLOG_DEBUG,"socat(): write(fd1=%d,offset=%d,len2=%d) = %d",fd1,buf2-buffer2,len2,count);

				if (count > 0) {
					buf2 += count;
					len2 -= count;
				}
				else if (count == -1) {
					if ((errno != EAGAIN) && (errno != EINTR))
						break;
				}

				if (len2 == 0) {
					buf2 = buffer2;
					left2 = bufsize;
				}
			}
		}
	}

	shutdown(fd2, SHUT_RDWR);
	close(fd2);
	free(buffer1);
	free(buffer2);

	return 0;
}

int
process_vnc_connect_method(apr_pool_t *pool, int sink, const char *remoteip, const char *token, int portbase,
	const apr_dbd_driver_t *driver, char *dbd_params, int send_response, int bufsize)
{
	int delay = 1;
	int maxdelay = 3600;
	struct timeval tv;
	double starttime = 0.0, endtime = 0.0;
	char query[1024];
	char bufstr[1024];
	int nrows;
	apr_dbd_results_t *res = NULL;
	apr_dbd_row_t *row = NULL;
	apr_status_t rv;
	int viewid = 0;
	pid_t pid = -1;
	int interval;
	char *c;
	int dispnum;
	int sessnum;
	int timeout;
	int id;
	apr_dbd_t *handle;
	int status;
	char hostname[41];
	char username[33];
	int port;
	int tries = 1;
	const char *etoken;
	const char *eremoteip;
	const char *value;

	if (bufsize <= 0) {
		bufsize = 0;
	}

	if (open_database(driver, pool, dbd_params, &handle) != 0) {
		logmsg(MLOG_ERR, "Unable to open database to lookup token");
		return -1;
	}

	etoken = apr_dbd_escape(driver, pool, token, handle);

	if (etoken == NULL) {
		logmsg(MLOG_ERR, "Unable to build query safe token string");
		return -1;
	}

	eremoteip = apr_dbd_escape(driver, pool, remoteip, handle);

	if (eremoteip == NULL) {
		logmsg(MLOG_ERR, "Unable to build query safe remoteip string");
		return -1;
	}

	rv = snprintf(query, sizeof (query), "SELECT viewperm.sessnum,viewuser,display.hostname,session.dispnum,"
		" session.timeout,portbase"
		" FROM viewperm"
		" JOIN display ON viewperm.sessnum = display.sessnum"
		" JOIN session ON session.sessnum = display.sessnum"
		" JOIN host ON display.hostname = host.hostname"
		" WHERE viewperm.viewtoken='%s' LIMIT 1;", etoken);

	if (rv < 0 || rv >= sizeof (query)) {
		logmsg(MLOG_ERR, "Failed to build query string");
		return -1;
	}

	rv = apr_dbd_select(driver, pool, handle, &res, query, 0);

	logmsg(MLOG_DEBUG, query);

	row = NULL;

	if ((rv != APR_SUCCESS) || (apr_dbd_get_row(driver, pool, res, &row, -1) != 0)) {
		logmsg(MLOG_WARNING, "No database entry found for token [%s]", token);

		if (send_response) {
			printf("HTTP/1.1 404 Not Found\n\n");
			fflush(stdout);
			logmsg(MLOG_INFO, "Response: HTTP/1.1 404 Not Found");
		}

		rv = apr_dbd_close(driver, handle);

		if (rv != APR_SUCCESS) {
			logmsg(MLOG_ERR, "Unable to close database connection");
			return -1;
		}

		return 0;
	}

	value = apr_dbd_get_entry(driver, row, 0);

	if (value == NULL) {
		logmsg(MLOG_ERR, "Unable to read sessnum column from result row");
		return -1;
	}

	sessnum = atoi(value);

	if (sessnum <= 0) {
		logmsg(MLOG_ERR, "Invalid sessnum column read from result row");
		return -1;
	}

	value = apr_dbd_get_entry(driver, row, 1);

	if (value == NULL) {
		logmsg(MLOG_ERR, "Unable to read viewuser column from result row");
		return -1;
	}

	strncpy(username, value, sizeof (username));

	value = apr_dbd_get_entry(driver, row, 2);

	if (value == NULL) {
		logmsg(MLOG_ERR, "Unable to read hostname column from result row");
		return -1;
	}

	strncpy(hostname, value, sizeof (hostname));

	value = apr_dbd_get_entry(driver, row, 3);

	if (value == NULL) {
		logmsg(MLOG_ERR, "Unable to read dispnum column from result row");
		return -1;
	}

	dispnum = atoi(value);

	if (dispnum <= 0) {
		logmsg(MLOG_ERR, "Invalid dispnum column read from result row");
		return -1;
	}

	value = apr_dbd_get_entry(driver, row, 4);

	if (value == NULL) {
		logmsg(MLOG_ERR, "Unable to read timeout column from result row");
		return -1;
	}

	timeout = atoi(value);

	if (timeout <= 0) {
		logmsg(MLOG_ERR, "Invalid timeout column read from result row");
		return -1;
	}

	port = dispnum + portbase;

	logmsg(MLOG_NOTICE, "Map %s@%s for %s to %s:%d", username, remoteip, token, hostname, port);

	rv = apr_dbd_get_row(driver, pool, res, &row, -1); // clears connection

	if (rv != -1) {
		logmsg(MLOG_ERR, "Database transaction not finished when expected.");
		return -1;
	}

	if (send_response) {
		printf("HTTP/1.0 200 Connection Established\n");
		printf("Proxy-agent: HUBzero connection redirector\n\n");
		fflush(stdout);
		logmsg(MLOG_INFO, "Response: HTTP/1.0 200 Connection Established");
	}

	pid = fork();

	if (pid == -1) {
		logmsg(MLOG_ERR, "Unable to fork() for exec of socat");
		return -1;
	}

	if (pid == 0) {
		logmsg(MLOG_DEBUG, "Starting new process [%d] to exec() socat tunnel in", getpid());

		/* dbd may be open, but closing it with apr_dbd_close here corrupts parent */
		/* maybe should close file descriptors here */

		rv = snprintf(query, sizeof (query), "tcp4:%s:%d", hostname, port);

		if (rv < 0 || rv >= sizeof (query)) {
			logmsg(MLOG_ERR, "Failed to build argument string");
			exit(2);
		} 
		
		rv = snprintf(bufstr, sizeof (bufstr), "%d", bufsize);

		if (rv < 0 || rv >= sizeof (bufstr)) {
			logmsg(MLOG_ERR, "Failed to build buffer size argument string");
			exit(2);
		} 
		
		logmsg(MLOG_DEBUG, "Executing: socat - %s.", query);

		if (bufsize <= 0)
			bufsize = 8192;

		socat(0,hostname,port,bufsize);
		exit(0);
	}

	if (sink) {
		rv = sink_until_ssl_handshake();

		if (rv == -1) {
			logmsg(MLOG_ERR, "Sinking until SSL handshake detected failed");
			return -1;
		}
	}

	if (freopen("/dev/null", "r", stdin) == NULL) {
		logmsg(MLOG_ERR, "Failed to redirect /dev/null to stdin");
		return -1;
	}

	if (freopen("/dev/null", "w", stdout) == NULL) {
		logmsg(MLOG_ERR, "Failed to redirect stdout to /dev/null");
		return -1;
	}

	rv = gettimeofday(&tv, NULL);

	if (rv == 0) {
		starttime = tv.tv_sec + (tv.tv_usec / 1000000.0);
	} else {
		starttime = 0;
	}

	rv = snprintf(query, sizeof (query), "UPDATE session SET accesstime=NOW() WHERE sessnum = '%d';", sessnum);

	if (rv < 0 || rv >= sizeof (query)) {
		logmsg(MLOG_ERR, "Failed to build query string");
		return -1;
	}

	rv = apr_dbd_query(driver, handle, &nrows, query);

	logmsg(MLOG_DEBUG, "%s", query);

	if (rv != APR_SUCCESS) {
		logmsg(MLOG_ERR, "Failed to update session accesstime");
		return -1;
	}

	rv = snprintf(query, sizeof (query), "INSERT INTO view(sessnum,username,remoteip,start,heartbeat) "
		" VALUE (%d,'%s','%s',now(),now());", sessnum, username, eremoteip);

	if (rv < 0 || rv >= sizeof (query)) {
		logmsg(MLOG_ERR, "Failed to build query string");
		return -1;
	}

	rv = apr_dbd_query(driver, handle, &nrows, query);

	logmsg(MLOG_DEBUG, "%s", query);

	if (rv != APR_SUCCESS) {
		logmsg(MLOG_ERR, "Failed to insert view record");
		return -1;
	}

	rv = snprintf(query, sizeof (query), "SELECT LAST_INSERT_ID();");

	if (rv < 0 || rv >= sizeof (query)) {
		logmsg(MLOG_ERR, "Failed to build query string");
		return -1;
	}

	rv = apr_dbd_select(driver, pool, handle, &res, query, 0);

	logmsg(MLOG_DEBUG, "%s", query);

	if (rv != APR_SUCCESS) {
		logmsg(MLOG_ERR, "Failed to query last insert id");
		return -1;
	}

	row = NULL;

	if (apr_dbd_get_row(driver, pool, res, &row, -1) != APR_SUCCESS) {
		logmsg(MLOG_ERR, "Fail to retrieve last insert id");
		return -1;
	}

	value = apr_dbd_get_entry(driver, row, 0);

	if (value == NULL) {
		logmsg(MLOG_ERR, "Unable to read insert id column from result row");
		return -1;
	}

	viewid = atoi(value);

	if (viewid <= 0) {
		logmsg(MLOG_ERR, "Invalid insert id column read from result row");
		return -1;
	}

	logmsg(MLOG_NOTICE, "View %d (%s@%s) started", viewid, username, remoteip);

	rv = apr_dbd_get_row(driver, pool, res, &row, -1); // clears connection

	if (rv != -1) {
		logmsg(MLOG_ERR, "Database transaction not finished when expected");
		return -1;
	}

	rv = apr_dbd_close(driver, handle);

	if (rv != APR_SUCCESS) {
		logmsg(MLOG_ERR, "Unable to close database connection");
		return -1;
	}

	interval = (int) (timeout * 0.40);

	if (signal(SIGALRM, do_proxy_vnc_sighandler) == SIG_ERR) {
		logmsg(MLOG_ERR, "Failed to assign signal handler to SIGALRM");
		return -1;
	}

	if (signal(SIGCHLD, do_proxy_vnc_sighandler) == SIG_ERR) {
		logmsg(MLOG_ERR, "Failed to assign signal handler to SIGCHLD");
		return -1;
	}

	if (signal(SIGHUP, do_proxy_vnc_sighandler) == SIG_ERR) {
		logmsg(MLOG_ERR, "Failed to assign signal handler to SIGHUP");
		return -1;
	}

	if (signal(SIGTERM, do_proxy_vnc_sighandler) == SIG_ERR) {
		logmsg(MLOG_ERR, "Failed to assign signal handler to SIGTERM");
		return -1;
	}

	if (signal(SIGINT, do_proxy_vnc_sighandler) == SIG_ERR) {
		logmsg(MLOG_ERR, "Failed to assign signal handler to SIGINT");
		return -1;
	}

	alarm(interval);

	while (1) {
		pause();

		if (gSignal) {
			while (waitpid(-1, &status, WNOHANG) > 0)
				; // reap child processes

			rv = gettimeofday(&tv, NULL);

			if (rv == 0) {
				endtime = tv.tv_sec + (tv.tv_usec / 1000000.0);
			} else {
				endtime = 0;
			}

			if (starttime == 0) {
				starttime = endtime;
			}

			if (endtime < starttime) {
				endtime = starttime;
			}

			logmsg(MLOG_NOTICE, "View %d (%s@%s) ended after %lf seconds",
				viewid, username, remoteip, endtime - starttime);

			if (open_database(driver, pool, dbd_params, &handle) != 0) {
				logmsg(MLOG_ERR, "Unable to open database to record end of view");
				return -1;
			}

			rv = snprintf(query, sizeof (query), "INSERT INTO viewlog(sessnum,username,remoteip,time,duration) "
				"SELECT sessnum, username, remoteip, start, %lf "
				"FROM view WHERE viewid='%d'", (double) (endtime - starttime), viewid);

			if (rv < 0 || rv >= sizeof (query)) {
				logmsg(MLOG_ERR, "Failed to build viewlog insert query string");
				return -1;
			}

			logmsg(MLOG_DEBUG, "%s", query);

			rv = apr_dbd_query(driver, handle, &nrows, query);

			if (rv != APR_SUCCESS) {
				logmsg(MLOG_ERR, "Failed to insert viewlog record");
				return -1;
			}

			snprintf(query, sizeof (query), "UPDATE session JOIN view ON session.sessnum=view.sessnum "
				"SET session.accesstime=now() "
				"WHERE viewid='%d'", viewid);

			if (rv < 0 || rv >= sizeof (query)) {
				logmsg(MLOG_ERR, "Failed to build session update accesstime query string");
				return -1;
			}

			logmsg(MLOG_DEBUG, "%s", query);

			rv = apr_dbd_query(driver, handle, &nrows, query);

			if (rv != APR_SUCCESS) {
				logmsg(MLOG_ERR, "Failed to update session accesstime record");
				return -1;
			}

			rv = snprintf(query, sizeof (query), "DELETE FROM view WHERE viewid='%d'", viewid);

			if (rv < 0 || rv >= sizeof (query)) {
				logmsg(MLOG_ERR, "Failed to build view deletion query string");
				return -1;
			}

			logmsg(MLOG_DEBUG, "%s", query);

			rv = apr_dbd_query(driver, handle, &nrows, query);

			if (rv != APR_SUCCESS) {
				logmsg(MLOG_ERR, "Failed to delete view record");
				return -1;
			}

			rv = apr_dbd_close(driver, handle);

			if (rv != APR_SUCCESS) {
				logmsg(MLOG_ERR, "Unable to close database connection");
				return -1;
			}

			break;
		}

		if (gAlarm == 1) {
			logmsg(MLOG_INFO, "Refreshing heartbeat for view %d", viewid);

			rv = snprintf(query, sizeof (query), "UPDATE view SET heartbeat=NOW() WHERE viewid='%d' LIMIT 1;", viewid);

			if (rv < 0 || rv >= sizeof (query)) {
				logmsg(MLOG_ERR, "Failed to build view heartbeat update query string");
				return -1;
			}

			if (open_database(driver, pool, dbd_params, &handle) != 0) {
				logmsg(MLOG_ERR, "Unable to open database to update heartbeat of view [%d]", viewid);
				return -1;
			}

			logmsg(MLOG_DEBUG, "%s", query);

			rv = apr_dbd_query(driver, handle, &nrows, query);

			if (rv != APR_SUCCESS) {
				logmsg(MLOG_ERR, "Failed to update view heartbeat");
				return -1;
			}

			apr_dbd_close(driver, handle);

			if (rv != APR_SUCCESS) {
				logmsg(MLOG_ERR, "Unable to close database connection after heartbeat update");
				return -1;
			}

			interval = (int) (timeout * 0.40);

			gAlarm = 0;

			alarm(interval);
		}
	}

	return 0;
}

