#!/usr/bin/env python
#
# @package      hubzero-submit-server
# @file         submit-server.py
# @copyright    Copyright (c) 2006-2020 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2006-2020 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#
import os
import sys
import logging

from hubzero.submit.LogMessage   import getLogIDMessage as getLogMessage
from hubzero.submit.SubmitServer import SubmitServer

LOGDIRECTORY       = os.path.join(os.sep,'var','log','submit')
HUBLOGFILE         = 'submit-server.log'
HUBLOGCRITICALFILE = 'submit-server_critical.log'
APPLICATIONLOGGER  = logging.getLogger('')

PIDFILE = os.path.join(os.sep,'var','run','submit-server.pid')

CONFIGURATIONDIRECTORY = os.path.join(os.sep,'etc','submit')
CONFIGURATIONFILE      = 'submit-server.conf'
CONFIGFILEPATH         = os.path.join(CONFIGURATIONDIRECTORY,CONFIGURATIONFILE)
INFOSCONFIGURATIONFILE = 'infos.conf'

DISTRIBUTORDIRECTORY = os.path.join(os.sep,'opt','submit')
DISTRIBUTORSCRIPT    = 'distributor'
HARVESTERDIRECTORY   = os.path.join(os.sep,'opt','submit')
HARVESTERSCRIPT      = 'harvester.py'
JOBSTATUSDIRECTORY   = os.path.join(os.sep,'opt','submit')
JOBSTATUSSCRIPT      = 'jobStatus.py'
JOBKILLDIRECTORY     = os.path.join(os.sep,'opt','submit')
JOBKILLSCRIPT        = 'jobKill.py'
VENUEPROBEDIRECTORY  = os.path.join(os.sep,'opt','submit')
VENUEPROBESCRIPT     = 'venueProbeR.py'

def daemonize(fdLogFile):
   if fdLogFile != sys.stdout.fileno():
      try:
         devnull = open("/dev/null",'r')
         try:
            os.dup2(devnull.fileno(),sys.stdin.fileno())
            os.dup2(fdLogFile,sys.stdout.fileno())
            os.dup2(fdLogFile,sys.stderr.fileno())
         except OSError:
            APPLICATIONLOGGER.log(logging.ERROR,getLogMessage("file descriptor dup failed"))
      except (IOError,OSError):
         APPLICATIONLOGGER.log(logging.ERROR,getLogMessage("%s could not be opened" % ("/dev/null")))

   if os.fork():
      os._exit(os.EX_OK)
   os.setsid()
   os.chdir("/")
   if os.fork():
      os._exit(os.EX_OK)


def openLogger(logDirectory,
               hubLogFile,
               hubLogCriticalFile):
   class EmptyFilter(logging.Filter):
      """
      This is a filter which rejects empty messages

      """

      def filter(self,record):
         if record.getMessage() == "":
            emptyRecord = True
         else:
            emptyRecord = False

         return(not emptyRecord)

   APPLICATIONLOGGER.setLevel(logging.INFO)

# all messages

   hubLogPath = os.path.join(logDirectory,hubLogFile)
   logHandler = logging.FileHandler(hubLogPath)
   fdLogFile = logHandler.stream.fileno()

   emptyFilter = EmptyFilter()
   logHandler.addFilter(emptyFilter)

   logFormatter = logging.Formatter('%(asctime)s %(message)s','%s [%a %b %d %H:%M:%S %Y]')
   logHandler.setFormatter(logFormatter)
   APPLICATIONLOGGER.addHandler(logHandler)

# critical messages only

   hubLogPath = os.path.join(logDirectory,hubLogCriticalFile)
   logCriticalHandler = logging.FileHandler(hubLogPath)
   logCriticalHandler.setLevel(logging.CRITICAL)

   emptyFilter = EmptyFilter()
   logCriticalHandler.addFilter(emptyFilter)

   logCriticalFormatter = logging.Formatter('%(asctime)s %(message)s','%s [%a %b %d %H:%M:%S %Y]')
   logCriticalHandler.setFormatter(logCriticalFormatter)
   APPLICATIONLOGGER.addHandler(logCriticalHandler)

   APPLICATIONLOGGER.log(logging.INFO,getLogMessage(80*'#'))
   APPLICATIONLOGGER.log(logging.INFO,getLogMessage("Backgrounding process."))

   return(fdLogFile)


def writePidFile(pidFile):
   pidSaved = False
   try:
      fpPid = open(pidFile,'w')
      try:
         fpPid.write("%d\n" % (os.getpid()))
      except (IOError,OSError):
         APPLICATIONLOGGER.log(logging.ERROR,getLogMessage("Unable to write pid (%d) to %s" % (os.getpid(),pidFile)))
      else:
         pidSaved = True
      finally:
         fpPid.close()
   except:
      APPLICATIONLOGGER.log(logging.ERROR,getLogMessage("Unable to open pid (%d) to %s" % (os.getpid(),pidFile)))

   if not pidSaved:
      os._exit(os.EX_CANTCREAT)


def server(configurationDirectory,
           configurationFile,
           infosConfigurationFile,
           distributorDirectory,
           distributorScript,
           harvesterDirectory,
           harvesterScript,
           jobStatusDirectory,
           jobStatusScript,
           jobKillDirectory,
           jobKillScript,
           venueProbeDirectory,
           venueProbeScript):
   submitServer = SubmitServer(configurationDirectory,configurationFile,infosConfigurationFile,
                               distributorDirectory,distributorScript,
                               harvesterDirectory,harvesterScript,
                               jobStatusDirectory,jobStatusScript,
                               jobKillDirectory,jobKillScript,
                               venueProbeDirectory,venueProbeScript)
   if submitServer.configure():
      if submitServer.setupMySQL():
         if submitServer.setupClientListeners():
            submitServer.run()
         else:
            APPLICATIONLOGGER.log(logging.ERROR,getLogMessage("Shutting down - no listeners configured."))
            sys.exit(1)
      else:
         APPLICATIONLOGGER.log(logging.ERROR,getLogMessage("Shutting down - could not configure MySQL database."))
         sys.exit(1)
   else:
      APPLICATIONLOGGER.log(logging.ERROR,getLogMessage("Shutting down - server configuration failed."))
      sys.exit(1)


if __name__ == '__main__':
   fdLogFile = openLogger(LOGDIRECTORY,
                          HUBLOGFILE,HUBLOGCRITICALFILE)

   daemonize(fdLogFile)

   writePidFile(PIDFILE)

   server(CONFIGURATIONDIRECTORY,CONFIGURATIONFILE,INFOSCONFIGURATIONFILE,
          DISTRIBUTORDIRECTORY,DISTRIBUTORSCRIPT,
          HARVESTERDIRECTORY,HARVESTERSCRIPT,
          JOBSTATUSDIRECTORY,JOBSTATUSSCRIPT,
          JOBKILLDIRECTORY,JOBKILLSCRIPT,
          VENUEPROBEDIRECTORY,VENUEPROBESCRIPT)

   sys.exit(0)


