#!/usr/bin/env python
#
# @package      hubzero-submit-server
# @file         submit-server.py
# @author       Steven Clark <clarks@purdue.edu>
# @author       Rick Kennell <kennell@purdue.edu>
# @copyright    Copyright (c) 2006-2013 HUBzero Foundation, LLC.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2006-2013 HUBzero Foundation, LLC.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#

import os
import sys

from hubzero.submit.LogMessage   import getLogMessageFileNo, openLog, logID as log
from hubzero.submit.SubmitServer import SubmitServer

LOGDIRECTORY = os.path.join(os.sep,'var','log','submit')
HUBLOGFILE   = 'submit-server.log'

PIDFILE = os.path.join(os.sep,'var','run','submit-server.pid')

CONFIGURATIONDIRECTORY = os.path.join(os.sep,'etc','submit')
CONFIGURATIONFILE      = 'submit-server.conf'
CONFIGFILEPATH         = os.path.join(CONFIGURATIONDIRECTORY,CONFIGURATIONFILE)

DISTRIBUTORDIRECTORY = os.path.join(os.sep,'opt','submit')
DISTRIBUTORSCRIPT    = 'distributor'
JOBSTATUSDIRECTORY   = os.path.join(os.sep,'opt','submit')
JOBSTATUSSCRIPT      = 'jobStatus.py'
JOBKILLDIRECTORY     = os.path.join(os.sep,'opt','submit')
JOBKILLSCRIPT        = 'jobKill.py'

def daemonize():
   logMessageFileNo = getLogMessageFileNo()
   if logMessageFileNo != sys.stdout.fileno():
      devnull = open("/dev/null",'r')
      os.dup2(devnull.fileno(),sys.stdin.fileno())
      os.dup2(logMessageFileNo,sys.stdout.fileno())
      os.dup2(logMessageFileNo,sys.stderr.fileno())

   if os.fork():
      os._exit(os.EX_OK)
   os.setsid()
   os.chdir("/")
   if os.fork():
      os._exit(os.EX_OK)


def openLogFile(logDirectory,
                hubLogFile):
   hubLogPath = os.path.join(logDirectory,hubLogFile)
   openLog(hubLogPath)
   log(80*'#')
   log("Backgrounding process.")


def writePidFile(pidFile):
   try:
      fpPid = open(pidFile,'w')
      fpPid.write("%d\n" % (os.getpid()))
      fpPid.close()
   except:
      log("Unable to write pid (%d) to %s" % (os.getpid(),pidFile))
      os._exit(os.EX_CANTCREAT)


def server(configFilePath,
           distributorDirectory,
           distributorScript,
           jobStatusDirectory,jobStatusScript,
           jobKillDirectory,jobKillScript):
   submitServer = SubmitServer(configFilePath,
                               distributorDirectory,distributorScript,
                               jobStatusDirectory,jobStatusScript,
                               jobKillDirectory,jobKillScript)
   if submitServer.configure():
      if submitServer.setupMySQL():
         if submitServer.setupClientListeners():
            submitServer.run()
         else:
            log("Shutting down - no listeners configured.")
            sys.exit(1)
      else:
         log("Shutting down - could not configure MySQL database.")
         sys.exit(1)
   else:
      log("Shutting down - server configuration failed.")
      sys.exit(1)


if __name__ == '__main__':
   openLogFile(LOGDIRECTORY,HUBLOGFILE)

   daemonize()

   writePidFile(PIDFILE)

   server(CONFIGFILEPATH,
          DISTRIBUTORDIRECTORY,DISTRIBUTORSCRIPT,
          JOBSTATUSDIRECTORY,JOBSTATUSSCRIPT,
          JOBKILLDIRECTORY,JOBKILLSCRIPT)

   sys.exit(0)


