#!/usr/bin/env python3
#
# @package      hubzero-submit-monitors
# @file         jobProcessor.py
# @copyright    Copyright (c) 2004-2020 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2004-2020 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#
# ----------------------------------------------------------------------
#  jobProcessor.py
#
#  script which processes registered jobs
#
import sys
import os
import signal
import time
import logging
from errno import EINTR

from hubzero.submit.LogMessage   import getLogPIDMessage as getLogMessage
from hubzero.submit.JobProcessor import JobProcessor

CONFIGURATIONDIRECTORY   = os.path.join(os.sep,'etc','submit')
DAEMONSCONFIGURATIONFILE = 'daemons.conf'

PROCESSORLOGLOCATION = os.path.join(os.sep,'var','log','submit','monitors')
PROCESSORLOGFILENAME = "jobProcessor.log"
APPLICATIONLOGGER    = logging.getLogger('')

PIDFILE = os.path.join(os.sep,'var','run','submit','submit-jobprocessor.pid')

def openLogger(logDirectory,
               hubLogFile):
   class EmptyFilter(logging.Filter):
      """
      This is a filter which rejects empty messages

      """

      def filter(self,record):
         if record.getMessage() == "":
            emptyRecord = True
         else:
            emptyRecord = False

         return(not emptyRecord)

   APPLICATIONLOGGER.setLevel(logging.DEBUG)

   hubLogPath = os.path.join(logDirectory,hubLogFile)
   logHandler = logging.FileHandler(hubLogPath)
   fdLogFile = logHandler.stream.fileno()

   emptyFilter = EmptyFilter()
   logHandler.addFilter(emptyFilter)

   logFormatter = logging.Formatter('%(asctime)s %(message)s','%s [%a %b %d %H:%M:%S %Y]')
   logHandler.setFormatter(logFormatter)
   APPLICATIONLOGGER.addHandler(logHandler)

   return(fdLogFile)


def daemonize(fdLogFile):
   if fdLogFile != sys.stdout.fileno():
      try:
         devnull = open("/dev/null",'r')
         try:
            os.dup2(devnull.fileno(),sys.stdin.fileno())
            os.dup2(fdLogFile,sys.stdout.fileno())
            os.dup2(fdLogFile,sys.stderr.fileno())
         except OSError:
            APPLICATIONLOGGER.log(logging.ERROR,getLogMessage("file descriptor dup failed"))
      except (IOError,OSError):
         APPLICATIONLOGGER.log(logging.ERROR,getLogMessage("%s could not be opened" % ("/dev/null")))

   if os.fork() != 0:
      os.wait()
      sys.exit(0)
   else:
      os.setsid()
      os.chdir("/")
      pid = os.fork()
      if pid != 0:
         sys.exit(0)

   time.sleep(2)


def writePidFile(pidFile):
   pidSaved = False
   try:
      fpPid = open(pidFile,'w')
      try:
         fpPid.write("%d\n" % (os.getpid()))
      except (IOError,OSError):
         APPLICATIONLOGGER.log(logging.ERROR,getLogMessage("Unable to write pid (%d) to %s" % (os.getpid(),pidFile)))
      else:
         pidSaved = True
      finally:
         fpPid.close()
   except:
      APPLICATIONLOGGER.log(logging.ERROR,getLogMessage("Unable to open pid (%d) to %s" % (os.getpid(),pidFile)))

   if not pidSaved:
      os._exit(os.EX_CANTCREAT)


class ProcessJobs:
   def __init__(self,
                configurationDirectory,
                daemonsConfigurationFile):
      self.logger = logging.getLogger(__name__)

      self.timeBetweenJobRelease = 5

      self.jobProcessor = JobProcessor(configurationDirectory,
                                       daemonsConfigurationFile)

      self.terminateJobProcessor = self.jobProcessor.setInfo()

      signal.signal(signal.SIGINT,self.sigINT_handler)
      signal.signal(signal.SIGHUP,self.sigHUP_handler)
      signal.signal(signal.SIGQUIT,self.sigQUIT_handler)
      signal.signal(signal.SIGABRT,self.sigABRT_handler)
      signal.signal(signal.SIGTERM,self.sigTERM_handler)


   def terminate(self):
      self.terminateJobProcessor = True


   def sigGEN_handler(self,
                      signalNumber,
                      frame):
      self.terminate()


   def sigINT_handler(self,
                      signalNumber,
                      frame):
      self.logger.log(logging.INFO,getLogMessage("Received SIGINT!"))
      self.sigGEN_handler(signalNumber,frame)


   def sigHUP_handler(self,
                      signalNumber,
                      frame):
      self.logger.log(logging.INFO,getLogMessage("Received SIGHUP!"))
      self.sigGEN_handler(signalNumber,frame)


   def sigQUIT_handler(self,
                       signalNumber,
                       frame):
      self.logger.log(logging.INFO,getLogMessage("Received SIGQUIT!"))
      self.sigGEN_handler(signalNumber,frame)


   def sigABRT_handler(self,
                       signalNumber,
                       frame):
      self.logger.log(logging.INFO,getLogMessage("Received SIGABRT!"))
      self.sigGEN_handler(signalNumber,frame)


   def sigTERM_handler(self,
                       signalNumber,
                       frame):
      self.logger.log(logging.INFO,getLogMessage("Received SIGTERM!"))
      self.sigGEN_handler(signalNumber,frame)


   def process(self):
      self.logger.log(logging.INFO,getLogMessage("*************************"))
      self.logger.log(logging.INFO,getLogMessage("* job processor started *"))
      self.logger.log(logging.INFO,getLogMessage("*************************"))

      while True:
         if self.terminateJobProcessor:
            self.jobProcessor.terminate()
            self.logger.log(logging.INFO,getLogMessage("*************************"))
            self.logger.log(logging.INFO,getLogMessage("* job processor stopped *"))
            self.logger.log(logging.INFO,getLogMessage("*************************"))
            break

         nReleasedJobs = self.jobProcessor.releaseRegisteredJobs()
         if nReleasedJobs > 0:
            self.logger.log(logging.INFO,getLogMessage("%d jobs released" % (nReleasedJobs)))

         nPostedJobs = self.jobProcessor.postEnrolledWSJobs()
         if nPostedJobs > 0:
            self.logger.log(logging.INFO,getLogMessage("%d jobs posted" % (nPostedJobs)))

         self.jobProcessor.markActiveJobsAsComplete()

         time.sleep(self.timeBetweenJobRelease)


if __name__ == '__main__':

   fdLogFile = openLogger(PROCESSORLOGLOCATION,PROCESSORLOGFILENAME)
   daemonize(fdLogFile)

   writePidFile(PIDFILE)

   __processJobs__ = ProcessJobs(CONFIGURATIONDIRECTORY,
                                 DAEMONSCONFIGURATIONFILE)
   __processJobs__.process()


