#!/usr/bin/env python
#
# @package      hubzero-submit-monitors
# @file         monitorProbeR.py
# @copyright    Copyright (c) 2004-2020 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2004-2020 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#
# ----------------------------------------------------------------------
#  monitorProbeR.py
#
#  script which reports monitor status for all probed sites
#
import os
import sys
import time
import logging

from hubzero.submit.DaemonsInfo        import DaemonsInfo
from hubzero.submit.RemoteProbeMonitor import RemoteProbeMonitor

CONFIGURATIONDIRECTORY   = os.path.join(os.sep,'etc','submit')
DAEMONSCONFIGURATIONFILE = 'daemons.conf'

APPLICATIONLOGGER = logging.getLogger('') 

def openLogger():
   class EmptyFilter(logging.Filter):
      """
      This is a filter which rejects empty messages

      """

      def filter(self,record):
         if record.getMessage() == "":
            emptyRecord = True
         else:
            emptyRecord = False

         return(not emptyRecord)

   APPLICATIONLOGGER.setLevel(logging.DEBUG)

   logHandler = logging.StreamHandler()

   emptyFilter = EmptyFilter()
   logHandler.addFilter(emptyFilter)

   logFormatter = logging.Formatter('%(asctime)s %(message)s','[%a %b %d %H:%M:%S %Y]')
   logHandler.setFormatter(logFormatter)
   APPLICATIONLOGGER.addHandler(logHandler)


def monitorProbeR(listenURI):
   """Report active probes"""
   remoteProbeMonitor = RemoteProbeMonitor(listenURI)

   reportSites = sys.argv[1:]

   response = remoteProbeMonitor.reportProbeSiteStatus()
   reportTime = int(time.time())

   if len(response) > 0:
      siteResponses = []
      for site in response:
         siteResponses.append(','.join([site,str(response[site]['timeEvaluated']),
                                             response[site]['status'],
                                             str(response[site]['waitRank'])]))

#     Sort by FAIL/PASS, score, age of score
      siteResponses.sort(key=lambda x:(x.split(',')[2],int(x.split(',')[3]),int(x.split(',')[1])))

      try:
         print("%-30s  %s  %s  %9s" % ('Site','State ','Speed','Age(min)'))
         print("%-30s  %s  %s  %9s" % (30*'-','------','-----',9*'-'))
         for siteResponse in siteResponses:
            site,scoreTime,state,score = siteResponse.split(',')
            if (len(reportSites) == 0) or (site in reportSites):
               print("%-30s  %s  %5d  %9.1f" % (site,state,int(score),(reportTime-int(scoreTime))/60.))
      except IOError:
         pass


if __name__ == '__main__':
   openLogger()

   configFilePath = os.path.join(CONFIGURATIONDIRECTORY,DAEMONSCONFIGURATIONFILE)
   daemonsInfo    = DaemonsInfo(configFilePath)
   listenURI      = daemonsInfo.getDaemonListenURI('probeMonitor','tcp')

   monitorProbeR(listenURI)


