# @package      hubzero-submit-monitors
# @file         MonitorsInfo.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2012-2013 HUBzero Foundation, LLC.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2012-2013 HUBzero Foundation, LLC.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#

import sys
import os.path
import pwd
import re
import socket
import select
import subprocess
import shlex
import traceback
from errno import EPIPE

from hubzero.submit.LogMessage import logJobId as log

class MonitorsInfo:
   def __init__(self,
                infoDirectory,
                monitorsFile):
      self.monitors   = {}
      self.childPid   = 0
      self.bufferSize = 4096

      monitorPattern  = re.compile('(\s*\[)([^\s]*)(]\s*)')
      keyValuePattern = re.compile('( *)(\w*)( *= *)(.*[^\s$])( *)')
      commentPattern  = re.compile('\s*#.*')
      monitorName     = ""

      monitorsPath = os.path.join(infoDirectory,monitorsFile)
      if os.path.exists(monitorsPath):
         fpInfo = open(monitorsPath,'r')
         if fpInfo:
            eof = False
            while not eof:
               record = fpInfo.readline()
               if record != "":
                  record = commentPattern.sub("",record)
                  if   monitorPattern.match(record):
                     monitorName = monitorPattern.match(record).group(2)
                     self.monitors[monitorName] = {'venue':'',
                                                   'gsiHost':socket.gethostname(),
                                                   'venueMechanism':'ssh',
                                                   'tunnelDesignator':'',
                                                   'remoteUser':'',
                                                   'x509ProxyPath':'',
                                                   'remoteBinDirectory':'',
                                                   'remoteMonitorCommand':'',
                                                   'state':'enabled'
                                                  }
                  elif keyValuePattern.match(record):
                     key,value = keyValuePattern.match(record).group(2,4)
                     if key in self.monitors[monitorName]:
                        if   isinstance(self.monitors[monitorName][key],list):
                           self.monitors[monitorName][key] = [e.strip() for e in value.split(',')]
                        elif isinstance(self.monitors[monitorName][key],bool):
                           self.monitors[monitorName][key] = bool(value.lower() == 'true')
                        else:
                           self.monitors[monitorName][key] = value
                     else:
                        log("Undefined key = value pair %s = %s for site %s" % (key,value,monitorName))
               else:
                  eof = True
            fpInfo.close()

            markedForDeletion = []
            for monitorName in self.monitors:
               if self.monitors[monitorName]['state'] != 'enabled':
                  markedForDeletion.append(monitorName)
            for monitorName in markedForDeletion:
               del self.monitors[monitorName]
            del markedForDeletion
      else:
         log("Monitors configuration file %s is missing" % (monitorsPath))


   @staticmethod
   def __writeToStdout(message):
      try:
         sys.stdout.write(message)
         sys.stdout.flush()
      except IOError,err:
         if not err[0] in [EPIPE]:
            log("Can't write to stdout: %s" % (message))


   @staticmethod
   def __writeToStderr(message):
      try:
         sys.stderr.write(message)
         sys.stderr.flush()
      except IOError,err:
         if not err[0] in [EPIPE]:
            log("Can't write to stderr: %s" % (message))


   def __executeCommand(self,
                        command,
                        streamOutput=False):
      commandArgs = shlex.split(command)
      child = subprocess.Popen(commandArgs,bufsize=self.bufferSize,
                               stdout=subprocess.PIPE,
                               stderr=subprocess.PIPE,
                               close_fds=True)
      self.childPid = child.pid
      childout      = child.stdout
      childoutFd    = childout.fileno()
      childerr      = child.stderr
      childerrFd    = childerr.fileno()

      outEOF = False
      errEOF = False

      outData = []
      errData = []

      while True:
         toCheck = []
         if not outEOF:
            toCheck.append(childoutFd)
         if not errEOF:
            toCheck.append(childerrFd)
         try:
            ready = select.select(toCheck,[],[]) # wait for input
         except select.error,err:
            ready = {}
            ready[0] = []
         if childoutFd in ready[0]:
            outChunk = os.read(childoutFd,self.bufferSize)
            if outChunk == '':
               outEOF = True
            outData.append(outChunk)
            if streamOutput:
               self.__writeToStdout(outChunk)

         if childerrFd in ready[0]:
            errChunk = os.read(childerrFd,self.bufferSize)
            if errChunk == '':
               errEOF = True
            errData.append(errChunk)
            if streamOutput:
               self.__writeToStderr(errChunk)

         if outEOF and errEOF:
            break

      pid,err = os.waitpid(self.childPid,0)
      self.childPid = 0
      if err != 0:
         if   os.WIFSIGNALED(err):
            log("%s failed w/ signal %d" % (command,os.WTERMSIG(err)))
         else:
            if os.WIFEXITED(err):
               err = os.WEXITSTATUS(err)
            log("%s failed w/ exit code %d" % (command,err))
         if not streamOutput:
            log("%s" % ("".join(errData)))

      return(err,"".join(outData),"".join(errData))


   def getSSHCommand(self,
                     monitorNameInput,
                     sshIdentity,
                     remoteTunnelMonitor):
      sshCommand       = ""
      tunnelDesignator = ""

      monitorName = ""
      if monitorNameInput in self.monitors:
         monitorName = monitorNameInput
      else:
         for monitorInfoName in self.monitors:
            if re.match(monitorInfoName,monitorNameInput):
               self.monitors[monitorNameInput] = {}
               self.monitors[monitorNameInput]['venue']                = self.monitors[monitorInfoName]['venue'].replace('@name@',monitorNameInput)
               self.monitors[monitorNameInput]['venueMechanism']       = self.monitors[monitorInfoName]['venueMechanism']
               self.monitors[monitorNameInput]['tunnelDesignator']     = self.monitors[monitorInfoName]['tunnelDesignator']
               self.monitors[monitorNameInput]['remoteUser']           = self.monitors[monitorInfoName]['remoteUser']
               self.monitors[monitorNameInput]['x509ProxyPath']        = self.monitors[monitorInfoName]['x509ProxyPath']
               self.monitors[monitorNameInput]['remoteMonitorCommand'] = self.monitors[monitorInfoName]['remoteMonitorCommand']
               self.monitors[monitorNameInput]['state']                = self.monitors[monitorInfoName]['state']
               monitorName = monitorNameInput
               break

      if monitorName:
         try:
            if   self.monitors[monitorName]['venueMechanism'] == 'ssh':
               venue                = self.monitors[monitorName]['venue']
               tunnelDesignator     = self.monitors[monitorName]['tunnelDesignator']
               user                 = self.monitors[monitorName]['remoteUser']
               remoteMonitorCommand = self.monitors[monitorName]['remoteMonitorCommand']
   
               sshBaseCommand = "ssh -T -x -a -i " + sshIdentity
               if tunnelDesignator == "":
                  sshCommand = sshBaseCommand + " " + user + "@" + venue + " '" + remoteMonitorCommand + "'"
               else:
                  tunnelAddress,tunnelPort = remoteTunnelMonitor.getTunnelAddressPort(tunnelDesignator)
                  sshCommand = sshBaseCommand + " -p " + tunnelPort + " " + user + "@" + tunnelAddress + \
                                                                      " '" + remoteMonitorCommand + "'"
            elif self.monitors[monitorName]['venueMechanism'] == 'gsissh':
               gsiHost = self.monitors[monitorName]['gsiHost']
               if gsiHost != socket.gethostname():
                  venue                = self.monitors[monitorName]['venue']
                  user                 = self.monitors[monitorName]['remoteUser']
                  remoteMonitorCommand = self.monitors[monitorName]['remoteMonitorCommand']
                  remoteBinDirectory   = self.monitors[monitorName]['remoteBinDirectory']
                  if user.startswith('USER:'):
                     keyfamily = user.split(':')[1]
                     user = pwd.getpwuid(os.getuid())[0]
                     command = 'genuserpki ' + keyfamily
                     log("command = " + command)
                     exitStatus,stdOutput,stdError = self.__executeCommand(command)
                     if not exitStatus:
                        sshIdentityPath = stdOutput.strip()
                        log("IDENTITY = " + sshIdentityPath)
                        sshBaseCommand = "ssh -T -x -a -i " + sshIdentityPath
                        sshCommand = sshBaseCommand + " " + user + "@" + gsiHost + \
                                     " '" + os.path.join(remoteBinDirectory,'monitorJob.sh') + " " + \
                                                                                       venue + " " + \
                                                                              remoteMonitorCommand + "'"
               else:
                  venue                = self.monitors[monitorName]['venue']
                  x509ProxyPath        = self.monitors[monitorName]['x509ProxyPath']
                  remoteMonitorCommand = self.monitors[monitorName]['remoteMonitorCommand']
    
                  sshBaseCommand = "X509_USER_PROXY=" + x509ProxyPath + " gsissh -T -x -a "
                  sshCommand = sshBaseCommand + " " + venue + " '" + remoteMonitorCommand + "'"
         except:
            log("Build sshCommand failed for %s" % (monitorNameInput))
            log(traceback.format_exc())
      else:
         log("Could not build sshCommand for %s.\nPerhaps the Site Monitor needs to be reloaded!" % (monitorNameInput))
   
      return(sshCommand,tunnelDesignator)


