# @package      hubzero-submit-monitors
# @file         ProbeMonitor.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2012-2013 HUBzero Foundation, LLC.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2012-2013 HUBzero Foundation, LLC.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#

import re

from hubzero.submit.LogMessage  import logID as log
from hubzero.submit.MessageCore import MessageCore

class ProbeMonitor(MessageCore):
   def __init__(self,
                host,
                port,
                repeatDelay=5,
                fixedBufferSize=128,
                activeProbeDumpPath="monitorProbe.dump"):
      bindLabel = "%s:%d" % (host if host else "localhost",port)
      MessageCore.__init__(self,bindHost=host,bindPort=port,bindLabel=bindLabel,repeatDelay=repeatDelay)
      self.fixedBufferSize     = fixedBufferSize
      self.activeProbeDumpPath = activeProbeDumpPath
      self.activeProbes        = {}


   def loadActiveProbes(self):
      dumpedProbes = []
      try:
         dumpFile = open(self.activeProbeDumpPath,"r")
         dumpedProbes = dumpFile.readlines()
         dumpFile.close()
      except:
         pass

      for dumpedProbe in dumpedProbes:
         site,timeEvaluated,status,waitRank = dumpedProbe.split()
         self.activeProbes[site] = (timeEvaluated,status,waitRank)


   def dumpActiveProbes(self):
      dumpFile = open(self.activeProbeDumpPath,'w')
      for activeProbe in self.activeProbes.items():
         dumpFile.write("%s\t%s\t%s\t%s\n" % (activeProbe[0],activeProbe[1][0],activeProbe[1][1],activeProbe[1][2]))
      dumpFile.close()


   def processRequest(self,
                      channel):
      channelClosed = False

      message = self.receiveMessage(channel,self.fixedBufferSize)
      if message:
         if re.match('[QUR]:',message):
            try:
               messageType,request = message.split(':')
            except:
               log("Failed QUR message request: " + message)
         else:
            log("Failed message request: " + message)
            messageType = ""

         if   messageType == 'Q':                        # site Query
            site = request.strip()
            if site in self.activeProbes:
               timeEvaluated,status,waitRank = self.activeProbes[site]
            else:
               timeEvaluated,status,waitRank = ("0","FAILED","0")
            self.sendMessage(channel,timeEvaluated + " " + status + " " + waitRank,self.fixedBufferSize)
         elif messageType == 'U':                        # update site status
            site,timeEvaluated,status,sWaitTime = request.split(',')
            waitTime = float(sWaitTime)/60.
            if   waitTime <= 5.:
               waitRank = 100
            elif waitTime <= 10.:
               waitRank = 90
            elif waitTime <= 15.:
               waitRank = 80
            elif waitTime <= 30.:
               waitRank = 70
            elif waitTime <= 60.:
               waitRank = 60
            elif waitTime <= 120.:
               waitRank = 50
            elif waitTime <= 180.:
               waitRank = 40
            elif waitTime <= 240.:
               waitRank = 30
            elif waitTime <= 300.:
               waitRank = 20
            elif waitTime <= 360.:
               waitRank = 10
            else:
               waitRank = 0
            status = status.strip()
            self.activeProbes[site] = (timeEvaluated,status,str(waitRank))
            self.sendMessage(channel,','.join([site,timeEvaluated,status,sWaitTime]),self.fixedBufferSize)
            self.dumpActiveProbes()
         elif messageType == 'R':                        # Report active probes
            report = ""
            for site in self.activeProbes:
               timeEvaluated,status,waitRank = self.activeProbes[site]
               if report == "":
                  report = ','.join([site,timeEvaluated,status,waitRank])
               else:
                  report += ":" + ','.join([site,timeEvaluated,status,waitRank])
            reportLength = len(report)
            if self.sendMessage(channel,str(reportLength),self.fixedBufferSize) > 0:
               if int(reportLength) > 0:
                  self.sendMessage(channel,report)
      else:
         try:
            channel.close()
            channelClosed = True
         except:
            log("close channel failed")

      return(channelClosed)


