#
# @package      hubzero-submit-distributor
# @file         Scripts/tapis3/uploadJobInputs.py
# @copyright    Copyright 2024-2025 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2024-2025 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#
import sys
import os
import time
import tarfile
import tempfile
import shutil

from TapisAPI import TapisAPI
from TapisAPI import BaseTapyException

TIMESTAMPTRANSFERRED = ".__timestamp_transferred"

exitStatus = 0

tapisAPI = TapisAPI()

systemId         = sys.argv[1]
instanceToken    = sys.argv[2]
jobInputTarPath  = sys.argv[3]
auxiliaryTarFile = sys.argv[4]

directory = "/JobInputs/%s" % (instanceToken)
jobInputTarFile = os.path.basename(jobInputTarPath)
localJobId,instanceId = jobInputTarFile.split('_')[0:2]

if len(sys.argv) > 5:
   timestampTransferred = os.path.basename(sys.argv[5])
else:
   timestampTransferred = "%s.%s_%s" % (TIMESTAMPTRANSFERRED,localJobId,instanceId)

try:
   uploadFileResponse = tapisAPI.upload(source_file_path=jobInputTarPath,
                                        system_id=systemId,
                                        dest_file_path=os.path.join(directory,jobInputTarFile))
except BaseTapyException as err:
   exitStatus = 1
   sys.stderr.write(err.message)
   sys.stderr.write("\n")
else:
   if uploadFileResponse:
      print(uploadFileResponse['status'])

if exitStatus == 0:
   try:
      tmpDirectory = tempfile.mkdtemp(prefix="prestage_",dir=os.path.join(os.sep,'tmp'))
   except:
      tmpDirectory = None
      exitStatus = 1
      sys.stderr.write("TMP directory creation failed\n")
   else:
      currentDirectory = os.getcwd()
      try:
         os.chdir(tmpDirectory)
      except:
         exitStatus = 1
         sys.stderr.write("cd to TMP directory creation failed\n")
      else:
         now = int(time.time())
         with open(timestampTransferred,'w') as fp:
            fp.write("%d\n" % (now+1))

         tarFile = "%s_%s_timetransferred.tar.gz" % (localJobId,instanceId)
         fpTarFile = tarfile.open(tarFile,mode="w:gz")
         fpTarFile.add(timestampTransferred)
         fpTarFile.close()

         try:
            uploadFileResponse = tapisAPI.upload(source_file_path=tarFile,
                                                 system_id=systemId,
                                                 dest_file_path=os.path.join(directory,tarFile))
         except BaseTapyException as err:
            exitStatus = 1
            sys.stderr.write(err.message)
            sys.stderr.write("\n")
         else:
            if uploadFileResponse:
               print(uploadFileResponse['status'])
      finally:
         os.chdir(currentDirectory)
         if tmpDirectory:
            shutil.rmtree(tmpDirectory,True)

if exitStatus == 0:
   if auxiliaryTarFile:
      if os.path.exists(auxiliaryTarFile):
         try:
            uploadFileResponse = tapisAPI.upload(source_file_path=auxiliaryTarFile,
                                                 system_id=systemId,        
                                                 dest_file_path=os.path.join(directory,os.path.basename(auxiliaryTarFile)))
         except BaseTapyException as err:
            exitStatus = 1
            sys.stderr.write(err.message)
            sys.stderr.write("\n")
         else:
            if uploadFileResponse:
               print(uploadFileResponse['status'])

sys.exit(exitStatus)
