#!/usr/bin/env python
#
# @package      hubzero-submit-distributor
# @file         venueProbeR.py
# @copyright    Copyright (c) 2015-2020 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2015-2020 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#
"""Report probe status for one or more venues.
   """
import os
import sys
import re
import time
import logging
from logging.handlers import SocketHandler,SysLogHandler

from hubzero.submit.LogMessage         import getLogIDMessage as getLogMessage, logSetJobId
from hubzero.submit.CommandParser      import CommandParser
from hubzero.submit.DaemonsInfo        import DaemonsInfo
from hubzero.submit.InfosInfo          import InfosInfo
from hubzero.submit.SitesInfo          import SitesInfo
from hubzero.submit.RemoteProbeMonitor import RemoteProbeMonitor

APPLICATIONLOGGER = logging.getLogger('')

CONFIGURATIONDIRECTORY       = os.path.join(os.sep,'etc','submit')
DISTRIBUTORCONFIGURATIONFILE = 'distributor.conf'
DAEMONSCONFIGURATIONFILE     = 'daemons.conf'
INFOSCONFIGURATIONFILE       = 'infos.conf'

def openLogger():
   class EmptyFilter(logging.Filter):
      """
      This is a filter which rejects empty messages

      """

      def filter(self,record):
         if record.getMessage() == "":
            emptyRecord = True
         else:
            emptyRecord = False

         return(not emptyRecord)

   APPLICATIONLOGGER.setLevel(logging.DEBUG)

   logHandler = logging.StreamHandler()

   emptyFilter = EmptyFilter()
   logHandler.addFilter(emptyFilter)

   logFormatter = logging.Formatter('%(asctime)s %(message)s','[%a %b %d %H:%M:%S %Y]')
   logHandler.setFormatter(logFormatter)
   APPLICATIONLOGGER.addHandler(logHandler)


class VenueProbe():
   def __init__(self,
                configurationDirectory,
                distributorConfigurationFile,
                daemonsConfigurationFile,
                infosConfigurationFile):
      self.logger = logging.getLogger(__name__)

      self.configurationDirectory       = configurationDirectory
      self.distributorConfigurationFile = distributorConfigurationFile
      self.daemonsConfigurationFile     = daemonsConfigurationFile
      self.infosConfigurationFile       = infosConfigurationFile

      self.configData     = {}
      self.dataDirectory  = ""

      self.daemonInfo = None
      self.infosInfo  = None
      self.sitesInfo  = None

      self.commandParser        = None
      self.doubleDashTerminator = bool(os.getenv("DOUBLE_DASH_TERMINATOR",'0'))
      self.pegasusVersion       = "4.x.x"

      self.remoteProbeMonitor = None

      self.statusVenues = []


   def configure(self):
      sectionPattern  = re.compile('(\s*\[)([^\s]*)(]\s*)')
      keyValuePattern = re.compile('( *)(\w*)( *= *)(.*[^\s$])( *)')
      commentPattern  = re.compile('\s*#.*')
      inDistributorSection = False

      configured = False
      configFilePath = os.path.join(self.configurationDirectory,self.distributorConfigurationFile)
      try:
         fpConfig = open(configFilePath,'r')
         try:
            eof = False
            while not eof:
               record = fpConfig.readline()
               if record != "":
                  record = commentPattern.sub("",record)
                  if   sectionPattern.match(record):
                     sectionName = sectionPattern.match(record).group(2)
                     inDistributorSection = (sectionName == 'distributor')
                     if inDistributorSection:
                        self.configData = {'probeMonitoringInstalled':False,
                                           'maximumSelectedSites':1,
                                           'allowedVenueMechanisms':['local','ssh'],
                                           'dataDirectory':os.path.join(os.sep,'opt','submit'),
                                           'binDirectory':os.path.join(os.sep,'opt','submit','bin'),
                                           'condorRoot':'',
                                           'condorConfig':'',
                                           'pbsRoot':''
                                          }
                  elif inDistributorSection:
                     if keyValuePattern.match(record):
                        key,value = keyValuePattern.match(record).group(2,4)
                        if key in self.configData:
                           if   isinstance(self.configData[key],list):
                              self.configData[key] = [e.strip() for e in value.split(',')]
                           elif isinstance(self.configData[key],bool):
                              self.configData[key] = bool(value.lower() == 'true')
                           elif isinstance(self.configData[key],float):
                              self.configData[key] = float(value)
                           elif isinstance(self.configData[key],int):
                              self.configData[key] = int(value)
                           elif isinstance(self.configData[key],dict):
                              try:
                                 sampleKey   = self.configData[key].keys()[0]
                                 sampleValue = self.configData[key][sampleKey]
                              except:
                                 sampleKey   = "key"
                                 sampleValue = "value"
                              self.configData[key] = {}
                              for e in value.split(','):
                                 dictKey,dictValue = e.split(':')
                                 if isinstance(sampleKey,int):
                                    dictKey = int(dictKey)
                                 if   isinstance(sampleValue,int):
                                    dictValue = int(dictValue)
                                 elif isinstance(sampleValue,float):
                                    dictValue = float(dictValue)
                                 elif isinstance(sampleValue,bool):
                                    dictValue = bool(dictValue.lower() == 'true')
                                 self.configData[key][dictKey] = dictValue
                           else:
                              self.configData[key] = value
                        else:
                           self.logger.log(logging.WARNING,getLogMessage("Undefined key = value pair %s = %s" % (key,value)))
               else:
                  eof = True
                  configured = True
         except (IOError,OSError):
            self.logger.log(logging.ERROR,getLogMessage("%s could not be read" % (configFilePath)))
         finally:
            fpConfig.close()
      except (IOError,OSError):
         self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (configFilePath)))

      if configured:
         if os.path.isdir(self.configData['dataDirectory']):
            self.dataDirectory = self.configData['dataDirectory']
         else:
            message = "Specified dataDirectory does not exist: %s" % (self.configData['dataDirectory'])
            self.logger.log(logging.ERROR,getLogMessage(message))
            configured = False

      return(configured)


   def setup(self):
      configFilePath = os.path.join(self.configurationDirectory,self.infosConfigurationFile)
      self.infosInfo = InfosInfo(configFilePath)

      self.sitesInfo = SitesInfo(self.infosInfo.getInfoPath('sites'),
                                 restrictionUser=os.getenv("USER"),
                                 templateDirectory=self.dataDirectory,
                                 allowedVenueMechanisms=self.configData['allowedVenueMechanisms'],
                                 pegasusVersion=self.pegasusVersion)

      configFilePath   = os.path.join(self.configurationDirectory,self.daemonsConfigurationFile)
      self.daemonsInfo = DaemonsInfo(configFilePath)
      listenURI = self.daemonsInfo.getDaemonListenURI('probeMonitor','tcp')
      self.remoteProbeMonitor = RemoteProbeMonitor(listenURI)


   def parseCommandArguments(self):
      exitCode = 0
      self.commandParser = CommandParser(self.doubleDashTerminator)
      self.commandParser.parseArguments(sys.argv[1:])
      self.operationMode = self.commandParser.getOperationMode()

      if self.operationMode & self.commandParser.OPERATIONMODERUNVENUESTATUS:
         enabledSites = self.sitesInfo.getEnabledSites()
         statusVenues = self.commandParser.getOption('statusVenues')
         if len(statusVenues) > 0:
            for statusVenue in statusVenues:
               if statusVenue in enabledSites:
                  self.statusVenues.append(statusVenue)
         else:
            self.statusVenues = enabledSites
      else:
         exitCode = 1

      return(exitCode)


   def report(self):
      """Report active probes"""
      exitCode = 1
      response = self.remoteProbeMonitor.reportProbeSiteStatus()
      reportTime = int(time.time())

      if len(response) > 0:
         siteResponses = []
         for venue in response:
            siteResponses.append(','.join([venue,str(response[venue]['timeEvaluated']),
                                                 response[venue]['status'],
                                                 str(response[venue]['waitRank'])]))

#        Sort by FAIL/PASS, score, age of score
         siteResponses.sort(key=lambda x:(x.split(',')[2],int(x.split(',')[3]),int(x.split(',')[1])))

         headerPrinted = False
         for siteResponse in siteResponses:
            venue,scoreTime,state,score = siteResponse.split(',')
            for statusVenue in self.statusVenues:
               probeResultSiteName,probeResult = self.sitesInfo.getProbeResult(statusVenue,self.remoteProbeMonitor)
               if probeResultSiteName == venue:
                  if not headerPrinted:
                     print("%-30s  %6s  %5s  %9s" % ('Venue','Status','Score','Age(min)'))
                     print("%s--%s--%s--%s" % ('-'*30,'-'*6,'-'*5,'-'*9))
                     headerPrinted = True
                  print("%-30s  %s  %5d  %9.1f" % (statusVenue,state,int(score),(reportTime-int(scoreTime))/60.))
                  if state == 'PASSED':
                     exitCode = 0

      return(exitCode)


if __name__ == '__main__':
   openLogger()

   exitCode = 1
   venueProbe = VenueProbe(CONFIGURATIONDIRECTORY,
                           DISTRIBUTORCONFIGURATIONFILE,
                           DAEMONSCONFIGURATIONFILE,
                           INFOSCONFIGURATIONFILE)

   if venueProbe.configure():
      venueProbe.setup()
      exitCode = venueProbe.parseCommandArguments()
      if exitCode:
         sys.exit(exitCode)

      exitCode = venueProbe.report()

   sys.exit(exitCode)


