# @package      hubzero-submit-distributor
# @file         RemoteBatchLSF.py
# @copyright    Copyright (c) 2012-2020 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2012-2020 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#
import os.path
import re
import logging

from hubzero.submit.LogMessage        import getLogJobIdMessage as getLogMessage 
from hubzero.submit.ParameterTemplate import ParameterTemplate
from hubzero.submit.TimeConversion    import minTohhmmss

class RemoteBatchLSF:
   SUBMISSIONSCRIPTCOMMANDPREFIX = '#BSUB'

   def __init__(self,
                hubUserName,
                hubUserId,
                submitterClass,
                runName,
                localJobId,
                instanceId,
                appScriptName,
                environment,
                transferExecutable,
                executable,
                arguments,
                stdinput,
                isMultiCoreRequest,
                siteInfo,
                toolFilesInfo,
                dockerImageInfo,
                submissionScriptsInfo,
                managerInfo,
                nNodes,
                ppn,
                wallTime,
                quotaLimit,
                timeHistoryLogs):
      self.logger                   = logging.getLogger(__name__)
      self.hubUserName              = hubUserName
      self.hubUserId                = hubUserId
      self.submitterClass           = submitterClass
      self.runName                  = runName
      self.localJobId               = localJobId
      self.instanceId               = instanceId
      self.appScriptName            = appScriptName
      self.environment              = environment
      self.stageFiles               = siteInfo['stageFiles']
      self.submissionScriptsInfo    = submissionScriptsInfo
      self.transferExecutable       = transferExecutable
      if transferExecutable:
         if self.stageFiles:
            self.executable         = os.path.join(".",os.path.basename(executable))
         else:
            self.executable         = executable
      else:
         if not executable.startswith('/') and not executable.startswith('$') and siteInfo['remoteApplicationRootDirectory'] != "":
            self.executable         = os.path.join(siteInfo['remoteApplicationRootDirectory'],executable)
         else:
            self.executable         = executable
      self.arguments                = arguments
      self.stdinput                 = stdinput
      self.isMultiCoreRequest       = isMultiCoreRequest
      self.computationMode          = managerInfo['computationMode']
      self.preManagerCommands       = managerInfo['preManagerCommands']
      self.managerCommand           = managerInfo['managerCommand']
      self.postManagerCommands      = managerInfo['postManagerCommands']
      self.nNodes                   = nNodes
      self.ppn                      = ppn
      self.wallTime                 = minTohhmmss(wallTime)
      if siteInfo['remoteBatchAccount'] != "":
         self.account               = self.SUBMISSIONSCRIPTCOMMANDPREFIX + ' -P ' + siteInfo['remoteBatchAccount']
      else:
         self.account               = ''
      self.submissionScriptCommands = siteInfo['submissionScriptCommands']
      self.toolFilesInfo            = toolFilesInfo
      self.dockerImageInfo          = dockerImageInfo
      self.quotaLimit               = quotaLimit
      self.timestampStart           = timeHistoryLogs['timestampStart']
      self.timestampFinish          = timeHistoryLogs['timestampFinish']
      self.timeResults              = timeHistoryLogs['timeResults']

      self.nodeFileName = ""
      self.nodeList     = []

      self.toolInputTemplateFileName  = ""
      self.toolInputTemplate          = ""
      self.toolOutputTemplateFileName = ""
      self.toolOutputTemplate         = ""


   def __buildSerialFile(self):
      rawSubmissionScript = self.submissionScriptsInfo.getSubmissionScript('Batch','LSF','serial')

      commandsLSF = ""
      if self.submissionScriptCommands:
         if self.SUBMISSIONSCRIPTCOMMANDPREFIX:
            commandSeparator = "\n%s " % self.SUBMISSIONSCRIPTCOMMANDPREFIX
            commandsLSF = self.SUBMISSIONSCRIPTCOMMANDPREFIX + " " + commandSeparator.join(self.submissionScriptCommands)
         else:
            commandSeparator = "\n"
            commandsLSF = commandSeparator.join(self.submissionScriptCommands)

      environmentExport = ""
      environmentVars = self.environment.split()
      for environmentVar in environmentVars:
         environmentExport += "export " + environmentVar + "\n"

      substitutions = {}
      if self.stdinput == "":
         substitutions["STDIN"]                 = '/dev/null'
      else:
         substitutions["STDIN"]                 = self.stdinput
      substitutions["WALLTIME"]                 = self.wallTime
      substitutions["ENVIRONMENT"]              = environmentExport
      substitutions["EXECUTABLE"]               = self.executable
      substitutions["ARGUMENTS"]                = self.arguments
      substitutions["TS_START"]                 = self.timestampStart
      substitutions["TS_FINISH"]                = self.timestampFinish
      substitutions["TIME_RESULTS"]             = self.timeResults
      substitutions["RUNNAME"]                  = self.runName
      substitutions["JOBID"]                    = self.localJobId
      substitutions["INSTANCEID"]               = self.instanceId
      substitutions["NNODES"]                   = self.nNodes
      substitutions["PPN"]                      = self.ppn
      substitutions["REMOTEBATCHACCOUNT"]       = self.account
      substitutions["SUBMISSIONSCRIPTCOMMANDS"] = commandsLSF
      substitutions["PREMANAGERCOMMANDS"]       = "\n".join(self.preManagerCommands)
      substitutions["POSTMANAGERCOMMANDS"]      = "\n".join(self.postManagerCommands)
      substitutions["HUBUSERNAME"]              = self.hubUserName
      substitutions["HUBUSERID"]                = str(self.hubUserId)
      if self.quotaLimit > 0:
         substitutions["QUOTA"]                 = "ulimit -f " + str(self.quotaLimit)
      else:
         substitutions["QUOTA"]                 = ""

      template = ParameterTemplate(rawSubmissionScript)
      try:
         submissionScript = template.substitute_recur(substitutions)
      except KeyError,err:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Pattern substitution failed for @@%s\n" % (err[0])))
      except TypeError,err:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Submission script substitution failed:\n%s\n" % (rawSubmissionScript)))
      else:
         if self.SUBMISSIONSCRIPTCOMMANDPREFIX:
            submissionScript = re.sub("(\n)*\n%s" % (self.SUBMISSIONSCRIPTCOMMANDPREFIX),
                                           "\n%s" % (self.SUBMISSIONSCRIPTCOMMANDPREFIX),submissionScript)

      return(submissionScript)


   def __buildMPIFile(self):
      rawSubmissionScript = self.submissionScriptsInfo.getSubmissionScript('Batch','LSF','mpi')

      commandsLSF = ""
      if self.submissionScriptCommands:
         if self.SUBMISSIONSCRIPTCOMMANDPREFIX:
            commandSeparator = "\n%s " % self.SUBMISSIONSCRIPTCOMMANDPREFIX
            commandsLSF = self.SUBMISSIONSCRIPTCOMMANDPREFIX + " " + commandSeparator.join(self.submissionScriptCommands)
         else:
            commandSeparator = "\n"
            commandsLSF = commandSeparator.join(self.submissionScriptCommands)

      environmentExport = ""
      environmentVars = self.environment.split()
      for environmentVar in environmentVars:
         environmentExport += "export " + environmentVar + "\n"

      substitutions = {}
      substitutions["PREMANAGERCOMMANDS"]       = "\n".join(self.preManagerCommands)
      substitutions["MANAGERCOMMAND"]           = self.managerCommand
      substitutions["POSTMANAGERCOMMANDS"]      = "\n".join(self.postManagerCommands)
      substitutions["WALLTIME"]                 = self.wallTime
      substitutions["NNODES"]                   = self.nNodes
      substitutions["PPN"]                      = self.ppn
      substitutions["ENVIRONMENT"]              = environmentExport
      substitutions["APPSCRIPTNAME"]            = self.appScriptName
      substitutions["TS_START"]                 = self.timestampStart
      substitutions["TS_FINISH"]                = self.timestampFinish
      substitutions["TIME_RESULTS"]             = self.timeResults
      substitutions["RUNNAME"]                  = self.runName
      substitutions["JOBID"]                    = self.localJobId
      substitutions["INSTANCEID"]               = self.instanceId
      substitutions["REMOTEBATCHACCOUNT"]       = self.account
      substitutions["SUBMISSIONSCRIPTCOMMANDS"] = commandsLSF
      substitutions["HUBUSERNAME"]              = self.hubUserName
      substitutions["HUBUSERID"]                = str(self.hubUserId)
      if self.quotaLimit > 0:
         substitutions["QUOTA"]                 = "ulimit -f " + str(self.quotaLimit)
      else:
         substitutions["QUOTA"]                 = ""

      template = ParameterTemplate(rawSubmissionScript)
      try:
         submissionScript = template.substitute_recur(substitutions)
      except KeyError,err:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Pattern substitution failed for @@%s\n" % (err[0])))
      except TypeError,err:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Submission script substitution failed:\n%s\n" % (rawSubmissionScript)))
      else:
         if self.SUBMISSIONSCRIPTCOMMANDPREFIX:
            submissionScript = re.sub("(\n)*\n%s" % (self.SUBMISSIONSCRIPTCOMMANDPREFIX),
                                           "\n%s" % (self.SUBMISSIONSCRIPTCOMMANDPREFIX),submissionScript)

      return(submissionScript)


   def buildBatchScript(self):
      batchLogName = ""
      batchScriptName = "%s_%s.lsf" % (self.localJobId,self.instanceId)
      if self.isMultiCoreRequest:
         if   self.computationMode == 'mpi':
            batchScript = self.__buildMPIFile()
         elif self.computationMode == 'parallel':
            batchScript = self.__buildSerialFile()
         elif self.computationMode == 'matlabmpi':
            batchScript = ""
            batchScriptName = ""
      else:
         batchScript = self.__buildSerialFile()
      batchScriptExecutable = False

      return(batchLogName,batchScriptName,batchScript,batchScriptExecutable)


   def getBatchNodeList(self):
      return(self.nodeFileName,self.nodeList)


   def getBatchToolInputTemplate(self):
      return(self.toolInputTemplateFileName,self.toolInputTemplate)


   def getBatchToolOutputTemplate(self):
      return(self.toolOutputTemplateFileName,self.toolOutputTemplate)


   def getRemoteJobIdNumber(self,
                            remoteJobId):
      try:
# Job <851126> is submitted to default queue <normal>.
         remoteJobIdNumber = remoteJobId.split('<')[1].split('>')[0]
      except:
         remoteJobIdNumber = "-1"

      return(remoteJobIdNumber)


