# @package      hubzero-submit-distributor
# @file         DockerImagesInfo.py
# @copyright    Copyright (c) 2012-2020 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2017-2020 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#
import os.path
import re
import glob
import logging

from hubzero.submit.LogMessage import getLogJobIdMessage as getLogMessage

class DockerImagesInfo:
   def __init__(self,
                dockerImagesPath):
      self.logger = logging.getLogger(__name__)

      self.dockerImages = {}

      if os.path.isdir(dockerImagesPath):
         for dockerImagesInfoPath in glob.iglob(os.path.join(dockerImagesPath,'*')):
            self.readDockerImagesInfoFile(dockerImagesInfoPath)
      else:
         for dockerImagesInfoPath in glob.iglob(dockerImagesPath):
            self.readDockerImagesInfoFile(dockerImagesInfoPath)

      markedForDeletion = []
      for dockerImageName in self.dockerImages:
         if self.dockerImages[dockerImageName]['state'] != 'enabled':
            markedForDeletion.append(dockerImageName)
      for dockerImageName in markedForDeletion:
         del self.dockerImages[dockerImageName]
      del markedForDeletion


   def readDockerImagesInfoFile(self,
                                dockerImagesInfoPath):
      dockerImagePattern = re.compile('(\s*\[)([^\s]*)(]\s*)')
      keyValuePattern    = re.compile('( *)(\w*)( *= *)(.*[^\s$])( *)')
      commentPattern     = re.compile('\s*#.*')
      dockerImageName    = ""

      if os.path.exists(dockerImagesInfoPath):
         try:
            fpInfo = open(dockerImagesInfoPath,'r')
            try:
               eof = False
               while not eof:
                  record = fpInfo.readline()
                  if record != "":
                     record = commentPattern.sub("",record)
                     if   dockerImagePattern.match(record):
                        dockerImageName = dockerImagePattern.match(record).group(2)
                        self.dockerImages[dockerImageName] = {'imageFile':'',
                                                              'layerFiles':[],
                                                              'state':'enabled'
                                                             }
                     elif keyValuePattern.match(record):
                        key,value = keyValuePattern.match(record).group(2,4)
                        if key in self.dockerImages[dockerImageName]:
                           if   isinstance(self.dockerImages[dockerImageName][key],list):
                              self.dockerImages[dockerImageName][key] = [e.strip() for e in value.split(',')]
                           elif isinstance(self.dockerImages[dockerImageName][key],bool):
                              self.dockerImages[dockerImageName][key] = bool(value.lower() == 'true')
                           elif isinstance(self.dockerImages[dockerImageName][key],float):
                              self.dockerImages[dockerImageName][key] = float(value)
                           elif isinstance(self.dockerImages[dockerImageName][key],int):
                              self.dockerImages[dockerImageName][key] = int(value)
                           elif isinstance(self.dockerImages[dockerImageName][key],dict):
                              try:
                                 sampleKey   = self.dockerImages[dockerImageName][key].keys()[0]
                                 sampleValue = self.dockerImages[dockerImageName][key][sampleKey]
                              except:
                                 sampleKey   = "key"
                                 sampleValue = "value"
                              self.dockerImages[dockerImageName][key] = {}
                              for e in value.split(','):
                                 dictKey,dictValue = e.split(':')
                                 if isinstance(sampleKey,int):
                                    dictKey = int(dictKey)
                                 if   isinstance(sampleValue,int):
                                    dictValue = int(dictValue)
                                 elif isinstance(sampleValue,float):
                                    dictValue = float(dictValue)
                                 elif isinstance(sampleValue,bool):
                                    dictValue = bool(dictValue.lower() == 'true')
                                 self.dockerImages[dockerImageName][key][dictKey] = dictValue
                           else:
                              self.dockerImages[dockerImageName][key] = value
                        else:
                           message = "Undefined key = value pair %s = %s for docker image %s" % (key,value,dockerImageName)
                           self.logger.log(logging.WARNING,getLogMessage(message))
                  else:
                     eof = True
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("DockerImages configuration file %s could not be read" % \
                                                                                             (dockerImagesInfoPath)))
            finally:
               fpInfo.close()
         except (IOError,OSError):
            self.logger.log(logging.ERROR,getLogMessage("DockerImages configuration file %s could not be opened" % \
                                                                                            (dockerImagesInfoPath)))
      else:
         self.logger.log(logging.ERROR,getLogMessage("DockerImages configuration file %s is missing" % \
                                                                                (dockerImagesInfoPath)))


   def getDockerImageNames(self):
      dockerImageNames = []
      for dockerImageName in self.dockerImages:
         dockerImageNames.append(dockerImageName)

      return(dockerImageNames)


   def dockerImageExists(self,
                         dockerImageName):
      return(dockerImageName in self.dockerImages)


   def getDefaultDockerImageInfo(self):
      dockerImageInfo = {}
      dockerImageInfo['dockerImageName'] = 'default'
      dockerImageInfo['imageFile']       = ''
      dockerImageInfo['layerFiles']      = []

      return(dockerImageInfo)


   def getDockerImageInfo(self,
                          dockerImageName):
      dockerImageInfo = self.getDefaultDockerImageInfo()
      if dockerImageName in self.dockerImages:
         dockerImageInfo['dockerImageName'] = dockerImageName
         if 'imageFile' in self.dockerImages[dockerImageName]:
            dockerImageInfo['imageFile']    = self.dockerImages[dockerImageName]['imageFile']
         if 'layerFiles' in self.dockerImages[dockerImageName]:
            dockerImageInfo['layerFiles']   = self.dockerImages[dockerImageName]['layerFiles']

      return(dockerImageInfo)


