# @package      hubzero-submit-distributor
# @file         RemoteBatchPBS.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2012-2015 HUBzero Foundation, LLC.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2012-2015 HUBzero Foundation, LLC.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#
import os.path
import re
import logging

from hubzero.submit.LogMessage        import getLogJobIdMessage as getLogMessage 
from hubzero.submit.ParameterTemplate import ParameterTemplate
from hubzero.submit.TimeConversion    import minTohhmmss

class RemoteBatchPBS:
   SUBMISSIONSCRIPTCOMMANDPREFIX = '#PBS'

   def __init__(self,
                hubUserName,
                hubUserId,
                submitterClass,
                runName,
                localJobId,
                instanceId,
                appScriptName,
                environment,
                transferExecutable,
                executable,
                arguments,
                stdinput,
                isMultiCoreRequest,
                siteInfo,
                toolFilesInfo,
                dockerImageInfo,
                submissionScriptsInfo,
                managerInfo,
                nNodes,
                ppn,
                wallTime,
                quotaLimit,
                timeHistoryLogs):
      self.logger                   = logging.getLogger(__name__)
      self.hubUserName              = hubUserName
      self.hubUserId                = hubUserId
      self.submitterClass           = submitterClass
      self.runName                  = runName
      self.localJobId               = localJobId
      self.instanceId               = instanceId
      self.appScriptName            = appScriptName
      self.environment              = environment
      self.stageFiles               = siteInfo['stageFiles']
      self.submissionScriptsInfo    = submissionScriptsInfo
      self.transferExecutable       = transferExecutable
      if transferExecutable:
         if self.stageFiles:
            self.executable         = os.path.join(".",os.path.basename(executable))
         else:
            self.executable         = executable
      else:
         if not executable.startswith('/') and not executable.startswith('$') and siteInfo['remoteApplicationRootDirectory'] != "":
            self.executable         = os.path.join(siteInfo['remoteApplicationRootDirectory'],executable)
         else:
            self.executable         = executable
      self.arguments                = arguments
      self.stdinput                 = stdinput
      self.isMultiCoreRequest       = isMultiCoreRequest
      self.computationMode          = managerInfo['computationMode']
      self.preManagerCommands       = managerInfo['preManagerCommands']
      self.managerCommand           = managerInfo['managerCommand']
      self.postManagerCommands      = managerInfo['postManagerCommands']
      self.nNodes                   = nNodes
      self.ppn                      = ppn
      self.hostAttributes           = siteInfo['hostAttributes']
      self.remoteBatchQueue         = siteInfo['remoteBatchQueue']
      self.wallTime                 = minTohhmmss(wallTime)
      if siteInfo['remoteBatchAccount'] != "":
         self.account               = self.SUBMISSIONSCRIPTCOMMANDPREFIX + ' -A ' + siteInfo['remoteBatchAccount']
      else:
         self.account               = ''
      self.submissionScriptCommands = siteInfo['submissionScriptCommands']
      self.toolFilesInfo            = toolFilesInfo
      self.dockerImageInfo          = dockerImageInfo
      self.quotaLimit               = quotaLimit
      self.timestampStart           = timeHistoryLogs['timestampStart']
      self.timestampFinish          = timeHistoryLogs['timestampFinish']
      self.timeResults              = timeHistoryLogs['timeResults']

      self.nodeFileName = ""
      self.nodeList     = []

      self.toolInputTemplateFileName  = ""
      self.toolInputTemplate          = ""
      self.toolOutputTemplateFileName = ""
      self.toolOutputTemplate         = ""


   def __buildSerialFile(self):
      rawSubmissionScript = self.submissionScriptsInfo.getSubmissionScript('Batch','PBS','serial')

      queuePBS = ""
      if self.remoteBatchQueue != "":
         queuePBS = self.SUBMISSIONSCRIPTCOMMANDPREFIX + " -q " + self.remoteBatchQueue

      commandsPBS = ""
      if self.submissionScriptCommands:
         if self.SUBMISSIONSCRIPTCOMMANDPREFIX:
            commandSeparator = "\n%s " % self.SUBMISSIONSCRIPTCOMMANDPREFIX
            commandsPBS = self.SUBMISSIONSCRIPTCOMMANDPREFIX + " " + commandSeparator.join(self.submissionScriptCommands)
         else:
            commandSeparator = "\n"
            commandsPBS = commandSeparator.join(self.submissionScriptCommands)

      environmentExport = ""
      environmentVars = self.environment.split()
      for environmentVar in environmentVars:
         environmentExport += "export " + environmentVar + "\n"

      substitutions = {}
      if self.stdinput == "":
         substitutions["STDIN"]                 = '/dev/null'
      else:
         substitutions["STDIN"]                 = self.stdinput
      substitutions["WALLTIME"]                 = self.wallTime
      substitutions["QUEUE"]                    = queuePBS
      substitutions["SUBMISSIONSCRIPTCOMMANDS"] = commandsPBS
      substitutions["ENVIRONMENT"]              = environmentExport
      substitutions["EXECUTABLE"]               = self.executable
      substitutions["ARGUMENTS"]                = self.arguments
      substitutions["TS_START"]                 = self.timestampStart
      substitutions["TS_FINISH"]                = self.timestampFinish
      substitutions["TIME_RESULTS"]             = self.timeResults
      substitutions["RUNNAME"]                  = self.runName
      substitutions["JOBID"]                    = self.localJobId
      substitutions["INSTANCEID"]               = self.instanceId
      substitutions["NNODES"]                   = self.nNodes
      substitutions["ATTRIBUTES"]               = self.hostAttributes
      substitutions["PPN"]                      = self.ppn
      nProcessors = str(int(self.nNodes)*int(self.ppn))
      substitutions["NPROCESSORS"]              = nProcessors
      substitutions["REMOTEBATCHACCOUNT"]       = self.account
      substitutions["PREMANAGERCOMMANDS"]       = "\n".join(self.preManagerCommands)
      substitutions["POSTMANAGERCOMMANDS"]      = "\n".join(self.postManagerCommands)
      substitutions["HUBUSERNAME"]              = self.hubUserName
      substitutions["HUBUSERID"]                = str(self.hubUserId)
      if self.quotaLimit > 0:
         substitutions["QUOTA"]                 = "ulimit -f " + str(self.quotaLimit)
      else:
         substitutions["QUOTA"]                 = ""

      template = ParameterTemplate(rawSubmissionScript)
      try:
         submissionScript = template.substitute_recur(substitutions)
      except KeyError,err:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Pattern substitution failed for @@%s\n" % (err[0])))
      except TypeError,err:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Submission script substitution failed:\n%s\n" % (rawSubmissionScript)))
      else:
         if self.SUBMISSIONSCRIPTCOMMANDPREFIX:
            submissionScript = re.sub("(\n)*\n%s" % (self.SUBMISSIONSCRIPTCOMMANDPREFIX),
                                           "\n%s" % (self.SUBMISSIONSCRIPTCOMMANDPREFIX),submissionScript)

      return(submissionScript)


   def __buildMatlabMPIFile(self):
      rawSubmissionScript = self.submissionScriptsInfo.getSubmissionScript('Batch','PBS','matlabmpi')

      queuePBS = ""
      if self.remoteBatchQueue != "":
         queuePBS = self.SUBMISSIONSCRIPTCOMMANDPREFIX + " -q " + self.remoteBatchQueue

      commandsPBS = ""
      if self.submissionScriptCommands:
         if self.SUBMISSIONSCRIPTCOMMANDPREFIX:
            commandSeparator = "\n%s " % self.SUBMISSIONSCRIPTCOMMANDPREFIX
            commandsPBS = self.SUBMISSIONSCRIPTCOMMANDPREFIX + " " + commandSeparator.join(self.submissionScriptCommands)
         else:
            commandSeparator = "\n"
            commandsPBS = commandSeparator.join(self.submissionScriptCommands)

      environmentExport = ""
      environmentVars = self.environment.split()
      for environmentVar in environmentVars:
         environmentExport += "export " + environmentVar + "\n"

      substitutions = {}
      substitutions["PREMANAGERCOMMANDS"]       = "\n".join(self.preManagerCommands)
      substitutions["MANAGERCOMMAND"]           = self.managerCommand
      substitutions["POSTMANAGERCOMMANDS"]      = "\n".join(self.postManagerCommands)
      substitutions["WALLTIME"]                 = self.wallTime
      substitutions["NNODES"]                   = self.nNodes
      substitutions["ATTRIBUTES"]               = self.hostAttributes
      substitutions["PPN"]                      = self.ppn
      nProcessors = str(int(self.nNodes)*int(self.ppn))
      substitutions["NPROCESSORS"]              = nProcessors
      substitutions["QUEUE"]                    = queuePBS
      substitutions["SUBMISSIONSCRIPTCOMMANDS"] = commandsPBS
      substitutions["ENVIRONMENT"]              = environmentExport
      substitutions["TS_START"]                 = self.timestampStart
      substitutions["TS_FINISH"]                = self.timestampFinish
      substitutions["RUNNAME"]                  = self.runName
      substitutions["JOBID"]                    = self.localJobId
      substitutions["INSTANCEID"]               = self.instanceId
      substitutions["EXECUTABLE"]               = self.executable
      substitutions["ARGUMENTS"]                = self.arguments
      substitutions["TIME_RESULTS"]             = self.timeResults
      substitutions["REMOTEBATCHACCOUNT"]       = self.account
      substitutions["HUBUSERNAME"]              = self.hubUserName
      substitutions["HUBUSERID"]                = str(self.hubUserId)
      if self.quotaLimit > 0:
         substitutions["QUOTA"]                 = "ulimit -f " + str(self.quotaLimit)
      else:
         substitutions["QUOTA"]                 = ""

      template = ParameterTemplate(rawSubmissionScript)
      try:
         submissionScript = template.substitute_recur(substitutions)
      except KeyError,err:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Pattern substitution failed for @@%s\n" % (err[0])))
      except TypeError,err:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Submission script substitution failed:\n%s\n" % (rawSubmissionScript)))
      else:
         if self.SUBMISSIONSCRIPTCOMMANDPREFIX:
            submissionScript = re.sub("(\n)*\n%s" % (self.SUBMISSIONSCRIPTCOMMANDPREFIX),
                                           "\n%s" % (self.SUBMISSIONSCRIPTCOMMANDPREFIX),submissionScript)

      return(submissionScript)


   def __buildParallelFile(self):
      rawSubmissionScript = self.submissionScriptsInfo.getSubmissionScript('Batch','PBS','parallel')

      queuePBS = ""
      if self.remoteBatchQueue != "":
         queuePBS = self.SUBMISSIONSCRIPTCOMMANDPREFIX + " -q " + self.remoteBatchQueue

      commandsPBS = ""
      if self.submissionScriptCommands:
         if self.SUBMISSIONSCRIPTCOMMANDPREFIX:
            commandSeparator = "\n%s " % self.SUBMISSIONSCRIPTCOMMANDPREFIX
            commandsPBS = self.SUBMISSIONSCRIPTCOMMANDPREFIX + " " + commandSeparator.join(self.submissionScriptCommands)
         else:
            commandSeparator = "\n"
            commandsPBS = commandSeparator.join(self.submissionScriptCommands)

      environmentExport = ""
      environmentVars = self.environment.split()
      for environmentVar in environmentVars:
         environmentExport += "export " + environmentVar + "\n"

      substitutions = {}
      if self.stdinput == "":
         substitutions["STDIN"]                 = '/dev/null'
      else:
         substitutions["STDIN"]                 = self.stdinput
      substitutions["PREMANAGERCOMMANDS"]       = "\n".join(self.preManagerCommands)
      substitutions["MANAGERCOMMAND"]           = self.managerCommand
      substitutions["POSTMANAGERCOMMANDS"]      = "\n".join(self.postManagerCommands)
      substitutions["WALLTIME"]                 = self.wallTime
      substitutions["NNODES"]                   = self.nNodes
      substitutions["ATTRIBUTES"]               = self.hostAttributes
      substitutions["PPN"]                      = self.ppn
      substitutions["QUEUE"]                    = queuePBS
      substitutions["SUBMISSIONSCRIPTCOMMANDS"] = commandsPBS
      substitutions["ENVIRONMENT"]              = environmentExport
      substitutions["EXECUTABLE"]               = self.executable
      substitutions["ARGUMENTS"]                = self.arguments
      substitutions["TS_START"]                 = self.timestampStart
      substitutions["TS_FINISH"]                = self.timestampFinish
      substitutions["TIME_RESULTS"]             = self.timeResults
      substitutions["RUNNAME"]                  = self.runName
      substitutions["JOBID"]                    = self.localJobId
      substitutions["INSTANCEID"]               = self.instanceId
      nProcessors = str(int(self.nNodes)*int(self.ppn))
      substitutions["NPROCESSORS"]              = nProcessors
      substitutions["REMOTEBATCHACCOUNT"]       = self.account
      substitutions["HUBUSERNAME"]              = self.hubUserName
      substitutions["HUBUSERID"]                = str(self.hubUserId)
      if self.quotaLimit > 0:
         substitutions["QUOTA"]                 = "ulimit -f " + str(self.quotaLimit)
      else:
         substitutions["QUOTA"]                 = ""

      template = ParameterTemplate(rawSubmissionScript)
      try:
         submissionScript = template.substitute_recur(substitutions)
      except KeyError,err:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Pattern substitution failed for @@%s\n" % (err[0])))
      except TypeError,err:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Submission script substitution failed:\n%s\n" % (rawSubmissionScript)))
      else:
         if self.SUBMISSIONSCRIPTCOMMANDPREFIX:
            submissionScript = re.sub("(\n)*\n%s" % (self.SUBMISSIONSCRIPTCOMMANDPREFIX),
                                           "\n%s" % (self.SUBMISSIONSCRIPTCOMMANDPREFIX),submissionScript)

      return(submissionScript)


   def __buildMPIFile(self):
      rawSubmissionScript = self.submissionScriptsInfo.getSubmissionScript('Batch','PBS','mpi')

      queuePBS = ""
      if self.remoteBatchQueue != "":
         queuePBS = self.SUBMISSIONSCRIPTCOMMANDPREFIX + " -q " + self.remoteBatchQueue

      commandsPBS = ""
      if self.submissionScriptCommands:
         if self.SUBMISSIONSCRIPTCOMMANDPREFIX:
            commandSeparator = "\n%s " % self.SUBMISSIONSCRIPTCOMMANDPREFIX
            commandsPBS = self.SUBMISSIONSCRIPTCOMMANDPREFIX + " " + commandSeparator.join(self.submissionScriptCommands)
         else:
            commandSeparator = "\n"
            commandsPBS = commandSeparator.join(self.submissionScriptCommands)

      environmentExport = ""
      environmentVars = self.environment.split()
      for environmentVar in environmentVars:
         environmentExport += "export " + environmentVar + "\n"

      substitutions = {}
      substitutions["PREMANAGERCOMMANDS"]       = "\n".join(self.preManagerCommands)
      substitutions["MANAGERCOMMAND"]           = self.managerCommand
      substitutions["POSTMANAGERCOMMANDS"]      = "\n".join(self.postManagerCommands)
      substitutions["WALLTIME"]                 = self.wallTime
      substitutions["NNODES"]                   = self.nNodes
      substitutions["ATTRIBUTES"]               = self.hostAttributes
      substitutions["PPN"]                      = self.ppn
      nProcessors = str(int(self.nNodes)*int(self.ppn))
      substitutions["NPROCESSORS"]              = nProcessors
      substitutions["QUEUE"]                    = queuePBS
      substitutions["SUBMISSIONSCRIPTCOMMANDS"] = commandsPBS
      substitutions["ENVIRONMENT"]              = environmentExport
      substitutions["APPSCRIPTNAME"]            = self.appScriptName
      substitutions["TS_START"]                 = self.timestampStart
      substitutions["TS_FINISH"]                = self.timestampFinish
      substitutions["TIME_RESULTS"]             = self.timeResults
      substitutions["RUNNAME"]                  = self.runName
      substitutions["JOBID"]                    = self.localJobId
      substitutions["INSTANCEID"]               = self.instanceId
      substitutions["REMOTEBATCHACCOUNT"]       = self.account
      substitutions["HUBUSERNAME"]              = self.hubUserName
      substitutions["HUBUSERID"]                = str(self.hubUserId)
      if self.quotaLimit > 0:
         substitutions["QUOTA"]                 = "ulimit -f " + str(self.quotaLimit)
      else:
         substitutions["QUOTA"]                 = ""

      template = ParameterTemplate(rawSubmissionScript)
      try:
         submissionScript = template.substitute_recur(substitutions)
      except KeyError,err:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Pattern substitution failed for @@%s\n" % (err[0])))
      except TypeError,err:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Submission script substitution failed:\n%s\n" % (rawSubmissionScript)))
      else:
         if self.SUBMISSIONSCRIPTCOMMANDPREFIX:
            submissionScript = re.sub("(\n)*\n%s" % (self.SUBMISSIONSCRIPTCOMMANDPREFIX),
                                           "\n%s" % (self.SUBMISSIONSCRIPTCOMMANDPREFIX),submissionScript)

      return(submissionScript)


   def buildBatchScript(self):
      batchLogName = ""
      batchScriptName = "%s_%s.pbs" % (self.localJobId,self.instanceId)
      if self.isMultiCoreRequest:
         if   self.computationMode == 'mpi':
            batchScript = self.__buildMPIFile()
         elif self.computationMode == 'parallel':
            batchScript = self.__buildParallelFile()
         elif self.computationMode == 'matlabmpi':
            batchScript = self.__buildMatlabMPIFile()
      else:
         batchScript = self.__buildSerialFile()
      batchScriptExecutable = False

      return(batchLogName,batchScriptName,batchScript,batchScriptExecutable)


   def getBatchNodeList(self):
      return(self.nodeFileName,self.nodeList)


   def getBatchToolInputTemplate(self):
      return(self.toolInputTemplateFileName,self.toolInputTemplate)


   def getBatchToolOutputTemplate(self):
      return(self.toolOutputTemplateFileName,self.toolOutputTemplate)


   def getRemoteJobIdNumber(self,
                            remoteJobId):
      try:
         remoteJobIdNumber = remoteJobId.split('.')[0]
      except:
         remoteJobIdNumber = "-1"

      return(remoteJobIdNumber)


