# @package      hubzero-submit-distributor
# @file         RemoteWorkflowPEGASUS.py
# @copyright    Copyright (c) 2012-2020 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2012-2020 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#
import os
import re
import logging
import warnings

def showWarning(message, category, filename, lineno, file=None, line=None):
   return

_default_show_warning = warnings.showwarning

warnings.showwarning = showWarning
try:
   from Pegasus.api import *
except:
   try:
      from Pegasus.DAX3 import *
   except:
      from hubzero.submit.DAX3 import *
warnings.showwarning = _default_show_warning

from hubzero.submit.LogMessage        import getLogIDMessage as getLogMessage
from hubzero.submit.ParameterTemplate import ParameterTemplate

class RemoteWorkflowPEGASUS:
   SUBMISSIONSCRIPTCOMMANDPREFIX = ''

   def __init__(self,
                hubUserName,
                hubUserId,
                submitterClass,
                session,
                instanceToken,
                wsJobId,
                runName,
                localJobId,
                instanceId,
                jobDirectory,
                scratchDirectory,
                useSetup,
                pegasusVersion,
                pegasusHome,
                pegasusTemplates,
                x509SubmitProxy,
                sshIdentityPath,
                gridsite,
                pegasusSiteInfo,
                nGpus,
                gpn,
                memoryMB,
                wallTime,
                siteInfo,
                submissionScriptsInfo,
                managerInfo,
                timeHistoryLogs):
      self.logger                   = logging.getLogger(__name__)
      self.hubUserName              = hubUserName
      self.hubUserId                = hubUserId
      self.submitterClass           = submitterClass
      self.session                  = session
      self.instanceToken            = instanceToken
      self.wsJobId                  = wsJobId
      self.runName                  = runName
      self.localJobId               = localJobId
      self.instanceId               = instanceId
      self.jobDirectory             = jobDirectory
      self.scratchDirectory         = scratchDirectory

      self.useSetup                 = useSetup
      self.pegasusVersion           = pegasusVersion
      self.pegasusHome              = pegasusHome
      self.pegasusTemplates         = pegasusTemplates
      if int(self.pegasusVersion[0]) > 4:
         self.daxPath               = os.path.join(self.jobDirectory,"%s.yml" % (self.localJobId))
      else:
         self.daxPath               = os.path.join(self.jobDirectory,"%s.dax" % (self.localJobId))
      self.executable               = 'pegasus-plan'
      self.arguments                = ""

      self.x509SubmitProxy          = x509SubmitProxy
      self.sshIdentityPath          = sshIdentityPath
      self.gridsite                 = gridsite
      self.pegasusSiteInfo          = pegasusSiteInfo
      self.nGpus                    = nGpus
      self.gpn                      = gpn
      self.memoryMB                 = memoryMB
      self.wallTime                 = wallTime
      self.submissionScriptCommands = siteInfo['submissionScriptCommands']
      self.timePaths                = siteInfo['timePaths']
      self.submissionScriptsInfo    = submissionScriptsInfo
      self.preManagerCommands       = managerInfo['preManagerCommands']
      self.postManagerCommands      = managerInfo['postManagerCommands']
      self.timestampStart           = timeHistoryLogs['timestampStart']
      self.timestampFinish          = timeHistoryLogs['timestampFinish']
      self.timeResults              = timeHistoryLogs['timeResults']


   def __buildWorkflowScript(self):
      rawSubmissionScript = self.submissionScriptsInfo.getSubmissionScript('Workflow','PEGASUS','plan')

      commandsPEGASUS = ""
      if self.submissionScriptCommands:
         if self.SUBMISSIONSCRIPTCOMMANDPREFIX:
            commandSeparator = "\n%s " % self.SUBMISSIONSCRIPTCOMMANDPREFIX
            commandsPEGASUS = self.SUBMISSIONSCRIPTCOMMANDPREFIX + " " + commandSeparator.join(self.submissionScriptCommands)
         else:
            commandSeparator = "\n"
            commandsPEGASUS = commandSeparator.join(self.submissionScriptCommands)

      notRecognizedArguments = []
      discardedArguments     = []
      auxiliaryArguments     = []
      userArguments = self.arguments.split()
      for userArgument in userArguments:
         auxiliaryArguments.append(userArgument)

      if os.path.exists(self.pegasusTemplates['rc']):
         try:
            fpRCTemplate = open(self.pegasusTemplates['rc'],'r')
            try:
               searchString = os.path.basename(self.executable) + '.arguments'
               settings = fpRCTemplate.readlines()
               for setting in settings:
#pegasus-plan.arguments = --nocleanup
                  if setting.count(searchString) > 0:
                     try:
                        parameter,value = setting.split('=')
                        auxiliaryArguments.append(value.strip())
                     except:
                        pass
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("%s could not be read" % (self.pegasusTemplates['rc'])))
            finally:
               fpRCTemplate.close()
         except (IOError,OSError):
            self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (self.pegasusTemplates['rc'])))

      substitutions = {}
      substitutions["USESETUP"]                 = self.useSetup
      substitutions["PEGASUSVERSION"]           = self.pegasusVersion
      substitutions["RUNNAME"]                  = self.runName
      substitutions["JOBID"]                    = self.localJobId
      substitutions["INSTANCEID"]               = self.instanceId
      substitutions["INSTANCEDIRECTORY"]        = self.jobDirectory
      substitutions["SCRATCHDIRECTORY"]         = self.scratchDirectory
      substitutions["EXECUTABLE"]               = self.executable
      substitutions["GRIDSITE"]                 = self.gridsite
      arguments = ' '.join(auxiliaryArguments)
      substitutions["ARGUMENTS"]                = arguments.strip()
      arguments = ' '.join(discardedArguments)
      substitutions["DISCARDED"]                = arguments.strip()
      arguments = ' '.join(notRecognizedArguments)
      substitutions["NOTRECOGNIZED"]            = arguments.strip()
      substitutions["TS_START"]                 = self.timestampStart
      substitutions["TS_FINISH"]                = self.timestampFinish
      substitutions["TIME_RESULTS"]             = self.timeResults
      substitutions["HUBUSERNAME"]              = self.hubUserName
      substitutions["HUBUSERID"]                = str(self.hubUserId)
      substitutions["SUBMISSIONSCRIPTCOMMANDS"] = commandsPEGASUS
      substitutions["PREMANAGERCOMMANDS"]       = "\n".join(self.preManagerCommands)
      substitutions["POSTMANAGERCOMMANDS"]      = "\n".join(self.postManagerCommands)
      if int(self.pegasusVersion[0]) > 4:
         substitutions["DAX"]                   = self.daxPath
      else:
         substitutions["DAX"]                   = "--dax %s" % (self.daxPath)
      substitutions["TIMEPATHS"]                = ' '.join(self.timePaths)

      template = ParameterTemplate(rawSubmissionScript)
      try:
         submissionScript = template.substitute_recur(substitutions)
      except KeyError as e:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Pattern substitution failed for @@{%s}\n" % (e.args[0])))
         self.logger.log(logging.ERROR,getLogMessage("in SubmissionScripts/Distributor/Workflow/PEGASUS/plan\n"))
      except TypeError:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Submission script substitution failed:\n%s\n" % (rawSubmissionScript)))
      else:
         if self.SUBMISSIONSCRIPTCOMMANDPREFIX:
            submissionScript = re.sub("(\n)*\n%s" % (self.SUBMISSIONSCRIPTCOMMANDPREFIX),
                                           "\n%s" % (self.SUBMISSIONSCRIPTCOMMANDPREFIX),submissionScript)

      return(submissionScript)


   def __buildCatalogs(self):
      re_instanceIdEmpty = re.compile("_@@{INSTANCEID}")
      substitutions = {}
      substitutions["SUBMITTERCLASS"]       = str(self.submitterClass)
      substitutions["SESSION"]              = self.session
      substitutions["INSTANCETOKEN"]        = self.instanceToken
      substitutions["WSJOBID"]              = self.wsJobId
      substitutions["RUNNAME"]              = self.runName
      substitutions["JOBID"]                = self.localJobId
      substitutions["INSTANCEDIRECTORY"]    = self.jobDirectory
      substitutions["BASESCRATCHDIRECTORY"] = os.path.basename(self.scratchDirectory)
      substitutions["SCRATCHDIRECTORY"]     = self.scratchDirectory
      if self.x509SubmitProxy != "":
         substitutions["X509SUBMITPROXY"]   = self.x509SubmitProxy
      else:
         substitutions["X509SUBMITPROXY"]   = os.path.join(os.sep,'tmp','hub-proxy.%s' % (self.hubUserName))
      substitutions["PEGASUSHOME"]          = self.pegasusHome
      substitutions["SSHPRIVATEKEYPATH"]    = self.sshIdentityPath
      substitutions["MEMORY"]               = self.memoryMB

      for templateType in self.pegasusTemplates:
         pegasusTemplatePath = self.pegasusTemplates[templateType]
         if pegasusTemplatePath != "":
            try:
               fpTemplate = open(pegasusTemplatePath,'r')
               try:
                  template = ''.join(fpTemplate.readlines())
               except (IOError,OSError):
                  self.logger.log(logging.ERROR,getLogMessage("%s could not be read" % (pegasusTemplatePath)))
               else:
                  if self.instanceId != "0":
                     substitutions["INSTANCEID"] = self.instanceId
                  else:
                     template = re_instanceIdEmpty.sub("",template)
                     substitutions["INSTANCEID"] = "00"

                  pegasusTemplate = ParameterTemplate(template)
                  try:
                     pegasusText = pegasusTemplate.substitute_recur(substitutions)
                  except KeyErroras as e:
                     pegasusText = ""
                     self.logger.log(logging.ERROR,getLogMessage("Pattern substitution failed for @@{%s}\n" % (e.args[0])))
                     self.logger.log(logging.ERROR,getLogMessage("in %s\n" % (pegasusTemplatePath)))
                  except TypeError:
                     pegasusText = ""
                     self.logger.log(logging.ERROR,getLogMessage("Pegasus template substitution failed:\n%s\n" % (template)))

                  if   templateType == 'rc':
                     pegasusFile = "%s.pegasusrc" % (self.localJobId)
                  elif templateType == 'sites':
                     if pegasusTemplatePath.endswith('.yml'): 
                        pegasusFile = "%s_sites.yml" % (self.localJobId)
                     else:
                        pegasusFile = "%s_sites.xml" % (self.localJobId)
                  elif templateType == 'tc':
                     if pegasusTemplatePath.endswith('.yml'): 
                        pegasusFile = "%s_tc.yml" % (self.localJobId)
                     else:
                        pegasusFile = "%s_tc.txt" % (self.localJobId)
                  pegasusPath = os.path.join(self.jobDirectory,pegasusFile)
                  try:
                     fpPegasusFile = open(pegasusPath,'w')
                     try:
                        fpPegasusFile.write(pegasusText)
                     except (IOError,OSError):
                        self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (pegasusPath)))
                     finally:
                        fpPegasusFile.close()
                  except (IOError,OSError):
                     self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (pegasusPath)))
               finally:
                  fpTemplate.close()
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (pegasusTemplatePath)))


   def __buildDAX(self):
      instanceIds = []
      dirFiles = os.listdir(self.jobDirectory)
      for dirFile in dirFiles:
         dirPath = os.path.join(self.jobDirectory,dirFile)
         if os.path.isdir(dirPath):
            instanceIds.append(dirFile)
      instanceIds.sort()

      nInstances = len(instanceIds)
      reAppScriptPath  = re.compile("[0-9_]+.sh")
      reStageInTarFile = re.compile("[0-9_]+_input.tar.gz")

      if int(self.pegasusVersion[0]) > 4:
         dax = Workflow(self.localJobId)
         tc  = TransformationCatalog()
         rc  = ReplicaCatalog()
         for instanceId in instanceIds:
            instanceDirectory = os.path.join(self.jobDirectory,instanceId)
            dirFiles = os.listdir(instanceDirectory)
            appScriptFiles  = list(filter(reAppScriptPath.search,dirFiles))
            stageInTarFiles = list(filter(reStageInTarFile.search,dirFiles))

            if len(appScriptFiles) == 1 and len(stageInTarFiles) == 1:
               appScriptName = appScriptFiles[0]
               appScriptPath = os.path.join(instanceDirectory,appScriptName)
               appScript = Transformation(appScriptName,site='local',
                                                        pfn=appScriptPath,
                                                        arch=Arch.X86_64,
                                                        os_type=OS.LINUX,
                                                        is_stageable=True)
               tc.add_transformations(appScript)

               stageInTarFile = stageInTarFiles[0]
               inputTarPath = os.path.join(instanceDirectory,stageInTarFile)
               inputTar = File(stageInTarFile)
               rc.add_replica("local",inputTar,inputTarPath)

               stageOutTarFile = stageInTarFile.replace('input','output')
               outputTar = File(stageOutTarFile)

               jobId = "%s-%s" % (self.runName,instanceId)
               job = Job(appScriptName,_id=jobId)
#              job.add_profiles(Namespace.HINTS,key='execution.site',value="%s" % (self.pegasusSiteInfo['name']))
               job.add_globus_profiles(max_wall_time=int(self.wallTime))
               job.add_inputs(inputTar)
               job.add_outputs(outputTar,register_replica=False,stage_out=True)
               dax.add_jobs(job)
         dax.add_transformation_catalog(tc)
         dax.add_replica_catalog(rc)
      else:
         dax = ADAG(self.localJobId)
         for instanceId in instanceIds:
            instanceDirectory = os.path.join(self.jobDirectory,instanceId)
            dirFiles = os.listdir(instanceDirectory)
            appScriptFiles  = list(filter(reAppScriptPath.search,dirFiles))
            stageInTarFiles = list(filter(reStageInTarFile.search,dirFiles))

            if len(appScriptFiles) == 1 and len(stageInTarFiles) == 1:
               arch      = self.pegasusSiteInfo['arch']
               osFlavor  = self.pegasusSiteInfo['osFlavor'].lower()
               osVersion = self.pegasusSiteInfo['osVersion']
               osRelease = self.pegasusSiteInfo['osRelease']

               appScriptName = appScriptFiles[0]
               appScriptPath = os.path.join(instanceDirectory,appScriptName)

               appScriptPath = "file://" + appScriptPath
               appScript = Executable(name=appScriptName,arch="%s" % (arch),
                                                         os="%s" % (osFlavor),
                                                         osrelease="%s" % (osRelease),
                                                         osversion="%s" % (osVersion),installed=False)
               appScript.addPFN(PFN(appScriptPath,"local"))
               dax.addExecutable(appScript)

               stageInTarFile = stageInTarFiles[0]
               inputTarPath = "file://" + os.path.join(instanceDirectory,stageInTarFile)
               inputTar = File(stageInTarFile)
               inputTar.addPFN(PFN(inputTarPath,"local"))
               dax.addFile(inputTar)

               stageOutTarFile = stageInTarFile.replace('input','output')
               outputTar = File(stageOutTarFile)

               jobId = "%s_%s" % (self.runName,instanceId)
               job = Job(name=appScriptName,id=jobId)
               job.addProfile(Profile(Namespace.HINTS, key='executionPool',value="%s" % (self.pegasusSiteInfo['name'])))
               job.addProfile(Profile(Namespace.GLOBUS,key='maxwalltime',value=self.wallTime))
               job.uses(inputTar,link=Link.INPUT,executable=False)
               job.uses(outputTar,link=Link.OUTPUT,register=True,transfer=True)
               dax.addJob(job)

      try:
         fpDAX = open(self.daxPath,'w')
         try:
            if int(self.pegasusVersion[0]) > 4:
               dax.write(fpDAX)
            else:
               dax.writeXML(fpDAX)
         except (IOError,OSError):
            self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (self.daxPath)))
         finally:
            fpDAX.close()
      except (IOError,OSError):
         self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (self.daxPath)))


   def buildWorkflowScript(self):
      workflowScriptName = "%s.pegasus" % (self.localJobId)
      workflowScript = self.__buildWorkflowScript()

      self.__buildCatalogs()
      self.__buildDAX()
      workflowScriptExecutable = True

      return(workflowScriptName,workflowScript,workflowScriptExecutable)


   def getRemoteJobIdNumber(self,
                            remoteJobId):
      try:
# 1 job(s) submitted to cluster 105.
         remoteJobIdNumber = re.search('cluster [0-9]+',remoteJobId).group().split()[1] + ".0"
      except:
         remoteJobIdNumber = "-1"

      return(remoteJobIdNumber)


