#
# @package      hubzero-submit-distributor
# @file         RemoteInstantSCRIPT.py
# @author       Steve Clark <clarks@purdue.edu>
# @copyright    Copyright 2004-2011 Purdue University. All rights reserved.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2004-2011 Purdue University
# All rights reserved.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of Purdue University.
#
import re
import random

class RemoteInstantSCRIPT:
   def __init__(self,
                localJobId,
                trial,
                workingDirectory,
                appScriptName,
                isMultiCoreRequest,
                computationMode,
                preManagerCommands,
                managerCommand,
                postManagerCommands,
                nNodes,
                ppn,
                venue,
                venues,
                venueIndex,
                timestampTransferred,
                timestampStart,
                timestampFinish,
                timeResults):
      self.localJobId           = localJobId
      self.trial                = trial
      self.workingDirectory     = workingDirectory
      self.appScriptName        = appScriptName
      self.isMultiCoreRequest   = isMultiCoreRequest
      self.computationMode      = computationMode
      self.preManagerCommands   = preManagerCommands
      self.managerCommand       = managerCommand
      self.postManagerCommands  = postManagerCommands
      self.nNodes               = nNodes
      self.ppn                  = ppn
      self.venue                = venue
      self.venues               = venues
      self.venueIndex           = venueIndex
      self.timestampStart       = timestampStart
      self.timestampFinish      = timestampFinish
      self.timestampTransferred = timestampTransferred
      self.timeResults          = timeResults

      self.nodeFileName = ""
      self.nodeList     = []


   def __makeMPITemplate(self):
      return """#!/bin/sh

trap cleanup HUP INT QUIT ABRT TERM

cleanup()
{
#  echo "Abnormal termination by signal"
#  kill -s TERM `jobs -p`
   mpirunPidTree=$(pstree -l -p `jobs -p`)
   mpirunPidTreeMaster=${mpirunPidTree%%-[rs]sh(*}
   appPidTree=${mpirunPidTreeMaster##*(}
   appPid=${appPidTree%%)*}
   kill -s TERM ${appPid}
   if [ ! -s TS_FINISH ] ; then
      date +"%s" > TS_FINISH
   fi
#  exit 1
}

# Change to directory where job was submitted.
cd WORKINGDIRECTORY
export PATH=WORKINGDIRECTORY:${PATH}

date +"%s" > TS_TRANSFERRED
date +"%s" > TS_START

PREMANAGERCOMMANDS
#MANAGERCOMMAND `pwd`/APPSCRIPTNAME > JOBID.stdout 2> JOBID.stderr
MANAGERCOMMAND `pwd`/APPSCRIPTNAME &
wait %1
POSTMANAGERCOMMANDS

date +"%s" > TS_FINISH

touch TIME_RESULTS-0
cat TIME_RESULTS-[0-9]* >> TIME_RESULTS
rm -f TIME_RESULTS-[0-9]*
"""


   def __buildMPIFile(self):
      # setup regex's for the template
      re_preManagerCommands  = re.compile("PREMANAGERCOMMANDS")
      re_managerCommand      = re.compile("MANAGERCOMMAND")
      re_postManagerCommands = re.compile("POSTMANAGERCOMMANDS")
      re_nnodes              = re.compile("NNODES")
      re_processors          = re.compile("NPROCESSORS")
      re_appScriptName       = re.compile("APPSCRIPTNAME")
      re_tsTransferred       = re.compile("TS_TRANSFERRED")
      re_tsStart             = re.compile("TS_START")
      re_tsFinish            = re.compile("TS_FINISH")
      re_timeResults         = re.compile("TIME_RESULTS")
      re_jobid               = re.compile("JOBID")
      re_workingDirectory    = re.compile("WORKINGDIRECTORY")
      re_nodeFile            = re.compile("\${PBS_NODEFILE}")

      template = self.__makeMPITemplate()

      template = re_preManagerCommands.sub("\n".join(self.preManagerCommands),template)
      template = re_postManagerCommands.sub("\n".join(self.postManagerCommands),template)
      template = re_managerCommand.sub(self.managerCommand,template)
      template = re_nnodes.sub(self.nNodes,template)
      nProcessors = str(int(self.nNodes)*int(self.ppn))
      template = re_processors.sub(nProcessors,template)
      template = re_appScriptName.sub(self.appScriptName,template)
      template = re_tsTransferred.sub(self.timestampTransferred,template)
      template = re_tsStart.sub(self.timestampStart,template)
      template = re_tsFinish.sub(self.timestampFinish,template)
      template = re_timeResults.sub(self.timeResults,template)
      template = re_jobid.sub("%s_%02d" % (self.localJobId,self.trial),template)
      template = re_workingDirectory.sub(self.workingDirectory,template)
      template = re_nodeFile.sub(self.nodeFileName,template)

      return(template)


   def __buildNodeList(self):
      del self.nodeList
      self.nodeList = []
      self.nodeFileName = "%s_%02d.machinelist" % (self.localJobId,self.trial)

      for core in xrange(int(self.ppn)):
         self.nodeList.append(self.venue)
      eligible = range(len(self.venues))
      eligible.remove(self.venueIndex)
      for node in xrange(int(self.nNodes) - 1):
         index = eligible[random.randint(0,len(eligible)-1)]
         enclosure = self.venues[index]
         for core in xrange(int(self.ppn)):
            self.nodeList.append(enclosure)
         eligible.remove(index)
         if len(eligible) == 0:
            eligible = range(len(self.venues))


   def __makeMatlabPCTTemplate(self):
      return """#!/bin/sh

trap cleanup HUP INT QUIT ABRT TERM

cleanup()
{
#  echo "Abnormal termination by signal"
   kill -s TERM `jobs -p`
   if [ ! -s TS_FINISH ] ; then
      date +"%s" > TS_FINISH
   fi
#  exit 1
}

# Change to directory where job was submitted.
cd WORKINGDIRECTORY
export PATH=WORKINGDIRECTORY:${PATH}

date +"%s" > TS_TRANSFERRED
date +"%s" > TS_START

PREMANAGERCOMMANDS
`pwd`/APPSCRIPTNAME &
wait %1
POSTMANAGERCOMMANDS

date +"%s" > TS_FINISH

touch TIME_RESULTS-0
cat TIME_RESULTS-[0-9]* >> TIME_RESULTS
rm -f TIME_RESULTS-[0-9]*
"""


   def __buildMatlabPCTFile(self):
      # setup regex's for the template
      re_preManagerCommands  = re.compile("PREMANAGERCOMMANDS")
      re_postManagerCommands = re.compile("POSTMANAGERCOMMANDS")
      re_appScriptName       = re.compile("APPSCRIPTNAME")
      re_tsTransferred       = re.compile("TS_TRANSFERRED")
      re_tsStart             = re.compile("TS_START")
      re_tsFinish            = re.compile("TS_FINISH")
      re_timeResults         = re.compile("TIME_RESULTS")
      re_workingDirectory    = re.compile("WORKINGDIRECTORY")

      template = self.__makeMatlabPCTTemplate()

      template = re_preManagerCommands.sub("\n".join(self.preManagerCommands),template)
      template = re_postManagerCommands.sub("\n".join(self.postManagerCommands),template)
      template = re_appScriptName.sub(self.appScriptName,template)
      template = re_tsTransferred.sub(self.timestampTransferred,template)
      template = re_tsStart.sub(self.timestampStart,template)
      template = re_tsFinish.sub(self.timestampFinish,template)
      template = re_timeResults.sub(self.timeResults,template)
      template = re_workingDirectory.sub(self.workingDirectory,template)

      return(template)


   def buildBatchScript(self):
      batchLogName = ""
      batchScriptName = ""
      if self.isMultiCoreRequest:
         if   self.computationMode == 'mpi':
            self.__buildNodeList()
            batchScript = self.__buildMPIFile()
            batchScriptName = "%s_%02d.cl" % (self.localJobId,self.trial)
         elif self.computationMode == 'parallel':
            batchScript = ""
         elif self.computationMode == 'matlabmpi':
            batchScript = ""
         elif self.computationMode == 'matlabpct':
            batchScript = self.__buildMatlabPCTFile()
            batchScriptName = "%s_%02d.cl" % (self.localJobId,self.trial)
      else:
         batchScript = ""

      return(batchLogName,batchScriptName,batchScript)


   def getBatchNodeList(self):
      return(self.nodeFileName,self.nodeList)

