#
# @package      hubzero-submit-distributor
# @file         RemoteBatchLSF.py
# @author       Steve Clark <clarks@purdue.edu>
# @copyright    Copyright 2004-2011 Purdue University. All rights reserved.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2004-2011 Purdue University
# All rights reserved.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of Purdue University.
#
import os.path
import re

from TimeConversion import minTohhmmss

class RemoteBatchLSF:
   def __init__(self,
                localJobId,
                trial,
                appScriptName,
                environment,
                stageFiles,
                transferExecutable,
                executable,
                applicationRootDirectory,
                arguments,
                stdinput,
                isMultiCoreRequest,
                computationMode,
                preManagerCommands,
                managerCommand,
                postManagerCommands,
                nNodes,
                ppn,
                wallTime,
                remoteBatchAccount,
                timestampStart,
                timestampFinish,
                timeResults):
      self.localJobId          = localJobId
      self.trial               = trial
      self.appScriptName       = appScriptName
      self.environment         = environment
      self.stageFiles          = stageFiles
      self.transferExecutable  = transferExecutable
      if transferExecutable:
         if stageFiles:
            self.executable    = os.path.join(".",os.path.basename(executable))
         else:
            self.executable    = executable
      else:
         if not executable.startswith('/') and not executable.startswith('$') and applicationRootDirectory != "":
            self.executable    = os.path.join(applicationRootDirectory,executable)
         else:
            self.executable    = executable
      self.arguments           = arguments
      self.stdinput            = stdinput
      self.isMultiCoreRequest  = isMultiCoreRequest
      self.computationMode     = computationMode
      self.preManagerCommands  = preManagerCommands
      self.managerCommand      = managerCommand
      self.postManagerCommands = postManagerCommands
      self.nNodes              = nNodes
      self.ppn                 = ppn
      self.wallTime            = minTohhmmss(wallTime)
      if remoteBatchAccount != "":
         self.account          = '#BSUB -P ' + remoteBatchAccount
      else:
         self.account          = ''
      self.timestampStart      = timestampStart
      self.timestampFinish     = timestampFinish
      self.timeResults         = timeResults


   def __makeSerialTemplate(self):
      return """#!/bin/sh
#BSUB -n NNODES
#BSUB -W WALLTIME
#BSUB -J JOB_JOBID
#BSUB -o lsf_JOBID.stdout
#BSUB -e lsf_JOBID.stderr
REMOTEBATCHACCOUNT

ENVIRONMENT

trap cleanup HUP INT QUIT ABRT TERM

cleanup()
{
   echo "Abnormal termination by signal"
   if [ ! -s TS_FINISH ] ; then
      date +"%s" > TS_FINISH
   fi
   exit 1
}

TIMEPATH=
for timePath in ${HOME}/bin/time /usr/bin/time /usr/local/bin/time /apps/submit/bin/time ; do
   if [ -x ${timePath} ] ; then
      TIMEPATH=${timePath}
      break
   fi
done

date +"%s" > TS_START

${TIMEPATH} -p -o TIME_RESULTS EXECUTABLE ARGUMENTS STDIN > JOBID.stdout 2> JOBID.stderr

date +"%s" > TS_FINISH
"""


   def __buildSerialFile(self):
      # setup regex's for the template
      re_stdinput    = re.compile("STDIN")
      re_wallTime    = re.compile("WALLTIME")
      re_environment = re.compile("ENVIRONMENT")
      re_executable  = re.compile("EXECUTABLE")
      re_arguments   = re.compile("ARGUMENTS")
      re_tsStart     = re.compile("TS_START")
      re_tsFinish    = re.compile("TS_FINISH")
      re_timeResults = re.compile("TIME_RESULTS")
      re_jobid       = re.compile("JOBID")
      re_nnodes      = re.compile("NNODES")
      re_ppn         = re.compile("PPN")
      re_account     = re.compile("REMOTEBATCHACCOUNT")

      template = self.__makeSerialTemplate()

      environmentExport = ""
      environmentVars = self.environment.split()
      for environmentVar in environmentVars:
         environmentExport += "export " + environmentVar + "\n"

      if self.stdinput == "":
         template = re_stdinput.sub(self.stdinput,template)
      else:
         template = re_stdinput.sub(" < "+self.stdinput,template)
      template = re_wallTime.sub(self.wallTime,template)
      template = re_environment.sub(environmentExport,template)
      template = re_executable.sub(self.executable,template)
      template = re_arguments.sub(self.arguments,template)
      template = re_tsStart.sub(self.timestampStart,template)
      template = re_tsFinish.sub(self.timestampFinish,template)
      template = re_timeResults.sub(self.timeResults,template)
      template = re_jobid.sub("%s_%02d" % (self.localJobId,self.trial),template)
      template = re_nnodes.sub(self.nNodes,template)
      template = re_ppn.sub(self.ppn,template)
      template = re_account.sub(self.account,template)

      return(template)


   def __makeMPITemplate(self):
      return """#!/bin/sh
#BSUB -n NNODES
#BSUB -W WALLTIME
#BSUB -J JOB_JOBID
#BSUB -o lsf_JOBID.stdout
#BSUB -e lsf_JOBID.stderr
REMOTEBATCHACCOUNT

ENVIRONMENT

trap cleanup HUP INT QUIT ABRT TERM

cleanup()
{
   echo "Abnormal termination by signal"
   if [ ! -s TS_FINISH ] ; then
      date +"%s" > TS_FINISH
   fi
   exit 1
}

chmod +x APPSCRIPTNAME

date +"%s" > TS_START

PREMANAGERCOMMANDS
MANAGERCOMMAND `pwd`/APPSCRIPTNAME > JOBID.stdout 2> JOBID.stderr
POSTMANAGERCOMMANDS

date +"%s" > TS_FINISH

touch TIME_RESULTS-0
cat TIME_RESULTS-[0-9]* >> TIME_RESULTS
rm -f TIME_RESULTS-[0-9]*
"""


   def __buildMPIFile(self):
      # setup regex's for the template
      re_preManagerCommands  = re.compile("PREMANAGERCOMMANDS")
      re_managerCommand      = re.compile("MANAGERCOMMAND")
      re_postManagerCommands = re.compile("POSTMANAGERCOMMANDS")
      re_wallTime            = re.compile("WALLTIME")
      re_nnodes              = re.compile("NNODES")
      re_ppn                 = re.compile("PPN")
      re_environment         = re.compile("ENVIRONMENT")
      re_appScriptName       = re.compile("APPSCRIPTNAME")
      re_tsStart             = re.compile("TS_START")
      re_tsFinish            = re.compile("TS_FINISH")
      re_timeResults         = re.compile("TIME_RESULTS")
      re_jobid               = re.compile("JOBID")
      re_account             = re.compile("REMOTEBATCHACCOUNT")

      template = self.__makeMPITemplate()

      environmentExport = ""
      environmentVars = self.environment.split()
      for environmentVar in environmentVars:
         environmentExport += "export " + environmentVar + "\n"

      template = re_preManagerCommands.sub("\n".join(self.preManagerCommands),template)
      template = re_postManagerCommands.sub("\n".join(self.postManagerCommands),template)
      template = re_managerCommand.sub(self.managerCommand,template)
      template = re_wallTime.sub(self.wallTime,template)
      template = re_nnodes.sub(self.nNodes,template)
      template = re_ppn.sub(self.ppn,template)
      template = re_environment.sub(environmentExport,template)
      template = re_appScriptName.sub(self.appScriptName,template)
      template = re_tsStart.sub(self.timestampStart,template)
      template = re_tsFinish.sub(self.timestampFinish,template)
      template = re_timeResults.sub(self.timeResults,template)
      template = re_jobid.sub("%s_%02d" % (self.localJobId,self.trial),template)
      template = re_account.sub(self.account,template)

      return(template)


   def buildBatchScript(self):
      batchLogName = ""
      batchScriptName = "%s_%02d.lsf" % (self.localJobId,self.trial)
      if self.isMultiCoreRequest:
         if   self.computationMode == 'mpi':
            batchScript = self.__buildMPIFile()
         elif self.computationMode == 'parallel':
            batchScript = self.__buildSerialFile()
         elif self.computationMode == 'matlabmpi':
            batchScript = ""
            batchScriptName = ""
      else:
         batchScript = self.__buildSerialFile()

      return(batchLogName,batchScriptName,batchScript)

