#
# @package      hubzero-submit-distributor
# @file         RemoteBatchLL.py
# @author       Steve Clark <clarks@purdue.edu>
# @copyright    Copyright 2004-2011 Purdue University. All rights reserved.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2004-2011 Purdue University
# All rights reserved.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of Purdue University.
#
import os.path
import re

from TimeConversion import minTohhmmss

class RemoteBatchLL:
   def __init__(self,
                localJobId,
                trial,
                appScriptName,
                environment,
                stageFiles,
                transferExecutable,
                executable,
                applicationRootDirectory,
                arguments,
                stdinput,
                isMultiCoreRequest,
                computationMode,
                preManagerCommands,
                managerCommand,
                postManagerCommands,
                nNodes,
                ppn,
                remoteBatchPartition,
                remoteBatchPartitionSize,
                remoteBatchConstraints,
                wallTime,
                quotaLimit,
                timestampStart,
                timestampFinish,
                timeResults):
      self.localJobId               = localJobId
      self.trial                    = trial
      self.appScriptName            = appScriptName
      self.environment              = environment
      self.stageFiles               = stageFiles
      self.transferExecutable       = transferExecutable
      if transferExecutable:
         if stageFiles:
            self.executable         = os.path.join(".",os.path.basename(executable))
         else:
            self.executable         = executable
      else:
         if not executable.startswith('/') and not executable.startswith('$') and applicationRootDirectory != "":
            self.executable         = os.path.join(applicationRootDirectory,executable)
         else:
            self.executable         = executable
      self.arguments                = arguments
      self.stdinput                 = stdinput
      self.isMultiCoreRequest       = isMultiCoreRequest
      self.computationMode          = computationMode
      self.preManagerCommands       = preManagerCommands
      self.managerCommand           = managerCommand
      self.postManagerCommands      = postManagerCommands
      self.nNodes                   = nNodes
      self.ppn                      = ppn
      self.remoteBatchPartition     = remoteBatchPartition
      self.remoteBatchPartitionSize = remoteBatchPartitionSize
      self.remoteBatchConstraints   = remoteBatchConstraints
      self.wallTime                 = minTohhmmss(wallTime)
      self.quotaLimit               = quotaLimit
      self.timestampStart           = timestampStart
      self.timestampFinish          = timestampFinish
      self.timeResults              = timeResults


   def __makeMPINoForkTemplate(self):
      return """#!/bin/sh
# @ job_type = bluegene
# @ class = PARTITION
#
# Specify partition using the following statement. This statement is the only way a partition should ever be
# specified in a LoadLeveler job control file.
# The partition you designate here indicates the number of nodes, so note that we do not specify a
# desired number of nodes anywhere in the job control file.
# @ bg_size = PARTITIONSIZE
#
# @ input = STDIN
# @ output = ll_JOBID.stdout
# @ error = ll_JOBID.stderr
#
# Maximum wall clock time for job will be in minutes.
# @ wall_clock_limit = WALLTIME
#
# @ notification = never
# @ queue

cd ${LOADL_STEP_INITDIR}

date +"%s" > TS_START

PREMANAGERCOMMANDS
MANAGERCOMMAND -cwd ${LOADL_STEP_INITDIR} ENVIRONMENT -exe EXECUTABLE ARGUMENTS > JOBID.stdout 2> JOBID.stderr
POSTMANAGERCOMMANDS

date +"%s" > TS_FINISH

startTime=`cat TS_START`
finishTime=`cat TS_FINISH`
let wallTime=finishTime-startTime
cnt=0
while [ "${cnt}" != "NPROCESSORS" ] ; do
   cat >> TIME_RESULTS <<EOF
real ${wallTime}.00
user ${wallTime}.00
sys 0.00
EOF
   cnt=`expr ${cnt} + 1`
done
"""


   def __buildMPINoForkFile(self):
      # setup regex's for the template
      re_stdinput            = re.compile("STDIN")
      re_executable          = re.compile("EXECUTABLE")
      re_arguments           = re.compile("ARGUMENTS")
      re_preManagerCommands  = re.compile("PREMANAGERCOMMANDS")
      re_managerCommand      = re.compile("MANAGERCOMMAND")
      re_postManagerCommands = re.compile("POSTMANAGERCOMMANDS")
      re_wallTime            = re.compile("WALLTIME")
      re_partitionsize       = re.compile("PARTITIONSIZE")
      re_processors          = re.compile("NPROCESSORS")
      re_partition           = re.compile("PARTITION")
      re_environment         = re.compile("ENVIRONMENT")
      re_tsStart             = re.compile("TS_START")
      re_tsFinish            = re.compile("TS_FINISH")
      re_timeResults         = re.compile("TIME_RESULTS")
      re_jobid               = re.compile("JOBID")

      template = self.__makeMPINoForkTemplate()

      environmentMpirun = ""
      environmentVars = self.environment.split()
      for environmentVar in environmentVars:
         environmentMpirun += environmentVar + " "

      if self.stdinput == "":
         template = re_stdinput.sub("/dev/null",template)
      else:
         template = re_stdinput.sub(self.stdinput,template)
      template = re_executable.sub(self.executable,template)
      if self.arguments != "":
         template = re_arguments.sub(' -args "' + self.arguments + '"',template)
      else:
         template = re_arguments.sub(self.arguments,template)
      template = re_preManagerCommands.sub("\n".join(self.preManagerCommands),template)
      template = re_postManagerCommands.sub("\n".join(self.postManagerCommands),template)
      template = re_managerCommand.sub(self.managerCommand,template)
      template = re_wallTime.sub(self.wallTime,template)
      nProcessors = str(int(self.nNodes)*int(self.ppn))
      template = re_processors.sub(nProcessors,template)
      template = re_partitionsize.sub(self.remoteBatchPartitionSize,template)
      template = re_partition.sub(self.remoteBatchPartition,template)
      if environmentMpirun != "":
         template = re_environment.sub(' -env "' + environmentMpirun.strip() + '"',template)
      else:
         template = re_environment.sub(environmentMpirun,template)
      template = re_tsStart.sub(self.timestampStart,template)
      template = re_tsFinish.sub(self.timestampFinish,template)
      template = re_timeResults.sub(self.timeResults,template)
      template = re_jobid.sub("%s_%02d" % (self.localJobId,self.trial),template)

      return(template)


   def buildBatchScript(self):
      batchLogName = ""
      batchScriptName = "%s_%02d.ll" % (self.localJobId,self.trial)
      if self.isMultiCoreRequest:
         if self.computationMode == 'mpiNoFork':
            batchScript = self.__buildMPINoForkFile()
         else:
            batchScript = ""
      else:
         batchScript = ""

      return(batchLogName,batchScriptName,batchScript)

