#
# @package      hubzero-submit-distributor
# @file         JobMonitor.py
# @author       Steve Clark <clarks@purdue.edu>
# @copyright    Copyright 2004-2011 Purdue University. All rights reserved.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2004-2011 Purdue University
# All rights reserved.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of Purdue University.
#
import re
import shelve
import time
import traceback

from LogMessage  import logID as log
from MessageCore import MessageCore

class JobMonitor(MessageCore):
   def __init__(self,
                host,
                port,
                repeatDelay=5,
                fixedBufferSize=64,
                activeJobDBPath="monitorJobDB",
                activeJobDumpPath=""):
      MessageCore.__init__(self,bindHost=host,bindPort=port,repeatDelay=repeatDelay)
      self.fixedBufferSize   = fixedBufferSize
      self.activeJobDBPath   = activeJobDBPath
      self.activeJobDumpPath = activeJobDumpPath
      self.activeJobs        = {}
      self.activeJobSites    = {}


   def loadActiveJobs(self):
      log("loading active jobs")
      if self.activeJobDBPath != "":
         try:
            self.activeJobs = shelve.open(self.activeJobDBPath)
            log("%d jobs loaded from DB file" % (len(self.activeJobs)))
            jobQueue = '?'
            for globalJobId,activeJob in self.activeJobs.items():
               if len(activeJob) == 3:
                  jobStatus,jobStage,timeRecorded = activeJob
                  self.updateActiveJob(globalJobId,jobStatus,jobStage,jobQueue)
         except IOError:
            log(traceback.format_exc())

      if self.activeJobDumpPath != "":
         dumpedJobs = []
         try:
            dumpFile = open(self.activeJobDumpPath,'r')
            dumpedJobs = dumpFile.readlines()
            dumpFile.close()
         except:
            pass

         for dumpedJob in dumpedJobs:
            timeRecorded = 0.
            jobStatus = 'D'
            jobStage = '?'
            jobQueue = '?'
            fields = dumpedJob.split()
            nFields = len(fields)
            if nFields > 0:
               globalJobId = fields[0]
               if nFields > 1:
                  jobStatus = fields[1]
                  if nFields > 2:
                     jobStage = fields[2]
                     if   nFields > 4:
                        jobQueue = fields[3]
                        timeRecorded = float(fields[4])
                     elif nFields > 3:
                        timeRecorded = float(fields[3])
               self.addActiveJob(globalJobId,jobStatus,jobStage,jobQueue,timeRecorded)

         log("%d jobs loaded from dump file" % (len(dumpedJobs)))


   def dumpActiveJobs(self):
      dumpFile = open(self.activeJobDumpPath,'w')
      for activeJob in self.activeJobs.items():
         dumpFile.write("%s %s %s %s %f\n" % (activeJob[0],activeJob[1][0],activeJob[1][1],activeJob[1][2],activeJob[1][3]))
      dumpFile.close()


   def close(self):
      self.activeJobs.close()


   def addActiveJobSite(self,
                        jobSite,
                        value):
      self.activeJobSites[jobSite] = value


   def deleteActiveJobSite(self,
                           jobSite):
      del self.activeJobSites[jobSite]


   def isJobSiteActive(self,
                       jobSite):
      return(jobSite in self.activeJobSites)


   def getActiveJobSiteQueues(self,
                              jobSite):
      activeJobSiteQueues = []
      for globalJobId,activeJob in self.activeJobs.items():
         site,localJobId = globalJobId.split(':')
         if site == jobSite:
            (jobStatus,jobStage,jobQueue,timeRecorded) = activeJob
            if jobQueue not in activeJobSiteQueues:
               activeJobSiteQueues.append(jobQueue)

      return(activeJobSiteQueues)


   def addActiveJob(self,
                    globalJobId,
                    jobStatus,
                    jobStage,
                    jobQueue,
                    timeRecorded=0.):
      if timeRecorded > 0.:
         self.activeJobs[globalJobId] = (jobStatus,jobStage,jobQueue,timeRecorded)
      else:
         self.activeJobs[globalJobId] = (jobStatus,jobStage,jobQueue,time.time())


   def updateActiveJob(self,
                       globalJobId,
                       jobStatus,
                       jobStage,
                       jobQueue):
      if globalJobId in self.activeJobs:
         self.activeJobs[globalJobId] = (jobStatus,jobStage,jobQueue,time.time())


   def markNewActiveJobsAsDone(self,
                               jobSite):
      siteMatch = re.compile('^'+jobSite+':')
      newJobs = filter(siteMatch.match,self.activeJobs.keys())
      for newJob in newJobs:
         jobStatus,jobStage,jobQueue,timeRecorded = self.activeJobs[newJob]
         if jobStatus == 'N':
            jobStatus = 'D'
            self.updateActiveJob(newJob,jobStatus,jobStage,jobQueue)


   def purgeActiveJobs(self,
                       purgeJobStatus,
                       cutOffAge=0.):
      cutOffTime = time.time()-cutOffAge
      markedForDeletion = []
      for globalJobId in self.activeJobs:
         jobStatus,jobStage,jobQueue,timeRecorded = self.activeJobs[globalJobId]
         if jobStatus == purgeJobStatus and timeRecorded <= cutOffTime:
            markedForDeletion.append(globalJobId)

      nPurgeActiveJobs = len(markedForDeletion)
      for globalJobId in markedForDeletion:
         del self.activeJobs[globalJobId]
      del markedForDeletion

      return(nPurgeActiveJobs)


   def getActiveJobCount(self):
      return(len(self.activeJobs))


   def processRequest(self,
                      channel):
      channelClosed = False
      newJobSite    = ""
      newJobId      = ""

      message = self.receiveMessage(channel,self.fixedBufferSize)
      if message != "":
         if re.match('[QSRT]:',message):
            try:
               messageType,siteId     = message.split(':')
               messageSite,localJobId = siteId.split()
            except:
               log("Failed QSRT message request: " + message)
               messageType = ''
         else:
            log("Failed message request: " + message)
            messageType = ''

         if   messageType == 'Q':                        # job Query
            globalJobId = messageSite + ':' + localJobId
            markReportedAsDone = False
            if globalJobId in self.activeJobs:
               jobStatus,jobStage,jobQueue,timeRecorded = self.activeJobs[globalJobId]
               if   jobStatus == 'D':
                  markReportedAsDone = True
               elif jobStatus == 'Dr':
                  jobStatus = 'D'
            else:
               jobStatus,jobStage,jobQueue = ('?','?','?')
            if self.sendMessage(channel,jobStatus + " " + jobStage,self.fixedBufferSize) > 0:
               newJobSite = messageSite
               if markReportedAsDone:
                  jobStatus = 'Dr'
                  self.updateActiveJob(globalJobId,jobStatus,jobStage,jobQueue)
         elif messageType == 'S':                        # new job Submission
            globalJobId = messageSite + ':' + localJobId
            if globalJobId in self.activeJobs:
               jobStatus,jobStage,jobQueue,timeRecorded = self.activeJobs[globalJobId]
            else:
               jobStatus,jobStage,jobQueue = ('N','Job','?')
               self.addActiveJob(globalJobId,jobStatus,jobStage,jobQueue)
            self.dumpActiveJobs()
            if self.sendMessage(channel,jobStatus + " " + jobStage,self.fixedBufferSize) > 0:
               newJobSite = messageSite
               newJobId   = localJobId
         elif messageType == 'T':                        # Terminate job
            globalJobId = messageSite + ':' + localJobId
            if globalJobId in self.activeJobs:
               jobStatus,jobStage,jobQueue,timeRecorded = self.activeJobs[globalJobId]
            else:
               jobStatus,jobStage,jobQueue = ('?','?','?')
            if self.sendMessage(channel,jobStatus + " " + jobStage,self.fixedBufferSize) > 0:
               if globalJobId in self.activeJobs:
                  jobStatus,jobStage = ('D','Job')
                  self.updateActiveJob(globalJobId,jobStatus,jobStage,jobQueue)
         elif messageType == 'R':                        # Report active jobs
            report = ""
            messageSites = []
            if messageSite == "*":
               for globalJobId in self.activeJobs.keys():
                  globalJobMessageSite,globalJobLocalJobId = globalJobId.split(':')
                  if not globalJobMessageSite in messageSites:
                     messageSites.append(globalJobMessageSite)
            else:
               messageSites.append(messageSite)

            siteDelimeter = ''
            maxLastReportTime = 0
            for messageSite in messageSites:
               if messageSite in self.activeJobSites:
                  lastReportTime = self.activeJobSites[messageSite]
               else:
                  lastReportTime = 0
               maxLastReportTime = max(maxLastReportTime,lastReportTime)
               siteReport = messageSite + " " + str(lastReportTime)

               queueDelimeter = ' @ '
               activeJobSiteQueues = self.getActiveJobSiteQueues(messageSite)
               for activeJobSiteQueue in activeJobSiteQueues:
                  queueReport = activeJobSiteQueue

                  jobDelimeter = ' : '
                  if localJobId == "*":
                     siteMatch = re.compile('^'+messageSite+':')
                     reportJobs = filter(siteMatch.match,self.activeJobs.keys())
                     for reportJob in reportJobs:
                        jobStatus,jobStage,jobQueue,timeRecorded = self.activeJobs[reportJob]
                        if jobQueue == activeJobSiteQueue:
                           queueReport += jobDelimeter + reportJob.split(':')[1] + " " + jobStatus + " " + jobStage
                  else:
                     globalJobId = messageSite + ':' + localJobId
                     if globalJobId in self.activeJobs:
                        jobStatus,jobStage,jobQueue,timeRecorded = self.activeJobs[globalJobId]
                     else:
                        jobStatus,jobStage,jobQueue = ('?','?','?')
                     if jobQueue == activeJobSiteQueue:
                        queueReport += jobDelimeter + localJobId + " " + jobStatus + " " + jobStage

                  siteReport += queueDelimeter + queueReport

               report += siteDelimeter + siteReport
               siteDelimeter = ' | '

            reportLength = len(report)
            if self.sendMessage(channel,str(reportLength) + " " + str(maxLastReportTime),self.fixedBufferSize) > 0:
               if reportLength > 0:
                  self.sendMessage(channel,report)
            del messageSites
      else:
         try:
            channel.close()
            channelClosed = True
         except:
            log("close channel failed")

      return(channelClosed,newJobSite,newJobId)

