#
# Copyright (c) 2004-2010 Purdue University All rights reserved.
# 
# Developed by: HUBzero Technology Group, Purdue University
#               http://hubzero.org
# 
# HUBzero is free software: you can redistribute it and/or modify it under the terms of the
# GNU Lesser General Public License as published by the Free Software Foundation, either
# version 3 of the License, or (at your option) any later version.
# 
# HUBzero is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
# without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
# See the GNU Lesser General Public License for more details.  You should have received a
# copy of the GNU Lesser General Public License along with HUBzero.
# If not, see <http://www.gnu.org/licenses/>.
# 
# GNU LESSER GENERAL PUBLIC LICENSE
# Version 3, 29 June 2007
# Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>
#
import re

class RemoteInstantAppScript:
   def __init__(self,
                localJobId,
                workingDirectory,
                executable,
                stdinput,
                arguments,
                useEnvironment,
                environment,
                isMultiCoreRequest,
                computationMode,
                mpiRankVariable,
                TIMESTAMPTRANSFERRED,
                TIMESTAMPSTART,
                TIMESTAMPFINISH,
                TIMERESULTS):
      self.localJobId           = localJobId
      self.workingDirectory     = workingDirectory
      self.executable           = executable
      self.stdinput             = stdinput
      self.arguments            = arguments
      self.useEnvironment       = useEnvironment
      self.environment          = environment
      self.isMultiCoreRequest   = isMultiCoreRequest
      self.computationMode      = computationMode
      self.mpiRankVariable      = mpiRankVariable
      self.TIMESTAMPTRANSFERRED = TIMESTAMPTRANSFERRED
      self.TIMESTAMPSTART       = TIMESTAMPSTART
      self.TIMESTAMPFINISH      = TIMESTAMPFINISH
      self.TIMERESULTS          = TIMERESULTS


   def __makeAppSerialScriptTemplate(self):
      return """#!/bin/sh
#
trap cleanup HUP INT QUIT ABRT TERM

cleanup()
{
#  kill -s TERM $(ps --no-headers --format pid --ppid `jobs -p`)
   kill -s TERM `jobs -p`
   if [ ! -s TS_FINISH ] ; then
      date +"%s" > TS_FINISH
   fi
}

USEENVIRONMENT

ENVIRONMENT

# Change to directory where job was submitted.
cd WORKINGDIRECTORY
PATH=WORKINGDIRECTORY:${PATH}
export PATH

TIMEPATH=
for timePath in ${HOME}/bin/time /usr/bin/time /usr/local/bin/time /apps/submit/bin/time ; do
   if [ -x ${timePath} ] ; then
      TIMEPATH=${timePath}
      break
   fi
done

date +"%s" > TS_TRANSFERRED
date +"%s" > TS_START

${TIMEPATH} -p -o TIME_RESULTS EXECUTABLE ARGUMENTS STDIN &
wait %1

date +"%s" > TS_FINISH
"""


   def __buildAppSerialScript(self):
      # setup regex's for the template
      re_stdinput         = re.compile("STDIN")
      re_useEnvironment   = re.compile("USEENVIRONMENT")
      re_environment      = re.compile("ENVIRONMENT")
      re_executable       = re.compile("EXECUTABLE")
      re_arguments        = re.compile("ARGUMENTS")
      re_tsTransferred    = re.compile("TS_TRANSFERRED")
      re_tsStart          = re.compile("TS_START")
      re_tsFinish         = re.compile("TS_FINISH")
      re_timeResults      = re.compile("TIME_RESULTS")
      re_jobid            = re.compile("JOBID")
      re_workingDirectory = re.compile("WORKINGDIRECTORY")

      template = self.__makeAppSerialScriptTemplate()

      environmentExport = ""
      environmentVars = self.environment.split()
      for environmentVar in environmentVars:
         environmentExport += "export " + environmentVar + "\n"

      if self.stdinput == "":
          template = re_stdinput.sub(self.stdinput,template)
      else:
          template = re_stdinput.sub(" < "+self.stdinput,template)
      template = re_useEnvironment.sub(self.useEnvironment,template)
      template = re_environment.sub(environmentExport,template)
      template = re_executable.sub(self.executable,template)
      template = re_arguments.sub(self.arguments,template)
      template = re_tsTransferred.sub(self.TIMESTAMPTRANSFERRED,template)
      template = re_tsStart.sub(self.TIMESTAMPSTART,template)
      template = re_tsFinish.sub(self.TIMESTAMPFINISH,template)
      template = re_timeResults.sub(self.TIMERESULTS,template)
      template = re_jobid.sub(self.localJobId,template)
      template = re_workingDirectory.sub(self.workingDirectory,template)

      return(template)


   def __makeAppMPIScriptTemplate(self):
      return """#!/bin/sh
#
SCRIPT=`basename $0`
WORKINGDIR=`dirname $0`

cd ${WORKINGDIR}

STD_INPUT=STDIN

USEENVIRONMENT
ENVIRONMENT

TIMEPATH=
for timePath in ${HOME}/bin/time /usr/bin/time /usr/local/bin/time /apps/submit/bin/time ; do
   if [ -x ${timePath} ] ; then
      TIMEPATH=${timePath}
      break
   fi
done

rankExtension=`printf "%06d" ${MPIRANKVARIABLE}`
${TIMEPATH} -p -o TIME_RESULTS-${rankExtension} EXECUTABLE ARGUMENTS < ${STD_INPUT}
"""


   def __makeAppMPIScriptTemplateNoRankVar(self):
      return """#!/bin/sh
#
SCRIPT=`basename $0`
WORKINGDIR=`dirname $0`

args=( "$@" )
for (( i1=0 ; i1 < ${#args[@]} ; i1++ )) ; do
   if   [ "${args[i1]}" = "-p4pg" ] ; then
      p4rank=0
   elif [ "${args[i1]}" = "-p4wd" ] ; then
      (( i1++ ))
      args[${i1}]=${WORKINGDIR}
   elif [ "${args[i1]}" = "-p4rmrank" ] ; then
      (( i1++ ))
      p4rank=${args[i1]}
   fi
done
if [ "${p4rank}" = "0" ] ; then
   set -- ${args[@]}
fi

cd ${WORKINGDIR}

STD_INPUT=STDIN

TIMEPATH=
for timePath in ${HOME}/bin/time /usr/bin/time /usr/local/bin/time /apps/submit/bin/time ; do
   if [ -x ${timePath} ] ; then
      TIMEPATH=${timePath}
      break
   fi
done

USEENVIRONMENT
ENVIRONMENT

rankExtension=`printf "%06d" ${p4rank}`
if [ "${p4rank}" = "0" ] ; then
   ${TIMEPATH} -p -o TIME_RESULTS-${rankExtension} EXECUTABLE ARGUMENTS "$@" < ${STD_INPUT}
else
   ${TIMEPATH} -p -o TIME_RESULTS-${rankExtension} EXECUTABLE "$@" < ${STD_INPUT}
fi
"""


   def __buildAppMPIScript(self):
      # setup regex's for the template
      re_jobid           = re.compile("JOBID")
      re_executable      = re.compile("EXECUTABLE")
      re_stdinput        = re.compile("STDIN")
      re_arguments       = re.compile("ARGUMENTS")
      re_useEnvironment  = re.compile("USEENVIRONMENT")
      re_environment     = re.compile("ENVIRONMENT")
      re_mpiRankVariable = re.compile("MPIRANKVARIABLE")
      re_timeResults     = re.compile("TIME_RESULTS")

      environmentExport = ""
      environmentVars = self.environment.split()
      if self.mpiRankVariable == "":
         template = self.__makeAppMPIScriptTemplateNoRankVar()
      else:
         template = self.__makeAppMPIScriptTemplate()
      re_cmdstdinput  = re.compile(" < \${STD_INPUT}")
      useEnvironmentExport = self.useEnvironment
      for environmentVar in environmentVars:
         environmentExport += "export " + environmentVar + "\n"

      template = re_executable.sub(self.executable,template)
      template = re_stdinput.sub(self.stdinput,template)
      if self.stdinput == "":
         template = re_cmdstdinput.sub("",template)
      template = re_arguments.sub(self.arguments,template)
      template = re_useEnvironment.sub(useEnvironmentExport,template)
      template = re_environment.sub(environmentExport,template)
      template = re_mpiRankVariable.sub(self.mpiRankVariable,template)
      template = re_timeResults.sub(self.TIMERESULTS,template)
      template = re_jobid.sub(self.localJobId,template)

      return (template)


   def buildAppScript(self):
      if self.isMultiCoreRequest:
         if   self.computationMode == 'mpi':
            appScript = self.__buildAppMPIScript()
         elif self.computationMode == 'parallel':
            appScript = ""
         elif self.computationMode == 'matlabmpi':
            appScript = ""
      else:
         appScript = self.__buildAppSerialScript()

      return(appScript)


