#!/usr/bin/env python
#
# Copyright (c) 2004-2010 Purdue University All rights reserved.
# 
# Developed by: HUBzero Technology Group, Purdue University
#               http://hubzero.org
# 
# HUBzero is free software: you can redistribute it and/or modify it under the terms of the
# GNU Lesser General Public License as published by the Free Software Foundation, either
# version 3 of the License, or (at your option) any later version.
# 
# HUBzero is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
# without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
# See the GNU Lesser General Public License for more details.  You should have received a
# copy of the GNU Lesser General Public License along with HUBzero.
# If not, see <http://www.gnu.org/licenses/>.
# 
# GNU LESSER GENERAL PUBLIC LICENSE
# Version 3, 29 June 2007
# Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>
#
# ----------------------------------------------------------------------
#  distributor
#
#  script which generates a pair of files for remote job submission.
#  The first file is the batch submission script, the second is a shell script
#  that wraps the application and generates time usage data.
#
#  RUN AS FOLLOWS:
#    distributor -j <jobId> -v <venue> -i <inputfile> -o <outputfile>
#                -n <nCpus> -N <ppn> -w <wallTime> -e <environment variable>
#
#  <jobId>                 is the job identifier
#  <executable>            is the MPI enabled executable to be run.
#  <inputfile>             is a file to be sent to the remote site
#  <outputfile>            is a file to be retrieved from the remote site
#  <nCpus>                 the number of processors to use
#  <ppn>                   the processors/node to use
#  <wallTime>              wall time limit for remote process
#  <environment variable>  variable=value
#  <venue>                 remote venue
#
#                                                 | cpu time    |
#                                                 | from time() |
#                                                 |             |
#                                     |  waiting  | real(wall)  |
#                                     |   time    |    time     |
#                                     |           |             |
#    |------------|---------|---------|-----------|-------------|----------|----
# simulate     condor     site      files     simulation    simulation   files
#  button        job    selected    staged     started        ended      staged
# pressed    submitted               in                                   out
#

from optparse import OptionParser,SUPPRESS_HELP,SUPPRESS_USAGE
import sys
import os
import re
import signal
import select
import time
import datetime
import popen2
import random
import stat
import traceback
import math
import sets

from LogMessage               import openLog, logJobId as log, logSetJobId
from SitesInfo                import SitesInfo
from ToolsInfo                import ToolsInfo
from ManagersInfo             import ManagersInfo
from TunnelsInfo              import TunnelsInfo
from EnvironmentWhitelistInfo import EnvironmentWhitelistInfo
from RemoteInstantSCRIPT      import RemoteInstantSCRIPT
from RemoteBatchPBS           import RemoteBatchPBS
from RemoteBatchPBS8          import RemoteBatchPBS8
from RemoteInstantAppScript   import RemoteInstantAppScript
from RemoteBatchAppScript     import RemoteBatchAppScript
from RemoteJobMonitor         import RemoteJobMonitor
from RemoteTunnelMonitor      import RemoteTunnelMonitor
from TimeConversion           import hhmmssTomin
from GroupMembership          import isSubmitMember
from TarCommand               import buildCreate as buildCreateTarCommand, buildAppend as buildAppendTarCommand

distributorVersion        = 'current'
distributorHome           = os.path.join(os.sep,'opt','submit')
distributorLogDir         = os.path.join(os.sep,'var','log','submit')
distributorLogFile        = "distributor.log"
distributorGridLogFile    = "gridjobid.log"
distributorHistoryLogFile = "gridjobhistory.log"

RUNJOB_ROOT   = ""
CONDOR_ROOT   = ""
CONDOR_CONFIG = ""
PBS_ROOT      = ""
runjobExists  = False

checkQuota = True
quotaLimit = -1

logUserRemotely = False

remoteBatchAccount = ""

remoteJobMonitorHost    = ""
remoteJobMonitorPort    = 5727
remoteProbeMonitorHost  = ""
remoteProbeMonitorPort  = 5728
remoteTunnelMonitorHost = ""
remoteTunnelMonitorPort = 5729

doubleDashTerminator = False

IDENTITY = ""

RECEIVEINPUTCOMMAND    = 'receiveinput.sh'
SUBMITBATCHJOBCOMMAND  = 'submitbatchjob.sh'
TRANSMITRESULTSCOMMAND = 'transmitresults.sh'
CLEANUPJOBCOMMAND      = 'cleanupjob.sh'
KILLBATCHJOBCOMMAND    = 'killbatchjob.sh'

TIMESTAMPTRANSFERRED = ".__timestamp_transferred."
TIMESTAMPFINISH      = ".__timestamp_finish."
TIMESTAMPSTART       = ".__timestamp_start."
TIMERESULTS          = ".__time_results."

GRIDRESOURCE = ".__grid_resource."
GRIDHISTORY  = ".__grid_history."
GRIDJOBID    = ".__grid_jobid."
LOCALJOBID   = ".__local_jobid."

jobId                  = 0
remoteJobIdNumber      = ''
remoteBatchSystem      = ''
venueMechanism         = ''
sshBaseCommand         = ''
jobSubmissionMechanism = 'u'    # unknown
venue                  = ''
nCpus                  = 1
workingDirectory       = ''
enteredCommand         = ''
manager                = ''
jobStatistics          = {}
maximumJobIndex        = 0
disableStateCheck      = False
disableScanCheck       = False
disableProbeCheck      = True
x509SubmitProxy        = ""
childPid               = 0
batchScriptName        = ""
appScriptName          = ""

fpUserLogPath = None

def initializeJobStatistic(nCores):
   statistic = {'jobSubmissionMechanism':'Unknown', \
                'remoteJobIdNumber':'', \
                'venue':'', \
                'realTime':0., \
                'userTime':0., \
                'sysTime':0., \
                'exitCode':0, \
                'transferCompleteTime':0., \
                'jobStartedTime':0., \
                'jobFinshedTime':0., \
                'waitingTime':0., \
                'elapsedRunTime':0., \
                'nCores':0, \
               }
   statistic['nCores'] = nCores

   return(statistic)


def logGridHistories():
   global localJobId
   global enteredCommand
   global maximumJobIndex
   global jobStatistics
   global nCpus

   gridHistoryLogFile = open(os.path.join(distributorHome,distributorLogDir,distributorHistoryLogFile),"a")

   re_Files = re.compile(GRIDHISTORY + "_[0-9]+$")
   dirFiles = os.listdir(os.getcwd())
   matchingFiles = filter(re_Files.search,dirFiles)
   if len(matchingFiles) > 0:
      matchingFiles.sort()
      for matchingFile in matchingFiles:
         if len(matchingFile) > len(GRIDHISTORY):
            jobIndex = int(matchingFile[len(GRIDHISTORY)+1:])
         else:
            jobIndex = 1
         maximumJobIndex = max(maximumJobIndex,jobIndex)

         if not jobIndex in jobStatistics:
            jobStatistics[jobIndex] = initializeJobStatistic(nCpus)

         if gridHistoryLogFile:
            fp = open(matchingFile,'r')
            if fp:
               gridHistory = fp.readline()
               gridHistoryLogFile.write("%s\t%d\t%s\t%s\n" % (localJobId,jobIndex,gridHistory.strip(),enteredCommand))
               fp.close()
         os.remove(matchingFile)

   re_Files = re.compile(GRIDHISTORY + "_[0-9]+_$")
   dirFiles = os.listdir(os.getcwd())
   matchingFiles = filter(re_Files.search,dirFiles)
   if len(matchingFiles) > 0:
      matchingFiles.sort()
      for matchingFile in matchingFiles:
         if len(matchingFile) > len(GRIDHISTORY):
            jobIndex = int(matchingFile[len(GRIDHISTORY)+1:-1])
         else:
            jobIndex = 1
         maximumJobIndex = max(maximumJobIndex,jobIndex)

         if not jobIndex in jobStatistics:
            jobStatistics[jobIndex] = initializeJobStatistic(nCpus)

         if gridHistoryLogFile:
            fp = open(matchingFile,'r')
            if fp:
               jobInProgressSite,jobInProgressSetupCompleted = fp.readline().rstrip().split()
               jobInProgressCompleted = int(time.mktime(datetime.datetime.utcnow().timetuple()))
               jobInProgressExitStatus = -3
               gridHistoryLogFile.write("%s\t%d\t%s\t%s\t%d\t%d\t%s\n" % (localJobId,jobIndex,jobInProgressSite,jobInProgressSetupCompleted,jobInProgressCompleted,jobInProgressExitStatus,enteredCommand))
               fp.close()
         os.remove(matchingFile)

   if gridHistoryLogFile:
      gridHistoryLogFile.close()


def reportExitCondition():
   global jobStatistics
   global maximumJobIndex
   global jobId

   jobIndices = jobStatistics.keys()
   jobIndices.sort()

   try:
      for jobIndex in jobIndices:
         mechanism      = jobStatistics[jobIndex]['jobSubmissionMechanism']
         remoteId       = jobStatistics[jobIndex]['remoteJobIdNumber']
         remoteLocation = jobStatistics[jobIndex]['venue']
         exitCode       = jobStatistics[jobIndex]['exitCode']
         if jobIndex != maximumJobIndex:
            if exitCode == 0:
               exitCode = 65533
         cpuTime        = jobStatistics[jobIndex]['userTime']+jobStatistics[jobIndex]['sysTime']
         elapsedRunTime = jobStatistics[jobIndex]['elapsedRunTime']
         waitTime       = jobStatistics[jobIndex]['waitingTime']
         nCpuUsed       = jobStatistics[jobIndex]['nCores']

         message = "venue=%d:%s:%s:%s status=%d cputime=%f realtime=%f waittime=%f ncpus=%d\n" % \
                   (jobIndex,mechanism,remoteId,remoteLocation,exitCode,cpuTime,elapsedRunTime,waitTime,nCpuUsed)
         log(message)
         if fpUserLogPath:
            fpUserLogPath.write("[%s] %d: %s" % (time.ctime(),jobId,message))
            fpUserLogPath.flush()
         os.write(3,message)
   except:
      pass

# SIGTERM is sent by Rappture Abort
# SIGHUP is sent by submit
# SIGHUP, SIGTERM are sent by session termination

abortAttempted = False

def sigGEN_handler(signal, frame):
   global abortAttempted
   global remoteJobIdNumber
   global remoteBatchSystem
   global venueMechanism
   global KILLBATCHJOB
   global CLEANUPJOB
   global sshBaseCommand
   global jobSubmissionMechanism
   global venue
   global nCpus
   global workingDirectory
   global jobStatistics
   global maximumJobIndex
   global childPid
   global batchScriptName
   global appScriptName

   if not abortAttempted:
      abortAttempted = True
      if   remoteJobIdNumber:
         logGridHistories()
         jobStatistics[maximumJobIndex]['exitCode'] = 1 << 8 | signal
         reportExitCondition()

         if   venueMechanism == 'local':
            command = KILLBATCHJOB + " " + remoteJobIdNumber + " " + remoteBatchSystem
            log("command = " + command)
            log(executeCommand(command)[1])
         elif venueMechanism == 'ssh':
            remoteWorkingDirectory = workingDirectory.replace("$","\$")
            command = sshBaseCommand + " \"" + KILLBATCHJOB + " " + remoteJobIdNumber + " " + remoteBatchSystem + "\""
            log("command = " + command)
            log(executeCommand(command)[1])
            command = sshBaseCommand + " \"" + CLEANUPJOB + " " + remoteWorkingDirectory + "\""
            log("command = " + command)
            log(executeCommand(command)[1])
         elif venueMechanism == 'runjob':
            command = "condor_hold " + remoteJobIdNumber + ";condor_rm -constraint DAGManJobId==" + remoteJobIdNumber + ";condor_rm " + remoteJobIdNumber
            log("command = " + command)
            log(executeCommand(command)[1])

      elif venueMechanism == 'ssh' and remoteBatchSystem == 'SCRIPT' and childPid != 0:
         jobStatistics[maximumJobIndex]['exitCode'] = 1 << 8 | signal
         reportExitCondition()
#        log("Send TERM to child script process %d" % (childPid))
#        os.kill(childPid,signal.SIGTERM)
         if batchScriptName != "":
            command = sshBaseCommand + " pkill -TERM -f " + batchScriptName
         else:
            command = sshBaseCommand + " pkill -TERM -f " + appScriptName
         log("command = " + command)
         log(executeCommand(command)[1])

      else:
         jobStatistics[maximumJobIndex]['exitCode'] = 1 << 8 | signal
         reportExitCondition()

      sys.exit(jobStatistics[maximumJobIndex]['exitCode'])


def sigINT_handler(signal, frame):
   log("Received SIGINT!")
   sigGEN_handler(signal, frame)

def sigHUP_handler(signal, frame):
   log("Received SIGHUP!")
   sigGEN_handler(signal, frame)

def sigQUIT_handler(signal, frame):
   log("Received SIGQUIT!")
   sigGEN_handler(signal, frame)

def sigABRT_handler(signal, frame):
   log("Received SIGABRT!")
   sigGEN_handler(signal, frame)

def sigTERM_handler(signal, frame):
   log("Received SIGTERM!")
   sigGEN_handler(signal, frame)

signal.signal(signal.SIGINT, sigINT_handler)
signal.signal(signal.SIGHUP, sigHUP_handler)
signal.signal(signal.SIGQUIT, sigQUIT_handler)
signal.signal(signal.SIGABRT, sigABRT_handler)
signal.signal(signal.SIGTERM, sigTERM_handler)


def getUserQuota():
   global quotaLimit

   quotaCommand = "quota -w | tail -n 1"
   log("quotaCommand = " + quotaCommand)
   exitStatus,stdOutput,stdError = executeCommand(quotaCommand)
   if exitStatus == 0:
      try:
         quotaResults = stdOutput.split()
         if quotaResults[-1] != 'none':
# quota limit is 90% of difference between hard limit and current usage, 1024 byte blocks
            currentUsage = int(quotaResults[1].strip("*"))
            quotaLimit = int((int(quotaResults[3])-currentUsage)*0.9)
      except:
         log(stdOutput)
         log(traceback.format_exc())


def getJobId():
   localJobIdFileName = os.path.join(distributorHome,"localJobId.dat")
   if not os.path.isfile(localJobIdFileName):
      fileObject = open(localJobIdFileName,'w')
      if fileObject:
         fileObject.write('0')
         fileObject.close()
      else:
         log("could not open %s for writing" % (localJobIdFileName))
         sys.stderr.write("could not open %s for writing\n" % (localJobIdFileName))
         sys.stderr.flush()
         jobStatistics[maximumJobIndex]['exitCode'] = 1
         jobStatistics[maximumJobIndex]['nCores']   = 0
         reportExitCondition()
         sys.exit(1)

   fileObject = open(localJobIdFileName,'r')
   previousLocalJobId = int(fileObject.readline().strip())
   fileObject.close()

   nextLocalJobId = previousLocalJobId+1

   fileObject = open(localJobIdFileName,'w')
   fileObject.write('%d' % (nextLocalJobId))
   fileObject.close()

   return(nextLocalJobId)


def executeCommand(command,
                   streamOutput=False):
   global childPid

   child      = popen2.Popen3(command,1)
   childPid   = child.pid
   child.tochild.close() # don't need to talk to child
   childout   = child.fromchild
   childoutFd = childout.fileno()
   childerr   = child.childerr
   childerrFd = childerr.fileno()

   outEOF = errEOF = 0
   BUFSIZ = 4096

   outData = []
   errData = []

   while 1:
      toCheck = []
      if not outEOF:
         toCheck.append(childoutFd)
      if not errEOF:
         toCheck.append(childerrFd)
      ready = select.select(toCheck,[],[]) # wait for input
      if childoutFd in ready[0]:
         outChunk = os.read(childoutFd,BUFSIZ)
         if outChunk == '':
            outEOF = 1
         outData.append(outChunk)
         if streamOutput:
            sys.stdout.write(outChunk)
            sys.stdout.flush()

      if childerrFd in ready[0]:
         errChunk = os.read(childerrFd,BUFSIZ)
         if errChunk == '':
            errEOF = 1
         errData.append(errChunk)
         if streamOutput:
            sys.stderr.write(errChunk)
            sys.stderr.flush()

      if outEOF and errEOF:
         break

   err = child.wait()
   childPid = 0
   if err != 0:
      if os.WIFSIGNALED(err):
         log("%s failed w/ exit code %d signal %d" % (command,os.WEXITSTATUS(err),os.WTERMSIG(err)))
      else:
         err = os.WEXITSTATUS(err)
         log("%s failed w/ exit code %d" % (command,err))
      if not streamOutput:
         log("%s" % ("".join(errData)))

   return(err,"".join(outData),"".join(errData))


def executeSSHCommand(sshCommand,
                      remoteTunnelMonitor,
                      tunnelDesignator,
                      streamOutput=False):

   minimumDelay = 1       #  1 2 4 8 16 32 64 128 256
   maximumDelay = 256
   updateFrequency = 1
   maximumDelayTime = 900

   delayTime = 0
   sleepTime = minimumDelay
   nDelays = 0

   if tunnelDesignator != "":
      remoteTunnelMonitor.incrementTunnelUse(tunnelDesignator)
   exitStatus,stdOutput,stdError = executeCommand(sshCommand,streamOutput)

   while exitStatus and (stdError.count("You don't exist, go away!") > 0):
      nDelays += 1
      time.sleep(sleepTime)
      delayTime += sleepTime
      if nDelays == updateFrequency:
         nDelays = 0
         sleepTime *= 2
         if sleepTime > maximumDelay:
            sleepTime = maximumDelay

      exitStatus,stdOutput,stdError = executeCommand(sshCommand,streamOutput)

      if delayTime >= maximumDelayTime:
         break

   if tunnelDesignator != "":
      remoteTunnelMonitor.decrementTunnelUse(tunnelDesignator)

   return(exitStatus,stdOutput,stdError)


def executeLaunchCommand(launchCommand,
                         streamOutput=False):

   minimumDelay = 1       #  1 2 4 8 16 32 64 128 256
   maximumDelay = 256
   updateFrequency = 1
   maximumDelayTime = 900

   delayTime = 0
   sleepTime = minimumDelay
   nDelays = 0
   exitStatus,stdOutput,stdError = executeCommand(launchCommand,streamOutput)

   while exitStatus and ((stdError.count("qsub: cannot connect to server") > 0) or (stdError.count("ldap-nss.c") > 0)):
      nDelays += 1
      time.sleep(sleepTime)
      delayTime += sleepTime
      if nDelays == updateFrequency:
         nDelays = 0
         sleepTime *= 2
         if sleepTime > maximumDelay:
            sleepTime = maximumDelay

      exitStatus,stdOutput,stdError = executeCommand(launchCommand,streamOutput)

      if delayTime >= maximumDelayTime:
         break

   return(exitStatus,stdOutput,stdError)


def parseCommandArguments():
   global doubleDashTerminator

   def grabMultipleArguments(option, opt_str, value, parser):
      assert value is None
      done = 0
      value = getattr(parser.values,option.dest)
      if not value:
         value = []
      rargs = parser.rargs
      while rargs:
         arg = rargs[0]

         # Stop if we hit an arg like "--foo", "-a", "-fx", "--file=f",
         # etc.  Note that this also stops on "-3" or "-3.0", so if
         # your option takes numeric values, you will need to handle
         # this.
         if  (arg == "--"):
            del rargs[0]
            break
         elif((arg[:2] == "--" and len(arg) > 2) or
              (arg[:1] == "-" and len(arg) > 1 and arg[1] != "-")):
            break
         else:
            if not arg in value:
               value.append(arg)
            del rargs[0]

      if len(value) > 0:
         setattr(parser.values,option.dest,value)


   def grabStringArgument(option, opt_str, value, parser):
      assert value is None
      done = 0
      value = getattr(parser.values,option.dest)
      if not value:
         value = []
      rargs = parser.rargs
      if rargs:
         arg = rargs[0]

         # Stop if we hit an arg like "--foo", "-a", "-fx", "--file=f",
         # etc.  Note that this also stops on "-3" or "-3.0", so if
         # your option takes numeric values, you will need to handle
         # this.
         if((arg[:2] == "--" and len(arg) > 2) or
            (arg[:1] == "-" and len(arg) > 1 and arg[1] != "-")):
            pass
         else:
            if not arg in value:
               value.append(arg)
            del rargs[0]

      if len(value) > 0:
         setattr(parser.values,option.dest,value)


   parser = OptionParser(prog="submit",add_help_option=False)
   parser.disable_interspersed_args()
#  parser.add_option("-v","--venue",action="store",type="string",dest="destination",help="Remote job destination")
#  parser.add_option("-j","--jobid",action="store",type="long",dest="jobId",help="Global job identifier")
   parser.add_option("-j","--jobid",action="store",type="long",dest="jobId",help=SUPPRESS_HELP)

   if doubleDashTerminator:
      parser.add_option("-v","--venue",action="callback",callback=grabMultipleArguments,dest="destinations", \
                                       help="Remote job destination list terminated by --")
      parser.add_option("-i","--inputfile",action="callback",callback=grabMultipleArguments,dest="inputFiles", \
                                           help="Input file list terminated by --")
#     parser.add_option("-o","--outputfile",action="callback",callback=grabMultipleArguments,dest="outputFiles", \
#                                           help="Output file list terminated by --")
      parser.add_option("-o","--outputfile",action="callback",callback=grabMultipleArguments,dest="outputFiles", \
                                            help=SUPPRESS_HELP)
   else:
#     parser.add_option("-v","--venue",action="append",dest="destinations",help="Remote job destination")
#     parser.add_option("-i","--inputfile",action="append",dest="inputFiles",help="Input file")
##    parser.add_option("-o","--outputfile",action="append",dest="outputFiles",help="Output file")
#     parser.add_option("-o","--outputfile",action="append",dest="outputFiles",help=SUPPRESS_HELP)
      parser.add_option("-v","--venue",action="callback",callback=grabStringArgument,dest="destinations",help="Remote job destination")
      parser.add_option("-i","--inputfile",action="callback",callback=grabStringArgument,dest="inputFiles",help="Input file")
#     parser.add_option("-o","--outputfile",action="callback",callback=grabStringArgument,dest="outputFiles",help="Output file")
      parser.add_option("-o","--outputfile",action="callback",callback=grabStringArgument,dest="outputFiles",help=SUPPRESS_HELP)

   parser.add_option("-n","--nCpus",action="store",type="int",dest="nCpus",help="Number of processors for MPI execution")
   parser.add_option("-N","--ppn",action="store",type="int",dest="ppn",help="Number of processors/node for MPI execution")
   parser.add_option("-w","--wallTime",action="store",type="string",dest="wallTime",help="Estimated walltime hh:mm:ss or minutes")
#  parser.add_option("-e","--env",action="append",type="string",dest="environment",help="Variable=value")
#  parser.add_option("-m","--manager",action="store",type="string",dest="manager",help="Multiprocessor job manager")
   parser.add_option("-e","--env",action="callback",callback=grabStringArgument,dest="environment",help="Variable=value")
   parser.add_option("-m","--manager",action="callback",callback=grabStringArgument,dest="manager",help="Multiprocessor job manager")
   parser.set_defaults(metrics=False)
   parser.add_option("-M","--metrics",action="store_true",dest="metrics",help="Report resource usage on exit")
   parser.set_defaults(waitSwitch=False)
   parser.add_option("-W","--wait",action="store_true",dest="waitSwitch",help="Wait for reduced job load before submission")
   parser.set_defaults(help=False)
   parser.add_option("-h","--help",action="store_true",dest="help",help="Report command usage")

   options,arguments = parser.parse_args(sys.argv[1:])

   if not options.help:
      if len(arguments) == 0:
         log("Command must be supplied")
         sys.stderr.write("Command must be supplied\n")
         sys.stderr.flush()
         jobStatistics[maximumJobIndex]['exitCode'] = 1
         jobStatistics[maximumJobIndex]['nCores']   = 0
         reportExitCondition()
         sys.exit(1)
   else:
      parser.print_help()

   return(options,arguments)


if __name__ == '__main__':

   openLog(os.path.join(distributorHome,distributorLogDir,distributorLogFile))

   if PBS_ROOT != "":
      os.environ['PATH'] = os.path.join(PBS_ROOT,'bin') + ':' + os.environ['PATH']
   if CONDOR_ROOT != "":
      os.environ['PATH'] = os.path.join(CONDOR_ROOT,'bin') + ':' + os.path.join(CONDOR_ROOT,'sbin') + ':' + os.environ['PATH']
      if CONDOR_CONFIG != "":
         os.environ['CONDOR_CONFIG'] = CONDOR_CONFIG
      if RUNJOB_ROOT != "":
         os.environ['PATH'] = os.path.join(RUNJOB_ROOT,'bin') + ':' + os.environ['PATH']
         if not executeCommand("which runjob.sh")[0]:
            runjobExists = True

   hubUserName = os.getenv("USER")

   jobIndex = 1
   maximumJobIndex = max(maximumJobIndex,jobIndex)
   if not jobIndex in jobStatistics:
      jobStatistics[jobIndex] = initializeJobStatistic(nCpus)

   opts,args = parseCommandArguments()
   enteredCommand = " ".join(args)

   remoteProbeMonitor = None

   sitesInfo = SitesInfo(distributorHome,"sites.dat")
   tunnelsInfo = TunnelsInfo(distributorHome,"tunnels.dat")

   toolsInfo = ToolsInfo(distributorHome,"tools.dat")
   toolsInfo.applyUserRestriction(hubUserName)
   toolsInfo.applyGroupRestriction()

   managersInfo = ManagersInfo(distributorHome,"managers.dat")

   environmentWhitelistInfo = EnvironmentWhitelistInfo(distributorHome,"environmentwhitelist.dat")

   if distributorVersion == 'dev':
      disableStateCheck = True

   if opts.help:
      if runjobExists:
         if disableStateCheck:
            command = "condor_status -format '%s\t' Name"
         else:
            command = "condor_status -avail -format '%s\t' Name"
         log("command = " + command)
         exitStatus,stdOutput,stdError = executeLaunchCommand(command)
         venues = stdOutput.split()
      else:
         venues = []
      for site in sitesInfo.getEnabledSites():
         venues.append(site)

      if len(venues) > 0:
         venues.sort()
         sys.stdout.write("\nCurrently available DESTINATIONs are:\n")
         for venue in venues:
            sys.stdout.write("   %s\n" % (venue))
         sys.stdout.flush()
      del venues

      validManagers = managersInfo.getManagerNames()
      if len(validManagers) > 0:
         validManagers.sort()
         sys.stdout.write("\nCurrently available MANAGERs are:\n")
         for validManager in validManagers:
            sys.stdout.write("   %s\n" % (validManager))
         sys.stdout.flush()
      del validManagers

      jobStatistics[maximumJobIndex]['jobSubmissionMechanism'] = "help"
      jobStatistics[maximumJobIndex]['exitCode'] = 1
      reportExitCondition()
      sys.exit(1)

   userDestinations    = []
   toolDestinations    = []
   venue               = "Unknown"
   inputFiles          = []
   outputFiles         = []
   isMultiCoreRequest  = False
   nNodes              = "1"
   ppn                 = ""
   wallTime            = 60
   environment         = ""
   managerSpecified    = False
   computationMode     = 'serial'
   preManagerCommands  = []
   managerCommand      = ""
   postManagerCommands = []
   mpiRankVariable     = ""
   softenvExtensions   = {}
   managedEnvironment  = []
   moduleInitialize    = ""
   modulesLoad         = []
   modulesUnload       = []

   if opts.jobId:
      jobId = opts.jobId
   if jobId == 0:
      jobId = getJobId()
   localJobId = "%08d" % (jobId)
   logSetJobId(jobId)

   submitVenues = os.getenv("SUBMITVENUES")
   if submitVenues == None:
      if opts.destinations:
         userDestinations = opts.destinations
   else:
      userDestinations = submitVenues.split(',')
   if len(userDestinations) > 0:
      sitesInfo.purgeDisabledSites(userDestinations)
      if len(userDestinations) == 0:
         log("All sites listed as possible execution hosts have been disabled.\nPlease select another site or attempt execution at a later time.")
         sys.stderr.write("All sites listed as possible execution hosts have been disabled.\nPlease select another site or attempt execution at a later time.\n")
         sys.stderr.flush()
         jobStatistics[maximumJobIndex]['exitCode'] = 7
         jobStatistics[maximumJobIndex]['nCores']   = 0
         reportExitCondition()
         sys.exit(1)

   if opts.inputFiles:
      inputFiles = opts.inputFiles
   if opts.outputFiles:
      outputFiles = opts.outputFiles
   if opts.manager:
      manager = opts.manager[0]
      managerExists,computationMode, \
                    preManagerCommands,managerCommand,postManagerCommands, \
                    mpiRankVariable,managedEnvironment,moduleInitialize,modulesLoad,modulesUnload = managersInfo.getManagerInfo(manager)
      if managerExists:
         managerSpecified = True
      else:
         validManagers = managersInfo.getManagerNames()
         log("Invalid manager %s specified. Valid managers are %s" % (manager,validManagers))
         sys.stderr.write("Invalid manager %s specified. Valid managers are %s\n" % (manager,validManagers))
         sys.stderr.flush()
         del validManagers
         jobStatistics[maximumJobIndex]['exitCode'] = 1
         jobStatistics[maximumJobIndex]['nCores']   = 0
         reportExitCondition()
         sys.exit(1)

   if opts.nCpus:
      isMultiCoreRequest = True
      nCpus = opts.nCpus
   if opts.ppn:
      if isMultiCoreRequest:
         ppn = str(opts.ppn)
   if opts.wallTime:
      if ":" in opts.wallTime:
         wallTime = hhmmssTomin(opts.wallTime)
      else:
         wallTime = float(opts.wallTime)
      if wallTime < 0.:
         log("wallTime must be > 0")
         sys.stderr.write("wallTime must be > 0\n")
         sys.stderr.flush()
         jobStatistics[maximumJobIndex]['exitCode'] = 1
         jobStatistics[maximumJobIndex]['nCores']   = 0
         reportExitCondition()
         sys.exit(1)
   if opts.environment:
      for environmentVariableValue in opts.environment:
         environmentVariable = ""
         value               = ""
         nParts = len(environmentVariableValue.split('='))
         if   nParts == 1:
            environmentVariable = environmentVariableValue.strip()
            if environmentVariable in os.environ:
               value = os.environ[environmentVariable]
         elif nParts == 2:
            environmentVariable,value = environmentVariableValue.split('=')
            environmentVariable = environmentVariable.strip()
            value               = value.strip()
            if value == "":
               if environmentVariable in os.environ:
                  value = os.environ[environmentVariable]
         if environmentVariable == "" or value == "":
            log("Invalid environment variable %s specified." % (environmentVariableValue))
            sys.stderr.write("Invalid environment variable %s specified.\n" % (environmentVariableValue))
            sys.stderr.flush()
            jobStatistics[maximumJobIndex]['exitCode'] = 1
            jobStatistics[maximumJobIndex]['nCores']   = 0
            reportExitCondition()
            sys.exit(1)

         if   environmentVariable == "IGNORESTATECHECK":
            disableStateCheck = True
         elif environmentVariable == "IGNORESCANCHECK":
            disableScanCheck = True
         elif environmentVariable == "IGNOREPROBECHECK":
            disableProbeCheck = True
         elif environmentVariable == "X509_SUBMIT_PROXY":
            x509SubmitProxy = value
         elif environmentVariable == "USER_LOG_PATH":
            try:
               fpUserLogPath = open(value,"a")
            except:
               log("User log %s could not be opened." % (value))
               sys.stderr.write("User log %s could not be opened.\n" % (value))
               sys.stderr.flush()
               jobStatistics[maximumJobIndex]['exitCode'] = 1
               jobStatistics[maximumJobIndex]['nCores']   = 0
               reportExitCondition()
               sys.exit(1)
         elif environmentVariable == "USER_JOBID_PATH":
            try:
               fp = open(value,"w")
               if fp:
                  fp.write("%d\n" % (jobId))
                  fp.close()
            except:
               log("User jobId file %s could not be created." % (value))
               sys.stderr.write("User jobId file %s could not be created.\n" % (value))
               sys.stderr.flush()
               jobStatistics[maximumJobIndex]['exitCode'] = 1
               jobStatistics[maximumJobIndex]['nCores']   = 0
               reportExitCondition()
               sys.exit(1)
         elif environmentVariable == "MODULES_INIT_SH":
            moduleInitialize = value
         elif environmentVariable == "MODULES_LOAD":
            for m in value.split(','):
               modulesLoad.append(m)
         elif environmentVariable == "MODULES_UNLOAD":
            for m in value.split(','):
               modulesUnload.append(m)
         else:
            if environmentWhitelistInfo.isVariableInWhiteList(environmentVariable):
               environment += environmentVariable + "=" + value + " "
            else:
               log("Environment variable %s could not be set." % (environmentVariable))
               sys.stderr.write("Environment variable %s could not be set.\n" % (environmentVariable))
               sys.stderr.flush()
      environment = environment.strip()

   if isMultiCoreRequest:
      if computationMode == 'serial':
         computationMode = 'mpi'

   if len(args) > 1:
      for arg in args[1:]:
         if not arg.startswith("-"):
            if os.path.isfile(arg):
               if not arg in inputFiles:
                  inputFiles.append(arg)


   venueMechanism             = ''
   stageFiles                 = True
   passUseEnvironment         = False
   hostAttributes             = ""
   executable                 = ""
   transferExecutable         = True
   arbitraryExecutableAllowed = True
   stdinput                   = ""
   arguments                  = ""
   batchScript                = ""
   batchScriptName            = ""
   appScript                  = ""
   appScriptName              = ""
   nodeFileName               = ""

   executable = args[0]
   if "/" in executable:
      if   not os.path.isfile(executable):
         log("Missing executable: %s" % (executable))
         sys.stderr.write("Missing executable: %s\n" % (executable))
         sys.stderr.flush()
         jobStatistics[maximumJobIndex]['exitCode'] = 1
         jobStatistics[maximumJobIndex]['nCores']   = 0
         reportExitCondition()
         sys.exit(1)
      elif not os.access(executable,os.X_OK):
         log("Permission denied on executable: %s" % (executable))
         sys.stderr.write("Permission denied on executable: %s\n" % (executable))
         sys.stderr.flush()
         jobStatistics[maximumJobIndex]['exitCode'] = 1
         jobStatistics[maximumJobIndex]['nCores']   = 0
         reportExitCondition()
         sys.exit(1)
   else:
      transferExecutable = False

   stdinput = '/dev/null'
   toCheck = []
   stdinFd = sys.stdin.fileno()
   toCheck.append(stdinFd)
   ready = select.select(toCheck,[],[],0.) # wait for input
   if stdinFd in ready[0]:
      content = sys.stdin.read()
      if content != "":
         stdinput = ".__%08d.stdin" % (jobId)
         stdinFile = open(stdinput,"w")
         stdinFile.write(content)
         stdinFile.close()
         inputFiles.append(stdinput)
      del content

   if transferExecutable:
      if not executable.startswith("/apps/"):
         if not isSubmitMember():
            log("%s not member of group 'submit' - Attempting to execute: %s" % (hubUserName,executable))
            sys.stderr.write("%s not member of group 'submit' - Attempting to execute: %s\n" % (hubUserName,executable))
            sys.stderr.flush()
            jobStatistics[maximumJobIndex]['exitCode'] = 1
            jobStatistics[maximumJobIndex]['nCores']   = 0
            reportExitCondition()
            sys.exit(1)
   else:
      applicationIsStaged = False
      if toolsInfo.isExecutableTool(executable):
         if toolsInfo.isPermissionGranted(executable):
            applicationIsStaged = True
      if not applicationIsStaged and not isSubmitMember():
         log("%s not member of group 'submit' - Attempting to execute: %s" % (hubUserName,executable))
         sys.stderr.write("%s not member of group 'submit' - Attempting to execute: %s\n" % (hubUserName,executable))
         sys.stderr.flush()
         jobStatistics[maximumJobIndex]['exitCode'] = 1
         jobStatistics[maximumJobIndex]['nCores']   = 0
         reportExitCondition()
         sys.exit(1)

   if not disableProbeCheck:
      ignoreProbeSites = sitesInfo.getIgnoreProbeSites()
      remoteProbeMonitor = RemoteProbeMonitor(remoteProbeMonitorHost,remoteProbeMonitorPort,ignoreProbeSites=ignoreProbeSites)

   if len(userDestinations) > 0:
      goToGrid = sitesInfo.purgeOfflineSites(userDestinations,remoteProbeMonitor)
      if len(userDestinations) == 0 and not goToGrid:
         log("All sites listed as possible execution hosts are out of service.\nPlease select another site or attempt execution at a later time.")
         sys.stderr.write("All sites listed as possible execution hosts are out of service.\nPlease select another site or attempt execution at a later time.\n")
         sys.stderr.flush()
         jobStatistics[maximumJobIndex]['exitCode'] = 8
         jobStatistics[maximumJobIndex]['nCores']   = 0
         reportExitCondition()
         sys.exit(1)

   destination  = ""
   destinations = []

   if not transferExecutable:
      if toolsInfo.isExecutableTool(executable):
         if toolsInfo.isPermissionGranted(executable):
            toolDestinations,executablePath,remoteManager,softenvExtensions = toolsInfo.selectTool(executable,userDestinations,remoteProbeMonitor)
            if len(toolDestinations) == 0:
               log("The combination of destination and executable is not properly specified.")
               sys.stderr.write("The combination of destination and executable is not properly specified.\n")
               sys.stderr.flush()
               jobStatistics[maximumJobIndex]['exitCode'] = 1
               jobStatistics[maximumJobIndex]['nCores']   = 0
               reportExitCondition()
               sys.exit(1)
            if len(toolDestinations) == 1:
               destination = toolDestinations[0]

            if not managerSpecified and (remoteManager != ""):
               managerExists,computationMode, \
                             preManagerCommands,managerCommand,postManagerCommands, \
                             mpiRankVariable,environmentManager,moduleInitializeManager,modulesLoadManager,modulesUnloadManager = managersInfo.getManagerInfo(remoteManager)
               if managerExists:
                  managerSpecified = True
                  if len(managedEnvironment) == 0:
                     managedEnvironment = environmentManager
                  if moduleInitialize == "":
                     moduleInitialize = moduleInitializeManager
                  if len(modulesLoad) == 0:
                     modulesLoad = modulesLoadManager
                  if len(modulesUnload) == 0:
                     modulesUnload = modulesUnloadManager
               else:
                  log("Invalid manager %s specified for tool %s." % (remoteManager,executable))
                  sys.stderr.write("Invalid manager %s specified for tool %s.\n" % (remoteManager,executable))
                  sys.stderr.flush()
                  jobStatistics[maximumJobIndex]['exitCode'] = 1
                  jobStatistics[maximumJobIndex]['nCores']   = 0
                  reportExitCondition()
                  sys.exit(1)
            executable = executablePath
         else:
            log("Access to %s is restricted by user or group" % (executable))
            sys.stderr.write("Access to %s is restricted by user or group\n" % (executable))
            sys.stderr.flush()
            jobStatistics[maximumJobIndex]['exitCode'] = 1
            jobStatistics[maximumJobIndex]['nCores']   = 0
            reportExitCondition()
            sys.exit(1)

   if len(toolDestinations) == 0:
      destinations = sitesInfo.selectSites(userDestinations)
      if len(destinations) == 1:
         destination = destinations[0]

   if destination == "":
      if len(toolDestinations) > 1:
         nonRunjobSites = sitesInfo.getSitesWithoutKeyValue('venueMechanism','runjob',toolDestinations)
         destinations = sitesInfo.selectSites(nonRunjobSites)
         if len(destinations) > 0:
            destinationIndex = random.randint(0,len(destinations)-1)
            destination = destinations[destinationIndex]

   if destination == "":
      if len(userDestinations) > 1:
         nonRunjobSites = sitesInfo.getSitesWithoutKeyValue('venueMechanism','runjob',userDestinations)
         destinations = sitesInfo.selectSites(nonRunjobSites)
         if len(destinations) > 0:
            destinationIndex = random.randint(0,len(destinations)-1)
            destination = destinations[destinationIndex]

   if   (sitesInfo.siteExists(destination)):
      venues,venueIndex,venue,venueMechanism,tunnelDesignator,siteMonitorDesignator,venueGatekeeper,remoteUser, \
             stageFiles,passUseEnvironment,logUserRemotely, \
             remoteBatchSystem,remoteBatchQueue,remoteBatchPartition,remoteBatchPartitionSize,remoteBatchConstraints, \
             binDirectory,scratchDirectory, \
             remoteManager,hostAttributes,arbitraryExecutableAllowed,remotePpn = sitesInfo.getSiteInfo(destination)
      if not managerSpecified and remoteManager != "":
         managerExists,computationMode, \
                       preManagerCommands,managerCommand,postManagerCommands, \
                       mpiRankVariable,environmentManager,moduleInitializeManager,modulesLoadManager,modulesUnloadManager = managersInfo.getManagerInfo(remoteManager)
         if managerExists:
            managerSpecified = True
            if len(managedEnvironment) == 0:
               managedEnvironment = environmentManager
            if moduleInitialize == "":
               moduleInitialize = moduleInitializeManager
            if len(modulesLoad) == 0:
               modulesLoad = modulesLoadManager
            if len(modulesUnload) == 0:
               modulesUnload = modulesUnloadManager
         else:
            log("Invalid manager %s specified for site %s." % (remoteManager,destination))
            sys.stderr.write("Invalid manager %s specified for site %s.\n" % (remoteManager,destination))
            sys.stderr.flush()
            jobStatistics[maximumJobIndex]['exitCode'] = 1
            jobStatistics[maximumJobIndex]['nCores']   = 0
            reportExitCondition()
            sys.exit(1)
      if ppn == "":
         ppn = remotePpn
      nNodes = str(int(math.ceil(float(nCpus) / float(ppn))))
      if int(nNodes) == 1:
         if int(nCpus) < int(ppn):
            ppn = str(nCpus)
   elif runjobExists:
      venueMechanism        = 'runjob'
      if   len(toolDestinations) > 0:
         gridsite           = " ".join(toolDestinations)
      elif len(userDestinations) > 0:
         gridsite           = " ".join(userDestinations)
      else:
         gridsite           = destination
      siteMonitorDesignator = 'condorG'
      remoteBatchSystem     = '?'
   elif transferExecutable:
      enabledSites = sitesInfo.getEnabledSites()
      toolsInfo.purgeOfflineSites(enabledSites,remoteProbeMonitor)
      arbitraryExecutableAllowedRequired = not executable.startswith("/apps/")
      destination = sitesInfo.selectSite(enabledSites,arbitraryExecutableAllowedRequired)
      if destination != "":
         venues,venueIndex,venue,venueMechanism,tunnelDesignator,siteMonitorDesignator,venueGatekeeper,remoteUser, \
                stageFiles,passUseEnvironment,logUserRemotely, \
                remoteBatchSystem,remoteBatchQueue,remoteBatchPartition,remoteBatchPartitionSize,remoteBatchConstraints, \
                binDirectory,scratchDirectory, \
                remoteManager,hostAttributes,arbitraryExecutableAllowed,remotePpn = sitesInfo.getSiteInfo(destination)
         if not managerSpecified and remoteManager != "":
            managerExists,computationMode, \
                          preManagerCommands,managerCommand,postManagerCommands, \
                          mpiRankVariable,environmentManager,moduleInitializeManager,modulesLoadManager,modulesUnloadManager = managersInfo.getManagerInfo(remoteManager)
            if managerExists:
               managerSpecified = True
               if len(managedEnvironment) == 0:
                  managedEnvironment = environmentManager
               if moduleInitialize == "":
                  moduleInitialize = moduleInitializeManager
               if len(modulesLoad) == 0:
                  modulesLoad = modulesLoadManager
               if len(modulesUnload) == 0:
                  modulesUnload = modulesUnloadManager
            else:
               log("Invalid manager %s specified for site %s." % (remoteManager,destination))
               sys.stderr.write("Invalid manager %s specified for site %s.\n" % (remoteManager,destination))
               sys.stderr.flush()
               jobStatistics[maximumJobIndex]['exitCode'] = 1
               jobStatistics[maximumJobIndex]['nCores']   = 0
               reportExitCondition()
               sys.exit(1)
         if ppn == "":
            ppn = remotePpn
         nNodes = str(int(math.ceil(float(nCpus) / float(ppn))))
         if int(nNodes) == 1:
            if int(nCpus) < int(ppn):
               ppn = str(nCpus)
      else:
         log("Currently no DESTINATIONs are available")
         sys.stderr.write("Currently no DESTINATIONs are available\n")
         sys.stderr.flush()
         jobStatistics[maximumJobIndex]['exitCode'] = 1
         jobStatistics[maximumJobIndex]['nCores']   = 0
         reportExitCondition()
         sys.exit(1)
   else:
      log("Currently no DESTINATIONs are available")
      sys.stderr.write("Currently no DESTINATIONs are available\n")
      sys.stderr.flush()
      jobStatistics[maximumJobIndex]['exitCode'] = 1
      jobStatistics[maximumJobIndex]['nCores']   = 0
      reportExitCondition()
      sys.exit(1)

   if transferExecutable:
      if not executable.startswith("/apps/"):
         if not arbitraryExecutableAllowed:
            log("submission of arbitrary executables is not permitted to selected site - Attempting to execute: %s" % (executable))
            sys.stderr.write("submission of arbitrary executables is not permitted to selected site - Attempting to execute: %s\n" % (executable))
            sys.stderr.flush()
            jobStatistics[maximumJobIndex]['jobSubmissionMechanism'] = venueMechanism
            jobStatistics[maximumJobIndex]['venue']                  = venue
            jobStatistics[maximumJobIndex]['exitCode']               = 1
            jobStatistics[maximumJobIndex]['nCores']                 = 0
            reportExitCondition()
            sys.exit(1)

   if isMultiCoreRequest:
      jobStatistics[maximumJobIndex]['nCores'] = nCpus

   TIMESTAMPTRANSFERRED += localJobId
   TIMESTAMPFINISH      += localJobId
   TIMESTAMPSTART       += localJobId
   TIMERESULTS          += localJobId
   GRIDRESOURCE         += localJobId
   GRIDHISTORY          += localJobId
   GRIDJOBID            += localJobId
   LOCALJOBID           += localJobId

   executeCommand("touch " + LOCALJOBID)
   inputFiles.append(LOCALJOBID)

   appScriptName = localJobId + ".sh"

   if transferExecutable:
      inputFiles.append(executable)

   if stageFiles:
      stageInTarFile = localJobId + "_input.tar"
      command = buildCreateTarCommand(stageInTarFile,inputFiles)
      log("command = " + command)
      tarExitStatus,tarStdOutput,tarStdError = executeCommand(command)
      if tarExitStatus:
         log(tarStdError)
         log(tarStdOutput)
         sys.stderr.write(tarStdError)
         sys.stderr.flush()
         jobStatistics[maximumJobIndex]['exitCode'] = 1
         jobStatistics[maximumJobIndex]['nCores']   = 0
         reportExitCondition()
         sys.exit(1)

      remoteArgs = []
      remoteArgs.append(args[0])
      if len(args) > 1:
         tarFiles = tarStdOutput.strip().split('\n')
         for arg in args[1:]:
            rarg = arg
            for tarFile in tarFiles:
               if re.match(".*" + re.escape(tarFile) + "$",arg):
                  rarg = "./" + tarFile
                  break
            remoteArgs.append(rarg)

      remoteCommand = " ".join(remoteArgs)
      log("remoteCommand " + remoteCommand)

      arguments = " ".join(remoteArgs[1:])
      arguments = arguments.strip()
   else:
      remoteCommand = " ".join(args)
      log("remoteCommand " + remoteCommand)

      arguments = " ".join(args[1:])
      arguments = arguments.strip()


   useEnvironment = ""
   if passUseEnvironment:
      reChoice = re.compile(".*_CHOICE$")
      environmentVars = filter(reChoice.search,os.environ.keys())
      if "PROMPT_CHOICE" in environmentVars:
         environmentVars.remove("PROMPT_CHOICE")
      if "EDITOR_CHOICE" in environmentVars:
         environmentVars.remove("EDITOR_CHOICE")
      if len(environmentVars) > 0:
         useEnvironment = ". /apps/environ/.setup\n"
         for environmentVar in environmentVars:
            useEnvironment += "use -e -r " + os.environ[environmentVar] + "\n"

   if len(managedEnvironment) > 0:
      for environmentVariableValue in managedEnvironment:
         environmentVariable = ""
         value               = ""
         nParts = len(environmentVariableValue.split('='))
         if   nParts == 1:
            environmentVariable = environmentVariableValue.strip()
            if environmentVariable in os.environ:
               value = os.environ[environmentVariable]
         elif nParts == 2:
            environmentVariable,value = environmentVariableValue.split('=')
            environmentVariable = environmentVariable.strip()
            value               = value.strip()
            if value == "":
               if environmentVariable in os.environ:
                  value = os.environ[environmentVariable]

         if environmentVariable != "" and value != "":
            if environment.find(environmentVariable + '=') < 0:
               environment += environmentVariable + "=" + value + " "

   if checkQuota:
      getUserQuota()

   if   venueMechanism == 'local':
      remoteBinDirectory = binDirectory.replace("$","\$")
      RECEIVEINPUT    = os.path.join(remoteBinDirectory,RECEIVEINPUTCOMMAND)
      SUBMITBATCHJOB  = os.path.join(remoteBinDirectory,SUBMITBATCHJOBCOMMAND)
      TRANSMITRESULTS = os.path.join(remoteBinDirectory,TRANSMITRESULTSCOMMAND)
      CLEANUPJOB      = os.path.join(remoteBinDirectory,CLEANUPJOBCOMMAND)
      KILLBATCHJOB    = os.path.join(remoteBinDirectory,KILLBATCHJOBCOMMAND)

      doInputOutputPacking = False
      workingDirectory = os.getcwd()
      log("workingDirectory " + workingDirectory)
      if   remoteBatchSystem == 'PBS':
         remoteBatch = RemoteBatchPBS(localJobId,appScriptName,environment,transferExecutable,executable,arguments,stdinput,
                                      isMultiCoreRequest,computationMode,preManagerCommands,managerCommand,postManagerCommands,
                                      nNodes,ppn,hostAttributes,remoteBatchQueue,wallTime,remoteBatchAccount,quotaLimit,
                                      TIMESTAMPSTART,TIMESTAMPFINISH,TIMERESULTS)
         batchScriptName,batchScript = remoteBatch.buildBatchScript()
         remoteAppScript = RemoteBatchAppScript(localJobId,transferExecutable,executable,stdinput,arguments,
                                                useEnvironment,environment,
                                                isMultiCoreRequest,computationMode,mpiRankVariable,
                                                TIMESTAMPSTART,TIMESTAMPFINISH,TIMERESULTS)
         appScript = remoteAppScript.buildAppScript()
      elif remoteBatchSystem == 'CONDOR':
         remoteBatch = LocalBatchCONDOR(localJobId,stageFiles,appScriptName,environment,
                                        isMultiCoreRequest,computationMode,
                                        wallTime)
         batchScriptName,batchScript = remoteBatch.buildBatchScript()
         remoteAppScript = RemoteBatchAppScript(localJobId,transferExecutable,executable,stdinput,arguments,
                                                useEnvironment,environment,
                                                isMultiCoreRequest,computationMode,mpiRankVariable,
                                                TIMESTAMPSTART,TIMESTAMPFINISH,TIMERESULTS)
         doInputOutputPacking = True
         appScript = remoteAppScript.buildAppScript(doInputOutputPacking)
      elif remoteBatchSystem == 'SCRIPT':
         remoteBatch = RemoteInstantSCRIPT(localJobId,workingDirectory,appScriptName,
                                           isMultiCoreRequest,computationMode,preManagerCommands,managerCommand,postManagerCommands,
                                           nNodes,ppn,venue,venues,venueIndex,
                                           TIMESTAMPTRANSFERRED,TIMESTAMPSTART,TIMESTAMPFINISH,TIMERESULTS)
         batchScriptName,batchScript,nodeFileName,nodeList = remoteBatch.buildBatchScript()
         if nodeFileName:
            fp = open(nodeFileName,"w")
            if fp:
               fp.write('\n'.join(nodeList)+'\n')
               fp.close()
         remoteAppScript = RemoteInstantAppScript(localJobId,workingDirectory,executable,stdinput,arguments,
                                                  useEnvironment,environment,
                                                  isMultiCoreRequest,computationMode,mpiRankVariable,
                                                  TIMESTAMPTRANSFERRED,TIMESTAMPSTART,TIMESTAMPFINISH,TIMERESULTS)
         appScript = remoteAppScript.buildAppScript()

      jobSubmissionMechanism = venueMechanism + remoteBatchSystem

   elif venueMechanism == 'ssh':
      remoteBinDirectory = binDirectory.replace("$","\$")
      RECEIVEINPUT    = os.path.join(remoteBinDirectory,RECEIVEINPUTCOMMAND)
      SUBMITBATCHJOB  = os.path.join(remoteBinDirectory,SUBMITBATCHJOBCOMMAND)
      TRANSMITRESULTS = os.path.join(remoteBinDirectory,TRANSMITRESULTSCOMMAND)
      CLEANUPJOB      = os.path.join(remoteBinDirectory,CLEANUPJOBCOMMAND)
      KILLBATCHJOB    = os.path.join(remoteBinDirectory,KILLBATCHJOBCOMMAND)

      if stageFiles:
         epoch = int(time.mktime(datetime.datetime.utcnow().timetuple()))
         workingDirectory = os.path.join(scratchDirectory,str(epoch) + '_' + localJobId)
      else:
         workingDirectory = os.getcwd()
      log("workingDirectory " + workingDirectory)

      if   remoteBatchSystem == 'PBS':
         remoteBatch = RemoteBatchPBS(localJobId,appScriptName,environment,transferExecutable,executable,arguments,stdinput,
                                      isMultiCoreRequest,computationMode,preManagerCommands,managerCommand,postManagerCommands,
                                      nNodes,ppn,hostAttributes,remoteBatchQueue,wallTime,remoteBatchAccount,quotaLimit,
                                      TIMESTAMPSTART,TIMESTAMPFINISH,TIMERESULTS)
         batchScriptName,batchScript = remoteBatch.buildBatchScript()
         remoteAppScript = RemoteBatchAppScript(localJobId,transferExecutable,executable,stdinput,arguments,
                                                useEnvironment,environment,
                                                isMultiCoreRequest,computationMode,mpiRankVariable,
                                                TIMESTAMPSTART,TIMESTAMPFINISH,TIMERESULTS)
         appScript = remoteAppScript.buildAppScript()
         jobSubmissionMechanism = venueMechanism + remoteBatchSystem
      elif remoteBatchSystem == 'PBS8':
         remoteBatch = RemoteBatchPBS8(localJobId,appScriptName,environment,transferExecutable,executable,arguments,stdinput,
                                       isMultiCoreRequest,computationMode,preManagerCommands,managerCommand,postManagerCommands,
                                       nNodes,ppn,hostAttributes,remoteBatchQueue,wallTime,remoteBatchAccount,quotaLimit,
                                       TIMESTAMPSTART,TIMESTAMPFINISH,TIMERESULTS)
         batchScriptName,batchScript = remoteBatch.buildBatchScript()
         remoteAppScript = RemoteBatchAppScript(localJobId,transferExecutable,executable,stdinput,arguments,
                                                useEnvironment,environment,
                                                isMultiCoreRequest,computationMode,mpiRankVariable,
                                                TIMESTAMPSTART,TIMESTAMPFINISH,TIMERESULTS)
         appScript = remoteAppScript.buildAppScript()
         jobSubmissionMechanism = venueMechanism + remoteBatchSystem
      elif remoteBatchSystem == 'LSF':
         remoteBatch = RemoteBatchLSF(localJobId,appScriptName,environment,transferExecutable,executable,arguments,stdinput,
                                      isMultiCoreRequest,computationMode,preManagerCommands,managerCommand,postManagerCommands,
                                      nNodes,ppn,wallTime,remoteBatchAccount,
                                      TIMESTAMPSTART,TIMESTAMPFINISH,TIMERESULTS)
         batchScriptName,batchScript = remoteBatch.buildBatchScript()
         remoteAppScript = RemoteBatchAppScript(localJobId,transferExecutable,executable,stdinput,arguments,
                                                useEnvironment,environment,
                                                isMultiCoreRequest,computationMode,mpiRankVariable,
                                                TIMESTAMPSTART,TIMESTAMPFINISH,TIMERESULTS)
         appScript = remoteAppScript.buildAppScript()
         jobSubmissionMechanism = venueMechanism + remoteBatchSystem
      elif remoteBatchSystem == 'LL':
         remoteBatch = RemoteBatchLL(localJobId,appScriptName,environment,transferExecutable,executable,arguments,stdinput,
                                     isMultiCoreRequest,computationMode,preManagerCommands,managerCommand,postManagerCommands,
                                     nNodes,ppn,remoteBatchPartition,remoteBatchPartitionSize,remoteBatchConstraints,
                                     wallTime,quotaLimit,
                                     TIMESTAMPSTART,TIMESTAMPFINISH,TIMERESULTS)
         batchScriptName,batchScript = remoteBatch.buildBatchScript()
         remoteAppScript = RemoteBatchAppScript(localJobId,transferExecutable,executable,stdinput,arguments,
                                                useEnvironment,environment,
                                                isMultiCoreRequest,computationMode,mpiRankVariable,
                                                TIMESTAMPSTART,TIMESTAMPFINISH,TIMERESULTS)
         appScript = remoteAppScript.buildAppScript()
         jobSubmissionMechanism = venueMechanism + remoteBatchSystem
      elif remoteBatchSystem == 'SLURM':
         remoteBatch = RemoteBatchSLURM(localJobId,appScriptName,environment,transferExecutable,executable,arguments,stdinput,
                                        isMultiCoreRequest,computationMode,preManagerCommands,managerCommand,postManagerCommands,
                                        nNodes,ppn,remoteBatchPartition,remoteBatchPartitionSize,remoteBatchConstraints,
                                        wallTime,quotaLimit,
                                        TIMESTAMPSTART,TIMESTAMPFINISH,TIMERESULTS)
         batchScriptName,batchScript = remoteBatch.buildBatchScript()
         remoteAppScript = RemoteBatchAppScript(localJobId,transferExecutable,executable,stdinput,arguments,
                                                useEnvironment,environment,
                                                isMultiCoreRequest,computationMode,mpiRankVariable,
                                                TIMESTAMPSTART,TIMESTAMPFINISH,TIMERESULTS)
         appScript = remoteAppScript.buildAppScript()
         jobSubmissionMechanism = venueMechanism + remoteBatchSystem
      elif remoteBatchSystem == 'CONDOR':
         remoteBatch = RemoteBatchCONDOR(localJobId,appScriptName,environment,
                                         isMultiCoreRequest,computationMode,
                                         wallTime)
         batchScriptName,batchScript = remoteBatch.buildBatchScript()
         remoteAppScript = RemoteBatchAppScript(localJobId,transferExecutable,executable,stdinput,arguments,
                                                useEnvironment,environment,
                                                isMultiCoreRequest,computationMode,mpiRankVariable,
                                                TIMESTAMPSTART,TIMESTAMPFINISH,TIMERESULTS)
         appScript = remoteAppScript.buildAppScript()
         jobSubmissionMechanism = venueMechanism + remoteBatchSystem
      elif remoteBatchSystem == 'SCRIPT':
         remoteBatch = RemoteInstantSCRIPT(localJobId,workingDirectory,appScriptName,
                                           isMultiCoreRequest,computationMode,preManagerCommands,managerCommand,postManagerCommands,
                                           nNodes,ppn,venue,venues,venueIndex,
                                           TIMESTAMPTRANSFERRED,TIMESTAMPSTART,TIMESTAMPFINISH,TIMERESULTS)
         batchScriptName,batchScript,nodeFileName,nodeList = remoteBatch.buildBatchScript()
         if nodeFileName:
            fp = open(nodeFileName,"w")
            if fp:
               fp.write('\n'.join(nodeList)+'\n')
               fp.close()
         remoteAppScript = RemoteInstantAppScript(localJobId,workingDirectory,executable,stdinput,arguments,
                                                  useEnvironment,environment,
                                                  isMultiCoreRequest,computationMode,mpiRankVariable,
                                                  TIMESTAMPTRANSFERRED,TIMESTAMPSTART,TIMESTAMPFINISH,TIMERESULTS)
         appScript = remoteAppScript.buildAppScript()
         jobSubmissionMechanism = venueMechanism + remoteBatchSystem

   elif venueMechanism == 'runjob':
      epoch = int(time.mktime(datetime.datetime.utcnow().timetuple()))
      workingDirectory = str(epoch) + '_' + localJobId
      log("workingDirectory " + workingDirectory)

      remoteBatch = RemoteBatchRUNJOB(localJobId,appScriptName,environment,transferExecutable,executable,arguments,
                                      isMultiCoreRequest,computationMode,
                                      nCpus,ppn,softenvExtensions,wallTime,
                                      gridsite,workingDirectory,
                                      x509SubmitProxy,disableProbeCheck,disableScanCheck,disableStateCheck,
                                      TIMESTAMPTRANSFERRED,TIMESTAMPSTART,TIMESTAMPFINISH,TIMERESULTS,
                                      GRIDHISTORY,GRIDJOBID,GRIDRESOURCE)
      batchScriptName,batchScript = remoteBatch.buildBatchScript()
      remoteAppScript = RemoteRunjobAppScript(localJobId,transferExecutable,executable,stdinput,arguments,environment,
                                              isMultiCoreRequest,computationMode,moduleInitialize,modulesLoad,modulesUnload,
                                              TIMESTAMPSTART,TIMESTAMPFINISH,TIMERESULTS)
      appScript = remoteAppScript.buildAppScript()
      jobSubmissionMechanism = venueMechanism

   scriptFiles = []
   if batchScript != "":
      fp = open(batchScriptName,"w")
      if fp:
         fp.write(batchScript)
         fp.close()
         if   venueMechanism == 'ssh':
            scriptFiles.append(batchScriptName)
            if remoteBatchSystem == 'SCRIPT':
               os.chmod(batchScriptName,stat.S_IRWXU|stat.S_IRGRP|stat.S_IXGRP|stat.S_IROTH|stat.S_IXOTH)
         elif venueMechanism == 'local':
            if remoteBatchSystem == 'SCRIPT':
               os.chmod(batchScriptName,stat.S_IRWXU|stat.S_IRGRP|stat.S_IXGRP|stat.S_IROTH|stat.S_IXOTH)
         elif venueMechanism == 'runjob':
            os.chmod(batchScriptName,stat.S_IRWXU|stat.S_IRGRP|stat.S_IXGRP|stat.S_IROTH|stat.S_IXOTH)
      else:
         log("could not open %s for writing" % (batchScriptName))
         sys.stderr.write("could not open %s for writing\n" % (batchScriptName))
         sys.stderr.flush()
         jobStatistics[maximumJobIndex]['jobSubmissionMechanism'] = jobSubmissionMechanism
         jobStatistics[maximumJobIndex]['exitCode']               = 1
         jobStatistics[maximumJobIndex]['nCores']                 = 0
         reportExitCondition()
         sys.exit(1)

   if appScript != "":
      fp = open(appScriptName,"w")
      if fp:
         fp.write(appScript)
         fp.close()
         os.chmod(appScriptName,stat.S_IRWXU|stat.S_IRGRP|stat.S_IXGRP|stat.S_IROTH|stat.S_IXOTH)
         scriptFiles.append(appScriptName)
      else:
         log("could not open %s for writing" % (appScriptName))
         sys.stderr.write("could not open %s for writing\n" % (appScriptName))
         sys.stderr.flush()
         jobStatistics[maximumJobIndex]['jobSubmissionMechanism'] = jobSubmissionMechanism
         jobStatistics[maximumJobIndex]['exitCode']               = 1
         jobStatistics[maximumJobIndex]['nCores']                 = 0
         reportExitCondition()
         sys.exit(1)

   if stageFiles:
      if len(scriptFiles) > 0:
         command = buildAppendTarCommand(stageInTarFile,scriptFiles)
         log("command = " + command)
         tarExitStatus,tarStdOutput,tarStdError = executeCommand(command)
         if tarExitStatus:
            log(tarStdError)
            log(tarStdOutput)
            sys.stderr.write(tarStdError)
            sys.stderr.flush()
            jobStatistics[maximumJobIndex]['exitCode'] = 1
            jobStatistics[maximumJobIndex]['nCores']   = 0
            reportExitCondition()
            sys.exit(1)

      command = "nice -n 19 gzip " + stageInTarFile
      log("command = " + command)
      gzipExitStatus,gzipStdOutput,gzipStdError = executeCommand(command)
      if gzipExitStatus:
         log(gzipStdError)
         log(gzipStdOutput)
         sys.stderr.write(gzipStdError)
         sys.stderr.flush()
         jobStatistics[maximumJobIndex]['exitCode'] = 1
         jobStatistics[maximumJobIndex]['nCores']   = 0
         reportExitCondition()
         sys.exit(1)
      stageInTarFile += ".gz"

   exitStatus = 0
   exitSignal = 0
   jobSubmitted = False

   remoteJobMonitor    = RemoteJobMonitor(remoteJobMonitorHost,remoteJobMonitorPort)
   remoteTunnelMonitor = RemoteTunnelMonitor(remoteTunnelMonitorHost,remoteTunnelMonitorPort)

   if   venueMechanism == 'local':
      command = RECEIVEINPUT + " " + workingDirectory + " " + TIMESTAMPTRANSFERRED
      log("command = " + command)
      log(executeCommand(command)[1])

      if remoteBatchSystem == 'SCRIPT':
         if batchScriptName != "":
            command = os.path.join(os.getcwd(),batchScriptName)
         else:
            command = os.path.join(os.getcwd(),appScriptName)
         log("command = " + command)
         exitStatus,stdOutput,stdError = executeCommand(command,True)
         jobSubmitted = True

         jobStatistics[jobIndex]['exitCode'] = exitStatus

         if nodeFileName:
            if os.path.isfile(nodeFileName):
               os.remove(nodeFileName)
      else:
         commandArgs = []
         commandArgs.append(SUBMITBATCHJOB)
         commandArgs.append(workingDirectory)
         commandArgs.append(batchScriptName)
         command = ' '.join(commandArgs)
         log("command = " + command)
         exitStatus,remoteJobId,stdError = executeLaunchCommand(command)
         remoteJobId = remoteJobId.strip()
         log("remoteJobId = " + remoteJobId)
         if not exitStatus:
            jobSubmitted = True
            if   remoteBatchSystem == 'PBS':
               remoteJobIdNumber = remoteJobId.split('.')[0]
            elif remoteBatchSystem == 'CONDOR':
#        Submitting job(s). Logging submit event(s). 1 job(s) submitted to cluster 469609.
               remoteJobIdNumber = re.search('cluster [0-9]+',remoteJobId).group().split()[1] + ".0"
            remoteJobMonitor.postNewJobSubmission(siteMonitorDesignator,remoteJobIdNumber)
            remoteJobMonitor.waitForBatchJob(siteMonitorDesignator,remoteJobIdNumber)
            jobStatistics[jobIndex]['remoteJobIdNumber'] = remoteJobIdNumber
         else:
            jobStatistics[maximumJobIndex]['exitCode'] = exitStatus

      jobStatistics[jobIndex]['venue']                  = venue
      jobStatistics[jobIndex]['jobSubmissionMechanism'] = jobSubmissionMechanism

      if doInputOutputPacking:
         stageOutTarFile = localJobId + "_output.tar.gz"
         if os.path.isfile(stageOutTarFile):
            command = "tar xzmf " + stageOutTarFile
            log("command = " + command)
            tarExitStatus,tarStdOutput,tarStdError = executeCommand(command)
            os.remove(stageOutTarFile)
         else:
            stageOutTarFile = localJobId + "_output.tar"
            if os.path.isfile(stageOutTarFile):
               command = "tar xmf " + stageOutTarFile
               log("command = " + command)
               tarExitStatus,tarStdOutput,tarStdError = executeCommand(command)
               os.remove(stageOutTarFile)

      if batchScriptName:
         if os.path.isfile(batchScriptName):
            os.remove(batchScriptName)
      if appScriptName:
         if os.path.isfile(appScriptName):
            os.remove(appScriptName)

   elif venueMechanism == 'ssh':
      if not stageFiles:
         if remoteBatchSystem == 'SCRIPT':
            jobStatistics[jobIndex]['venue']                  = venue
            jobStatistics[jobIndex]['jobSubmissionMechanism'] = jobSubmissionMechanism

            sshBaseCommand = "ssh -T -x -a " + venue
            if batchScriptName != "":
               command = sshBaseCommand + " " + os.path.join(os.getcwd(),batchScriptName)
            else:
               command = sshBaseCommand + " " + os.path.join(os.getcwd(),appScriptName)
            log("command = " + command)
            exitStatus,stdOutput,stdError = executeCommand(command,True)
            jobSubmitted = True

            jobStatistics[jobIndex]['exitCode'] = exitStatus

            if batchScriptName:
               if os.path.isfile(batchScriptName):
                  os.remove(batchScriptName)
            if appScriptName:
               if os.path.isfile(appScriptName):
                  os.remove(appScriptName)
            if nodeFileName:
               if os.path.isfile(nodeFileName):
                  os.remove(nodeFileName)

      else:
         jobStatistics[jobIndex]['venue']                  = venue
         jobStatistics[jobIndex]['jobSubmissionMechanism'] = jobSubmissionMechanism

         removeIdentity = False
         if not IDENTITY:
            if tunnelDesignator == "":
               command = os.path.join(distributorHome,"update-known-hosts") + " " + venue
               log("command = " + command)
               exitStatus,stdOutput,stdError = executeCommand(command)
            else:
               gatewayHost,localHost = tunnelsInfo.getSSHTunnelHosts(tunnelDesignator)
               if gatewayHost != "" and localHost != "":
                  command = os.path.join(distributorHome,"update-known-hosts") + " \'" + gatewayHost + "\'"
                  log("command = " + command)
                  exitStatus,stdOutput,stdError = executeCommand(command)
                  if exitStatus == 0:
                     command = os.path.join(distributorHome,"update-known-hosts") + " \'" + localHost + "\'"
                     log("command = " + command)
                     exitStatus,stdOutput,stdError = executeCommand(command)

            if not exitStatus:
               command = os.path.join(distributorHome,"genuserid")
               log("command = " + command)
               exitStatus,stdOutput,stdError = executeCommand(command)
               if not exitStatus:
                  IDENTITY = stdOutput.strip()
                  log("IDENTITY = " + IDENTITY)
                  if IDENTITY:
                     removeIdentity = True

         remoteWorkingDirectory = workingDirectory.replace("$","\$")
         if IDENTITY:
            if tunnelDesignator == "":
               sshBaseCommand = "ssh -T -x -a -i " + IDENTITY + " " + remoteUser + "@" + venue
            else:
               tunnelAddress,tunnelPort = remoteTunnelMonitor.getTunnelAddressPort(tunnelDesignator)
               sshBaseCommand = "ssh -T -x -a -i " + IDENTITY + " -p " + tunnelPort + " " + remoteUser + "@" + tunnelAddress

            command = "cat " + stageInTarFile + " | " + sshBaseCommand + " \"" + RECEIVEINPUT + " " + remoteWorkingDirectory + " " + TIMESTAMPTRANSFERRED + "\""
            log("command = " + command)
            exitStatus,stdOutput,stdError = executeSSHCommand(command,remoteTunnelMonitor,tunnelDesignator)
            log(stdOutput)

            if not exitStatus:
               if remoteBatchSystem == 'SCRIPT':
                  commandArgs = []
                  commandArgs.append(SUBMITBATCHJOB)
                  commandArgs.append(remoteWorkingDirectory)
                  if batchScriptName != "":
                     commandArgs.append(os.path.join('.',batchScriptName))
                  else:
                     commandArgs.append(os.path.join('.',appScriptName))
                  if logUserRemotely:
                     commandArgs.append(str(jobId))
                     commandArgs.append(hubUserName)
                  command = sshBaseCommand + " \"" + ' '.join(commandArgs) + "\""

                  log("command = " + command)
                  exitStatus,stdOutput,stdError = executeSSHCommand(command,remoteTunnelMonitor,tunnelDesignator,True)
                  jobSubmitted = True
               else:
                  commandArgs = []
                  commandArgs.append(SUBMITBATCHJOB)
                  commandArgs.append(remoteWorkingDirectory)
                  commandArgs.append(batchScriptName)
                  if logUserRemotely:
                     commandArgs.append(str(jobId))
                     commandArgs.append(hubUserName)
                  command = sshBaseCommand + " \"" + ' '.join(commandArgs) + "\""
                  log("command = " + command)
                  exitStatus,remoteJobId,stdError = executeSSHCommand(command,remoteTunnelMonitor,tunnelDesignator)
                  remoteJobId = remoteJobId.strip()
                  log("remoteJobId = " + remoteJobId)
                  if not exitStatus:
                     jobSubmitted = True
                     if   remoteBatchSystem == 'PBS':
                        remoteJobIdNumber = remoteJobId.split('.')[0]
                     elif remoteBatchSystem == 'PBS8':
                        remoteJobIdNumber = remoteJobId.split('.')[0]
                     elif remoteBatchSystem == 'LSF':
#                 Job <851126> is submitted to default queue <normal>.
                        remoteJobIdNumber = remoteJobId.split('<')[1].split('>')[0]
                     elif remoteBatchSystem == 'LL':
                        remoteJobIdNumber = remoteJobId
                     elif remoteBatchSystem == 'SLURM':
                        remoteJobIdNumber = remoteJobId.split()[-1]
                     elif remoteBatchSystem == 'CONDOR':
#                 Submitting job(s). Logging submit event(s). 1 job(s) submitted to cluster 469609.
                        remoteJobIdNumber = re.search('cluster [0-9]+',remoteJobId).group().split()[1] + ".0"
                     remoteJobMonitor.postNewJobSubmission(siteMonitorDesignator,remoteJobIdNumber)
                     remoteJobMonitor.waitForBatchJob(siteMonitorDesignator,remoteJobIdNumber,destination)
                     jobStatistics[jobIndex]['remoteJobIdNumber'] = remoteJobIdNumber
            else:
               jobStatistics[jobIndex]['exitCode'] = 11

         if jobSubmitted:
            command = sshBaseCommand + " \"" + TRANSMITRESULTS + " " + remoteWorkingDirectory + "\"" + " | tar xzmf -"
            log("command = " + command)
            exitStatus,stdOutput,stdError = executeSSHCommand(command,remoteTunnelMonitor,tunnelDesignator)
            log(stdOutput)
            if exitStatus:
               jobStatistics[jobIndex]['exitCode'] = 12

            command = sshBaseCommand + " \"" + CLEANUPJOB + " " + remoteWorkingDirectory + "\""
            log("command = " + command)
            log(executeSSHCommand(command,remoteTunnelMonitor,tunnelDesignator)[1])
         else:
            jobStatistics[jobIndex]['exitCode'] = exitStatus

         if removeIdentity:
            if os.path.isfile(IDENTITY):
               os.remove(IDENTITY)
         if batchScriptName:
            if os.path.isfile(batchScriptName):
               os.remove(batchScriptName)
         if appScriptName:
            if os.path.isfile(appScriptName):
               os.remove(appScriptName)

   elif venueMechanism == 'runjob':
      command = os.path.join(".",batchScriptName)
      log("command = " + command)
      exitStatus,remoteJobId,stdError = executeCommand(command,True)
      remoteJobId = remoteJobId.strip()
      if not exitStatus:
         jobSubmitted = True
         log("remoteJobId = " + remoteJobId)
         remoteJobIdNumber = re.search('with id [0-9]+',remoteJobId).group().split()[2] + ".0"
         log("remoteJobIdNumber = " + remoteJobIdNumber)

         remoteJobMonitor.postNewJobSubmission(siteMonitorDesignator,remoteJobIdNumber)
         remoteJobMonitor.waitForBatchJob(siteMonitorDesignator,remoteJobIdNumber)

         re_Files = re.compile(GRIDRESOURCE + "_[0-9]+$")
         dirFiles = os.listdir(os.getcwd())
         matchingFiles = filter(re_Files.search,dirFiles)
         if len(matchingFiles) > 0:
            gridJobIdLogFile = open(os.path.join(distributorHome,distributorLogDir,distributorGridLogFile),"a")
            matchingFiles.sort()
            for matchingFile in matchingFiles:
               if len(matchingFile) > len(GRIDRESOURCE):
                  jobIndex = int(matchingFile[len(GRIDRESOURCE)+1:])
               else:
                  jobIndex = 1
               maximumJobIndex = max(maximumJobIndex,jobIndex)

               if not jobIndex in jobStatistics:
                  jobStatistics[jobIndex] = initializeJobStatistic(nCpus)

               fp = open(matchingFile,'r')
               if fp:
                  gridResource = fp.readline().strip()
                  fp.close()
                  gridType = gridResource.split()[0]
                  if   gridType == "gt2":
                     gateKeeper = gridResource.split()[1].split('/')[0]
                     jobManager = gridResource.split()[1].split('/')[1].split('-')[1].upper()
                  elif gridType == "gt4":
                     gateKeeper = gridResource.split()[1]
                     jobManager = gridResource.split()[2]

                  venue = gateKeeper
                  jobSubmissionMechanism = venueMechanism + jobManager

               jobStatistics[jobIndex]['venue']                  = venue
               jobStatistics[jobIndex]['remoteJobIdNumber']      = remoteJobIdNumber
               jobStatistics[jobIndex]['jobSubmissionMechanism'] = jobSubmissionMechanism

               os.remove(matchingFile)
         else:
            log(GRIDRESOURCE + " not present")


         re_Files = re.compile(GRIDJOBID + "_[0-9]+$")
         dirFiles = os.listdir(os.getcwd())
         matchingFiles = filter(re_Files.search,dirFiles)
         if len(matchingFiles) > 0:
            gridJobIdLogFile = open(os.path.join(distributorHome,distributorLogDir,distributorGridLogFile),"a")
            matchingFiles.sort()
            for matchingFile in matchingFiles:
               if len(matchingFile) > len(GRIDJOBID):
                  jobIndex = int(matchingFile[len(GRIDJOBID)+1:])
               else:
                  jobIndex = 1
               maximumJobIndex = max(maximumJobIndex,jobIndex)

               if not jobIndex in jobStatistics:
                  jobStatistics[jobIndex] = initializeJobStatistic(nCpus)

               if gridJobIdLogFile:
                  fp = open(matchingFile,'r')
                  if fp:
                     gridJobId = fp.readline()
                     gridJobIdLogFile.write("%d %s\n" % (jobId,gridJobId.strip()))
                     fp.close()
               os.remove(matchingFile)
            if gridJobIdLogFile:
               gridJobIdLogFile.close()
         else:
            log(GRIDJOBID + " not present")

         logGridHistories()
      else:
         jobStatistics[maximumJobIndex]['exitCode'] = exitStatus

      if batchScriptName:
         if os.path.isfile(batchScriptName):
            os.remove(batchScriptName)
      if appScriptName:
         if os.path.isfile(appScriptName):
            os.remove(appScriptName)


   if not jobSubmitted:
#     if stdOutput != "":
#        fileObject = open(localJobId + ".stdout",'w')
#        if fileObject:
#           fileObject.write(stdOutput)
#           fileObject.close()
      if stdError != "":
         fileObject = open(localJobId + ".stderr",'w')
         if fileObject:
            fileObject.write(stdError)
            fileObject.close()

   if stageFiles:
      if os.path.isfile(stageInTarFile):
         os.remove(stageInTarFile)

   if os.path.isfile(LOCALJOBID):
      os.remove(LOCALJOBID)

   if os.path.isfile(stdinput):
      os.remove(stdinput)

   for batchQueue in 'pbs','lsf','ll','slurm','condor':
      for outFile in 'stdout','stderr':
         batchQueueOutputFile = batchQueue + '_' + localJobId + "." + outFile
         if(os.path.exists(batchQueueOutputFile)):
            if(os.path.getsize(batchQueueOutputFile) == 0):
               os.remove(batchQueueOutputFile)


   re_Files = re.compile(TIMERESULTS + "(_[0-9]+)?$")
   dirFiles = os.listdir(os.getcwd())
   matchingFiles = filter(re_Files.search,dirFiles)
   if len(matchingFiles) > 0:
      foundIndexes = []
      matchingFiles.sort()
      for matchingFile in matchingFiles:
         if len(matchingFile) > len(TIMERESULTS):
            jobIndex = int(matchingFile[len(TIMERESULTS)+1:])
         else:
            jobIndex = 1
         maximumJobIndex = max(maximumJobIndex,jobIndex)
         foundIndexes.append(jobIndex)

         if not jobIndex in jobStatistics:
            jobStatistics[jobIndex] = initializeJobStatistic(nCpus)

         realTime = 0.
         userTime = 0.
         sysTime  = 0.
         exitStatus = 0
         exitSignal = 0

         fp = open(matchingFile,'r')
         if fp:
            while 1:
               line = fp.readline()
               if not line:
                  break
               line = line.strip()

               if line != "":
                  parts = line.split()
                  if len(parts) == 2:
                     type = parts[0]
                     timeUsed = float(parts[1])

                     if   type == "real":
                        realTime = max(realTime,timeUsed)
                     elif type == "user":
                        userTime = userTime + timeUsed
                     elif type == "sys":
                        sysTime  = sysTime + timeUsed
                  else:
                     if len(parts) > 2:
                        if parts[-2] == "status":
                           exitStatus = int(float(parts[-1]))
                        if parts[-2] == "signal":
# Killed by signal 2.
                           exitStatus = 1
                           exitSignal = int(float(parts[-1]))
                     if re.search("Killed",line):
                        exitStatus = 1
                        exitSignal = 15

            if exitSignal > 0:
               jobStatistics[jobIndex]['exitCode'] = exitStatus << 8 | exitSignal
            else:
               jobStatistics[jobIndex]['exitCode'] = exitStatus
            jobStatistics[jobIndex]['realTime'] = realTime
            jobStatistics[jobIndex]['userTime'] = userTime
            jobStatistics[jobIndex]['sysTime']  = sysTime

            fp.close()
         os.remove(matchingFile)
      if len(foundIndexes) != maximumJobIndex:
         for missingIndex in sets.Set(range(1,maximumJobIndex+1)).difference(sets.Set(foundIndexes)):
             log(TIMERESULTS + '_' + str(missingIndex) + " is missing")
      del foundIndexes
   else:
      if jobSubmitted:
         log(TIMERESULTS + " not present")

   if jobSubmitted:
      re_Files = re.compile(TIMESTAMPTRANSFERRED + "(_[0-9]+)?$")
      dirFiles = os.listdir(os.getcwd())
      matchingFiles = filter(re_Files.search,dirFiles)
      if len(matchingFiles) > 0:
         foundIndexes = []
         matchingFiles.sort()
         for matchingFile in matchingFiles:
            if len(matchingFile) > len(TIMESTAMPTRANSFERRED):
               jobIndex = int(matchingFile[len(TIMESTAMPTRANSFERRED)+1:])
            else:
               jobIndex = 1
            maximumJobIndex = max(maximumJobIndex,jobIndex)
            foundIndexes.append(jobIndex)

            if not jobIndex in jobStatistics:
               jobStatistics[jobIndex] = initializeJobStatistic(nCpus)

            if(os.path.getsize(matchingFile)):
               fp = open(matchingFile,'r')
               if fp:
                  jobStatistics[jobIndex]['transferCompleteTime'] = float(fp.readline())
                  fp.close()
            os.remove(matchingFile)
         if len(foundIndexes) != maximumJobIndex:
            for missingIndex in sets.Set(range(1,maximumJobIndex+1)).difference(sets.Set(foundIndexes)):
                log(TIMESTAMPTRANSFERRED + '_' + str(missingIndex) + " is missing")
         del foundIndexes
      else:
         log(TIMESTAMPTRANSFERRED + " not present")

      re_Files = re.compile(TIMESTAMPSTART + "(_[0-9]+)?$")
      dirFiles = os.listdir(os.getcwd())
      matchingFiles = filter(re_Files.search,dirFiles)
      if len(matchingFiles) > 0:
         foundIndexes = []
         matchingFiles.sort()
         for matchingFile in matchingFiles:
            if len(matchingFile) > len(TIMESTAMPSTART):
               jobIndex = int(matchingFile[len(TIMESTAMPSTART)+1:])
            else:
               jobIndex = 1
            maximumJobIndex = max(maximumJobIndex,jobIndex)
            foundIndexes.append(jobIndex)

            if not jobIndex in jobStatistics:
               jobStatistics[jobIndex] = initializeJobStatistic(nCpus)

            if(os.path.getsize(matchingFile)):
               fp = open(matchingFile,'r')
               if fp:
                  jobStatistics[jobIndex]['jobStartedTime'] = float(fp.readline())
                  fp.close()
            os.remove(matchingFile)
         if len(foundIndexes) != maximumJobIndex:
            for missingIndex in sets.Set(range(1,maximumJobIndex+1)).difference(sets.Set(foundIndexes)):
                log(TIMESTAMPSTART + '_' + str(missingIndex) + " is missing")
         del foundIndexes
      else:
         log(TIMESTAMPSTART + " not present")

      re_Files = re.compile(TIMESTAMPFINISH + "(_[0-9]+)?$")
      dirFiles = os.listdir(os.getcwd())
      matchingFiles = filter(re_Files.search,dirFiles)
      if len(matchingFiles) > 0:
         foundIndexes = []
         matchingFiles.sort()
         for matchingFile in matchingFiles:
            if len(matchingFile) > len(TIMESTAMPFINISH):
               jobIndex = int(matchingFile[len(TIMESTAMPFINISH)+1:])
            else:
               jobIndex = 1
            maximumJobIndex = max(maximumJobIndex,jobIndex)
            foundIndexes.append(jobIndex)

            if not jobIndex in jobStatistics:
               jobStatistics[jobIndex] = initializeJobStatistic(nCpus)

            if(os.path.getsize(matchingFile)):
               fp = open(matchingFile,'r')
               if fp:
                  jobStatistics[jobIndex]['jobFinshedTime'] = float(fp.readline())
                  fp.close()
            os.remove(matchingFile)
         if len(foundIndexes) != maximumJobIndex:
            for missingIndex in sets.Set(range(1,maximumJobIndex+1)).difference(sets.Set(foundIndexes)):
                log(TIMESTAMPFINISH + '_' + str(missingIndex) + " is missing")
         del foundIndexes
      else:
         log(TIMESTAMPFINISH + " not present")

   for jobIndex in jobStatistics:
      waitingTime = jobStatistics[jobIndex]['jobStartedTime'] - jobStatistics[jobIndex]['transferCompleteTime']
      elapsedRunTime = jobStatistics[jobIndex]['jobFinshedTime'] - jobStatistics[jobIndex]['jobStartedTime']

      jobStatistics[jobIndex]['waitingTime']    = max(waitingTime,0.)
      jobStatistics[jobIndex]['elapsedRunTime'] = max(elapsedRunTime,0.)

   reportExitCondition()

   sys.exit(0)

