# @package      hubzero-submit-common
# @file         SubmitServer.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2012 HUBzero Foundation, LLC.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2012 HUBzero Foundation, LLC.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#

import os
import sys
import signal
import select
import re
import pwd
import subprocess
import ldap
import traceback
import math
import shutil
import time

from hubzero.submit.LogMessage     import openLog, logJobId as log, logSetJobId
from hubzero.submit.CommandParser  import CommandParser
from hubzero.submit.ClientListener import ClientListener
from hubzero.submit.MySQLDatabase  import MySQLDatabase

class SubmitServer:
   def __init__(self,
                distributorDirectory,
                distributorScript):
      self.configData = {'listenURIs':[],
                         'mysqlHost':"",
                         'mysqlUser':"",
                         'mysqlPassword':"",
                         'mysqlDB':"",
                         'ldapHosts':[],
                         'ldapBaseDN':"",
                         'ldapUserDN':"",
                         'loadLimit':510,
                         'loadHalflife':3600,
                         'loadHorizon':86400
                        }

      self.mySQLDatabase  = None
      self.clientListener = None
      self.commandParser  = None

      self.clientData                       = {}
      self.clientData['username']           = ""
      self.clientData['password']           = ""
      self.clientData['token']              = ""
      self.clientData['args']               = []
      self.clientData['envs']               = {}
      self.clientData['umask']              = 027
      self.clientData['workDirectory']      = ""
      self.clientData['workDirectoryInode'] = None
      self.clientData['isClientTTY']        = True
      self.clientData['pegasusVersion']     = None
      self.clientData['waitForLowLoad']     = False
      self.clientData['reportMetrics']      = False
      self.clientData['showHelp']           = False
      self.clientData['isParametric']       = False
      self.clientData['localExecution']     = False
      self.clientData['inputFileMapping']   = {}

      self.serverData                         = {}
      self.serverData['distributorDirectory'] = distributorDirectory
      self.serverData['distributorScript']    = distributorScript
      self.serverData['pegasusVersion']       = None
      self.serverData['pegasusHome']          = None
      self.serverData['pegasusPythonPath']    = None
      self.serverData['type']                 = 'Listener'
      self.serverData['exitCode']             = 2
      self.serverData['uid']                  = None
      self.serverData['gid']                  = None
      self.serverData['gids']                 = None
      self.serverData['homeDirectory']        = None
      self.serverData['authAttempts']         = 0
      self.serverData['authz']                = 0
      self.serverData['jobId']                = 0
      self.serverData['superJobId']           = 0
      self.serverData['args']                 = []
      self.serverData['envs']                 = {}
      self.serverData['event']                = "simulation"
      self.serverData['ncpus']                = 1
      self.serverData['venue']                = "any"
      self.serverData['metricsRecorded']      = False
      self.serverData['metrics']              = ""
# Assume that a db entry exists.  That's where the jobid came from.
# If we've improperly assumed it exists, then finalizeJob() will
# indicate failure when a job is aborted prematurely.
      self.serverData['readyToExit']          = False
      self.serverData['dbJobEntryExists']     = True
      self.serverData['mysqlAuthenticated']   = False
      self.serverData['ldapAuthenticated']    = False
      self.serverData['sessionLimit']         = 0
      self.serverData['timeoutInterval']      = 60
      self.serverData['heartbeatInterval']    = 3600
      self.serverData['jobScanInterval']      = 15
      self.serverData['disconnectTime']       = 0
      self.serverData['disconnectMax']        = 4*3600
      self.serverData['selectTimeout']        = 5*60.
      self.serverData['childPid']             = None
      self.serverData['workDirectory']        = ""
      self.serverData['workDirectoryShared']  = True
      self.serverData['inputFileMapping']     = {}
      self.serverData['argMapping']           = {}
      self.serverData['inputObjects']         = {}
      self.serverData['outputObjects']        = {}
      self.serverData['importObjects']        = {}
      self.serverData['exportObjects']        = {}
      self.serverData['closePendingObjects']  = []
      self.serverData['childHasExited']       = False
      self.serverData['childKillPending']     = False
      self.serverData['jobScanPath']          = ""

      signal.signal(signal.SIGINT,self.handleSignal)
      signal.signal(signal.SIGHUP,self.handleSignal)
      signal.signal(signal.SIGQUIT,self.handleSignal)
      signal.signal(signal.SIGABRT,self.handleSignal)
      signal.signal(signal.SIGTERM,self.handleSignal)
      signal.signal(signal.SIGCHLD,self.handleChild)


   def exit(self,
            exitCode=0):
      if   self.serverData['type'] == 'JobExecuter':
         if self.serverData['readyToExit']:
            self.killChild()
            self.unsetupWorkDirectory()
            exitCode = self.serverData['exitCode']
            log("Server(%s) exiting(%d)." % (self.serverData['type'],exitCode))
            sys.exit(exitCode)
         else:
            self.finalizeJob()
            if self.serverData['exitCode'] == 2:
               self.serverData['exitCode'] = exitCode
            self.clientListener.postClientMessage('serverExit %d\n' % (self.serverData['exitCode']))
            self.serverData['readyToExit'] = True
      elif self.serverData['type'] == 'Listener':
         log("Server(%s) exiting(%d)." % (self.serverData['type'],exitCode))
         sys.exit(exitCode)


   def killChild(self):
      if self.serverData['childPid']:
         self.serverData['childKillPending'] = True
         try:
            os.kill(-self.serverData['childPid'],signal.SIGINT)
            for retry in range(0,12):
               time.sleep(10)
               os.kill(self.serverData['childPid'],0)
               if not self.serverData['childKillPending']:
                  break
            if self.serverData['childKillPending']:
               os.kill(-self.serverData['childPid'],signal.SIGTERM)
               for retry in range(0,6):
                  time.sleep(10)
                  os.kill(self.serverData['childPid'],0)
                  if not self.serverData['childKillPending']:
                     break
            if self.serverData['childKillPending']:
               os.kill(-self.serverData['childPid'],signal.SIGKILL)
         except:
            pass


   def stop(self):
      if self.serverData['readyToExit']:
         self.killChild()
         self.clientListener.postClientMessage('wait\n')
      log("Server stopping.")


   def handleSignal(self,
                    signalNumber,
                    frame):
      log("Server(%d) was terminated by a signal %d." % (os.getpid(),signalNumber))
      exitCode = (1 << 7) + signalNumber
      self.serverData['exitCode'] = exitCode
      self.exit(exitCode)


   def handleChild(self,
                   signalNumber,
                   frame):
      self.serverData['childKillPending'] = False
#     log("Child of %d (%s) has exited." % (os.getpid(),self.serverData['type']))


   def scheduleHeartbeat(self):
      signal.signal(signal.SIGALRM,self.heartbeat)
      signal.alarm(self.serverData['heartbeatInterval'])


   def heartbeat(self,
                 signalNumber,
                 frame):
      if self.mySQLDatabase.connect():
         sqlCommand = """UPDATE job SET heartbeat=now()
                          WHERE jobid='%d'
                      """ % (self.serverData['jobId'])
         result = self.mySQLDatabase.update(sqlCommand)
         if len(result) != 0:
            log("ERROR: Heartbeat: %s" % (result))

         self.mySQLDatabase.disconnect()

      self.scheduleHeartbeat()

      if self.clientListener.isClientConnected():
         self.serverData['disconnectTime'] = 0
      else:
         log("Server is in disconnected state.")
         self.serverData['disconnectTime'] += self.serverData['heartbeatInterval']
         if self.serverData['disconnectTime'] >= self.serverData['disconnectMax']:
            log("disconnect time (%d) >= disconnect max (%d)" % (self.serverData['disconnectTime'],
                                                                 self.serverData['disconnectMax']))
            self.exit(1)


   def scheduleTimeout(self,
                       interval=0):
      if interval == -1:
         signal.alarm(0)
      else:
         if interval == 0:
            interval = self.serverData['timeoutInterval']

         if self.serverData['authAttempts'] > 3:
            self.handleTimeout(0,0)
         signal.signal(signal.SIGALRM,self.handleTimeout)
         interval = int(math.ceil(interval))
         signal.alarm(interval)
         log("Server will time out in %d seconds." % (interval))


   def handleTimeout(self,
                     signalNumber,
                     frame):
      if   self.serverData['authz'] == 0:
         log("User '%s' failed to authenticate (%d) and timed out." % (self.clientData['username'],
                                                                       self.serverData['authAttempts']))
         self.exit(1)
      elif self.clientData['waitForLowLoad']:
         self.checkLoadAndLaunch()
      else:
         log("Unknown reason for timeout.")
         self.exit(1)


   def scheduleJobScan(self):
      signal.signal(signal.SIGALRM,self.handleJobScan)
      signal.alarm(self.serverData['jobScanInterval'])


   def handleJobScan(self,
                     signalNumber,
                     frame):
      if self.clientData['isParametric'] and self.clientData['isClientTTY']:
         if self.serverData['jobScanPath'] == "":
            jobScanPath = os.path.join(self.serverData['workDirectory'],
                                       "%08d" % (self.serverData['jobId']),
                                       "remoteParameterCombinations.csv")
            self.serverData['jobScanPath'] = jobScanPath
         if os.path.exists(self.serverData['jobScanPath']):
            fpJobScan = open(self.serverData['jobScanPath'],'r')
            if fpJobScan:
               jobScanText = fpJobScan.readlines()
               fpJobScan.close()
               self.clientListener.postClientMessageBySize('jobScanUpdate',''.join(jobScanText))

         self.scheduleJobScan()


   def configure(self,
                 configFilePath):
      keyValuePattern = re.compile('( *)(\w*)( *= *)(.*[^\s$])( *)')
      commentPattern  = re.compile('\s*#.*')

      fpConfig = open(configFilePath,'r')
      if fpConfig:
         eof = False
         while not eof:
            record = fpConfig.readline()
            if record != "":
               record = commentPattern.sub("",record)
               if keyValuePattern.match(record):
                  key,value = keyValuePattern.match(record).group(2,4)
                  if key in self.configData:
                     if   isinstance(self.configData[key],list):
                        self.configData[key] = [e.strip() for e in value.split(',')]
                     elif isinstance(self.configData[key],bool):
                        self.configData[key] = bool(value.lower() == 'true')
                     elif isinstance(self.configData[key],float):
                        self.configData[key] = float(value)
                     elif isinstance(self.configData[key],int):
                        self.configData[key] = int(value)
                     else:
                        self.configData[key] = value
                  else:
                     log("Undefined key = value pair %s = %s" % (key,value))
            else:
               eof = True
         fpConfig.close()

      configured = True
      if len(self.configData['listenURIs']) == 0:
         log("listenURIs missing from %s" % (configFilePath))
         configured = False
      if self.configData['mysqlHost'] == "" or \
         self.configData['mysqlUser'] == "" or \
         self.configData['mysqlPassword'] == "" or \
         self.configData['mysqlDB'] == "":
         log("MySQL information missing from %s" % (configFilePath))
         configured = False
      if len(self.configData['ldapHosts']) == 0 or \
         self.configData['ldapBaseDN'] == "" or \
         self.configData['ldapUserDN'] == "":
         log("LDAP information missing from %s" % (configFilePath))
         configured = False

      return(configured)


   def setupMySQL(self):
      setup = False
      self.mySQLDatabase = MySQLDatabase(mysqlHost=self.configData['mysqlHost'],
                                         mysqlUser=self.configData['mysqlUser'],
                                         mysqlPassword=self.configData['mysqlPassword'],
                                         mysqlDB=self.configData['mysqlDB'])
      if self.mySQLDatabase.connect():
         setup = True
         self.mySQLDatabase.disconnect()

      return(setup)


   def setupListeners(self):
      self.clientListener = ClientListener(self.configData['listenURIs'])

      return(self.clientListener.isListening())


   def parseCommandArguments(self,
                             doubleDashTerminator=False):
      argumentsOK = False
      log("Args are:" + str(self.clientData['args']))
      self.commandParser = CommandParser(doubleDashTerminator)
      self.commandParser.parseArguments(self.clientData['args'][1:])
      self.clientData['showHelp'] = self.commandParser.getOption('help') or \
                                    self.commandParser.getOption('helpManagers') or \
                                    self.commandParser.getOption('helpVenues') or \
                                    self.commandParser.getOption('helpTools')
      if   self.clientData['showHelp']:
         argumentsOK = True
      elif self.commandParser.validateArguments():
         exitCode = self.commandParser.setSweepParameters()
         if exitCode == 0:
            exitCode = self.commandParser.setCSVDataParameters()
            if exitCode == 0:
               if self.commandParser.getParameterCombinationCount() > 0:
                  self.clientData['isParametric'] = True

               enteredCommand = self.commandParser.getEnteredCommand()
               if enteredCommand == "":
                  log("Command must be supplied")
               else:
                  self.clientData['localExecution'] = self.commandParser.getOption('localExecution')
                  self.clientData['reportMetrics']  = self.commandParser.getOption('metrics')
                  argumentsOK = True

      return(argumentsOK)


   def ldapAuthenticate(self,
                        username,
                        password):
      if username and password:
         # Check the following note for more info on ldaps...
         # http://sourceforge.net/mailarchive/forum.php?forum_id=4346&max_rows=25&style=flat&viewmonth=200508

         for ldapHost in self.configData['ldapHosts']:
            try:
               if ldapHost.startswith("ldaps://"):
                  ldap.set_option(ldap.OPT_X_TLS_REQUIRE_CERT,ldap.OPT_X_TLS_ALLOW)
                  ldapObject = ldap.initialize(ldapHost)
               elif ldapHost.startswith("ldap://"):
                  ldapObject = ldap.initialize(ldapHost)
               else:
                  ldapObject = ldap.initialize("ldap://"+ldapHost)
            except ldap.LDAPError,msg:
               log("%s: %s" % (ldapHost,msg))
               continue

            try:
               status = ldapObject.simple_bind(self.configData['ldapUserDN'] % (username),password)
               if status != 1:
                  log("Unable to bind to LDAP server as user '%s'" % (username))
                  continue
            except ldap.LDAPError,msg:
               log("%s: %s" % (ldapHost,msg))
               continue

            try:
               arr = ldapObject.search_s(self.configData['ldapBaseDN'],ldap.SCOPE_SUBTREE,"uid=%s" % (username))
               if len(arr) != 1:
                  log("Non-singular result (%d) from LDAP for %s" % (len(arr),username))
                  continue
            except ldap.NO_SUCH_OBJECT,msg:
               log("%s: %s" % (ldapHost,msg))
               continue

            (dn,attrs) = arr[0]
            if 'jobsAllowed' in attrs:
               sessionLimit = int(attrs['jobsAllowed'][0])
               self.serverData['ldapAuthenticated'] = True
               self.serverData['sessionLimit']      = sessionLimit
            else:
               log("Can't get LDAP session_limit")
            ldapObject.unbind_s()
            break

      return(self.serverData['ldapAuthenticated'])


   def isValidUsername(self):
      validUsername = False
      username = self.clientData['username']
      try:
         (login,pw,uid,gid,name,homedir,shell) = pwd.getpwnam(username)
         self.serverData['uid']           = uid
         self.serverData['gid']           = gid
         self.serverData['homeDirectory'] = homedir
         validUsername = True
      except:
         log("Unable to get info for user '%s'" % (username))

      return(validUsername)


   def getUserGroups(self):
      userInGroups = False
      self.serverData['gids'] = []
      username = self.clientData['username']
      try:
         idCommand = ['/usr/bin/id',username,'--groups']
         child = subprocess.Popen(idCommand,
                                  stdout=subprocess.PIPE,
                                  stderr=subprocess.PIPE,
                                  close_fds=True)
         idStdOutput,idStdError = child.communicate()
         idExitStatus = child.returncode
         if idExitStatus == 0:
            idStdOutput = str(idStdOutput).strip()
            for gid in idStdOutput.split():
               self.serverData['gids'].append(int(gid))
            userInGroups = True
         else:
            log("Unable to get gidlist for user '%s'" % (username))
            if idStdOutput:
               log(idStdOutput)
            if idStdError:
               log(idStdError)
      except:
         # Not fatal...
         log("Unable to get gidlist for user '%s'" % (username))
         log(traceback.format_exc())

      return(userInGroups)


   def signon(self):
      authorized = False
      if self.isValidUsername() and self.getUserGroups():
         try:
            username = self.clientData['username']
         except:
            username = ""
         try:
            password = self.clientData['password']
         except:
            password = ""
         try:
            token = self.clientData['token']
         except:
            token = ""

         if   username and token:
            if self.mySQLDatabase.connect():
               sqlCommand = """SELECT sessnum FROM session
                               WHERE username='%s' AND sesstoken='%s'
                            """ % (username,token)
               result = self.mySQLDatabase.select(sqlCommand)
               if len(result) != 0:
                  row = result[0]
                  self.clientData['session'] = int(row[0])
                  authorized = True
                  self.serverData['mysqlAuthenticated'] = True
               self.mySQLDatabase.disconnect()
         elif username and password:
            authorized = self.ldapAuthenticate(username,password)
            if authorized:
               remoteIP = self.clientListener.getRemoteIP()
               appname = "submit"

               if self.mySQLDatabase.connect():
                  sqlCommand = """SELECT COUNT(*) FROM session WHERE username="%s"
                               """ % (username)
                  result = self.mySQLDatabase.select(sqlCommand)
                  if len(result) == 1:
                     row = result[0]
                     sessionCount = int(row[0])
                     if sessionCount >= self.serverData['sessionLimit']:
                        log("User %s cannot exceed session limit of %d" % (username,self.serverData['sessionLimit']))
                        self.clientListener.postClientMessageBySize('writeStderr',"You cannot exceed your session limit of %d." %
                                                                                                (self.serverData['sessionLimit']))
                        authorized = False
                     else:
                        sqlCommand = """INSERT INTO session(username,remoteip,exechost,dispnum,start,appname,sessname,sesstoken)
                                                     VALUES(    '%s',    '%s',    '%s',      0,now(),'%s'   ,    '%s',md5(rand()))
                                     """ %                 (username,remoteIP,"submit",              appname,appname)
                        result = self.mySQLDatabase.insert(sqlCommand)
                        if result != "":
                           log("ERROR: Unable to create session record for '%s'" % (username))
                           log("Error was: %s" % (result))
                           self.clientListener.postClientMessageBySize('message',"Internal database problem.")
                           authorized = False
                        else:
                           sqlCommand = """SELECT sessnum,sesstoken
                                             FROM session
                                            WHERE sessnum=LAST_INSERT_ID()"""
                           result = self.mySQLDatabase.select(sqlCommand)
                           row = result[0]
                           self.clientData['session'] = int(row[0])

                           sqlCommand = """INSERT INTO viewperm(                   sessnum, viewuser, viewtoken)
                                                         VALUES(                        %d,     '%s',md5(rand()))
                                        """ %                  (self.clientData['session'], username)
                           result = self.mySQLDatabase.insert(sqlCommand)
                           if result != "":
                              log("ERROR: Unable to create viewperm record for '%s'" % (username))
                              log("Error was: %s" % result)
                              authorized = False
                  self.mySQLDatabase.disconnect()
            else:
               log("LDAP authentication failed for user '%s'" % username)

         if 'session' in self.clientData:
            self.clientListener.postClientMessage('session %d\n' % (self.clientData['session']))

      return(authorized)


   def getInputObjects(self):
      listeningSockets,clientReader = self.clientListener.getInputObjects()

      return(listeningSockets,
             clientReader,
             self.serverData['inputObjects'].keys(),
             self.serverData['exportObjects'].keys())


   def getOutputObjects(self):
      clientWriter = self.clientListener.getOutputObjects()

      importObjects = []
      for importObject in self.serverData['importObjects']:
         if self.serverData['importObjects'][importObject]['buffer'] != "":
            importObjects.append(importObject)

      outputObjects = []
      for outputObject in self.serverData['outputObjects']:
         if self.serverData['outputObjects'][outputObject]['buffer'] != "":
            outputObjects.append(outputObject)

      return(clientWriter,
             importObjects,
             outputObjects)


   def readFile(self,
                fileObject):
      if   fileObject in self.serverData['inputObjects']:
         inputFile = self.serverData['inputObjects'][fileObject]
         if   inputFile == '#STDOUT#' or inputFile == '#STDERR#':
            try:
               text = os.read(fileObject.fileno(),1024)
               if text == "":
                  self.closeFile(fileObject)
               else:
                  if inputFile == '#STDOUT#':
                     self.clientListener.postClientMessageBySize('writeStdout',text)
                  else:
                     self.clientListener.postClientMessageBySize('writeStderr',text)
            except:
               self.exit(1)
         elif inputFile == '#METRICS#':
            try:
               text = os.read(fileObject.fileno(),1024)
               if text == "":
                  self.closeFile(fileObject)
               else:
                  self.serverData['metrics'] += text
            except:
               self.exit(1)
         else:
            try:
               text = os.read(fileObject.fileno(),1024)
               if text == "":
                  self.clientListener.postClientMessage('close %s\n' % (inputFile))
                  self.closeFile(fileObject)
               else:
                  self.clientListener.postClientMessageBySize('write %s' % (inputFile),text)
            except:
               self.exit(1)
      elif fileObject in self.serverData['exportObjects']:
         inputFile = self.serverData['exportObjects'][fileObject]
         try:
            text = os.read(fileObject.fileno(),1024)
            if text == "":
               self.clientListener.postClientMessage('close %s\n' % (inputFile))
               self.closeFile(fileObject)
            else:
               self.clientListener.postClientMessageBySize('write %s' % (inputFile),text)
         except:
            self.exit(1)


   def writeFile(self,
                 fileObject):
      if   fileObject in self.serverData['importObjects']:
         writeBuffer = self.serverData['importObjects'][fileObject]['buffer']
         try:
            textLength = os.write(fileObject.fileno(),writeBuffer)
            self.serverData['importObjects'][fileObject]['buffer'] = writeBuffer[textLength:]
         except:
            self.exit(1)
      elif fileObject in self.serverData['outputObjects']:
         writeBuffer = self.serverData['outputObjects'][fileObject]['buffer']
         try:
            textLength = os.write(fileObject.fileno(),writeBuffer)
            self.serverData['outputObjects'][fileObject]['buffer'] = writeBuffer[textLength:]
         except:
            self.exit(1)


   def closeFile(self,
                 fileObject):
      if fileObject in self.serverData['inputObjects']:
         del self.serverData['inputObjects'][fileObject]

      if fileObject in self.serverData['exportObjects']:
         exportFile = self.serverData['exportObjects'][fileObject]
         exportPath = os.path.join(os.sep,"tmp",exportFile)
         if(os.path.exists(exportPath)):
            os.remove(exportPath)
         del self.serverData['exportObjects'][fileObject]

      transferTarFile = ""
      if fileObject in self.serverData['importObjects']:
         transferTarFile = self.serverData['importObjects'][fileObject]['path']
         del self.serverData['importObjects'][fileObject]
         if len(self.serverData['importObjects']) == 0:
            self.clientListener.postClientMessage('exportFilesComplete\n')

      if fileObject in self.serverData['outputObjects']:
         del self.serverData['outputObjects'][fileObject]

      try:
         fileObject.close()
         if transferTarFile:
            tarCommand = ['tar','xv','-C',self.serverData['workDirectory'],
                                     '-f',os.path.join(self.serverData['workDirectory'],transferTarFile)]
            log("command: %s" % (tarCommand))
            child = subprocess.Popen(tarCommand,
                                     stdout=subprocess.PIPE,
                                     stderr=subprocess.PIPE,
                                     close_fds=True)
            tarStdOutput,tarStdError = child.communicate()
            tarExitStatus = child.returncode
            if tarExitStatus == 0:
               os.remove(os.path.join(self.serverData['workDirectory'],transferTarFile))
            else:
               log("Failed to extract transfer tarfile %s" % (transferTarFile))
               if tarStdOutput:
                  log(tarStdOutput)
               if tarStdError:
                  log(tarStdError)
      except:
         log(traceback.format_exc())


   def checkClosePendingObjects(self):
      markedForDeletion = []
      for fileObject in self.serverData['closePendingObjects']:
         if   fileObject in self.serverData['importObjects']:
            if self.serverData['importObjects'][fileObject]['buffer'] == "":
               self.closeFile(fileObject)
               markedForDeletion.append(fileObject)
         elif fileObject in self.serverData['outputObjects']:
            if self.serverData['outputObjects'][fileObject]['buffer'] == "":
               self.closeFile(fileObject)
               markedForDeletion.append(fileObject)
      for fileObject in markedForDeletion:
         self.serverData['closePendingObjects'].remove(fileObject)
      del markedForDeletion


   def checkLoad(self):
      loadOK = False
      if self.mySQLDatabase.connect():
         #
         # Check that the load is low enough
         #
         sqlCommand = """SELECT SUM( POW(.5, timestampdiff(second,start,now())/%d) )
                           FROM job WHERE username='%s' AND timestampdiff(second,start,now()) < %d
                      """ % (self.configData['loadHalflife'],self.clientData['username'],self.configData['loadHorizon'])
         result = self.mySQLDatabase.select(sqlCommand)
         if len(result) != 1:
            log("Error retrieving load for user %s" % (self.clientData['username']))
            self.scheduleTimeout()
         else:
            row = result[0]
            try:
               load = float(row[0])
            except TypeError:
               # If there are no entries for this user, result will be NULL
               load = 0.

            if load < self.configData['loadLimit']:
               log("Cumulative job load is %.2f.  (Max: %.2f)" % (load,self.configData['loadLimit']))
               loadOK = True
            else:
               log("User %s cannot exceed load limit %d." % (self.clientData['username'],self.configData['loadLimit']))
               if not self.clientData['waitForLowLoad']:
                  msg = "You cannot exceed your active job limit of %f. (cur: %f)" % (self.configData['loadLimit'],load)
                  self.clientListener.postClientMessageBySize('message',msg)
                  self.clientListener.postClientMessage('jobId 0\n')
                  self.scheduleTimeout()
               else:
                  loadRatio = load/self.configData['loadLimit']
                  waitPeriod = math.ceil(self.configData['loadHalflife'] * math.log(loadRatio) / math.log(2))
                  msg = "Cumulative job load is %.2f.  (Max: %.2f)  Sleeping %d seconds." % (load,self.configData['loadLimit'],
                                                                                             waitPeriod)
                  log(msg)
                  self.clientListener.postClientMessageBySize('message',msg)
                  self.scheduleTimeout(waitPeriod)

         self.mySQLDatabase.disconnect()

      return(loadOK)


   def checkLoadAndLaunch(self):
      if self.checkLoad():
         self.insertJob()
         self.launchJob()


   def insertJob(self):
      if self.mySQLDatabase.connect():
         #
         # Check that the superjob is a job that we own.
         #
         if self.serverData['superJobId'] != 0:
            sqlCommand = """SELECT jobid FROM job
                             WHERE jobid=%d AND sessnum=%d
                         """ % (self.serverData['superJobId'],self.clientData['session'])
            result = self.mySQLDatabase.select(sqlCommand)
            if len(result) != 1:
               log("ERROR: Trying to claim superjob of %d." % (self.serverData['superJobId']))
               self.serverData['superJobId'] = 0
         #
         # Insert the job into the job table.
         #
         sqlCommand = """INSERT INTO job(sessnum,superjob,username,event,ncpus,venue,start,heartbeat)
                              VALUES(%d,%d,'%s','%s',%d,'%s',now(),now())
                      """ % (self.clientData['session'],self.serverData['superJobId'],self.clientData['username'],
                             self.serverData['event'],self.serverData['ncpus'],self.serverData['venue'])
         result = self.mySQLDatabase.insert(sqlCommand)
         if result != "":
            log("ERROR: Unable to insert job.")
            log("Error was: %s" % (result))
         else:
            sqlCommand = """SELECT LAST_INSERT_ID()"""
            result = self.mySQLDatabase.select(sqlCommand)
            row = result[0]
            self.serverData['jobId'] = int(row[0])
            logSetJobId(self.serverData['jobId'])

            sqlCommand = """UPDATE session SET accesstime=now()
                             WHERE sessnum=%d
                         """ % (self.clientData['session'])
            result = self.mySQLDatabase.update(sqlCommand)
            if result != "":
               log("ERROR: Unable to update session accesstime for jobid %d." % (self.serverData['jobId']))
               log("Error was: %s" % (result))

         self.mySQLDatabase.disconnect()


   def updateJob(self):
      if self.mySQLDatabase.connect():
         sqlCommand = """UPDATE job SET event='%s',ncpus=%d
                          WHERE jobid=%d
                      """ % (self.serverData['event'],self.serverData['ncpus'],self.serverData['jobId'])
         result = self.mySQLDatabase.update(sqlCommand)
         if result != "":
            log("ERROR: Unable to update job fields for jobid %d." % (self.serverData['jobId']))
            log("Error was: %s" % (result))

         self.mySQLDatabase.disconnect()


   def invokeLocal(self):
      if self.clientData['isParametric']:
         pegasusRCPath    = os.path.join(self.serverData['distributorDirectory'],'pegasus_local.rc')
         pegasusSitesPath = os.path.join(self.serverData['distributorDirectory'],'sites_local.xml')
         if os.path.exists(pegasusRCPath) and os.path.exists(pegasusSitesPath):
            fpPegasus = open(pegasusRCPath,'r')
            if fpPegasus:
               pegasusRCText = fpPegasus.readlines()
               fpPegasus.close()
               self.clientListener.postClientMessageBySize('pegasusRC',''.join(pegasusRCText))
            fpPegasus = open(pegasusSitesPath,'r')
            if fpPegasus:
               pegasusSitesText = fpPegasus.readlines()
               fpPegasus.close()
               self.clientListener.postClientMessageBySize('pegasusSites',''.join(pegasusSitesText))
         else:
            log("Pegasus local configuration files are missing")
      self.clientListener.postClientMessage('serverReadyForIO\n')
      self.scheduleHeartbeat()

   @staticmethod
   def __isClientFileShared(clientFile,
                            clientInode):
      clientFileShared = True
      if os.path.exists(clientFile):
         inode = os.lstat(clientFile).st_ino
         if inode != clientInode:
            clientFileShared = False
      else:
         clientFileShared = False

      return(clientFileShared)


   def setupWorkDirectory(self):
      if self.__isClientFileShared(self.clientData['workDirectory'],self.clientData['workDirectoryInode']):
         self.serverData['workDirectory'] = self.clientData['workDirectory']
      else:
         self.serverData['workDirectory'] = os.path.join(os.sep,"tmp","dir_%s_%08d" % (self.clientData['username'],
                                                                                       self.serverData['jobId']))
         os.mkdir(self.serverData['workDirectory'],0700)
         self.serverData['workDirectoryShared'] = False

      os.chdir(self.serverData['workDirectory'])


   def unsetupWorkDirectory(self):
      if not self.serverData['workDirectoryShared']:
         os.chdir("/tmp")
         if os.path.isdir(self.serverData['workDirectory']):
            shutil.rmtree(self.serverData['workDirectory'],True)


   def filterEnvironment(self):
      self.serverData['envs'] = {}
      for env in self.clientData['envs']:
         if not env[0].isalpha():
            continue
         if env == "SUBMIT_JOB":
            self.serverData['envs'][env] = self.clientData['envs'][env]
            continue
         if env == "SUBMIT_APPLICATION_REVISION":
            self.serverData['envs'][env] = self.clientData['envs'][env]
            continue
         if env == "SUBMITVENUES":
            self.serverData['envs'][env] = self.clientData['envs'][env]
            continue
         if env.endswith("_CHOICE"):
            self.serverData['envs'][env] = self.clientData['envs'][env]
            continue

      if self.clientData['isClientTTY']:
         self.serverData['envs']['SUBMIT_ISCLIENTTTY'] = '1'

      self.serverData['envs']['HOME']    = self.serverData['homeDirectory']
      self.serverData['envs']['USER']    = self.clientData['username']
      self.serverData['envs']['LOGNAME'] = self.clientData['username']
      self.serverData['envs']['PATH']    = "/usr/bin:/bin"
      if self.serverData['pegasusVersion']:
         self.serverData['envs']['PEGASUS_VERSION'] = self.serverData['pegasusVersion']
      if self.serverData['pegasusHome']:
         self.serverData['envs']['PEGASUS_HOME'] = self.serverData['pegasusHome']
      if self.serverData['pegasusPythonPath']:
         self.serverData['envs']['PYTHONPATH'] = self.serverData['pegasusPythonPath']


   def mapArguments(self):
      if self.clientData['showHelp']:
         for arg in self.clientData['args']:
            self.serverData['args'].append(arg)
      else:
         enteredExecutable = self.commandParser.getEnteredExecutable()
         if self.commandParser.getOption('inputFiles'):
            inputFiles = self.commandParser.getOption('inputFiles')
         else:
            inputFiles = []

         reachedExecutable = False
         location = 0
         checkLocations = []
         for arg in self.clientData['args']:
            self.serverData['args'].append(arg)
            if arg == enteredExecutable:
               reachedExecutable = True
            if not reachedExecutable:
               if arg in inputFiles:
                  checkLocations.append(location)
            else:
               if not arg.startswith('-'):
                  checkLocations.append(location)
            location += 1

         for location in checkLocations:
            arg = self.serverData['args'][location]
            for clientArg in self.serverData['argMapping']:
               if arg == clientArg:
                  self.serverData['args'][location] = self.serverData['argMapping'][clientArg]
                  break

      distributorPath = os.path.join(self.serverData['distributorDirectory'],
                                     self.serverData['distributorScript'])
      self.serverData['args'][0] = distributorPath
      self.serverData['args'].insert(1, "--jobid")
      self.serverData['args'].insert(2, str(self.serverData['jobId']))


   def invokeDistributor(self):
      if not self.serverData['workDirectoryShared']:
         try:
            fpMarker = open('.__fileTimeMarkerSubmit','w')
            fpMarker.close()
            time.sleep(1)
         except:
            pass
      stdinPipe  = os.pipe()
      stdoutPipe = os.pipe()
      stderrPipe = os.pipe()
      statPipe   = os.pipe()
      self.serverData['childPid'] = os.fork()
      if self.serverData['childPid'] == 0:
         os.setsid()
         os.dup2(stdinPipe[0],0)
         os.dup2(stdoutPipe[1],1)
         os.dup2(stderrPipe[1],2)
         os.dup2(statPipe[1],3)
         os.close(stdinPipe[1])
         os.close(stdoutPipe[0])
         os.close(stderrPipe[0])
         os.close(statPipe[0])

         try:
            os.execve(self.serverData['args'][0],self.serverData['args'],self.serverData['envs'])
         except OSError,err:
            os.write(2,"Cannot invoke distributor.")
            os.write(2,err.args[1])
            sys.exit(err.args[0])
      else:
         fpStdinPipe  = os.fdopen(stdinPipe[1],'w',0)
         fpStdoutPipe = os.fdopen(stdoutPipe[0],'r',0)
         fpStderrPipe = os.fdopen(stderrPipe[0],'r',0)
         fpMetricPipe = os.fdopen(statPipe[0],'r',0)
         os.close(stdinPipe[0])
         os.close(stdoutPipe[1])
         os.close(stderrPipe[1])
         os.close(statPipe[1])
         self.serverData['outputObjects'][fpStdinPipe]           = {}
         self.serverData['outputObjects'][fpStdinPipe]['path']   = '#STDIN#'
         self.serverData['outputObjects'][fpStdinPipe]['buffer'] = ""
         self.serverData['inputObjects'][fpStdoutPipe] = '#STDOUT#'
         self.serverData['inputObjects'][fpStderrPipe] = '#STDERR#'
         self.serverData['inputObjects'][fpMetricPipe]   = '#METRICS#'

         # At this point, the I/O is ready to receive things from the client.
         # Inform the client.
         self.clientListener.postClientMessage('serverReadyForIO\n')

         if self.clientData['isParametric'] and self.clientData['isClientTTY']:
            self.scheduleJobScan()
         else:
            self.scheduleHeartbeat()


   def startCommand(self):
      self.scheduleHeartbeat()

      self.setUserGroupIds()
      self.filterEnvironment()
      self.mapArguments()
      os.umask(self.clientData['umask'])
      self.invokeDistributor()


   def launchJob(self):
      self.updateJob()
      self.clientListener.postClientMessage('jobId %d\n' % (self.serverData['jobId']))
      self.clientListener.postClientMessage('event %s\n' % (self.serverData['event']))
      if self.clientData['localExecution']:
         self.invokeLocal()
      else:
         self.startCommand()


   def updateJobMetrics(self,
                        metricsRecord):
      if not self.serverData['metricsRecorded']:
         if self.mySQLDatabase.connect():
            status   = 0
            waittime = 0.
            cputime  = 0.
            realtime = 0.
            ncpus    = 0
            program  = self.serverData['event']

            metricsRecord = metricsRecord.strip()
            metrics = metricsRecord.split()
            for metric in metrics:
               try:
                  key,value = metric.split('=')
                  if   key == 'status':
                     status = int(value)
                     self.serverData['exitCode'] = status
                  elif key == 'cputime':
                     cputime = float(value)
                  elif key == 'realtime':
                     realtime = float(value)
                  elif key == 'waittime':
                     waittime = float(value)
                  elif key == 'ncpus':
                     ncpus = int(value)
                  elif key == 'venue':
                     self.serverData['venue'] = value
                  elif key == 'event':
                     program = value
                  else:
                     log("Unknown status item: '%s'" % (metric))
               except:
                  log("Erroneous status item: '%s'" % (metric))

            metricsMessage = " venue=%s status=%d cpu=%f real=%f wait=%f" % (self.serverData['venue'],
                                                                             status,cputime,realtime,waittime)
            log("Job Status:" + metricsMessage)
            if self.clientData['reportMetrics'] and self.serverData['venue'] != "":
               log("Sending requested metrics.")
               msg = "=SUBMIT-METRICS=>"
               msg += " job=%d" % (self.serverData['jobId'])
               msg += metricsMessage
               self.clientListener.postClientMessageBySize('writeStderr',msg)

            if waittime > 0:
               sqlCommand = """INSERT INTO joblog(sessnum,job,superjob,event,start,walltime,venue)
                                    SELECT sessnum,jobid,superjob,'%s',start,%f,     '%s'
                                      FROM job WHERE jobid=%d
                            """ % ("[waiting]",waittime,self.serverData['venue'],self.serverData['jobId'])
               result = self.mySQLDatabase.insert(sqlCommand)
               if result != "":
                  log("ERROR: Unable to create wait time record. (%f)" % (waittime))
                  log("Error was: %s" % (result))

            sqlCommand = """INSERT INTO joblog(sessnum,job,superjob,event,start,walltime,cputime,ncpus,status,venue)
                                 SELECT sessnum,jobid,superjob,'%s',start,%f,     %f,     ncpus,%d,'%s'
                                   FROM job WHERE jobid=%d
                         """ % (program,realtime,cputime,status,self.serverData['venue'],self.serverData['jobId'])
            result = self.mySQLDatabase.insert(sqlCommand)
            if result != "":
               log("ERROR: Unable to copy job %d to joblog" % (self.serverData['jobId']))
               log("Error was: %s" % (result))

            self.serverData['metricsRecorded'] = True

            self.mySQLDatabase.disconnect()


   def finalizeJob(self):
      if self.serverData['dbJobEntryExists']:
         if self.serverData['jobId'] > 0:
            if self.mySQLDatabase.connect():
               sqlCommand = """SELECT sessnum FROM job
                               WHERE jobid=%d
                            """ % (self.serverData['jobId'])
               result = self.mySQLDatabase.select(sqlCommand)
               if len(result) == 0:
                  log("ERROR: Unable to find session for job %d." % (self.serverData['jobId']))
               else:
                  row = result[0]
                  session = row[0]

                  sqlCommand = """UPDATE session SET accesstime=now()
                                   WHERE sessnum=%d
                               """ % (session)
                  result = self.mySQLDatabase.update(sqlCommand)
                  if result != "":
                     log("ERROR: Unable to update session accesstime for job.")
                     log("Error was: %s" % result)

               sqlCommand = """UPDATE job SET active=0
                                WHERE jobid=%d
                            """ % (self.serverData['jobId'])
               result = self.mySQLDatabase.update(sqlCommand)
               if result != "":
                  log("Unable to deactivate job.")
                  log("Error was: " + result)

               #
               # Clear inactive jobs beyond the load horizon.
               #
               sqlCommand = """DELETE FROM job
                                WHERE active=0 AND timestampdiff(second,job.start,now()) > %d
                            """ % (self.configData['loadHorizon'])
               result = self.mySQLDatabase.delete(sqlCommand)
               if result != "":
                  log("Unable to clear old inactive jobs.")
                  log("Error was: " + result)

               #
               # Mark inactive any jobs that have not had a recent heartbeat.
               #
               sqlCommand = """UPDATE job SET active=0
                                WHERE active=1 AND timestampdiff(second,job.heartbeat,now()) > %d
                            """ % (self.serverData['heartbeatInterval']*3)
               if result != "":
                  log("Unable to deactivate moribund job entries.")
                  log("Error was: " + result)

               self.mySQLDatabase.disconnect()

         self.serverData['dbJobEntryExists'] = False


   def setUserGroupIds(self):
      if os.getuid() == 0:
         os.setregid(self.serverData['gid'],self.serverData['gid'])
         os.setgroups(self.serverData['gids'])
         os.setreuid(self.serverData['uid'],self.serverData['uid'])


   def processClientRequests(self):
      message = self.clientListener.pullClientMessage(0)
      while message:
         args = message.split()
#        if args[0] != 'null':
#           log("client request = %s" % (args[0]))

         if   args[0] == 'username':
            self.clientData['username'] = args[1]
         elif args[0] == 'password':
            self.clientData['password'] = args[1]
            self.serverData['authAttempts'] += 1
         elif args[0] == 'token':
            self.clientData['token'] = args[1]
            self.serverData['authAttempts'] += 1
         elif args[0] == 'signon':
            self.serverData['authz'] = self.signon()
            self.clientListener.postClientMessage('authz %d\n' % (self.serverData['authz']))
            if self.serverData['authz'] != 0:
               self.setUserGroupIds()
               self.clientListener.postClientMessageBySize('message',"Congratulations - you have successfully authenticated.")
               self.scheduleHeartbeat()
               if self.serverData['jobId'] == 0:
                  if self.checkLoad():
                     self.insertJob()
            else:
               self.scheduleTimeout()
         elif args[0] == 'arg':
            argLength = int(args[1])
            if argLength > 0:
               argText = self.clientListener.pullClientMessage(argLength)
               if len(argText) > 0:
                  self.clientData['args'].append(argText)
               else:
                  self.clientListener.pushClientMessage(message + '\n')
                  break
         elif args[0] == 'var':
            envNameLength = int(args[1])
            envValueLength = int(args[2])
            if envNameLength+envValueLength > 0:
               envText = self.clientListener.pullClientMessage(envNameLength+envValueLength)
               if len(envText) > 0:
                  envName  = envText[0:envNameLength]
                  envValue = envText[envNameLength:]
                  self.clientData['envs'][envName] = envValue
               else:
                  self.clientListener.pushClientMessage(message + '\n')
                  break
         elif args[0] == 'inputFile':
            clientInputInode         = long(args[1])
            clientInputFileLength    = int(args[2])
            clientInputFileArgLength = int(args[3])
            if clientInputFileLength+clientInputFileArgLength > 0:
               clientInputFileText = self.clientListener.pullClientMessage(clientInputFileLength+clientInputFileArgLength)
               if clientInputFileText > 0:
                  clientInputFile    = clientInputFileText[0:clientInputFileLength]
                  clientInputFileArg = clientInputFileText[clientInputFileLength:]

                  mapInputFile = not self.__isClientFileShared(clientInputFile,clientInputInode)
#                 log("Input file: %s %s" % (clientInputFile,mapInputFile))
                  if mapInputFile:
                     serverInputFile = os.path.join(self.serverData['workDirectory'],os.path.basename(clientInputFile))
                     self.clientData['inputFileMapping'][clientInputFile] = serverInputFile
                     self.serverData['inputFileMapping'][serverInputFile] = clientInputFile
                     serverArg = os.path.basename(clientInputFileArg)
                     self.serverData['argMapping'][clientInputFileArg] = serverArg
                     if clientInputFileArg.startswith('./'):
                        self.serverData['argMapping'][clientInputFileArg] = './' + serverArg
                     log("Arg map: %s %s" % (clientInputFileArg,self.serverData['argMapping'][clientInputFileArg]))
               else:
                  self.clientListener.pushClientMessage(message + '\n')
                  break
         elif args[0] == 'inputFileInodesSent':
            if len(self.clientData['inputFileMapping']) > 0:
               for inputFileMapping in self.clientData['inputFileMapping']:
                  self.clientListener.postClientMessage('addExportFile %s\n' % (inputFileMapping))
               transferTarFile = "%s_transfer.tar" % (self.serverData['jobId'])
               transferTarPath = os.path.join(self.serverData['workDirectory'],transferTarFile)
               fp = open(transferTarPath,'w')
               if fp:
                  self.serverData['importObjects'][fp]           = {}
                  self.serverData['importObjects'][fp]['path']   = transferTarFile
                  self.serverData['importObjects'][fp]['buffer'] = ""
                  self.clientListener.postClientMessage('exportFiles %s\n' % (transferTarFile))
               else:
                  log("input file open failed: %s" % (transferTarPath))
            else:
               self.clientListener.postClientMessage('exportFilesComplete\n')
         elif args[0] == 'read':
            clientInputFile  = args[1]
            textLength = int(args[2])
            if textLength > 0:
               text = self.clientListener.pullClientMessage(textLength)
               if len(text) > 0:
                  found = False
                  if not found:
                     for importObject in self.serverData['importObjects']:
                        if self.serverData['importObjects'][importObject]['path'] == clientInputFile:
                           self.serverData['importObjects'][importObject]['buffer'] += text
                           found = True
                           break
                  if not found:
                     for outputObject in self.serverData['outputObjects']:
                        if self.serverData['outputObjects'][outputObject]['path'] == clientInputFile:
                           self.serverData['outputObjects'][outputObject]['buffer'] += text
                           found = True
                           break
                  if not found:
                     log("File object not found to match %s" % (clientInputFile))
               else:
                  self.clientListener.pushClientMessage(message + '\n')
                  break
         elif args[0] == 'close':
            clientInputFile  = args[1]
            found = False
            for importObject in self.serverData['importObjects']:
               if self.serverData['importObjects'][importObject]['path'] == clientInputFile:
                  self.serverData['closePendingObjects'].append(importObject)
                  found = True
                  break
            if not found:
               for outputObject in self.serverData['outputObjects']:
                  if self.serverData['outputObjects'][outputObject]['path'] == clientInputFile:
                     self.serverData['closePendingObjects'].append(outputObject)
                     found = True
                     break
         elif args[0] == 'importFilesComplete':
            self.exit()
         elif args[0] == 'umask':
            self.clientData['umask'] = int(args[1],0)
         elif args[0] == 'pwd':
            self.clientData['workDirectory']      = args[1]
            self.clientData['workDirectoryInode'] = long(args[2])
         elif args[0] == 'isClientTTY':
            self.clientData['isClientTTY'] = bool(int(args[1]))
         elif args[0] == 'pegasusVersion':
            self.clientData['pegasusVersion'] = args[1]
            pegasusCommand = ". /etc/environ.sh\n" + \
                             "use -e -r pegasus-%s\n" % (self.clientData['pegasusVersion']) + \
                             "pegasus-version\n" + \
                             "pegasus-config --bin\n" + \
                             "pegasus-config --python"
            child = subprocess.Popen(pegasusCommand,shell=True,
                                     stdout=subprocess.PIPE,
                                     stderr=subprocess.PIPE,
                                     close_fds=True)
            pegasusStdOutput,pegasusStdError = child.communicate()
            pegasusExitStatus = child.returncode
            if pegasusExitStatus == 0:
               try:
                  pegasusVersion,pegasusBin,pegasusPythonPath = pegasusStdOutput.strip().split()
                  self.serverData['pegasusVersion']    = pegasusVersion
                  self.serverData['pegasusHome']       = os.path.dirname(pegasusBin)
                  self.serverData['pegasusPythonPath'] = pegasusPythonPath
               except:
                  pass
         elif args[0] == 'startLocal':
            self.setupWorkDirectory()
            if self.parseCommandArguments():
               self.launchJob()
            else:
               log("Command line argument parsing failed")
               self.exit(1)
         elif args[0] == 'setupRemote':
            self.setupWorkDirectory()
            if self.parseCommandArguments():
               self.clientListener.postClientMessage('serverReadyForInputMapping\n')
            else:
               log("Command line argument parsing failed")
         elif args[0] == 'startRemote':
            self.launchJob()
         elif args[0] == 'localWait':
            self.serverData['readyToExit'] = True
            self.stop()
         elif args[0] == 'jobId':
            self.serverData['jobId'] = int(args[1])
            logSetJobId(self.serverData['jobId'])
         elif args[0] == 'event':
            self.serverData['event'] = args[1]
         elif args[0] == 'localMetrics':
            instanceIndex = int(args[1])
            status        = int(args[2])
            cpuTime       = float(args[3])
            realTime      = float(args[4])
            self.updateJobMetrics("venue=%d:local status=%d cputime=%f realtime=%f" % (instanceIndex,status,cpuTime,realTime))
            self.serverData['metricsRecorded'] = False
         elif args[0] == 'localExit':
            status   = int(args[1])
            cpuTime  = float(args[2])
            realTime = float(args[3])
            venue = ""
            self.updateJobMetrics("venue=%s status=%d cputime=%f realtime=%f" % (venue,status,cpuTime,realTime))
            self.finalizeJob()
            self.clientListener.postClientMessage('exit %d\n' % (status))
            self.serverData['readyToExit'] = True
         elif args[0] == 'clientSignal':
            signalNumber = int(args[1])
            if self.serverData['childPid']:
               self.killChild()
            else:
               self.handleSignal(signalNumber,None)
         elif args[0] == 'clientExit':
            exitCode = int(args[1])
            self.exit(exitCode)
         elif args[0] == 'null':
            pass
         else:
            log("Discarded client message: %s" % (message))

         message = self.clientListener.pullClientMessage(0)


   def run(self):
      while True:
         try:
            clientListenerSockets,clientReader,inputObjects,exportObjects = self.getInputObjects()
            clientWriter,importObjects,outputObjects                      = self.getOutputObjects()
            if self.serverData['childHasExited']:
               selectTimeout = 0.1
            else:
               selectTimeout = self.serverData['selectTimeout']
            readyReaders,readyWriters,readyExceptions = select.select(clientListenerSockets+clientReader+inputObjects+exportObjects,
                                                                      clientWriter+importObjects+outputObjects,[],
                                                                      selectTimeout)
         except select.error,err:
            readyReaders = []
            readyWriters = []

         for readyReader in readyReaders:
            if   readyReader in clientListenerSockets:
               self.clientListener.acceptClientConnection(readyReader)

               # Do a double-fork to dissociate from the listening server.
               if os.fork() != 0:
                  self.clientListener.closeClientConnection()  # Close the client socket in the listening server
                  os.wait()                                    # Wait for the intermediate child to exit
               else:
                  if os.fork() != 0:
                     sys.exit(0) # This is the intermediate child.  Exit.
                  else:
                     # This is the real child.
                     os.setsid()
                     self.serverData['type'] = 'JobExecuter'
                     self.scheduleTimeout()
                     if self.clientListener.handshake(readyReader):
                        self.clientListener.closeListeningConnections()
                     else:
                        log("Client connection handshake failed.")
                        self.exit(1)
            elif readyReader in clientReader:
               self.clientListener.receiveClientMessage()
            else:
               self.readFile(readyReader)

         self.processClientRequests()

         for readyWriter in readyWriters:
            if   readyWriter in clientWriter:
               self.clientListener.sendClientMessage()
            elif readyWriter in importObjects:
               self.writeFile(readyWriter)
            elif readyWriter in outputObjects:
               self.writeFile(readyWriter)
         self.checkClosePendingObjects()

         if self.serverData['readyToExit']:
            if not self.clientListener.isClientMessagePending():
               self.exit()

         if self.serverData['childPid']:
            pid = 0
            try:
               (pid,exitCode) = os.waitpid(self.serverData['childPid'],os.WNOHANG)
            except:
               try:
                  os.kill(self.serverData['childPid'],0)
               except:
                  log("Child has exited.")
                  pid = self.serverData['childPid']
                  exitCode = 1 << 8
            if pid != 0:
               self.serverData['childHasExited'] = True
               self.serverData['childPid'] = 0
               exitCode = exitCode >> 8

               while len(self.serverData['inputObjects']) > 0:
                  inputObjects = self.serverData['inputObjects'].keys()
                  for inputObject in inputObjects:
                     self.readFile(inputObject)

               metricsRecords = self.serverData['metrics'].split('\n')
               metricsParsed = False
               for metricsRecord in metricsRecords:
                  if metricsRecord != "":
                     self.serverData['metricsRecorded'] = False
                     self.updateJobMetrics(metricsRecord)
                     metricsParsed = True
               if not metricsParsed:
                  self.updateJobMetrics("status=%d" % (self.serverData['exitCode']))

               if os.path.exists(self.serverData['jobScanPath']):
                  fpJobScan = open(self.serverData['jobScanPath'],'r')
                  if fpJobScan:
                     jobScanText = fpJobScan.readlines()
                     fpJobScan.close()
                     self.clientListener.postClientMessageBySize('jobScanUpdate',''.join(jobScanText))
                  os.remove(self.serverData['jobScanPath'])

               if self.serverData['workDirectoryShared']:
                  self.clientListener.postClientMessage('importFile None\n')
               else:
                  try:
                     transferTarFile = "%s_output.tar" % (self.serverData['jobId'])
                     transferTarPath = os.path.join(os.sep,"tmp",transferTarFile)
                     findCommand = ['find','.','-newer','.__fileTimeMarkerSubmit','-not','-name','.','-print0']
                     tarCommand = ['xargs','--null','tar','cvf',transferTarPath]
                     findChild = subprocess.Popen(findCommand,
                                                  stdout=subprocess.PIPE,
                                                  stderr=subprocess.PIPE,
                                                  close_fds=True)
                     tarChild = subprocess.Popen(tarCommand,
                                                 stdin=findChild.stdout,
                                                 stdout=subprocess.PIPE,
                                                 stderr=subprocess.PIPE,
                                                 close_fds=True)
                     tarStdOutput,tarStdError = tarChild.communicate()
                     tarExitStatus = tarChild.returncode
                     if tarExitStatus != 0:
                        log("Failed to write transfer tarfile %s" % (transferTarPath))
                        if tarStdOutput:
                           log(tarStdOutput)
                        if tarStdError:
                           log(tarStdError)
                        exitCode = 1
                     else:
                        fp = open(transferTarPath,'r')
                        if fp:
                           self.serverData['exportObjects'][fp] = transferTarFile
                           self.clientListener.postClientMessage('importFile %s\n' % (transferTarFile))
                        else:
                           log("input file open failed: %s" % (transferTarPath))
                  except OSError,err:
                     log("Failed to create transfer tarfile %s" % (transferTarPath))
                     log(err.args[1])


