# @package      hubzero-submit-client
# @file         JobScanner.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2012-2013 HUBzero Foundation, LLC.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2012-2013 HUBzero Foundation, LLC.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#

import sys
import os
import os.path
import math
import re      # Regexp: http://docs.python.org/library/re.html

import curses # <-- docs at: http://docs.python.org/library/curses.html
import curses.ascii   # <-- docs at http://docs.python.org/library/curses.ascii.html
import curses.textpad
import curses.wrapper

from numpy import empty, ones # The matrices use this
import csv

import textwrap

# ===============
class SearchBoard:
   def __init__(self, nrow, ncol):
      self.hit   = empty( (nrow, ncol), dtype=bool)
      self.dirty =  ones( (nrow, ncol), dtype=bool)

   def clear(self):
      self.dirty = ones( (self.numRows(), self.numCols()), dtype=bool)

   def set(self, row, col, value):
      self.hit[row][col] = value
      self.cleanify(row, col) # This cell has been legitimately set...

   def matches(self, row, col):
      if (self.dirty[row][col]):
         raise # It's not a "clean" value yet
      else:
         return self.hit[row][col]

   def isDirty(self, row, col):
      return self.dirty[row][col]

   def dirtify(self, row, col):
      self.dirty[row][col] = True

   def cleanify(self, row, col):
      self.dirty[row][col] = False

   def numRows(self):
      return self.hit.shape[0]

   def numCols(self):
      return self.hit.shape[1]

# ===============

class Point:
   def __init__(self, argX, argY):
      self.x = argX
      self.y = argY

#class RowCol: # Like a Point, but ROW (Y) comes first, then COL (X)
#   def __init__(self, argRow, argCol):
#      self.row = argRow
#      self.col = argCol
#
#   def __init__(self, rowColList): # Used to init size from "getmaxyx()"
#      (self.row, self.col) = rowColList

class Size:
   def __init__(self, argWidth, argHeight):
      self.width  = argWidth
      self.height = argHeight

   def resizeYX(self, rowColList):
      (self.height, self.width) = rowColList


class AGW_File_Data_Collection:
   '''
   This object stores all the state about a file. It's a lot like a "window" on a normal GUI.
   Confusingly, it is NOT the same as an AGW_Win, which is a "CURSES" terminal sub-window.
   Sorry for the confusion.
   '''
   def __init__(self):
      self.collection = [] # <-- contants: a bunch of AGW_File_Data objects
      self.currentFileIdx = None # which one are we currently looking at?
                                 # probably should be not part of the object, come to think of it.

   def size(self):
      '''Basically, how many files did we load.'''
      return len(self.collection)

   def getCurrent(self): # AGW_File_Data_Collection
      '''Get the current data object that backs the spreadsheet we are looking at right this second.'''
      return self.getInfoAtIndex(self.currentFileIdx)

   def getInfoAtIndex(self, i):
      return self.collection[i]

   def addFileInfo(self, fileInfoObj):
      if (not isinstance(fileInfoObj, AGW_File_Data)):
         print("Uh oh, someone tried to add some random object into the file info collection.")
         raise
      else:
         self.collection.append(fileInfoObj)

class AGW_File_Data:
   def __init__(self, argFilename):
      self.filename = argFilename
      self.table = AGW_Table()
      self.defaultCellProperty = curses.A_NORMAL #REVERSE
      self.hasColHeader = False
      self.hasRowHeader = False
      self.cursorPos = Point(0, 0) # what is the selected cell
      self.__regex = None
      self.__compiledRegex = None
      self.regexIsCaseSensitive = False
      self.boolHighlightNumbers = False

   def getNumCols(self):
      return self.table.getNumCols()
   def getNumRows(self):
      return self.table.getNumRows()

   def getActiveCellX(self):
      return self.cursorPos.x
   def getActiveCellY(self):
      return self.cursorPos.y

   def toggleNumericHighlighting(self):
      self.boolHighlightNumbers = (not self.boolHighlightNumbers)

   def stringFromAny(self, argString): # Returns a string, even if given None as an input
      if argString is None:
         return ""
      else:
         return str(argString)

   def getRegexString(self):
      return self.stringFromAny(self.__regex)

   def appendToCurrentSearchTerm(self, newThing):
      self.changeCurrentSearchTerm(self.stringFromAny(self.__regex) + self.stringFromAny(newThing))
      return

   def lenFromAny(self, argThing): # returns 0 for None's length. Otherwise you get an exception
      if argThing is None:
         return 0
      else:
         return len(argThing)

   # In class AGW_File_Data
   def changeCurrentSearchTerm(self, argSearchString, argIsCaseSens=None):
      self.__regex = argSearchString

      if (argIsCaseSens is not None):
         self.regexIsCaseSensitive = argIsCaseSens

      if (self.lenFromAny(self.__regex) <= 0):
         self.__regex         = None
         self.__compiledRegex = None
      else:
         if (self.regexIsCaseSensitive):
            self.__compiledRegex = re.compile(self.__regex)
         else:
            self.__compiledRegex = re.compile(self.__regex, re.IGNORECASE)

   def clearCurrentSearchTerm(self):
      self.changeCurrentSearchTerm(None, None)

   def trimRegex(self, numChars=1):
      '''Remove the last <numChars> characters from the end of the search term
      (i.e., it is like pressing backspace). Clears the search term
      (which sets it to None) if it is going to be zero-length.'''
      currentRegexLength = self.lenFromAny(self.__regex)
      if currentRegexLength <= 1:
         self.clearCurrentSearchTerm()
      else:
         self.changeCurrentSearchTerm(self.__regex[:(currentRegexLength-1)])

   def regexIsActive(self):
      '''Tells us whether we should be highlighting the search terms or not'''
      return (self.__regex is not None)

   # In class AGW_File_Data
   def stringDoesMatchRegex(self, stringToCheck):
      if (self.__regex is None or stringToCheck is None):
         return False

      if (self.__compiledRegex.search(stringToCheck)): # Note the difference between *search* and *match*
                                                       # (match does not do partial results)
         return True
      else:
         return False


# =======================================
# End of class AGW_File_Data
# =======================================

# =======================================
# Start of class AGW_Win
# These are "curses" (terminal) "windows"--basically just regions in the terminal
# that can be written to separately with their own coordinate systems.
# =======================================
class AGW_Win:
   def __init__(self):
      self.win = None
      self.pos = Point(0,0) # where is the top-left of this window?
      self.windowWidth = 0
      self.windowHeight = 0

   # Make a new window with the height, width, and at the location specified
   def initWindow(self, argHeight, argWidth, atY, atX):
      self.pos  = Point(atX, atY)
      self.windowWidth  = argWidth
      self.windowHeight = argHeight
      try:
         self.win = curses.newwin(argHeight, argWidth, atY, atX)
      except:
         raise "Cannot allocate a curses window with height " + \
               str(argHeight) + " and width " + str(argWidth) + \
               " at Y=" + str(atY) + " and X=" + str(atX) # + ". Error was: " + err.message

   # safeAddCh: Safely adds a single character to an AGW_Win object
   # It is "safe" because it does not throw an error if it overruns
   # the pad (instead, it just doesn't draw anything at all)
   def safeAddCh(self, y, x, argChar, attr=0):
      if (y >= self.windowHeight or x >= self.windowWidth-1): # out of bounds!
         return

      try:
         self.win.addch(y, x, argChar, attr)
      except(curses.error):
         #1, "safeAddCh is messed up: " + err.message)
         print("ERROR: Unable to print a character at", y, x, \
               "with window dimensions (in chars): ", self.windowHeight, self.windowWidth)
         raise
      except:
         raise

   # safeAddStr: Safely adds a string to a CURSES "win" object.
   # It is "safe" because it does not throw an error if it overruns
   # Additionally, it does NOT WRAP TEXT. This is different from default addstr.
   def safeAddStr(self, y, x, string, attr=0):
      try:
         if (string is None):
            return
         if (x < 0 or y < 0):
            return

         if y >= self.windowHeight:
            return # off screen!

         if (x+len(string) >= self.windowWidth): # Runs off to the right.
            newLength = max(0, (self.windowWidth - x - 1))
            string = string[:newLength]

         if (len(string) > 0):
            self.win.addstr(y, x, string, attr)

      except(curses.error):
         print("safeAddStr is messed up!")
         raise
      except:
         raise


## Data window (the main window with the cells in it)
class AGW_DataWin(AGW_Win):
   def __init__(self, isSearchable=False):
      AGW_Win.__init__(self) # parent constructor
      self.info = None
      self.defaultCellProperty = curses.A_NORMAL #REVERSE
      self.cellHeight = 1 ## How many lines of text each cell takes up, vertically. 1 is normal. Integer.
      self.RAGGED_END_ID = 1
      self.SELECTED_CELL_ID = 2
      self.COL_HEADER_ID = 3
      self.ROW_HEADER_ID = 4
      self.BOX_COLOR_ID = 5 # The borders of the cells
      self.SEARCH_MATCH_COLOR_ID = 7 # Highlighted search results
      self.HEADER_NUM_DELIMITER_STRING = ": " # the separator between "ROW_1" and "here is the header".
                                              # Example: "ROW_982-->Genomes". In that case, the string would have been "-->"
      self.isSearchable = isSearchable


   def getTable(self):
      return self.info.table

   def setInfo(self, whichInfo):
      if (not isinstance(whichInfo, AGW_File_Data)):
         print("### Someone passed in a not-an-AGW_File_Data object to AGW_DataWin--->setInfo()\n")
         raise

      self.info = whichInfo
      return

   def getInfo(self):
      return self.info

   def padStrToLength(self, argString, argMaxlen, padChar):
      # pad out the length with blanks so that the ENTIRE
      # length is taken up. However! curses does not like to
      # draw just plain things unfortunately, so we have to
      # trick it with a -. This is very hackish. Maybe I
      # should draw colored boxes instead?
      numBlankSpacesToAdd = argMaxlen - len(argString)
      if   numBlankSpacesToAdd > 0:
         return (argString + str(padChar*numBlankSpacesToAdd))
      elif numBlankSpacesToAdd < 0:
         return ("..." + argString[-(argMaxlen-3):])
      else:
         return argString


   def calculateBorderChar(self, r, c, topRow, leftCol, bottomRow, rightCol):
      if (r == 0): # TOP R
         if (c == 0):
            ch = curses.ACS_ULCORNER
         elif (c == rightCol):
            ch = curses.ACS_URCORNER
         else:       
            ch = curses.ACS_TTEE
      elif (r == bottomRow):
         if (c == 0):
            ch = curses.ACS_LLCORNER
         elif (c == rightCol):
            ch = curses.ACS_LRCORNER
         else:
            ch = curses.ACS_BTEE
      else:
         if (c == 0):
            ch = curses.ACS_LTEE
         elif (c == rightCol):
            ch = curses.ACS_RTEE
         else:
            ch = curses.ACS_PLUS

      return ch


   def stringFromAny(self, argString): # Returns a string, even if given None as an input
      if argString is None:
         return ""
      else:
         return str(argString)


   def drawTable(self,
                 whichInfo,
                 topCell,
                 leftCell,
                 cellBorders,
                 nRowsToDraw=None,
                 nColsToDraw=None,
                 boolPrependRowCoordinate=False,
                 boolPrependColCoordinate=False):
      self.win.erase() # or clear()

      theTable = self.getTable()

      if nColsToDraw is not None:
         numToDrawX = nColsToDraw
      else:
         numToDrawX = theTable.getNumCols()

      if nRowsToDraw is not None:
         numToDrawY = nRowsToDraw
      else:
         numToDrawY = theTable.getNumRows()

      start = Point(leftCell, topCell) # "which cells to draw"
      end   = Point(min(leftCell+numToDrawX, theTable.getNumCols()),
                    min(topCell+numToDrawY , theTable.getNumRows()))

      ## See if we need to read more lines from the file
      if (end.y >= theTable.getNumRows()):
         theTable.readFromCurrentFile() ## Ok, read some more lines!

      cellTextPos = Point(None, None)
      for r in range(start.y, end.y): ## Go through each row, one at a time, top to bottom
         cellTextPos.y = cellBorders.height + (cellBorders.height+self.cellHeight)*(r - start.y) # all cells are the same HEIGHT,
                                                                                                 # so this can be computed in
                                                                                                 # one equation

         if cellTextPos.y >= self.windowHeight:
            break

         cellTextPos.x = cellBorders.width # initialization! (update is way down below)

         for c in range(start.x, end.x): ## Go through ALL the cells on this row
            if cellTextPos.x >= self.windowWidth:
               break

            cell = theTable.cellValue(r, c) ## <-- this is the actual text that is in the cell
            maxLenForThisCell = theTable.getColWidth(c) ## This is how long this cell text can be.

            if (boolPrependColCoordinate):
               cell = str(c+1) + self.HEADER_NUM_DELIMITER_STRING + self.stringFromAny(cell)

            if (boolPrependRowCoordinate):
               cell = str(r+1) + self.HEADER_NUM_DELIMITER_STRING + self.stringFromAny(cell)

            selectedPt = whichInfo.cursorPos # Where is the cursor...

            cellIsSelected = (c == selectedPt.x and r == selectedPt.y)
            shouldHighlightCell = cellIsSelected # or (self.highlightEntireCol and c == selectedPt.x) or
                                                 # (self.highlightEntireRow and r == selectedPt.y)

            cellAttr = self.defaultCellProperty

            drawCheckerboard = False

            if (cell is None):
               cellAttr = curses.color_pair(self.RAGGED_END_ID)       # (indicate that there isn't a cell here at all
               cell = self.padStrToLength("", maxLenForThisCell, '~') #  distinct from an *empty* cell)

            if (r == 0 and c == 0 and (self.info.hasRowHeader and self.info.hasColHeader)):
               cellAttr = curses.A_NORMAL # there is an "odd man out" in the top left, for files
               drawCheckerboard = True     # with both a column AND a row header
            elif (r == 0 and self.info.hasColHeader):
               cellAttr = curses.color_pair(self.COL_HEADER_ID)
               drawCheckerboard = True
            elif (c == 0 and self.info.hasRowHeader):
               cellAttr = curses.color_pair(self.ROW_HEADER_ID)
               drawCheckerboard = True

            if (shouldHighlightCell):
               cellAttr = curses.color_pair(self.SELECTED_CELL_ID) + curses.A_NORMAL

            cell = " " + self.padStrToLength(cell, maxLenForThisCell-2, " ") + " " # <-- needed so that the cells don't randomly
                                                                                   # get garbage everywhere when they don't re-draw
                                                                                   # all the way. Very annoying. If you remove this,
                                                                                   # then long cells will randomly overwrite text
                                                                                   # of shorter cells when you move left and right.

            if drawCheckerboard:
               bgAttr = cellAttr
               self.win.attron(bgAttr)
               hLineLength = max(0, min(self.windowWidth-cellTextPos.x, maxLenForThisCell))
               self.win.hline(cellTextPos.y, cellTextPos.x, ' ', hLineLength) # checkerboard character
               self.win.attroff(bgAttr)

            if (whichInfo.regexIsActive() and self.isSearchable):
               theTable.initRegexTable()
               if (theTable.regTab.isDirty(r, c)):
                  # calcluate the regex...
                  boolRegexMatched = whichInfo.stringDoesMatchRegex(cell)
                  theTable.regTab.set(row=r, col=c, value=boolRegexMatched)

               if (theTable.regTab.matches(row=r, col=c)):
                  cellAttr = curses.color_pair(self.SEARCH_MATCH_COLOR_ID) #+ curses.A_REVERSE #curses.A_BLINK

            self.safeAddStr(cellTextPos.y, cellTextPos.x, cell, cellAttr) # Draw text
            self.win.attron(curses.color_pair(self.BOX_COLOR_ID))

            if (cellBorders.height > 0): # horizontal lines
               hLineLength = max(0, min(self.windowWidth - cellTextPos.x - 1, maxLenForThisCell))

               try:
                  self.win.hline(cellTextPos.y-1, cellTextPos.x, curses.ACS_HLINE, hLineLength)
               except:
                  print("problem when cellTextPos is y=" + str(cellTextPos.y-1) +
                        ", x=" + str(cellTextPos.x) + 
                        " and also the rows and cols are " + str(r) + " and " + str(c))
                  raise

            if (cellBorders.width > 0): # vertical lines
               vLineLength = max(0, min(self.windowHeight-cellTextPos.y, cellBorders.width))
               try:
                  self.win.vline(cellTextPos.y, cellTextPos.x-1, curses.ACS_VLINE, vLineLength)
               except:
                  print("problem when cellTextPos is y=" + str(cellTextPos.y) +
                        ", x=" + str(cellTextPos.x-1) +
                        " and also the rows and cols are " + str(r) + " and " + str(c))
                  raise

            if (cellBorders.width > 0 and cellBorders.height > 0):
               ch = self.calculateBorderChar(r=r, c=c, topRow=0, leftCol=0,
                                             bottomRow=theTable.getNumRows(), rightCol=theTable.getNumCols())
               self.safeAddCh(cellTextPos.y-1, cellTextPos.x-1, ch)

            self.win.attroff(curses.color_pair(self.BOX_COLOR_ID))

            cellTextPos.x += (cellBorders.width+maxLenForThisCell)

         if (cellBorders.width > 0): # vertical lines
            vLineLength = max(0, min(self.windowHeight-cellTextPos.y, cellBorders.width))
            try:
               self.win.attron(curses.color_pair(self.BOX_COLOR_ID))
               self.win.vline(cellTextPos.y, cellTextPos.x-1, curses.ACS_VLINE, vLineLength)
               self.win.attroff(curses.color_pair(self.BOX_COLOR_ID))
            except:
               print("problem when cellTextPos is y=" + str(cellTextPos.y) +
                     ", x=" + str(cellTextPos.x-1) +
                     " and also the rows is " + str(r))
               raise

      self.win.refresh()
      # end of "drawTable"


class AGW_Table:
   def __init__(self):
      self.clearTable()
      self.currentFile = None
      self.currentFilename = None
      self.csvReader = None
      self.currentNumLinesLoaded = 0
      self.MAX_COL_WIDTH = 30  ## If a column is wider than this, then clip it to this width
      self.HEADER_NUM_DELIMITER_STRING = ": " # the separator between "ROW_1" and "here is the header".
      self.submitCommand = None
      self.submitStarted = None
      self.submitFinished = None
      self.submitInputFiles = None
      self.submitCompleted = None
      self.nInstances = None
      self.nCompleted = None
      self.__nCells = Size(0,0)
      self.__cells = []
      self.colWidth = []
      self.isRagged = False
      self.regTab = None


   def clearTable(self):
      self.__nCells = Size(0,0) # size in cells
      self.__cells = []
      self.colWidth = [] # maximum char length in a col
      self.isRagged = False # Does the table have "ragged" ends? (differing col counts).
                            # Ragged means "some rows have more cols than others"
      self.regTab = None # this will be a table of "None" (not yet checked) "True" or "False",
                         # depending on whether the cell currently matches the regex!

   def getColWidth(self, colIdx):
      ## Report out how wide each column is
      try: 
         if (self.colWidth[colIdx] > self.MAX_COL_WIDTH):
            return self.MAX_COL_WIDTH
         else:
            return self.colWidth[colIdx]
      except:
         print("### Someone passed in an invalid column index, " + str(colIdx) + ". Max was " + str(self.getNumCols()) + ".\n")
         raise #return 0 #raise #return 0

   def getNumCols(self):
      return self.__nCells.width  # table width in number of cells

   def getNumRows(self):
      return self.__nCells.height  # table height in number of cells

   def getHeaderCellForCol(self, colIdx):
      return self.cellValue(0, colIdx)
   def getHeaderCellForRow(self, rowIdx):
      return self.cellValue(rowIdx, 0)

   def cellValue(self, row, col):
      try:
         return self.__cells[row][col]
      except:
         return "~~~~~" #("R" + str(row) + ", C" + str(col) + " out of bounds")

   def initRegexTable(self):
      self.regTab = SearchBoard(nrow=self.getNumRows(), ncol=self.getNumCols())
      return

   def appendRowOfCellContents(self, contents, boolPrependRowCoordinate=False, boolPrependColCoordinate=False):
      ## Turns a line from a file into a properly-formatted internal
      ## representation of row of cells.

      self.__cells.append(contents)

      if (self.__nCells.width > 0 and self.__nCells.width != len(contents)):
         self.isRagged = True # Ragged table! (not a "true" table)

      numItemsInThisNewRow = len(contents)
      self.__nCells.width = max(self.__nCells.width, numItemsInThisNewRow) # Set the table width to that of the MAXIMUM number
                                                                           # of cols that a row has
      self.__nCells.height += 1 # Read another row...

      for c in range(0, self.__nCells.width):
         # Ok, find the **longest** item in each column. This could be slow for a tall column!
         cellWidth = None
         if (c >= numItemsInThisNewRow or (contents[c] is None)):
            cellWidth = 0
         else:
            cellWidth = len(contents[c])+2

         numColsWithWidth = len(self.colWidth)
         if (c == 0 and c < numColsWithWidth): # and self.hasRowHeader):
            # It's the ROW header (the leftmost column)! Gotta account for the ": " in the row header
            if boolPrependRowCoordinate:
               cellWidth += len(str(self.getNumCols())) + len(self.HEADER_NUM_DELIMITER_STRING)

         if (c >= numColsWithWidth):
            if boolPrependColCoordinate:
               cellWidth += len(str(c+1)) + len(self.HEADER_NUM_DELIMITER_STRING)  # Accounting for the col header!
            self.colWidth.append(cellWidth)
         else:
            self.colWidth[c] = max(self.colWidth[c], cellWidth)

   def loadNewFile(self, theFilename):
      self.closeCurrentFile() ## Close the old table first, if there is one

      self.clearTable()
      ## We want to see which file we should read from.
      ## 1. We either got a HYPHEN, which is the UNIX shorthand for "read from STDIN"
      ## 2. or we got a real filename, in which case we try to read the file from the filesystem
      self.currentFile = open(theFilename, 'rb')

      self.csvReader = csv.reader(self.currentFile)
      self.currentFilename = theFilename


   def closeCurrentFile(self):
      if (self.currentFile is not None):
         self.currentFile.close()

      self.currentFile = None
      self.currentFilename = None
      self.csvReader = None
      self.currentNumLinesLoaded = 0


   def readFromCurrentFile(self, nLinesToRead=10000):
      try:
         ## Read lines from a file into our data structure
         for row in self.csvReader:
            if len(row) == 1:
               if   row[0].split(':')[0] == '# command':
                  self.submitCommand = ':'.join(row[0].split(':')[1:]).strip()
               elif row[0].split(':')[0] == '# started':
                  submitStarted = row[0].split(':')[1:]
                  self.submitStarted = ':'.join(submitStarted).strip()
               elif row[0].split(':')[0] == '# finished':
                  submitFinished = row[0].split(':')[1:]
                  self.submitFinished = ':'.join(submitFinished).strip()
               elif row[0].split(':')[0] == '# input files':
                  self.submitInputFiles = row[0].split(':')[1].strip()
               elif row[0].split(':')[0] == '# completed':
                  self.submitCompleted = row[0].split(':')[1].strip()
                  jobsInfo = self.submitCompleted.split()[0]
                  jobsComplete,jobsTotal = jobsInfo.split('/')
                  self.nInstances = int(jobsTotal)
                  self.nCompleted = int(jobsComplete)
            else:
               self.appendRowOfCellContents( row )
               self.currentNumLinesLoaded += 1
               if self.currentNumLinesLoaded > nLinesToRead:
                  break
      except:
         print("[statussheet.py]: Error: Problem attempting to read from the file named " + self.currentFilename)
         print("[statussheet.py]:        We were able to read " + str(self.currentNumLinesLoaded) + " lines from it.")
         print("[statussheet.py]:        Verify that this file exists and is readable.")
         raise

      return # end of function


   def reloadFile(self, theFilename):
      previousCurrentNumLinesLoaded = self.currentNumLinesLoaded
      self.closeCurrentFile() ## Close the old table first, if there is one

      self.clearTable()

      self.currentFile = open(theFilename, 'rb')

      self.csvReader = csv.reader(self.currentFile)

      self.currentFilename = theFilename

      self.readFromCurrentFile(previousCurrentNumLinesLoaded)


class JobScanner:
   '''
Sheet.py

Found at http://timeforscience.googlecode.com/hg/Lab_Code/Python/Tools/sheet.py

A python program for viewing tab-delimited files in spreadsheet-like format. Only a viewer--you cannot edit files with it!



Uses the "CURSES" terminal interaction library to talk to the terminal.

by Alex Williams, 2009

Use it like this:   statussheet.py  yourFile.tab

Example usage: "statussheet.py ~/T" (tab-delimted file)

Note the line at the top of this file (the one that looks like "#!/somewhere/python")!
That points to the current Python installation. It must be a real python distribution
in order for you to be able to run statussheet.py!
(Otherwise try typing "which python" or "python statussheet.py")
   '''

   #http://www.amk.ca/python/howto/curses/curses.html
   # Semi-decent docs on how to use curses: http://www.amk.ca/python/howto/curses/

   def __init__(self,
                jobScanPath):
      self.ROW_HEADER_MAXIMUM_COLUMN_FRACTION_OF_SCREEN = 0.25 # The row header column (i.e., the leftmost column)
                                                               # cannot be any wider than this fraction of the total screen width.
                                                               # 1.0 means "do not change--it can be the entire screen,"
                                                               # 0.25 means "one quarter of the screen is the max, etc.
                                                               # 0.5 was the default before.
      self.KEY_MODE_NORMAL_INPUT     = 0
      self.KEY_MODE_SHOWRESULT_INPUT = 1
      self.KEY_MODE_SEARCH_INPUT     = 2
      self.KEY_MODE_QUESTION_INPUT   = 3

      self.STANDARD_BG_COLOR = curses.COLOR_BLACK

      # If a table ends with a "ragged end," and some rows aren't even the proper
      # length, then the straggling cells get this color.
      # You will see it a lot in ragged-end files, like list files.
      self.RAGGED_END_ID = 1
      self.RAGGED_END_TEXT_COLOR   = curses.COLOR_GREEN #BLUE #WHITE
      self.RAGGED_END_BG_COLOR     = self.STANDARD_BG_COLOR #BLUE

      self.SELECTED_CELL_ID = 2
      self.SELECTED_CELL_TEXT_COLOR = curses.COLOR_CYAN
      self.SELECTED_CELL_BG_COLOR = curses.COLOR_MAGENTA

      self.COL_HEADER_ID = 3
      self.COL_HEADER_TEXT_COLOR = curses.COLOR_YELLOW #BLACK
      self.COL_HEADER_BG_COLOR = self.STANDARD_BG_COLOR #curses.COLOR_BLUE #BLACK #YELLOW

      self.ROW_HEADER_ID = 4
      self.ROW_HEADER_TEXT_COLOR = curses.COLOR_GREEN #BLACK
      self.ROW_HEADER_BG_COLOR = self.STANDARD_BG_COLOR #curses.COLOR_BLUE #BLACK #GREEN

      self.BOX_COLOR_ID = 5 # The borders of the cells
      self.BOX_COLOR_TEXT_COLOR = curses.COLOR_YELLOW
      self.BOX_COLOR_BG_COLOR = self.STANDARD_BG_COLOR

      self.BLANK_COLOR_ID = 6
      self.BLANK_COLOR_TEXT_COLOR = curses.COLOR_CYAN
      self.BLANK_COLOR_BG_COLOR = self.STANDARD_BG_COLOR

      self.SEARCH_MATCH_COLOR_ID = 7 # Highlighted search results
      self.SEARCH_MATCH_COLOR_TEXT_COLOR = curses.COLOR_YELLOW
      self.SEARCH_MATCH_COLOR_BG_COLOR = curses.COLOR_RED

      self.WARNING_COLOR_ID = 8 # "Error message" color
      self.WARNING_COLOR_TEXT_COLOR = curses.COLOR_YELLOW
      self.WARNING_COLOR_BG_COLOR = curses.COLOR_RED

      self.NUMERIC_NEGATIVE_COLOR_ID = 9 # Negative numbers are this style
      self.NUMERIC_NEGATIVE_COLOR_TEXT_COLOR = curses.COLOR_RED
      self.NUMERIC_NEGATIVE_COLOR_BG_COLOR = self.STANDARD_BG_COLOR

      self.NUMERIC_POSITIVE_COLOR_ID = 10 # Positive numbers are this style
      self.NUMERIC_POSITIVE_COLOR_TEXT_COLOR = curses.COLOR_CYAN
      self.NUMERIC_POSITIVE_COLOR_BG_COLOR = self.STANDARD_BG_COLOR

      self.ACTIVE_FILENAME_COLOR_ID = 11
      self.ACTIVE_FILENAME_COLOR_TEXT_COLOR = curses.COLOR_YELLOW
      self.ACTIVE_FILENAME_COLOR_BG_COLOR = self.STANDARD_BG_COLOR

      self.HELP_AREA_ID = 12
      self.HELP_AREA_TEXT_COLOR = curses.COLOR_CYAN
      self.HELP_AREA_BG_COLOR   = self.STANDARD_BG_COLOR

      self.termSize = Size(None, None) ## Size of the actual xterm terminal window that statussheet.py lives in

      self.standardScreen = None  # The main screen

      self.mainInfo = AGW_File_Data_Collection() # An array of AGW_File_Datas

      self.sheetWin    = AGW_DataWin(isSearchable=True)
      self.infoWin     = AGW_Win()
      self.helpWin     = AGW_Win()
      self.questionWin = AGW_Win()
      self.resultWin   = AGW_Win()

      self.colHeaderWin = AGW_DataWin() ## "Window" for the column headers at the top of the screen
      self.rowHeaderWin = AGW_DataWin() ## "Window" for the row headers at the left side of the screen

      self.cellBorders = Size(1, 0) #Size(1,1) # Width, Height
      ## Size(1, 0) means there is a border to the left/right of each cell, but no border above/below

      self.currentMode = self.KEY_MODE_NORMAL_INPUT ## Start in normal input (not search) mode

      self.wantToQuit = False ## False, since, by default, the user does not want to immediately quit!
      self.firstTimeComplete = False

      #windowPos = Point(0,0) ## Initial position in the table of data: Point(0,0) is the top left

      self.truncationSuffix = "..." ## Suffix for "this cell is too long to fit on screen"

#     self.delimiter = "\t"
      self.delimiter = ","

      self.fastMoveSpeed = Point(10, 10) ## How many cells to scroll when the user is moving with shift-arrows

      self.warningMessage = None ## A message that we can display in the "warning" region in the info pane
      self.commandStr = None     ## The current command, which we sometimes display in the info pane
      self.questionStr = None    ## The current question, which we sometimes display in the info pane

      self.submitId = ""
      self.resultBasePaths = []
      self.resultPaths = ("","")
      self.resultOutputLines = None
      self.resultnOutputLines = 0
      self.resultTopLine = 0
      self.resultCurrentLine = 0

      ## Check to make sure the specified filenames are valid.
      ## We also allow a hyphen, which is UNIX shorthand for "read from STDIN."
      if (os.path.isfile(jobScanPath)):
         # If we got here, then it's a valid file or STDIN to read from...
         singleFileInfo = AGW_File_Data(jobScanPath)
         singleFileInfo.hasColHeader = True
         singleFileInfo.hasRowHeader = False
         self.mainInfo.addFileInfo(singleFileInfo)
      else:
         print("[statussheet.py]: Error in reading a file: Tried to read <" +
               jobScanPath +
               ">, but that file either did not exist, or could not be read!")

      self.mainInfo.currentFileIdx = 0

      self.cursesInitialized   = False
      self.cursesEcho          = True
      self.cursesCbreak        = False
      self.cursesKeypadSetting = False


   def stringFromAny(self, argString): # Returns a string, even if given None as an input
      if argString is None:
         return ""
      else:
         return str(argString)


   def cursesClearLine(self, window, lineYPos):
      window.move(lineYPos, 0)
      window.clrtoeol()


   def agwEnglishPlural(self, string, numOf, suffix="s"):
      '''You pass in a string like "squid" and a number
      indicating how many squid there are. If the number
      is one, then "squid" is returned, otherwise "squids"
      is returned.'''
      if (1 == numOf):
         return string
      else:
         return string + suffix

   def setWarning(self, string):
      self.warningMessage = string

   def setCommandStr(self, string):
      self.commandStr = string

   def clearCommandStr(self):
      self.commandStr = None

   def setQuestion(self, string, func):
      self.questionStr  = string
      self.questionFunc = func

   def clearQuestion(self):
      self.questionStr  = None
      self.questionFunc = None

   def setWantToQuit(self, wantToQuit):
      self.wantToQuit = wantToQuit

   def setShowResultInstance(self, showResultInstance):
      self.showResultInstance = showResultInstance


   def initializeWindowSettings(self, fileInfoToReadFrom):
      if (not isinstance(fileInfoToReadFrom, AGW_File_Data)):
         print("### Init window: Someone passed in a not-an-AGW_File_Data object to initializeWindowSettings\n")
         raise

      ## =========== Specify the dimensions of the windows here! ==================
#     INFO_PANEL_HEIGHT   = 5 ## How many vertical lines of terminal is the info panel in total?
      INFO_PANEL_HEIGHT   = 6 ## How many vertical lines of terminal is the info panel in total?
      COL_HEADER_HEIGHT   = 2 # How many vertical lines of terminal is the column header?
                              # Answer, it should be 2: one line for the header itself,
                              # and one below for the horizontal line that separates it from the main window (sheetWin).
      HELP_WIN_HEIGHT     = 2 ## How tall is the "help" window at the bottom?
      QUESTION_WIN_HEIGHT = 6 ## How tall is the "question" window inside the sheet window?
      ## =========== Specify the dimensions of the windows here! ==================

      self.termSize.resizeYX(self.standardScreen.getmaxyx()) # Resize the curses window to use the whole available screen

      self.sheetWin.setInfo(fileInfoToReadFrom)
      fileInfoToReadFrom.table.loadNewFile(fileInfoToReadFrom.filename)
      fileInfoToReadFrom.table.readFromCurrentFile()

      resultBasePath = os.path.dirname(fileInfoToReadFrom.filename)
      self.submitId = os.path.basename(resultBasePath)
      self.resultBasePaths.append(resultBasePath)
      self.resultBasePaths.append(os.path.join(resultBasePath,'InProcessResults'))

      if (fileInfoToReadFrom.hasColHeader and fileInfoToReadFrom.getNumRows() >= 2):
         fileInfoToReadFrom.cursorPos.y = 1 # start off with the column to the RIGHT of the col header selected,
                                            # instead of having the column header show up twice

      if (fileInfoToReadFrom.hasRowHeader and fileInfoToReadFrom.getNumCols() >= 2):
         fileInfoToReadFrom.cursorPos.x = 1 # start off with the first not-a-header row highlighted

      ## ===============================================
      ## Figure out how wide the row header should be.
      rowHeaderMaxWidth = 0
      if (self.sheetWin.getTable().getNumCols() > 0):
         # If there is at least one column, then the row header is as wide as the first column
         # (but note that the maximum size is adjusted below)
         if fileInfoToReadFrom.hasRowHeader:
            rowHeaderMaxWidth = self.sheetWin.getTable().getColWidth(0) + (self.cellBorders.width*2)

      # However: the maximum allowed rowHeaderWidth is some fraction of the total screen width
      if (rowHeaderMaxWidth > (self.termSize.width*self.ROW_HEADER_MAXIMUM_COLUMN_FRACTION_OF_SCREEN)):
         rowHeaderMaxWidth = int(self.termSize.width*self.ROW_HEADER_MAXIMUM_COLUMN_FRACTION_OF_SCREEN)

      ## Done figuring out how wide the row header should be
      ## ===============================================

      ## How tall is the main data window?
      SHEET_WIN_HEIGHT = (self.termSize.height - INFO_PANEL_HEIGHT - COL_HEADER_HEIGHT - HELP_WIN_HEIGHT)
      self.sheetWin.initWindow(SHEET_WIN_HEIGHT, ## height
                               (self.termSize.width - rowHeaderMaxWidth), ## width
                               (INFO_PANEL_HEIGHT + COL_HEADER_HEIGHT), ## location (y)
                               rowHeaderMaxWidth) ## location (x)

      self.infoWin.initWindow(INFO_PANEL_HEIGHT,  # height
                              self.termSize.width,    # width
                              0,      # location--y
                              0)      # location--x

      self.helpWin.initWindow(HELP_WIN_HEIGHT, ## height
                              self.termSize.width, ## width
                              (INFO_PANEL_HEIGHT + COL_HEADER_HEIGHT + SHEET_WIN_HEIGHT), ## location (y)
                              0) ## location (x)

      self.questionWin.initWindow(QUESTION_WIN_HEIGHT, ## height
                                  min(50,self.termSize.width), ## width
                                  int(INFO_PANEL_HEIGHT + COL_HEADER_HEIGHT + SHEET_WIN_HEIGHT/4.), ## location (y)
                                  0) ## location (x)

      # Goes along the left side!
      ROW_HEADER_HEIGHT = SHEET_WIN_HEIGHT
      ROW_HEADER_LOC_X  = 0
      ROW_HEADER_LOC_Y  = INFO_PANEL_HEIGHT + COL_HEADER_HEIGHT
      self.rowHeaderWin.initWindow(ROW_HEADER_HEIGHT,
                                   rowHeaderMaxWidth,
                                   ROW_HEADER_LOC_Y,
                                   ROW_HEADER_LOC_X)
      self.rowHeaderWin.setInfo(fileInfoToReadFrom)

      # Goes along the TOP
      COL_HEADER_WIDTH = self.termSize.width-rowHeaderMaxWidth
      COL_HEADER_LOC_X = rowHeaderMaxWidth
      COL_HEADER_LOC_Y = INFO_PANEL_HEIGHT
      self.colHeaderWin.initWindow(COL_HEADER_HEIGHT,
                                   COL_HEADER_WIDTH,
                                   COL_HEADER_LOC_Y,
                                   COL_HEADER_LOC_X)
      self.colHeaderWin.setInfo(fileInfoToReadFrom)

      RESULT_WIN_HEIGHT = (self.termSize.height - INFO_PANEL_HEIGHT - HELP_WIN_HEIGHT)
      self.resultWin.initWindow(RESULT_WIN_HEIGHT, ## height
                               self.termSize.width, ## width
                               INFO_PANEL_HEIGHT, ## location (y)
                               0) ## location (x)

      self.fastMoveSpeed.x = 5 #self.termSize.width // 2 # <-- measured in CELLS, not characters!
      self.fastMoveSpeed.y = self.sheetWin.windowHeight // 2 # measured in CELLS, not characters!


   def drawHelpWin(self):
      self.helpWin.win.erase()
      lineAttr = curses.color_pair(self.HELP_AREA_ID) # <-- set the border color
      self.helpWin.safeAddStr(0, 0
                              ,("Help: [Q]uit   [hjkl]: Move cursor (faster with [Shift])   [/]: Search ")
                              , lineAttr)

      self.helpWin.safeAddStr(1, 0
                              ,("      [^/$/g/G]: Go to left/right/top/bottom of table  [x]: Examine Result ")
                              , lineAttr)

      self.helpWin.win.refresh()    


   def drawQuestionWin(self):
      questionStr = self.stringFromAny(self.questionStr)
      if len(questionStr) > 0:
         self.questionWin.win.erase()
         actionStatement = "Press: Y / N"
         lenQuestion = len(questionStr)
         lenActionStatement = len(actionStatement)
         xQuestion = max(0,int((self.questionWin.windowWidth-4-lenQuestion)/2.))
         xActionStatement = max(0,int((lenQuestion-lenActionStatement)/2.))
         self.questionWin.safeAddStr(2,xQuestion,questionStr)
         self.questionWin.safeAddStr(3,xQuestion+xActionStatement,actionStatement)

         self.questionWin.win.border()    
         self.questionWin.win.refresh()    

   #
   #
   # File " " is 2 rows X 3 cols. (Ragged ends)
   # File list:
   # Row 1: <FDSFSDF>
   # Col 8: <LKJOIJWE>
   # Value: <Value>
   # > Command
   #
   def drawInfoWin(self, theInfo, inTab):
      '''inTab: the actual table of data that is going to be drawn'''

      activeCellPos = theInfo.cursorPos

      self.infoWin.win.erase()

      FILE_INFO_ROW = 0
#     ROW_HEADER_ROW = 1
#     COL_HEADER_ROW = 2
#     VALUE_ROW = 3
#     COMMAND_ROW = 4
      SUBMIT_COMMAND_ROW   = 1
      SUBMIT_STARTED_ROW   = 2
      SUBMIT_COMPLETED_ROW = 3
      COMMAND_ROW = 4
      WARNING_ROW = 5

      if inTab.isRagged:
         raggedText = " The table is ragged--some rows have differing numbers of columns."
      else:
         raggedText = ""

      filename = 'RUN      : ' + self.submitId

      fileStatusStr = filename + " has " + \
                      str(inTab.getNumRows()-1) + self.agwEnglishPlural(" job", inTab.getNumRows()) + " and " + \
                      str(inTab.getNumCols()-2) + self.agwEnglishPlural(" variable", inTab.getNumCols()-2) + "." + raggedText

      self.infoWin.safeAddStr(FILE_INFO_ROW, 0, fileStatusStr) ## self.infoWin is the topmost "window" pane

#     rowStr1 = "Row #" + str(activeCellPos.y) + ": " + self.stringFromAny(inTab.getHeaderCellForRow(activeCellPos.y))
#     self.cursesClearLine(self.infoWin.win, ROW_HEADER_ROW)
#     self.infoWin.safeAddStr(ROW_HEADER_ROW, 0, rowStr1, curses.color_pair(self.ROW_HEADER_ID))

#     colStr1 = "Col #" + str(activeCellPos.x+1) + ": " + self.stringFromAny(inTab.getHeaderCellForCol(activeCellPos.x))
#     self.cursesClearLine(self.infoWin.win, COL_HEADER_ROW)
#     self.infoWin.safeAddStr(COL_HEADER_ROW, 0, colStr1, curses.color_pair(self.COL_HEADER_ID))

#     cellText = self.stringFromAny(inTab.cellValue(activeCellPos.y, activeCellPos.x))
#     valueStr1 = "(R_" + str(activeCellPos.y) + ", C_" + str(activeCellPos.x+1) + ") = " + cellText
#     self.cursesClearLine(self.infoWin.win, VALUE_ROW)
#     self.infoWin.safeAddStr(VALUE_ROW, 0, valueStr1)

      if inTab.submitCommand:
         commandStr1 = 'RUNNING  : ' + self.stringFromAny(inTab.submitCommand)
         self.infoWin.safeAddStr(SUBMIT_COMMAND_ROW, 0, commandStr1, curses.A_NORMAL)

      if   inTab.submitFinished:
         commandStr1 = 'FINISHED : ' + self.stringFromAny(inTab.submitFinished)
         self.infoWin.safeAddStr(SUBMIT_STARTED_ROW, 0, commandStr1, curses.A_NORMAL)
      elif inTab.submitStarted:
         commandStr1 = 'STARTED  : ' + self.stringFromAny(inTab.submitStarted)
         self.infoWin.safeAddStr(SUBMIT_STARTED_ROW, 0, commandStr1, curses.A_NORMAL)

      if inTab.submitCompleted:
         commandStr1 = 'COMPLETED: ' + self.stringFromAny(inTab.submitCompleted) + " "
         self.infoWin.safeAddStr(SUBMIT_COMPLETED_ROW, 0, commandStr1, curses.A_NORMAL)
         jobsInfo = inTab.submitCompleted.split()[0]
         jobsComplete,jobsTotal = jobsInfo.split('/')
         jobsPercentComplete = int(float(jobsComplete)/float(jobsTotal)*100.)
         maximumGaugeWidth = self.infoWin.windowWidth - len(commandStr1)
         gaugeWidth = min(30,maximumGaugeWidth)
         if gaugeWidth > 13:
            column = len(commandStr1)
            self.infoWin.safeAddStr(SUBMIT_COMPLETED_ROW, column, str(jobsPercentComplete)+"%", curses.A_NORMAL)
            column += len(str(jobsPercentComplete))+1
            self.infoWin.safeAddStr(SUBMIT_COMPLETED_ROW, column, " [", curses.A_NORMAL)
            column += 2
            gaugeCompleteWidth = int(jobsPercentComplete/100.*gaugeWidth)
            gaugeIncompleteWidth = gaugeWidth-gaugeCompleteWidth
            gaugeComplete = " "*gaugeCompleteWidth
            gaugeIncomplete = "="*gaugeIncompleteWidth
            self.infoWin.safeAddStr(SUBMIT_COMPLETED_ROW, column, gaugeComplete, curses.A_REVERSE)
            column += gaugeCompleteWidth
            self.infoWin.safeAddStr(SUBMIT_COMPLETED_ROW, column, gaugeIncomplete, curses.A_NORMAL)
            column += gaugeIncompleteWidth
            self.infoWin.safeAddStr(SUBMIT_COMPLETED_ROW, column, "]", curses.A_NORMAL)
 
      commandStr1 = "" + self.stringFromAny(self.commandStr)
      self.infoWin.safeAddStr(COMMAND_ROW, 0, commandStr1, curses.A_NORMAL)

      if (self.warningMessage is not None):
         self.cursesClearLine(self.infoWin.win, WARNING_ROW)
         attr = curses.color_pair(self.WARNING_COLOR_ID)
         self.infoWin.safeAddStr(WARNING_ROW, 0, self.warningMessage, attr)

         self.setWarning(None) # And then clear the warning

      self.infoWin.win.refresh()

      return


   def drawSheetWin(self):
      activeFI = self.mainInfo.getCurrent()   # get the current meta-info storage thing for the main window
      activeCellPos = activeFI.cursorPos # Ask where the cursor is...
      theTable = self.sheetWin.getTable()

      # The main table...
      self.sheetWin.drawTable(activeFI, activeCellPos.y, activeCellPos.x, self.cellBorders)

      # Row header... (the leftmost, vertically-oriented pane along the left edge of the table)
      if activeFI.hasRowHeader:
         self.rowHeaderWin.drawTable(activeFI, activeCellPos.y, 0, self.cellBorders, nColsToDraw=1, boolPrependRowCoordinate=False)

      # Column header... (the column header along the top of the table)
      self.colHeaderWin.drawTable(activeFI, 0, activeCellPos.x, self.cellBorders, nRowsToDraw=1, boolPrependColCoordinate=False)

      lineAttr = curses.color_pair(self.BOX_COLOR_ID) # <-- set the border color
      self.colHeaderWin.win.attron(lineAttr) # "start drawing lines"
      start = activeCellPos.x
      end = theTable.getNumCols()
      hLineLength = self.cellBorders.width
      for c in range(start, end):
         hLineLength += theTable.getColWidth(c) + self.cellBorders.width
      hLineLength = max(0,min(hLineLength, self.colHeaderWin.windowWidth))
      self.colHeaderWin.win.hline(self.colHeaderWin.windowHeight-1, 0, curses.ACS_HLINE, hLineLength) # Draw a horizontal line
                                                                                                      # below the column header...
      self.colHeaderWin.win.attroff(lineAttr) # "stop drawing lines"
      self.colHeaderWin.win.refresh()


   def drawResult(self):
      self.resultWin.win.clear()

      nInstances = self.sheetWin.getTable().nInstances
      nInstanceIdDigits = max(2,int(math.log10(nInstances)+1))
      instanceId = str(int(self.showResultInstance)).zfill(nInstanceIdDigits)

      stdoutPath = None
      stdoutText = []
      stderrPath = None
      stderrText = []

      resultPaths = []
      for resultBasePath in self.resultBasePaths:
         if os.path.isdir(resultBasePath):
            resultPaths.append(resultBasePath)
         instancePath = os.path.join(resultBasePath,instanceId)
         if os.path.isdir(instancePath):
            resultPaths.append(instancePath)

      for resultPath in resultPaths:
         if not stdoutPath:
            reFiles = re.compile(self.submitId + "_" + instanceId + ".stdout")
            try:
               dirFiles = os.listdir(resultPath)
               matchingFiles = filter(reFiles.search,dirFiles)
               if len(matchingFiles) > 0:
                  for matchingFile in matchingFiles:
                     stdoutPath = os.path.join(resultPath,matchingFile)
                     fpResult = open(stdoutPath,'r')
                     if fpResult:
                        stdoutText += [x.strip() for x in fpResult.readlines()]
                        fpResult.close()
            except:
               stdoutPath = None

         if not stderrPath:
            reFiles = re.compile(self.submitId + "_" + instanceId + ".stderr")
            try:
               dirFiles = os.listdir(resultPath)
               matchingFiles = filter(reFiles.search,dirFiles)
               if len(matchingFiles) > 0:
                  for matchingFile in matchingFiles:
                     stderrPath = os.path.join(resultPath,matchingFile)
                     fpResult = open(stderrPath,'r')
                     if fpResult:
                        stderrText += [x.strip() for x in fpResult.readlines()]
                        fpResult.close()
            except:
               stderrPath = None

      resultPaths = (stdoutPath,stderrPath)
      if resultPaths != self.resultPaths or resultPaths == (None,None):
         self.resultPaths = resultPaths
         del self.resultOutputLines
         self.resultOutputLines = stdoutText + stderrText
         if len(self.resultOutputLines) == 0:
            self.resultOutputLines.append("Results are not available for instance %s" % (instanceId))
         self.resultnOutputLines = len(self.resultOutputLines)
         self.resultTopLine = 0
         self.resultCurrentLine = 0
      del stdoutText
      del stderrText

      if self.resultnOutputLines > 0:
         top = self.resultTopLine
         bottom = top+self.resultWin.windowHeight-1
         resultRow = 0
         for line in self.resultOutputLines[top:bottom]:
            if resultRow != self.resultCurrentLine:
               self.resultWin.safeAddStr(resultRow,0,line)
            else:
               self.resultWin.safeAddStr(resultRow,0,line,curses.A_BOLD)
            resultRow += 1
      self.resultWin.win.refresh()


   def drawEverything(self):
      activeFI = self.mainInfo.getCurrent()   # get the current meta-info storage thing for the main window

      self.standardScreen.refresh() # Refresh it first...

      # The "info" pane at the top...
      theTable = self.sheetWin.getTable()
      self.drawInfoWin(activeFI, theTable)

      # The "help" window at the bottom...
      self.drawHelpWin()

      # The main table...
      if   self.currentMode == self.KEY_MODE_NORMAL_INPUT:
         self.drawSheetWin()
         if not self.firstTimeComplete:
            if theTable.submitFinished:
               self.setQuestion("All jobs have completed. Quit?",self.setWantToQuit)
               self.setUserInteractionMode(self.KEY_MODE_QUESTION_INPUT)
               self.firstTimeComplete = True
      elif self.currentMode == self.KEY_MODE_SHOWRESULT_INPUT:
         self.drawResult()

      self.drawQuestionWin()


   def setUpCurses(self): # initialize the curses environment
      if (not curses.has_colors()):
         print("UH OH, this terminal does not support color! We might crash.",
               " Quitting now anyway until I figure out what to do. Sorry.",
               " This might not actually be a problem, but I will need to test",
               " it to see what happens in a non-color terminal!")
#        sys.exit(1)
      else:
         curses.start_color()

         curses.init_pair(self.RAGGED_END_ID,self.RAGGED_END_TEXT_COLOR,
                                             self.RAGGED_END_BG_COLOR)
         curses.init_pair(self.SELECTED_CELL_ID,self.SELECTED_CELL_TEXT_COLOR,
                                                self.SELECTED_CELL_BG_COLOR)
         curses.init_pair(self.COL_HEADER_ID,self.COL_HEADER_TEXT_COLOR,
                                             self.COL_HEADER_BG_COLOR)
         curses.init_pair(self.ROW_HEADER_ID,self.ROW_HEADER_TEXT_COLOR,
                                             self.ROW_HEADER_BG_COLOR)
         curses.init_pair(self.BOX_COLOR_ID,self.BOX_COLOR_TEXT_COLOR,
                                            self.BOX_COLOR_BG_COLOR)
         curses.init_pair(self.BLANK_COLOR_ID,self.BLANK_COLOR_TEXT_COLOR,
                                              self.BLANK_COLOR_BG_COLOR)
         curses.init_pair(self.SEARCH_MATCH_COLOR_ID,self.SEARCH_MATCH_COLOR_TEXT_COLOR,
                                                     self.SEARCH_MATCH_COLOR_BG_COLOR)
         curses.init_pair(self.WARNING_COLOR_ID,self.WARNING_COLOR_TEXT_COLOR,
                                                self.WARNING_COLOR_BG_COLOR)
         curses.init_pair(self.NUMERIC_NEGATIVE_COLOR_ID,self.NUMERIC_NEGATIVE_COLOR_TEXT_COLOR,
                                                         self.NUMERIC_NEGATIVE_COLOR_BG_COLOR)
         curses.init_pair(self.NUMERIC_POSITIVE_COLOR_ID,self.NUMERIC_POSITIVE_COLOR_TEXT_COLOR,
                                                         self.NUMERIC_POSITIVE_COLOR_BG_COLOR)
         curses.init_pair(self.ACTIVE_FILENAME_COLOR_ID,self.ACTIVE_FILENAME_COLOR_TEXT_COLOR,
                                                        self.ACTIVE_FILENAME_COLOR_BG_COLOR)
         curses.init_pair(self.HELP_AREA_ID,self.HELP_AREA_TEXT_COLOR,
                                            self.HELP_AREA_BG_COLOR)

      CURSES_INVISIBLE_CURSOR = 0
      CURSES_VISIBLE_CURSOR = 1
      CURSES_HIGHLIGHTED_CURSOR = 2
      try:
         curses.curs_set(CURSES_INVISIBLE_CURSOR) # Don't show a blinking cursor
      except(curses.error):
         print("Unable to set cursor state to \"invisible\"")

      curses.meta(1)  # Allow 8-bit chars


   def handleKeysForSearchMode(self, argCh, currentTable):
      ## If we are in search mode, then when the user types, that text is added to the query.

      KEYS_SEARCH_MODE_FINISHED = (curses.KEY_ENTER, curses.ascii.LF, curses.ascii.CR )
      KEYS_SEARCH_MODE_CANCEL = (curses.ascii.ESC, curses.ascii.ctrl(ord('g')), curses.KEY_CANCEL)
      KEYS_SEARCH_MODE_BACKSPACE = (curses.KEY_BACKSPACE, curses.ascii.BS)
      KEYS_SEARCH_MODE_DELETE_FORWARD = (curses.KEY_DC, curses.ascii.DEL)

      finishedSearch = False
      cancelSearch = False

      if (argCh in KEYS_SEARCH_MODE_FINISHED): ## User wants to STOP entering search text
         # ----------------------------------
         if (len(self.mainInfo.getCurrent().getRegexString()) > 0):
            finishedSearch = True # If there *is* a search string
         else:
            cancelSearch = True # Search string is blank--user cancelled the search
         # ----------------------------------
      elif (argCh in KEYS_SEARCH_MODE_CANCEL): ## User wants to CANCEL searching, deleting any query that was there
         cancelSearch = True
         # ----------------------------------
      elif argCh in KEYS_SEARCH_MODE_BACKSPACE or argCh in KEYS_SEARCH_MODE_DELETE_FORWARD:
         # we don't support forward-delete yet. sorry.
         # Pretend it's regular delete for now...
         if (self.mainInfo.getCurrent().regexIsActive() is False):
            # The user deleted PAST the beginning of the string
            cancelSearch = True
         else:
            self.mainInfo.getCurrent().trimRegex(numChars=1)
            currentTable.regTab.clear()
         # ----------------------------------
      else:
         # Append whatever the user typed to the search string...
         try:
            charToAdd = chr(argCh)
            self.mainInfo.getCurrent().appendToCurrentSearchTerm(charToAdd)
            currentTable.regTab.clear()
         except(ValueError):
            # we don't care if "charToAdd" is not in the range of add-able chars
            pass
            # ----------------------------------

      if (cancelSearch):
         self.setCommandStr("Search Cancelled")
         self.mainInfo.getCurrent().clearCurrentSearchTerm()
         currentTable.regTab.clear()
         self.exitSearchMode()
      elif (finishedSearch):
         self.setCommandStr("Searching for \"" + self.mainInfo.getCurrent().getRegexString() + '"')
         self.exitSearchMode()
      else:
         self.setCommandStr("Search (press enter when done): " + self.mainInfo.getCurrent().getRegexString())


   def handleKeysForQuestionMode(self, argCh, currentTable):
      ## If we are in question mode, then when the user types, the question is answered.

      KEYS_QUESTION_MODE_YES = (ord('y'), ord('Y') ) 
      KEYS_QUESTION_MODE_NO  = (ord('n'), ord('N') ) 

      if   argCh in KEYS_QUESTION_MODE_YES:
         self.questionFunc(True)
         self.clearQuestion()
         self.setUserInteractionMode(self.KEY_MODE_NORMAL_INPUT)
      elif argCh in KEYS_QUESTION_MODE_NO:
         self.questionFunc(False)
         self.clearQuestion()
         self.setUserInteractionMode(self.KEY_MODE_NORMAL_INPUT)


   def handleKeysForNormalMode(self, argCh, currentTable):
      ## Handle user keyboard input when we are *not* in search mode.
      ## This will handle the majority of user interactions.

      KEYS_QUIT = (ord('q'), ord('Q'), curses.ascii.ESC)
      KEYS_TOGGLE_HIGHLIGHT_NUMBERS_MODE = ()
      KEYS_MOVE_TO_TOP_IN_LESS = ord('g')
      KEYS_MOVE_TO_BOTTOM_IN_LESS = ord('G')
      KEYS_MOVE_TO_TOP  = (curses.KEY_HOME, KEYS_MOVE_TO_TOP_IN_LESS)
      KEYS_MOVE_TO_BOTTOM = (curses.KEY_END, KEYS_MOVE_TO_BOTTOM_IN_LESS)
      KEYS_MOVE_RIGHT = (curses.KEY_RIGHT, ord('l'))
      KEYS_MOVE_LEFT = (curses.KEY_LEFT, ord('h'))
      KEYS_MOVE_UP   = (curses.KEY_UP, ord('k'))
      KEYS_MOVE_DOWN = (curses.KEY_DOWN, ord('j'))
      KEYS_MOVE_RIGHT_FAST = (ord('L'),)
      KEYS_MOVE_LEFT_FAST  = (ord('H'),)
      FAST_UP_KEY_IN_LESS   = ord('b')
      FAST_DOWN_KEY_IN_LESS = curses.ascii.SP # space
      FAST_DOWN_KEY_IN_EMACS = curses.ascii.ctrl(ord('v'))
      FAST_UP_KEY_IN_EMACS   = curses.ascii.alt(ord('v'))
      FAST_DOWN_KEY_IN_VI = curses.ascii.ctrl(ord('f'))
      FAST_UP_KEY_IN_VI   = curses.ascii.ctrl(ord('b'))
      KEYS_MOVE_UP_FAST    = (ord('K'), curses.KEY_PPAGE, FAST_UP_KEY_IN_LESS, FAST_UP_KEY_IN_VI)
      KEYS_MOVE_DOWN_FAST  = (ord('J'), curses.KEY_NPAGE, FAST_DOWN_KEY_IN_LESS, FAST_DOWN_KEY_IN_VI)
      KEYS_PREVIOUS_FILE = ()
      KEYS_NEXT_FILE     = ()
      KEYS_RELOAD_FILE   = (-1,)
      KEYS_GOTO_NEXT_MATCH = ()
      KEYS_GOTO_PREVIOUS_MATCH = ()
      GOTO_LINE_START_KEY_IN_EMACS = curses.ascii.ctrl(ord('a'))
      GOTO_LINE_END_KEY_IN_EMACS   = curses.ascii.ctrl(ord('e'))
      GOTO_LINE_START_KEY_IN_VI = ord('^')
      GOTO_LINE_END_KEY_IN_VI   = ord('$')
      KEYS_GOTO_LINE_END   = (GOTO_LINE_END_KEY_IN_VI,)
      KEYS_GOTO_LINE_START = (GOTO_LINE_START_KEY_IN_VI,)
      KEYS_WANT_TO_ENTER_SEARCH_MODE = (ord('/'),)
      KEYS_WANT_TO_ENTER_SHOWRESULT_MODE = (ord('x'),)

      kWANT_TO_ADJUST_CURSOR         = 0
      kWANT_TO_DIRECTLY_MOVE_CURSOR  = 1
      kWANT_TO_MOVE_TO_SEARCH_RESULT = 2
      kWANT_TO_CHANGE_FILE           = 3

      theAction = None  ## What did the user want to do?

      wantToMove = Point(0,0)    ## See if we need to move the cursor / scroll up / down / left /right
      wantToChangeFileIdx = None ## Do we want to go to the next/previous file?

      activeCellPos = self.mainInfo.getCurrent().cursorPos ## Where are we currently, in the table?

      #if (argCh is not None and argCh is not ''): raise "ArgCh: " + str(argCh)

      if   argCh in KEYS_QUIT:
         if self.sheetWin.getTable().submitFinished:
            self.setQuestion("Really quit?",self.setWantToQuit)
         else:
            self.setQuestion("Abort all remaining jobs and exit?",self.setWantToQuit)
         self.setUserInteractionMode(self.KEY_MODE_QUESTION_INPUT)
      elif argCh in KEYS_TOGGLE_HIGHLIGHT_NUMBERS_MODE:
         self.mainInfo.getCurrent().toggleNumericHighlighting()
      elif argCh in KEYS_MOVE_TO_TOP:
         activeCellPos.y = 1
      elif argCh in KEYS_MOVE_TO_BOTTOM:
         activeCellPos.y = (currentTable.getNumRows()-1)
      elif argCh in KEYS_MOVE_RIGHT:
         wantToMove.x = 1
      elif argCh in KEYS_MOVE_LEFT:
         wantToMove.x = -1
      elif argCh in KEYS_MOVE_UP:
         wantToMove.y = -1
      elif argCh in KEYS_MOVE_DOWN:
         wantToMove.y = 1
      elif argCh in KEYS_MOVE_RIGHT_FAST:
         wantToMove.x = self.fastMoveSpeed.x
      elif argCh in KEYS_MOVE_LEFT_FAST:
         wantToMove.x = -self.fastMoveSpeed.x
      elif argCh in KEYS_MOVE_UP_FAST:
         wantToMove.y = -self.fastMoveSpeed.y
      elif argCh in KEYS_MOVE_DOWN_FAST:
         wantToMove.y = self.fastMoveSpeed.y
      elif argCh in KEYS_PREVIOUS_FILE:
         wantToChangeFileIdx = -1
      elif argCh in KEYS_NEXT_FILE:
         wantToChangeFileIdx = 1
      elif argCh in KEYS_RELOAD_FILE:
         wantToChangeFileIdx = 0
      elif argCh in KEYS_GOTO_NEXT_MATCH:
         theAction = kWANT_TO_MOVE_TO_SEARCH_RESULT
         theActionParam = -1
      elif argCh in KEYS_GOTO_PREVIOUS_MATCH:
         theAction = kWANT_TO_MOVE_TO_SEARCH_RESULT
         theActionParam = 1
      elif argCh in KEYS_GOTO_LINE_END:
         activeCellPos.x = (currentTable.getNumCols()-1)
      elif argCh in KEYS_GOTO_LINE_START:
         activeCellPos.x = 0
      elif argCh in KEYS_WANT_TO_ENTER_SEARCH_MODE:
         self.setUserInteractionMode(self.KEY_MODE_SEARCH_INPUT)
         self.mainInfo.getCurrent().clearCurrentSearchTerm()
         currentTable.initRegexTable()
         self.setCommandStr("Search (press enter when done): ")
      elif argCh in KEYS_WANT_TO_ENTER_SHOWRESULT_MODE:
         resultRow = activeCellPos.y
         resultColumn = 0
         instanceId = currentTable.cellValue(resultRow,resultColumn)
         self.setShowResultInstance(instanceId)
         self.setUserInteractionMode(self.KEY_MODE_SHOWRESULT_INPUT)
      else:
         pass # unrecognized key


      if (theAction is None):
         pass
      elif (theAction == kWANT_TO_MOVE_TO_SEARCH_RESULT):
         self.setCommandStr("Sorry! Not implemented yet.")

      if (wantToMove.x != 0 or wantToMove.y != 0):
         activeCellPos.y = max(1, min(currentTable.getNumRows()-1, (activeCellPos.y+wantToMove.y)))
         activeCellPos.x = max(0, min(currentTable.getNumCols()-1 , (activeCellPos.x+wantToMove.x)))
         self.clearCommandStr() # whatever the previous command was, it no longer applies now that we have moved

      if (wantToChangeFileIdx is not None):
         possibleNewIndex = (self.mainInfo.currentFileIdx + wantToChangeFileIdx)
         if   (possibleNewIndex < 0):
            self.setWarning(">>> Cannot go to the previous file, because we are already at the beginning of the file list.")
         elif (possibleNewIndex >= self.mainInfo.size()):
            self.setWarning(">>> Cannot go to the next file, because we are already at the end of the file list.")
         else:
            if wantToChangeFileIdx != 0:
               self.resultWin.win.clear()
               self.sheetWin.win.clear()
               self.infoWin.win.clear()
               self.colHeaderWin.win.clear()
               self.colHeaderWin.win.refresh()
               self.rowHeaderWin.win.clear()
               self.mainInfo.currentFileIdx = possibleNewIndex # update which file we are currently examining
               self.initializeWindowSettings(self.mainInfo.getCurrent())
               self.setCommandStr("Changed to the file named \"" + self.mainInfo.getCurrent().filename + '".')
            else:
               self.mainInfo.getCurrent().table.reloadFile(self.mainInfo.getCurrent().filename)


   def handleKeysForShowResultMode(self, argCh, currentTable):
      ## If we are in show result mode, then when the user types scroll up and down.
      KEYS_QUIT = (ord('q'), ord('Q'), curses.ascii.ESC)
      KEYS_MOVE_UP   = (curses.KEY_UP, ord('k'))
      KEYS_MOVE_DOWN = (curses.KEY_DOWN, ord('j'))

      nScrollRows = 0
      if   argCh in KEYS_QUIT:
         self.setUserInteractionMode(self.KEY_MODE_NORMAL_INPUT)
      elif argCh in KEYS_MOVE_UP:
         nScrollRows = -1
      elif argCh in KEYS_MOVE_DOWN:
         nScrollRows = 1

      nRows = self.resultWin.windowHeight-1
      if nScrollRows != 0:
         nextLine = self.resultCurrentLine + nScrollRows
         if   nScrollRows < 0 and \
              self.resultCurrentLine == 0 and \
              self.resultTopLine != 0:
            self.resultTopLine += nScrollRows
         elif nScrollRows > 0 and \
              nextLine == nRows and \
              self.resultTopLine+nRows != self.resultnOutputLines:
            self.resultTopLine += nScrollRows
         else:
            if   nScrollRows < 0 and \
                 (self.resultTopLine != 0 or self.resultCurrentLine != 0):
               self.resultCurrentLine = nextLine
            elif nScrollRows > 0 and \
                 (self.resultTopLine+self.resultCurrentLine+1) != self.resultnOutputLines and \
                 self.resultCurrentLine != nRows:
               self.resultCurrentLine = nextLine


   def exitSearchMode(self):
      if (self.currentMode != self.KEY_MODE_SEARCH_INPUT):
         raise "Uh oh, tried to exit search mode... but we were not even IN search mode!!"
      else:
         self.setUserInteractionMode(self.KEY_MODE_NORMAL_INPUT)

   def setUserInteractionMode(self, argNewMode):
      self.currentMode = argNewMode

   def start(self):
      started = False
      try:
         # Initialize curses
         self.standardScreen = curses.initscr()
         self.cursesInitialized = True

         # Turn off echoing of keys, and enter cbreak mode,
         # where no buffering is performed on keyboard input
         curses.noecho()
         self.cursesEcho = False
         curses.cbreak()
         self.cursesCbreak = True

         # In keypad mode, escape sequences for special keys
         # (like the cursor keys) will be interpreted and
         # a special value like curses.KEY_LEFT will be returned
         self.standardScreen.keypad(1)
         self.cursesKeypadSetting = True

         # Start color, too.  Harmless if the terminal doesn't have
         # color; user can test with has_color() later on.  The try/catch
         # works around a minor bit of over-conscientiousness in the curses
         # module -- the error return from C start_color() is ignorable.
         try:
            curses.start_color()
         except:
            pass

         self.setUpCurses()
         self.initializeWindowSettings(self.mainInfo.getCurrent())
         self.drawEverything()
         started = True
      except curses.error, err:
         print err
      except:
         pass

      return(started)


   def processInput(self):
      self.standardScreen.nodelay(True) # <-- makes "getch" non-blocking
      theTable = self.sheetWin.getTable()
      ch = None
      eoi = False
      while (not self.wantToQuit) and (not eoi):
         try:
            self.drawEverything() # Draw the screen...

            ch = self.standardScreen.getch()    # Now get input from the user...

            if ch >= 0:
               if   self.currentMode == self.KEY_MODE_NORMAL_INPUT:
                  self.handleKeysForNormalMode(ch,theTable)
               elif self.currentMode == self.KEY_MODE_SHOWRESULT_INPUT:
                  self.handleKeysForShowResultMode(ch,theTable)
               elif self.currentMode == self.KEY_MODE_SEARCH_INPUT:
                  self.handleKeysForSearchMode(ch,theTable)
               elif self.currentMode == self.KEY_MODE_QUESTION_INPUT:
                  self.handleKeysForQuestionMode(ch,theTable)
            else:
               if self.currentMode == self.KEY_MODE_NORMAL_INPUT:
                  self.handleKeysForNormalMode(ch,theTable)
               eoi = True

         except KeyboardInterrupt:
            break # Exit the program on a Ctrl-C as well.
         except:
            raise # Something unexpected has happened. Better report it!

      jobsFinished = theTable.submitFinished != None

      return(self.wantToQuit,jobsFinished)


   def finish(self):
      # Set everything back to normal
      try:
         if self.standardScreen:
            if self.cursesKeypadSetting:
               self.standardScreen.keypad(0)
               self.cursesKeypadSetting = False
         if not self.cursesEcho:
            curses.echo()
            self.cursesEcho = True
         if self.cursesCbreak:
            curses.nocbreak()
            self.cursesCbreak = False
         if self.cursesInitialized:
            curses.endwin()
            self.cursesInitialized = False
      except:
         pass

      self.standardScreen = None


