# @package      hubzero-python
# @file         user.py
# @copyright    Copyright (c) 2012-2020 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2012-2020 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#

import datetime
import hubzero.config.passwords
import hubzero.utilities.group
import hubzero.utilities.misc
import hubzero.config.ldapconfig
import hubzero.config.webconfig
import hubzero.data.db
import ldap
import os
import sys
import traceback

pwhashtypes = ["{CRYPT}", "{MD5}", "{SSHA}", "{SHA}"]


def hashPassword(pw, hashtype='MD5'):
    """
    We allow users to pass in pre hashed passwords or plaintext. If * is passed in,
    that means a disabled password. No MD5 hashed password will hash to *

    This function analyzes a password and returns the hashed value

Note, only md5 supported for now from here
    """
    if any(pw.upper().startswith(x) for x in pwhashtypes):
        rv = pw
    elif pw == '*':
        rv = '{MD5}*'
    else:
        if hashtype == 'MD5':
            rv = hubzero.config.passwords.MD5PasswordHash(pw)
        else:
            raise Exception("hashPassword hashtype not supported:" + hashtype)

    return rv


def userExists(username, hubname=""):
    dbHost = hubzero.config.webconfig.getWebConfigDBHost(hubname)
    dbPW = hubzero.config.webconfig.getWebConfigDBPassword(hubname)
    dbUserName = hubzero.config.webconfig.getWebConfigDBUsername(hubname)
    dbName = hubzero.config.webconfig.getWebConfigDBName(hubname)

    db = hubzero.data.db.MySQLConnection(dbHost, dbName, dbUserName, dbPW)
    if _userExists(db, username):
        return True
    else:
        return False


def _userExists(db, username):
    sql = 'select id from `jos_users` where username = %s'
    data = [username]
    userID = db.query_selectscalar(sql, data)

    return userID


def userLookup(username, hubname=""):
    dbHost = hubzero.config.webconfig.getWebConfigDBHost(hubname)
    dbPW = hubzero.config.webconfig.getWebConfigDBPassword(hubname)
    dbUserName = hubzero.config.webconfig.getWebConfigDBUsername(hubname)
    dbName = hubzero.config.webconfig.getWebConfigDBName(hubname)

    db = hubzero.data.db.MySQLConnection(dbHost, dbName, dbUserName, dbPW)
    rv = _userExists(db, username)
    return rv


def _getUserID(db, username):
    sql = 'select id from `jos_users` where username = %s'
    data = [username]

    userID = db.query_selectscalar(sql, data)

    if userID is None:
        return None
    else:
        return userID


def _del_jos_users(db, username):
    sql = 'select id from `jos_users` where username = %s'
    data = [username]
    userID = db.query_selectscalar(sql, data)

    sql = 'delete from `jos_users` where id = %s'
    data = [userID]
    userID = db.query_selectscalar(sql, data)

    return userID


def _del_jos_user_usergroup_map(db, userID):

    sql = "DELETE FROM `jos_user_usergroup_map` WHERE `user_id` = %s"
    data = [userID]
    db.query_rowcount(sql, data)


def _del_jos_core_acl_aro(db, userID):

    sql = 'select id from `jos_core_acl_aro` where `value` = %s'
    data = [userID]
    aroID = db.query_selectscalar(sql, data)

    sql = "DELETE FROM `jos_core_acl_aro` where `value` = %s"
    data = [userID]
    db.query_rowcount(sql, data)

    return aroID


def _del_jos_core_acl_groups_aro_map(db, aroID):
    sql = "DELETE FROM `jos_core_acl_groups_aro_map` WHERE `aro_id` = %s"
    data = [aroID]
    db.query_rowcount(sql, data)


def _del_jos_xprofiles(db, uidNumber):
    sql = "DELETE FROM `jos_xprofiles` WHERE `uidNumber` = %s"
    data = [uidNumber]
    db.query_rowcount(sql, data)


def _update_jos_xprofiles_pw(db, userid, password):

    pw = hashPassword(password)
    sql = "UPDATE `jos_xprofiles` SET `userPassword`=%s WHERE `uidNumber`=%s AND `userPassword`<>%s"
    data = [pw, userid, pw]

    if db.query_rowcount(sql, data):
        return True
    
    return False

def _update_jos_users_pw(db, userid, password):

    pw = hashPassword(password)
    sql = "UPDATE `jos_users` SET `password` = %s WHERE `id`=%s AND `password`<>%s"
    data = [pw, userid, pw]

    if db.query_rowcount(sql, data):
        return True

    return False


def _update_jos_users_password(db, userid, password):

    pw = hashPassword(password)
    sql = "UPDATE `jos_users_password` SET `passhash` = %s WHERE `user_id`=%s AND `passhash`<>%s"
    data = [pw, userid, pw]

    if db.query_rowcount(sql, data):
        return True

    return False


# TODO jos_password_history??


def _update_ldap_user_pw(username, uidNumber, password):
    """
    Pass raw pw here, we'll hash it
    """
    hubLDAPAcctMgrDN = hubzero.config.webconfig.getComponentParam( "com_system", "ldap_managerdn")
    hubLDAPAcctMgrPW = hubzero.config.webconfig.getComponentParam( "com_system", "ldap_managerpw")
    hubLDAPBaseDN = hubzero.config.webconfig.getComponentParam( "com_system", "ldap_basedn")

    ldapServer = hubzero.config.webconfig.getComponentParam( "com_system", "ldap_primary")
    l = ldap.initialize(ldapServer)
    l.simple_bind_s(hubLDAPAcctMgrDN, hubLDAPAcctMgrPW)

    pw = hashPassword(password)

    dn = "uid=" + username + ",ou=users," + hubLDAPBaseDN

    attrs = [(ldap.MOD_REPLACE, 'userPassword', pw)]

    result = l.compare_s(dn, 'userPassword', pw)

    if not result:
        l.modify_s(dn, attrs)
        return 1

    return 0

def _insert_jos_users(db, joomlauserid, name, username, email, password, jgidNumber):

    pw = hashPassword(password)

    # xlate numeric jgidNumber to the string (Joomla needs both for some reason)
    # 18=Registered, 23=Manager, 24=Administrator, 25=Super Administrator
    userTypeLookupDict = {18: "Registered", 23: "Manager",
                          24: "Administrator", 25: "Super Administrator"}

    fullname = name
    fns = fullname.split(' ')

    firstname = ""
    middlename = ""
    lastname = ""

    if len(fns) == 1:
        firstname = fns[0]
    elif len(fns) == 2:
        firstname = fns[0]
        lastname = fns[1]
    elif len(fns) == 3:
        firstname = fns[0]
        middlename = fns[1]
        lastname = fns[2]

    if hubzero.utilities.misc.JOOMLA_25:
        if joomlauserid == -1:
            try:
                sql = "INSERT INTO `jos_users` (`name`,`givenName`,`middleName`,`surname`,`username`,`email`,`password`,`usertype`                           ,`block`,`approved`,`sendEmail`,`registerDate`             ,`registerIP`,`lastvisitDate`         ,`activation`,`params`,`lastResetTime`        ,`resetCount`,`access`,`usageAgreement`,`homeDirectory`,`loginShell`,`ftpShell`)"
                sql += "VALUES (" + "%s,"*22 + '%s' ")"
                data = [name, firstname, middlename, lastname, username, email, pw, userTypeLookupDict[int(
                    jgidNumber)], 0, 1, 0, datetime.datetime.now(), '127.0.0.1', '0000-00-00 00:00:00', 1, '', '0000-00-00 00:00:00', 0, 1, 1, '', '', '']
                userID = db.query_lastrowid(sql, data)
            except:
                sql = "INSERT INTO `jos_users` (`name`, `username`, `email`, `password`, `usertype`, `block`, `sendEmail`, `registerDate`, `lastvisitDate`, `activation`, `params`)"
                sql += "VALUES (%s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s);"
                data = [name, username, email, pw, userTypeLookupDict[int(
                    jgidNumber)], 0, 1, datetime.datetime.now(), '0000-00-00 00:00:00', '', '']
                userID = db.query_lastrowid(sql, data)
        else:
            try:
                sql = "INSERT INTO `jos_users` (`id`, `name`,`givenName`,`middleName`,`surname`,`username`,`email`,`password`,`usertype`                           ,`block`,`approved`,`sendEmail`,`registerDate`             ,`registerIP`,`lastvisitDate`         ,`activation`,`params`,`lastResetTime`        ,`resetCount`,`access`,`usageAgreement`,`homeDirectory`,`loginShell`,`ftpShell`)"
                sql += "VALUES (" + "%s,"*23 + '%s' ")"
                data = [joomlauserid, name, firstname, middlename, lastname, username, email, pw, userTypeLookupDict[int(
                    jgidNumber)], 0, 1, 0, datetime.datetime.now(), '127.0.0.1', '0000-00-00 00:00:00', 1, '', '0000-00-00 00:00:00', 0, 1, 1, '', '', '']
                userID = db.query_lastrowid(sql, data)
            except:
                sql = "INSERT INTO `jos_users` (`id`, `name`, `username`, `email`, `password`, `usertype`, `block`, `sendEmail`, `registerDate`, `lastvisitDate`, `activation`, `params`)"
                sql += "VALUES (%s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s);"
                data = [joomlauserid, name, username, email, pw, userTypeLookupDict[int(
                    jgidNumber)], 0, 1, datetime.datetime.now(), '0000-00-00 00:00:00', '', '']
                userID = db.query_lastrowid(sql, data)

    else:
        if joomlauserid == -1:
            sql = "INSERT INTO `jos_users` (`name`, `username`, `email`, `password`, `usertype`, `block`, `sendEmail`, `gid`, `registerDate`, `lastvisitDate`, `activation`, `params`)"
            sql += "VALUES (%s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s);"
            data = [name, username, email, pw, userTypeLookupDict[int(
                jgidNumber)], 0, 1, jgidNumber, datetime.datetime.now(), '0000-00-00 00:00:00', '', '']
            userID = db.query_lastrowid(sql, data)
        else:
            sql = "INSERT INTO `jos_users` (`id`, `name`, `username`, `email`, `password`, `usertype`, `block`, `sendEmail`, `gid`, `registerDate`, `lastvisitDate`, `activation`, `params`)"
            sql += "VALUES (%s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s);"
            data = [joomlauserid, name, username, email, pw, userTypeLookupDict[int(
                jgidNumber)], 0, 1, jgidNumber, datetime.datetime.now(), '0000-00-00 00:00:00', '', '']
            userID = db.query_lastrowid(sql, data)

    return userID


def _insert_jos_core_acl_aro(db, userID, fullname):
    sql = "INSERT INTO `jos_core_acl_aro` (`section_value`, `value`, `order_value`, `name`, `hidden`) "
    sql += "VALUES (%s, %s, %s, %s, %s)"
    data = ['users',  userID, 0, fullname, 0]
    aroID = db.query_lastrowid(sql, data)
    return aroID


def _insert_jos_core_acl_groups_aro_map(db, jgidNumber, aroID):
    sql = "INSERT INTO `jos_core_acl_groups_aro_map` (`group_id`, `section_value`, `aro_id`) "
    sql += "VALUES (%s, %s, %s)"
    data = [jgidNumber, '', aroID]
    db.query_rowcount(sql, data)


def _insert_jos_user_usergroup_map(db, userID, groupID=2):
    sql = "INSERT INTO `jos_user_usergroup_map` (`user_id`, `group_id`) VALUES (%s, %s)"
    data = [userID, groupID]
    db.query_rowcount(sql, data)


def _insert_jos_xprofiles(db, hubname, uidNumber, fname, mname, lname, username, email, password, gid, gidNumber, loginShell, homeDir):

    pw = hashPassword(password)

    # have to support multiple database schemas for this query
    sql = "SELECT count(1) FROM information_schema.COLUMNS WHERE TABLE_NAME = 'jos_xprofiles' AND COLUMN_NAME = 'jobsAllowed'"
    rowCount = db.query_selectscalar(sql, None)

    # cursor.execute(sql)
    ##rs = cursor.fetchone()

    # this is a hack, @TODO: something less hackish would be good here. Not sure why I did this, datatype issue?
    tou = 1

    # newer table version requires different insert (so far jobsAllowed is the only exception)
    if rowCount == 1:
        sql = "INSERT INTO `jos_xprofiles` (`uidNumber`, `name`, `username`, `email`, `registerDate`, `gidNumber`, `homeDirectory`, `loginShell`, "
        sql += "`ftpShell`, `userPassword`, `gid`, `orgtype`, `organization`, `countryresident`, `countryorigin`, `gender`, `url`, `reason`, "
        sql += "`mailPreferenceOption`, `usageAgreement`, `jobsAllowed`, `modifiedDate`, `emailConfirmed`, `regIP`, `regHost`, `nativeTribe`, `phone`, "
        sql += "`proxyPassword`, `proxyUidNumber`, `givenName`, `middleName`, `surname`, `picture`, `vip`, `public`, `params`, `note`, `shadowExpire`) "
        sql += "VALUES (" + '%s,'*37 + "%s );"

        data = [uidNumber, fname + ' ' + lname, username, email, datetime.datetime.now(), gidNumber, homeDir, loginShell,
                '/usr/lib/sftp-server', pw, gid, '', '', '', '', '', '', '',
                0, tou, 3, '0000-00-00 00:00:00', 1, '', '', '', '',
                '', '', fname, mname, lname, '', 0, 0, '', '', None]

    else:
        sql = "INSERT INTO `jos_xprofiles` (`uidNumber`, `name`, `username`, `email`, `registerDate`, `gidNumber`, `homeDirectory`, `loginShell`, "
        sql += "`ftpShell`, `userPassword`, `gid`, `orgtype`, `organization`, `countryresident`, `countryorigin`, `gender`, `url`, `reason`, "
        sql += "`mailPreferenceOption`, `usageAgreement`, `modifiedDate`, `emailConfirmed`, `regIP`, `regHost`, `nativeTribe`, `phone`, "
        sql += "`proxyPassword`, `proxyUidNumber`, `givenName`, `middleName`, `surname`, `picture`, `vip`, `public`, `params`, `note`, `shadowExpire`) "
        sql += "VALUES (" + '%s,'*36 + "%s );"

        data = [uidNumber, fname + ' ' + lname, username, email, datetime.datetime.now(), gidNumber, homeDir, loginShell,
                '/usr/lib/sftp-server', pw, gid, '', '', '', '', '', '', '',
                0, tou, '0000-00-00 00:00:00', 1, '', '', '', '',
                '', '', fname, mname, lname, '', 0, 0, '', '', None]

    db.query_rowcount(sql, data)


def _insert_jos_users_password(db, userid, password):
    sql = "INSERT INTO jos_users_password (`user_id`, `passhash`) "
    sql += "VALUES (%s, %s)"

    if password != '*':
        pw = hubzero.config.passwords.MD5PasswordHash(password)
    elif '{MD5}' in password:
        pw = password
    else:
        # can user generate a value that will MD5 hash to, we think not '*'?
        pw = '{MD5}*'

    data = [userid, pw]
    db.query_rowcount(sql, data)


def add_ldap_user(uid,
                  gecos,
                  password,
                  uidNumber,
                  homeDir,
                  loginShell,
                  gid,
                  gidNumber,
                  hubname=""):
    """ depecated, will eventaully replace with add_ldap_user2"""

    # get ldapManagerUserDN and PW
    hubLDAPAcctMgrDN = hubzero.config.webconfig.getComponentParam(
        "com_system", "ldap_managerdn", hubname)
    hubLDAPAcctMgrPW = hubzero.config.webconfig.getComponentParam(
        "com_system", "ldap_managerpw", hubname)
    hubLDAPBaseDN = hubzero.config.webconfig.getComponentParam(
        "com_system", "ldap_basedn", hubname)

    ldapServer = hubzero.config.webconfig.getComponentParam(
        "com_system", "ldap_primary")
    l = ldap.initialize(ldapServer)
    l.simple_bind_s(hubLDAPAcctMgrDN, hubLDAPAcctMgrPW)

    dn = "uid=" + str(uid) + ",ou=users," + hubLDAPBaseDN
    attrs = {}
    attrs['objectclass'] = ['top', 'account', 'posixAccount', 'shadowAccount']
    attrs['cn'] = str(uid)

    attrs['userPassword'] = hashPassword(password)
    attrs['uid'] = uid.encode('utf-8')
    attrs['uidNumber'] = str(uidNumber)
    attrs['homeDirectory'] = homeDir.encode('utf-8')
    attrs['cn'] = gecos.encode('utf-8')
    attrs['loginShell'] = loginShell

    # TODO: grab this from the com_members component parm field
    attrs['gidNumber'] = str(gidNumber)
    #attrs['gid'] = gid

    ldif = ldap.modlist.addModlist(attrs)
    l.add_s(dn, ldif)


def delete_ldap_user(username, hubname=""):
    # get ldapManagerUserDN and PW
    hubLDAPAcctMgrDN = hubzero.config.webconfig.getComponentParam(
        "com_system", "ldap_managerdn", hubname)
    hubLDAPAcctMgrPW = hubzero.config.webconfig.getComponentParam(
        "com_system", "ldap_managerpw", hubname)
    hubLDAPBaseDN = hubzero.config.webconfig.getComponentParam(
        "com_system", "ldap_basedn", hubname)

    ldapServer = hubzero.config.webconfig.getComponentParam(
        "com_system", "ldap_primary")
    l = ldap.initialize(ldapServer)
    l.simple_bind_s(hubLDAPAcctMgrDN, hubLDAPAcctMgrPW)
    dn = "uid=" + username + ",ou=users," + hubLDAPBaseDN

    try:
        l.delete_s(dn)
    except ldap.NO_SUCH_OBJECT:
        pass


def addhubuser(hubname,
               username,
               email,
               pw,
               pwdisabled=False,
               grouptype=1,
               joomlagidNumber=18,
               gid="users",
               gidNumber=100,
               uidNumber=-1,
               gecos="",
               loginShell="/bin/bash",
               skipCreateHome=False,
               homeDir="",
               createGroup=False,
               skipCreateLDAPUser=False,
               skipCreateCMSUser=False,
               uid_min=1000,
               uid_max=65533):
    """
    hubname - for machines that have more than one hub, otherwise this will use the default hub for a server
    username - username to add
    email - user email
    pw - unencrypted password, we'll hash it as necessary using MD5
    pwdisabled - should the PW be disabled?
    grouptype - 0 (system), 1 (regular)
    joomlagidNumber - 18=Registered, 23=Manager, 24=Administrator, 25=Super Administrator
    gid - linux gid membership
    gidNumber - linux gidNumber, corresponds to the group with the previous gid
    uidNumber - specify the jos_users PK ID, also is the uidNumber for the ldap (same thing)
    gecos - comma delimited info list on user, we curretnly only use first entry, which is a full name (fname, mname, lname)
    loginShell - user's login shell
    skipCreateHome - skip creating user's home directory
    homeDir - path to user's home directory
    createGroup - skip creating a default group for new user, group name matches the username
    skipCreateLDAPUser - skip creating ldap info for this user
    skipCreateCMSUser - skip ldap info for this user
    """

    # grab config options
    dbHost = hubzero.config.webconfig.getWebConfigDBHost(hubname)
    dbPW = hubzero.config.webconfig.getWebConfigDBPassword(hubname)
    dbUserName = hubzero.config.webconfig.getWebConfigDBUsername(hubname)
    dbName = hubzero.config.webconfig.getWebConfigDBName(hubname)

    db = hubzero.data.db.MySQLConnection(dbHost, dbName, dbUserName, dbPW)

    # Parse gecos, it's a massive mess, but it's a holdover from the wild west days of unix and it's a comma delimited field
    # used in the /etc/passwd file to store personal info. It's comma delimited, and name is the first item in the list...
    # right now we don't care about what else gets passed, just the first entry
    gecosFields = gecos.split(",")
    fullName = gecosFields[0]
    fullNameList = gecosFields[0].split(" ")

    if len(fullNameList) == 3:
        firstName = fullNameList[0]
        middleName = fullNameList[1]
        lastName = fullNameList[2]
    elif len(fullNameList) == 2:
        firstName = fullNameList[0]
        middleName = ""
        lastName = fullNameList[1]
    elif len(fullNameList) == 1:
        firstName = fullNameList[0]
        middleName = ""
        lastName = ""
    else:
        firstName = ""
        middleName = ""
        lastName = ""

    # usually homeDir will be blank, but not always
    if homeDir == "":
        homeDir = '/home/' + hubname + "/" + username

    if pwdisabled:
        pw = "*"

    # check for unique username
    if not skipCreateCMSUser:
        # when inserting a cms user, with or without a ldap user, test the mysql database for unique usernames
        if _userExists(db, username):
            print "User '" + username + "' already exists in CMS"
            return (1)
    elif not skipCreateLDAPUser:
        # no cms and ldap, check ldap
        if uidExists(username):
            print "User '" + username + "' already exists in ldap"
            return (2)
    else:
        print "skip cms insert and ldap"
        return(3)

    # if the user doesn't want custom group for this user, they still need to be added to some group
    if not createGroup:
        gidNumber = 100  # default to the users group
        gid = 'users'

    if not skipCreateCMSUser:
        # create group for user named same as username (need group insert here for xprofiles addition below)
        # Guess if user specifies a specific already created group for the new user and says create a new group for the
        # user, we'll override and create the new group with this logic
        if createGroup:
            gidNumber = hubzero.utilities.group._insert_jos_xgroups(
                db, username, username + " group", grouptype)
            gid = username  # group name same as user

        uidNumber = _insert_jos_users(
            db, uidNumber, fullName, username, email, pw, joomlagidNumber)

        if hubzero.utilities.misc.JOOMLA_25:
            if (joomlagidNumber == 25):
                _insert_jos_user_usergroup_map(db, uidNumber, 8)
            else:
                _insert_jos_user_usergroup_map(db, uidNumber, 2)
        else:
            aroID = _insert_jos_core_acl_aro(db, uidNumber, fullName)
            _insert_jos_core_acl_groups_aro_map(db, joomlagidNumber, aroID)

        _insert_jos_xprofiles(db, hubname, uidNumber, firstName, middleName,
                              lastName, username, email, pw, gid, gidNumber, loginShell, homeDir)
        _insert_jos_users_password(db, uidNumber, pw)

        # add user to new group (need xgroups insert after user creation because we need userid, I'd prefer to keep all group logic in one place)
        if createGroup:
            hubzero.utilities.group._insert_jos_xgroups_members(
                db, gidNumber, uidNumber)

        # create home directory
        if not skipCreateHome:
            rc, procStdOut, procStdErr = hubzero.utilities.misc.exShellCommand(
                ["mkdir", homeDir, "--mode", "0750"])
            if rc:
                print procStdErr

            print "creating homedir: " + homeDir

            # set ownership
            if createGroup:
                os.chown(homeDir, uidNumber, gidNumber)
            else:
                os.chown(homeDir, uidNumber, -1)

            # change permissions user full, noone else has anything
            rc, procStdOut, procStdErr = hubzero.utilities.misc.exShellCommand(
                ["chmod", '700', homeDir])
            if rc:
                print procStdErr

    if not skipCreateLDAPUser:

        # if no uidNumber specified, find an available one
        if uidNumber == -1:
            uidNumber = hubzero.utilities.user.getLDAPMinUIDNumber(
                hubname, uid_min, uid_max)

        if createGroup:
            hubzero.utilities.group.add_ldap_group(
                username, username + " group", gidNumber)

        add_ldap_user(username, gecos, pw, uidNumber,
                      homeDir, loginShell, gid, gidNumber)

    return(0)


def delhubuser(username, hubname=""):
    dbHost = hubzero.config.webconfig.getWebConfigDBHost(hubname)
    dbPW = hubzero.config.webconfig.getWebConfigDBPassword(hubname)
    dbUserName = hubzero.config.webconfig.getWebConfigDBUsername(hubname)
    dbName = hubzero.config.webconfig.getWebConfigDBName(hubname)
    db = hubzero.data.db.MySQLConnection(dbHost, dbName, dbUserName, dbPW)

    sql = 'select id from `jos_users` where username = %s'
    data = [username]
    userID = db.query_selectscalar(sql, data)

    if userID:
        print "deleting found user"

        print userID

        if hubzero.utilities.misc.JOOMLA_25:
            _del_jos_user_usergroup_map(db, userID)
        else:
            aroID = _del_jos_core_acl_aro(db, userID)
            if aroID:
                _del_jos_core_acl_groups_aro_map(db, aroID)

        _del_jos_xprofiles(db, userID)
        _del_jos_users(db, username)

    else:
        print "user not found"

    delete_ldap_user(username)


def updateUserPW(username, password, updateCMS=True, updateLDAP=True, hubname=""):
    """
    pass plaintext pw, we'll hash it
    """
    changed = 0
    dbHost = hubzero.config.webconfig.getWebConfigDBHost(hubname)
    dbPW = hubzero.config.webconfig.getWebConfigDBPassword(hubname)
    dbUserName = hubzero.config.webconfig.getWebConfigDBUsername(hubname)
    dbName = hubzero.config.webconfig.getWebConfigDBName(hubname)
    db = hubzero.data.db.MySQLConnection(dbHost, dbName, dbUserName, dbPW)

    userid = _getUserID(db, username)

    pw = hashPassword(password)

    if updateCMS:
        changed += _update_jos_xprofiles_pw(db, userid, pw)
        changed += _update_jos_users_pw(db, userid, pw)
        changed += _update_jos_users_password(db, userid, pw)

    if updateLDAP:
        changed += _update_ldap_user_pw(username, userid, pw)

    if changed:
        return True

    return False

def uidNumberExists(uidNumber, hubname=""):
    """
    See if specified uidNumber is in the ldap user ou
    """

    hubLDAPAcctMgrDN = hubzero.config.webconfig.getComponentParam(
        "com_system", "ldap_managerdn", hubname)
    hubLDAPAcctMgrPW = hubzero.config.webconfig.getComponentParam(
        "com_system", "ldap_managerpw", hubname)
    hubLDAPBaseDN = hubzero.config.webconfig.getComponentParam(
        "com_system", "ldap_basedn", hubname)

    l = ldap.initialize('ldap://localhost')
    l.simple_bind_s(hubLDAPAcctMgrDN, hubLDAPAcctMgrPW)

    results = l.search_s('ou=users,' + hubLDAPBaseDN,
                         ldap.SCOPE_SUBTREE, '(uidNumber=' + uidNumber + ')', None)

    if not results:
        return False
    else:
        return True


def uidExists(uid, hubname=""):
    """
    See if specified uid is in the ldap user ou
    """
    hubLDAPAcctMgrDN = hubzero.config.webconfig.getComponentParam(
        "com_system", "ldap_managerdn", hubname)
    hubLDAPAcctMgrPW = hubzero.config.webconfig.getComponentParam(
        "com_system", "ldap_managerpw", hubname)
    hubLDAPBaseDN = hubzero.config.webconfig.getComponentParam(
        "com_system", "ldap_basedn", hubname)

    l = ldap.initialize('ldap://localhost')
    l.simple_bind_s(hubLDAPAcctMgrDN, hubLDAPAcctMgrPW)

    results = l.search_s('ou=users,' + hubLDAPBaseDN,
                         ldap.SCOPE_SUBTREE, '(uid=' + uid + ')', None)

    if not results:
        return False
    else:
        return True


def getLDAPMinUIDNumber(hubname="", minUIDNumber=1, maxUIDNumber=1000):
    """
    get max(uid) of all objects of class posixAccount in ou=Users
    """

    hubLDAPAcctMgrDN = hubzero.config.webconfig.getComponentParam(
        "com_system", "ldap_managerdn", hubname)
    hubLDAPAcctMgrPW = hubzero.config.webconfig.getComponentParam(
        "com_system", "ldap_managerpw", hubname)
    hubLDAPBaseDN = hubzero.config.webconfig.getComponentParam(
        "com_system", "ldap_basedn", hubname)

    l = ldap.initialize('ldap://localhost')
    l.simple_bind_s(hubLDAPAcctMgrDN, hubLDAPAcctMgrPW)

    results = l.search_s('ou=users,' + hubLDAPBaseDN, ldap.SCOPE_SUBTREE,
                         '(objectclass=posixAccount)', ['uidNumber'])

    # store all uidNumbers in list
    allUIDNumbers = []
    for result in results:
        allUIDNumbers.append(int(result[1]['uidNumber'][0]))

    rv = -1
    for x in range(int(minUIDNumber), int(maxUIDNumber)):
        if x not in allUIDNumbers:
            rv = x
            break

    return rv


def _syncuser(username, hubname="", syncAttributes=False):
    """
    sync attributes syncs all available non essential account attributes.
    """

    if not hubname:
        hubname = hubzero.config.webconfig.getDefaultSite()

    hubLDAPAcctMgrDN = hubzero.config.webconfig.getComponentParam(
        "com_system", "ldap_managerdn", hubname)
    hubLDAPAcctMgrPW = hubzero.config.webconfig.getComponentParam(
        "com_system", "ldap_managerpw", hubname)
    hubLDAPBaseDN = hubzero.config.webconfig.getComponentParam(
        "com_system", "ldap_basedn", hubname)

    dbHost = hubzero.config.webconfig.getWebConfigDBHost(hubname)
    dbPW = hubzero.config.webconfig.getWebConfigDBPassword(hubname)
    dbUserName = hubzero.config.webconfig.getWebConfigDBUsername(hubname)
    dbName = hubzero.config.webconfig.getWebConfigDBName(hubname)
    db = hubzero.data.db.MySQLConnection(dbHost, dbName, dbUserName, dbPW)

    query = """SELECT u.id AS uidNumber, u.username AS uid, u.name AS cn, p.gidNumber, u.homeDirectory, p.loginShell, 
    pwd.passhash AS userPassword, pwd.shadowLastChange, pwd.shadowMin, pwd.shadowMax, pwd.shadowWarning, 
    pwd.shadowInactive, pwd.shadowExpire, pwd.shadowFlag 
    FROM jos_users AS u 
    LEFT JOIN jos_users_password AS pwd ON u.id = pwd.user_id 
    LEFT JOIN jos_xprofiles AS p ON u.id = p.uidNumber 
    WHERE u.username = %s; 
    """
    data = [username]
    user = db.query(query, data)

    if len(user[0].uid) == 0 or user[0].uid[0] == '-':
        print "Skipping illegal uidNumber %s(%s)" % (user[0].uid, user[0].cn)
        return

    if user[0].uid.startswith('anonUsername_'):
        print "Deleting anonymized username %s(%s)" % (user[0].uid, user[0].cn)
        delete_ldap_user(user[0].uid)
        return

    attrsToSync = {}
    if syncAttributes:

        attributesToCheck = ['loginShell', 'userPassword', 'shadowLastChange', 'shadowMin',
                             'shadowMax', 'shadowWarning', 'shadowInactive', 'shadowExpire', 'shadowFlag']

        # set all the attributes
        for a in attributesToCheck:
            av = getattr(user[0], a, None)
            if av:
                attrsToSync[str(a)] = str(av)

    else:
        attrsToSync['userPassword'] = user[0].userPassword

    if user[0].homeDirectory == "":
        user[0].homeDirectory = '/home/' + hubname + "/" + user[0].uid

    if user[0].gidNumber == "" or user[0].gidNumber  == None:
        user[0].gidNumber = 3000

    if user:

        print ("adding user " + user[0].cn.encode('utf-8'))
        add_ldap_user2(user[0].cn,
                       user[0].uid,
                       user[0].uidNumber,
                       user[0].homeDirectory,
                       user[0].gidNumber,
                       hubname,
                       attrsToSync)
    else:
        print "_syncuser cannot find info on username: " + \
            username + " from the jos_users table"


def add_ldap_user2(cn,
                   uid,
                   uidNumber,
                   homeDir,
                   gidNumber,
                   hubname="",
                   passedAttributes={}):
    """
    Following ldap attributes are required by the follwing objectclasses
    # account - uid
    # posixAccount - cn, uid, uidNumber, gidNumber, homeDirectory
    # shadowAccount - uid

    All other optional attributes can be passed in the list and we'll grab
    them. If the attribute is invalid, shame on you
    """

    if (cn == ''):
        cn = uid

    # get ldapManagerUserDN and PW
    hubLDAPAcctMgrDN = hubzero.config.webconfig.getComponentParam(
        "com_system", "ldap_managerdn", hubname)
    hubLDAPAcctMgrPW = hubzero.config.webconfig.getComponentParam(
        "com_system", "ldap_managerpw", hubname)
    hubLDAPBaseDN = hubzero.config.webconfig.getComponentParam(
        "com_system", "ldap_basedn", hubname)

    l = ldap.initialize('ldap://localhost')
    l.simple_bind_s(hubLDAPAcctMgrDN, hubLDAPAcctMgrPW)

    dn = "uid=" + str(uid) + ",ou=users," + hubLDAPBaseDN

    attrs = {}
    attrs['objectclass'] = ['top', 'account', 'posixAccount', 'shadowAccount']
    attrs['cn'] = cn.encode('utf-8')
    attrs['uid'] = uid.encode('utf-8')
    attrs['uidNumber'] = str(uidNumber)
    attrs['homeDirectory'] = homeDir.encode('utf-8')
    attrs['gidNumber'] = str(gidNumber)

    allAttrs = dict(attrs.items() + passedAttributes.items())
    ldif = ldap.modlist.addModlist(allAttrs)

    try:
        l.add_s(dn, ldif)
    except ldap.ALREADY_EXISTS:
        # delete then readd
        delete_ldap_user(uid)
        l.add_s(dn, ldif)

        # code below was for an in place update, kept around just in case
        # get previous attribute list for object
        #results = l.search_s('ou=users,' + hubLDAPBaseDN, ldap.SCOPE_SUBTREE, '(uidNumber=' + str(uidNumber) + ')', None)

        # second item [1] is the attribute list of the first ldap entry [0] returned by the search
        # construct modified attribute list
        #ldif_mod = ldap.modlist.modifyModlist(results[0][1], dict(attrs.items() + passedAttributes.items()))

        # try:
        #	l.modify_s(dn, ldif_mod)
        # except Exception as e1:
        #	print "error updating " + uid
        #	print e1
        #	traceback.print_exc()


def syncuser(user, hubname="", purgeorphans=True, syncattributes=True, verbose=False, debug=False):
    _syncusers(hubname, purgeorphans, syncattributes, user, verbose, debug)


def syncusers(hubname="", purgeorphans=True, syncattributes=True, verbose=False, debug=False):
    _syncusers(hubname, purgeorphans, syncattributes, '*', verbose, debug)


def _syncusers(hubname="", purgeorphans=True, syncattributes=True, user='*', verbose=False, debug=False):
    dbHost = hubzero.config.webconfig.getWebConfigDBHost(hubname)
    dbPW = hubzero.config.webconfig.getWebConfigDBPassword()
    dbUserName = hubzero.config.webconfig.getWebConfigDBUsername()
    dbName = hubzero.config.webconfig.getWebConfigDBName()
    ldapServer = hubzero.config.webconfig.getComponentParam(
        "com_system", "ldap_primary")
    db = hubzero.data.db.MySQLConnection(dbHost, dbName, dbUserName, dbPW)

    # normally this would be accomplished with a complete purge and a readd, but we can't have our larger sites with an empty user database for
    # the length of time it would take this script to run, so we do it this way
    if purgeorphans:

        if debug:
            print "Orphaned users in ldap without a corresponding group in the database will be DELETED"

        # grab all users from db and store them off for easy lookup
        if user == '*':
            sql = "SELECT `username` from jos_users WHERE username NOT LIKE '-%' AND username NOT LIKE 'anonUsername_%'"
        else:
            sql = "SELECT `username` from jos_users WHERE username='" + user + "' and username NOT LIKE '-%' AND username NOT LIKE 'anonUsername_%'"

        usersFromDB = {}
        for duser in db.query(sql, None):
            usersFromDB[duser.username] = 1

        hubLDAPAcctMgrDN = hubzero.config.webconfig.getComponentParam(
            "com_system", "ldap_managerdn")
        hubLDAPAcctMgrPW = hubzero.config.webconfig.getComponentParam(
            "com_system", "ldap_managerpw")
        hubLDAPBaseDN = hubzero.config.webconfig.getComponentParam(
            "com_system", "ldap_basedn")

        # grab all users from ldap
        ldapServer = hubzero.config.webconfig.getComponentParam(
            "com_system", "ldap_primary")
        l = ldap.initialize(ldapServer)

        l.simple_bind_s(hubLDAPAcctMgrDN, hubLDAPAcctMgrPW)

        r = None
        try:
            if user == '*':
                r = l.search_s("ou=users," + hubLDAPBaseDN, ldap.SCOPE_SUBTREE,
                               '(objectClass=posixAccount)', ['uid'])
            else:
                r = l.search_s("uid="+user+",ou=users," + hubLDAPBaseDN,
                               ldap.SCOPE_SUBTREE, '(objectClass=posixAccount)', ['uid'])
        except ldap.NO_SUCH_OBJECT:
            pass

        if r:
            for dn, entry in r:
                uid = str(entry['uid'][0])

                if uid not in usersFromDB:
                    if verbose:
                        print "deleting user '" + uid + "' from ldap"
                    try:
                        hubzero.utilities.user.delete_ldap_user(uid)
                    except ldap.NO_SUCH_OBJECT:
                        if debug:
                            print " user not found"

    # now do a mass add of all users from db
    if user == '*':
        sql = "SELECT `username` from jos_users"
    else:
        sql = "SELECT `username` from jos_users WHERE username='" + user + "'"

    for duser in db.query(sql, None):
        if verbose:
            print "syncing user '" + duser.username + "' to ldap"
        _syncuser(duser.username, hubname, syncattributes)


def updateuser(username, hubname="", fname="", mname="", lname="", email="", homedir=""):

    if not hubname:
        hubname = hubzero.config.webconfig.getDefaultSite()

    dbHost = hubzero.config.webconfig.getWebConfigDBHost(hubname)
    dbPW = hubzero.config.webconfig.getWebConfigDBPassword()
    dbUserName = hubzero.config.webconfig.getWebConfigDBUsername()
    dbName = hubzero.config.webconfig.getWebConfigDBName()
    ldapServer = hubzero.config.webconfig.getComponentParam(
        "com_system", "ldap_primary")
    db = hubzero.data.db.MySQLConnection(dbHost, dbName, dbUserName, dbPW)
    uidNumber = _getUserID(db, username)

    _update_jos_xprofiles(db, hubname, uidNumber, fname,
                          mname, lname, email, '', '', '', '', '', '', homedir)

    if homedir:
        _update_ldap_user_attribue(username, 'homeDirectory', homedir)


def _update_ldap_user_attribue(uid, attributename, attributevalue):

    hubLDAPAcctMgrDN = hubzero.config.webconfig.getComponentParam(
        "com_system", "ldap_managerdn")
    hubLDAPAcctMgrPW = hubzero.config.webconfig.getComponentParam(
        "com_system", "ldap_managerpw")
    hubLDAPBaseDN = hubzero.config.webconfig.getComponentParam(
        "com_system", "ldap_basedn")
    ldapServer = hubzero.config.webconfig.getComponentParam(
        "com_system", "ldap_primary")
    l = ldap.initialize(ldapServer)
    l.simple_bind_s(hubLDAPAcctMgrDN, hubLDAPAcctMgrPW)

    dn = "uid=" + uid + ",ou=users," + hubLDAPBaseDN
    attrs = [(ldap.MOD_REPLACE, attributename, attributevalue)]
    l.modify_s(dn, attrs)


def _update_jos_xprofiles(db, hubname, uid, fname='', mname='', lname='',  email='', pw='', joomlagid='', joomlagidNumber='', gid='', gidNumber='', loginShell='', homeDir=''):

    nvlist = {}

    if fname:
        nvlist['givenName'] = fname
    if lname:
        nvlist['surname'] = lname
    if email:
        nvlist['email'] = email
    if homeDir:
        nvlist['homeDirectory'] = homeDir

    sql = "UPDATE `jos_xprofiles`"
    sql += " SET " + "=%s, ".join(nvlist.keys()) + "=%s"
    sql += " WHERE uidNumber = %s "

    values = nvlist.values()
    values.append(uid)

    db.query_rowcount(sql, values)
