# @package      hubzero-mw2-common
# @file         ldap_mirror.py
# @author       Juan Pineda <jpinedac@purdue.edu>
# @copyright    Copyright (c) 2016-2017 HUBzero Foundation, LLC.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2016-2017 HUBzero Foundation, LLC.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#

""" 
Small library to deal with LDAP on the HUBzero mirror
"""

import ldap
import ldap.modlist as modlist
import copy

#@TODO: be careful with the next variable, get it from the configuration.
LDAP_HOST = 'localhost'
LDAP_HOST_URI = 'ldap://' + LDAP_HOST

#HUBLDAPACCTMGRDN = hubzero.config.webconfig.getComponentParam('com_system',
                                                            'ldap_managerdn')
#HUBLDAPACCTMGRPW = hubzero.config.webconfig.getComponentParam('com_system',
                                                            'ldap_managerpw')
#HUBLDAPBASEDN = hubzero.config.webconfig.getComponentParam('com_system',
                                                        'ldap_basedn')
execfile("/etc/mw-client/ldap.conf")

def _exist_ldap_user(username):
    """ Verifies the existence of a user in the LDAP. Returns
    True if found, False otherwise."""
    try:
        ldh = ldap.initialize(LDAP_HOST_URI)
        ldh.simple_bind_s(HUBLDAPACCTMGRDN, HUBLDAPACCTMGRPW)
        dnq = "(&(objectClass=shadowAccount)(uid=" + username + "))"
        result_id = ldh.search_s(HUBLDAPBASEDN, ldap.SCOPE_SUBTREE, dnq, [''])

        if not result_id:
            return False
        else:
            return True
        ldh.unbind_s()
    except ldap.LDAPError, error:
        print 'E: Problem with LDAP: %s' % error


def _exist_ldap_group(groupid):
    """ Verifies the existence of a group in the LDAP. Returns True if found,
    False otherwise."""
    try:
        ldh = ldap.initialize(LDAP_HOST_URI)
        ldh.simple_bind_s(HUBLDAPACCTMGRDN, HUBLDAPACCTMGRPW)
        dnq = "(&(objectClass=posixGroup)(cn=" + groupid + "))"
        result_id = ldh.search_s(HUBLDAPBASEDN, ldap.SCOPE_SUBTREE, dnq, [''])
        if result_id == []:
            return False
        else:
            return True
        ldh.unbind_s()
    except ldap.LDAPError, error:
        print 'E: Problem with LDAP: %s' % error

def _list_groups(user):
    """List all groups registered to wich the users belongs in the LDAP. 
    Returns a list with the names of the groups or an empty list otherwise"""
    groups = []
    try:
        ldh = ldap.initialize(LDAP_HOST_URI)
        ldh.simple_bind_s(HUBLDAPACCTMGRDN, HUBLDAPACCTMGRPW)
        dnq = "(&(objectClass=posixGroup)(memberUid=" + user + "))"
        result_id = ldh.search_s(HUBLDAPBASEDN, ldap.SCOPE_SUBTREE, dnq,
                                 ['cn'])

        for i in result_id:
            groups += i[1]['cn']

        ldh.unbind_s()
    except ldap.LDAPError, error:
        print 'E: Problem with LDAP: %s' % error

    return groups

def _create_ldap_user(username, attrs):
    """ Create a new ldap user based on given attributes. It verifies the
    existence of the user in LDAP. Return none if success. """
    if not _exist_ldap_user(username):
        try:
            ldh = ldap.initialize(LDAP_HOST_URI)
            ldh.simple_bind_s(HUBLDAPACCTMGRDN, HUBLDAPACCTMGRPW)

            dnq = "(&(objectClass=shadowAccount)(uidNumber=*))"
            result_id = ldh.search_s(HUBLDAPBASEDN, ldap.SCOPE_SUBTREE, dnq,
                                   ['uidNumber'])
            uids = []

            for i in result_id:
                uids.append(int(i[1]['uidNumber'][0]))

            userid = max(uids)
            attrs['uidNumber'] = str(userid + 1)
            dnq = 'uid=' + username + ',ou=users,' + HUBLDAPBASEDN
            ldif = modlist.addModlist(attrs)
            ldh.add_s(dnq, ldif)
            ldh.unbind_s()
        except ldap.LDAPError, error:
            print 'E: Problem with LDAP: %s' % error

def _create_ldap_group(groupname):
    """ Creates a group on LDAP """
    if not _exist_ldap_group(groupname):
        attrs = {}
        attrs['objectclass'] = ['posixGroup', 'top']
        attrs['cn'] = groupname
        try:
            ldh = ldap.initialize(LDAP_HOST_URI)
            ldh.simple_bind_s(HUBLDAPACCTMGRDN, HUBLDAPACCTMGRPW)
            dnq = "(&(objectClass=posixGroup)(gidNumber=*))"
            result_id = ldh.search_s(HUBLDAPBASEDN, ldap.SCOPE_SUBTREE, dnq,
                                   ['gidNumber'])
            gids = [1000]
            for i in result_id:
                gids.append(int(i[1]['gidNumber'][0]))

            gid = max(gids)
            attrs['gidNumber'] = str(gid + 1)
            dnq = 'cn=' + groupname + ',ou=groups,' + HUBLDAPBASEDN
            ldif = modlist.addModlist(attrs)
            ldh.add_s(dnq, ldif)
            ldh.unbind_s()
        except ldap.LDAPError, error:
            print 'E: Problem with LDAP: %s' % error

def _user_exist_in_group(user, group):
    """ In this function it's assumed that the user and the group exists"""
    try:
        ldh = ldap.initialize(LDAP_HOST_URI)
        ldh.simple_bind_s(HUBLDAPACCTMGRDN, HUBLDAPACCTMGRPW)
        dnq = "(&(objectClass=posixGroup)(cn=" + group + "))"
        result_id = ldh.search_s(HUBLDAPBASEDN, ldap.SCOPE_SUBTREE, dnq, 
                                 ['memberUid'])
        ldh.unbind_s()

        if result_id[0][1]:
            users = result_id[0][1]['memberUid']

            for i in users:
                if i == user:
                    return True
            return False
        else:
            return False
    except ldap.LDAPError, error:
        print 'E: Problem with LDAP: %s' % error

def _add_user_to_group(user, group):
    """ In this function it's assumed that 
    the user and the group exists
    """
    if not _user_exist_in_group(user, group):
        try:
            ldh = ldap.initialize(LDAP_HOST_URI)
            ldh.simple_bind_s(HUBLDAPACCTMGRDN, HUBLDAPACCTMGRPW)
            dnq = "(&(objectClass=posixGroup)(cn=" + group + "))"
            result_id = ldh.search_s(HUBLDAPBASEDN, ldap.SCOPE_SUBTREE, dnq, 
                                     ['memberUid'])
            if result_id[0][1]:
                new_r = copy.deepcopy(result_id[0][1])
                new_r['memberUid'].append(user)
                dnq = 'cn=' + group + ',ou=groups,' + HUBLDAPBASEDN
                ldif = modlist.modifyModlist(result_id[0][1], new_r)
                ldh.modify_s(dnq, ldif)
            else:
                new_r = {'memberUid': [user]}
                dnq = 'cn=' + group + ',ou=groups,' + HUBLDAPBASEDN
                ldif = modlist.modifyModlist({}, new_r)
                ldh.modify_s(dnq, ldif)
            ldh.unbind_s()
        except ldap.LDAPError, error:
            print 'E: Problem with LDAP: %s' % error

def _delete_user_from_group(user, group):
    """ In this function it's assumed that the user and the group exists"""
    if _user_exist_in_group(user, group):
        try:
            ldh = ldap.initialize(LDAP_HOST_URI)
            ldh.simple_bind_s(HUBLDAPACCTMGRDN, HUBLDAPACCTMGRPW)
            dnq = "(&(objectClass=posixGroup)(cn=" + group + "))"
            result_id = ldh.search_s(HUBLDAPBASEDN, ldap.SCOPE_SUBTREE, dnq,
                                   ['memberUid'])
            new_r = copy.deepcopy(result_id[0][1])
            new_r['memberUid'].remove(user)
            dnq = 'cn=' + group + ',ou=groups,' + HUBLDAPBASEDN
            ldif = modlist.modifyModlist(result_id[0][1], new_r)
            ldh.modify_s(dnq, ldif)
            ldh.unbind_s()
        except ldap.LDAPError, error:
            print 'E: Problem with LDAP: %s' % error

def main():
    """ Main function to test the library
    """
    pass

if __name__ == '__main__':
    main()
