# @package      hubzero-mw2-common
# @file         heartbeat.py
# @author       Pascal Meunier <pmeunier@purdue.edu>
# @copyright    Copyright (c) 2016-2017 HUBzero Foundation, LLC.
# @license      http://opensource.org/licenses/MIT MIT
#
# Based on previous work by Richard L. Kennell and Nicholas Kisseberth
#
# Copyright (c) 2016-2017 HUBzero Foundation, LLC.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#

"""
This module supports a heartbeat every x seconds by running
an SQL query, until a UNIX signal is received.  This module defines a global.  
The code below is simpler than using a class.  Doing it with a class 
ends up being just as messy because the signal handlers need to be 
passed a class instance, which means a global would be necessary.  
Setting that global from outside this module would also be messy.  
"""

import signal
from errors import MaxwellError
from mw_db import MW_DB

# globals
timed_out = False
done = False

signals = [signal.SIGCHLD, signal.SIGHUP, signal.SIGINT, signal.SIGQUIT, signal.SIGTERM]
def done_handler(signum, dummy):
  global done
  done = True

def alarm_handler(signum, dummy):
  global timed_out
  timed_out = True

def do_heartbeat(sql, sql_param, db_params, delay):
  global timed_out
  global done

  if len(db_params) != 5:
    raise MaxwellError("incorrect database parameters")
  timed_out = False
  done = False

  # save and set signals
  saved_sigs = {}
  for sig in signals:
    saved_sigs[sig] = signal.signal(sig, done_handler)
  signal.signal(signal.SIGALRM, alarm_handler)

  # Wait for any signal, other than alarm.
  while not (done):
    while not (done or timed_out):
      # reset the time delay to avoid a race condition which makes it possible for
      # the alarm signal to happen after the check for self.timed_out and before the call to signal.pause
      signal.alarm(delay)
      # wait for any signal
      signal.pause()
    if timed_out:
      timed_out = False
      # register heartbeat
      db = MW_DB(*db_params)
      db.c.execute(sql, sql_param)
      db.db.close()

  # restore signals
  for sig in signals:
    signal.signal(sig, saved_sigs[sig])
  signal.signal(signal.SIGALRM, signal.SIG_IGN)
