<?php

use League\Flysystem\Config;
use League\Flysystem\Dropbox\DropboxAdapter as Dropbox;

class DropboxTests extends PHPUnit_Framework_TestCase
{
    public function getClientMock()
    {
        $mock = Mockery::mock('Dropbox\Client');
        $mock->shouldReceive('__toString')->andReturn('Dropbox\Client');

        return $mock;
    }

    public function testInstantiable()
    {
        new Dropbox($this->getClientMock(), 'prefix');
    }

    public function dropboxProvider()
    {
        $mock = $this->getClientMock();

        return [
            [new Dropbox($mock, 'prefix'), $mock],
        ];
    }

    /**
     * @dataProvider  dropboxProvider
     */
    public function testWrite($adapter, $mock)
    {
        $mock->shouldReceive('uploadFileFromString')->andReturn([
            'is_dir'   => false,
            'modified' => '10 September 2000',
        ], false);

        $result = $adapter->write('something', 'contents', new Config());
        $this->assertInternalType('array', $result);
        $this->assertArrayHasKey('type', $result);
        $this->assertEquals('file', $result['type']);
        $this->assertFalse($adapter->write('something', 'something', new Config()));
    }

    /**
     * @dataProvider  dropboxProvider
     */
    public function testUpdate(Dropbox $adapter, $mock)
    {
        $mock->shouldReceive('uploadFileFromString')->andReturn([
            'is_dir'   => false,
            'modified' => '10 September 2000',
        ], false);

        $result = $adapter->update('something', 'contents', new Config());
        $this->assertInternalType('array', $result);
        $this->assertArrayHasKey('type', $result);
        $this->assertEquals('file', $result['type']);
        $this->assertFalse($adapter->update('something', 'something', new Config()));
    }

    /**
     * @dataProvider  dropboxProvider
     */
    public function testWriteStream(Dropbox $adapter, $mock)
    {
        $mock->shouldReceive('uploadFile')->andReturn([
            'is_dir'   => false,
            'modified' => '10 September 2000',
        ], false);

        $result = $adapter->writeStream('something', tmpfile(), new Config());
        $this->assertInternalType('array', $result);
        $this->assertArrayHasKey('type', $result);
        $this->assertEquals('file', $result['type']);
        $this->assertFalse($adapter->writeStream('something', tmpfile(), new Config()));
    }

    /**
     * @dataProvider  dropboxProvider
     */
    public function testUpdateStream(Dropbox $adapter, $mock)
    {
        $mock->shouldReceive('uploadFile')->andReturn([
            'is_dir'   => false,
            'modified' => '10 September 2000',
        ], false);

        $result = $adapter->updateStream('something', tmpfile(), new Config());
        $this->assertInternalType('array', $result);
        $this->assertArrayHasKey('type', $result);
        $this->assertEquals('file', $result['type']);
        $this->assertFalse($adapter->updateStream('something', tmpfile(), new Config()));
    }

    public function metadataProvider()
    {
        return [
            ['getMetadata'],
            ['getMimetype'],
            ['getTimestamp'],
            ['getSize'],
            ['has'],
        ];
    }

    /**
     * @dataProvider  metadataProvider
     */
    public function testMetadataCalls($method)
    {
        $mock = $this->getClientMock();
        $mock->shouldReceive('getMetadata')->twice()->andReturn([
            'is_dir'   => false,
            'modified' => '10 September 2000',
        ], false);

        $adapter = new Dropbox($mock);
        $this->assertInternalType('array', $adapter->{$method}('one', 'two'));
        $this->assertFalse($adapter->{$method}('one', 'two'));
    }

    /**
     * @dataProvider  dropboxProvider
     */
    public function testRead($adapter, $mock)
    {
        $stream = tmpfile();
        fwrite($stream, 'something');
        $mock->shouldReceive('getFile')->andReturn($stream, false);
        $this->assertInternalType('array', $adapter->read('something'));
        $this->assertFalse($adapter->read('something'));
        fclose($stream);
    }

    /**
     * @dataProvider  dropboxProvider
     */
    public function testReadStream(Dropbox $adapter, $mock)
    {
        $stream = tmpfile();
        fwrite($stream, 'something');
        $mock->shouldReceive('getFile')->andReturn($stream, false);
        $this->assertInternalType('array', $adapter->readStream('something'));
        $this->assertFalse($adapter->readStream('something'));
        fclose($stream);
    }

    /**
     * @dataProvider  dropboxProvider
     */
    public function testDelete(Dropbox $adapter, $mock)
    {
        $mock->shouldReceive('delete')->andReturn(true);
        $this->assertTrue($adapter->delete('something'));
        $this->assertTrue($adapter->deleteDir('something'));
    }

    /**
     * @dataProvider  dropboxProvider
     */
    public function testCreateDir(Dropbox $adapter, $mock)
    {
        $mock->shouldReceive('createFolder')->with('/prefix/fail/please')->andReturn(null);
        $mock->shouldReceive('createFolder')->with('/prefix/pass/please')->andReturn([
            'is_dir' => true,
            'path'   => 'pass/please',
        ]);
        $this->assertFalse($adapter->createDir('fail/please', new Config()));
        $expected = ['path' => 'pass/please', 'type' => 'dir'];
        $this->assertEquals($expected, $adapter->createDir('pass/please', new Config()));
    }

    /**
     * @dataProvider  dropboxProvider
     */
    public function testListContents(Dropbox $adapter, $mock)
    {
        $mock->shouldReceive('getMetadataWithChildren')->andReturn(
            ['contents' => [
                ['is_dir' => true, 'path' => 'dirname'],
            ]],
            ['contents' => [
                ['is_dir' => false, 'path' => 'dirname/file'],
            ]],
            false
        );

        $result = $adapter->listContents('', true);
        $this->assertCount(2, $result);
        $this->assertEquals([], $adapter->listContents('', false));
    }

    /**
     * @dataProvider  dropboxProvider
     */
    public function testRename($adapter, $mock)
    {
        $mock->shouldReceive('move')->andReturn(['is_dir' => false, 'path' => 'something']);
        $this->assertTrue($adapter->rename('something', 'something'));
    }

    /**
     * @dataProvider  dropboxProvider
     */
    public function testRenameFail($adapter, $mock)
    {
        $mock->shouldReceive('move')->andReturnUsing(function () {
            throw new \Dropbox\Exception('Message');
        });

        $this->assertFalse($adapter->rename('something', 'something'));
    }

    /**
     * @dataProvider  dropboxProvider
     */
    public function testCopy($adapter, $mock)
    {
        $mock->shouldReceive('copy')->andReturn(['is_dir' => false, 'path' => 'something']);
        $this->assertTrue($adapter->copy('something', 'something'));
    }

    /**
     * @dataProvider  dropboxProvider
     */
    public function testCopyFail($adapter, $mock)
    {
        $mock->shouldReceive('copy')->andReturnUsing(function () {
            throw new \Dropbox\Exception('Message');
        });

        $this->assertFalse($adapter->copy('something', 'something'));
    }

    /**
     * @dataProvider  dropboxProvider
     */
    public function testGetClient($adapter, $mock)
    {
        $this->assertEquals($mock, $adapter->getClient());
    }
}
