#!/bin/sh
# @package      hubzero-chuse
# @file         chuse.tcl
# @copyright    Copyright (c) 2005-2020 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2005-2020 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#
#\
. /etc/environ.sh
#\
use -e -r rappture
#\
exec wish "$0" $*
# ----------------------------------------------------------------------
# wish executes everything from here on...

package require Img
package require RapptureGUI

proc expandPath {args} {
    set path ""
    set dirs [file split [lindex $args 0]]

    while {[llength $dirs] > 0} {
        set d [lindex $dirs 0]
        set dirs [lrange $dirs 1 end]
        if {[catch {file link [file join $path $d]} out] == 0} {
            # directory d is a link, follow it
            set outdirs [file split $out]
            if {[string compare "/" [lindex $outdirs 0]] == 0} {
                # directory leads back to root
                # clear path
                # reset dirs list
                set path ""
                set dirs $outdirs
            } else {
                # relative path for the link
                # prepend directory to dirs list
                set dirs [concat $outdirs $dirs]
            }
        } else {
            set path [file join $path $d]
        }
    }
    return $path
}

set installdir [file dirname [expandPath [info script]]]



source [file join $installdir cblist.tcl]
source [file join $installdir env_files.tcl]


option add *.font -*-helvetica-medium-r-normal-*-20-*
# option add *background #ffffff
 option add *bordercolor #ffffff
 option add *outline #ffffff

proc introHTML {} {
    return [format "
<html>
<body>
<p><br></p>
<p>Welcome to Chuse!</p>
<p><br></p>
<p>Browse through the environments to the left and check off
which ones you would like to activate. Once an environment has
been activated, all future terminal windows will have the
environment available. You can also deactivate environments
by unchecking them. All changes occur immediately and only affect
windows opened after the change has been made.</p>
<p><br></p>
<p><br></p>
</body>
</html>"]
}

proc formatHTML {fname ver desc help conflicts} {
    if {[string compare "" $ver] == 0} {
        set ver "?"
    }
    if {[string compare "" $conflicts] == 0} {
        set conflicts "None"
    }
    return [format "
<html>
<body>
<p><br></p>
<p>%s</p>
<p>Version: %s</p>
<p><br></p>
<p>%s</p>
<p><br></p>
<p>%s</p>
<p><br></p>
<p>Conflicts with: %s</P>
<p><br></p>
</body>
</html>" $fname $ver $desc $help $conflicts]
}

proc onChoose {name} {
    global conflictDB

    # find conflicting files, remove dup name
    set conflictChoice [lindex $conflictDB($name) 0]
    if {[string length $conflictChoice] != 0} {
        set conflictList $conflictDB($conflictChoice)
        set idx [lsearch $conflictList $name]
        foreach conflict [lreplace $conflictList $idx $idx] {
            .env.cblist unchoose_ele $conflict onUnChoose
        }
        exec /etc/environ sh use -r -p $name
    }

}

proc onUnChoose {name} {
    exec /etc/environ sh unuse -r -p $name
}

proc doSearch args {
    global searchStr defaultSearchStr

    if {[string length $searchStr] == 0 ||
        [string compare $searchStr $defaultSearchStr] == 0} {
        .search.e configure -foreground "#b3b3b3"
    } else {
        .search.e configure -foreground "#000000"
    }

    # show matching entries
    # enable/disable prev and next buttons
    if {[.env.cblist highlight_matching "$searchStr"] > 1} {
        .search.prevpb enable
        .search.nextpb enable
    } else {
        .search.prevpb disable
        .search.nextpb disable
    }

}

proc togglePushButtons {pbvar tags} {
    global pbvlist searchStr defaultSearchStr
    upvar #0 $pbvar inState

    # clear all previous tags
    .env.cblist active_tags_remove

    if {$inState == 1} {
        # unpush previously pushed buttons
        foreach pbv $pbvlist {
            if {[string compare $pbvar $pbv] != 0} {
                upvar #0 $pbv var
                set var 0
            }
        }
        # add to the active tags
        .env.cblist active_tags_add $tags
    }

    # show all environments tagged with selected tags
    .env.cblist show_tagged

    # update match highlighting
    set searchStr $searchStr
}

# tags: DEVEL MATHSCI RP EDITORS

wm title . chuse
wm withdraw .

set defaultSearchStr " Search "
set searchStr $defaultSearchStr

frame .search
entry .search.e -textvariable searchStr -background "#ffffff" -foreground "#b3b3b3"
bind .search.e <Button-1> {
    if {[string compare $searchStr $defaultSearchStr] == 0} {
        set searchStr ""
    }
}

# have to use a trace because .search.e's textvariable value does not seem to
# get updated until after the binding between .search.e and <KeyPress> is
# resolved, so the value sent to doSearch is incorrect.

trace add variable searchStr write doSearch

label .search.filtersl -text "Filters:"

set imagesDir [file join $installdir images]

# setup previous search button
set prevImg_active [image create photo -file [file join $imagesDir "green_left_arrow.png"]]
set prevImg_disabled [image create photo -file [file join $imagesDir "grey30_left_arrow.png"]]
Rappture::PushButton .search.prevpb \
    -onimage $prevImg_active \
    -offimage $prevImg_active \
    -disabledimage $prevImg_disabled \
    -command {
        after 250 {set prevPbVar 0}
        .env.cblist hilight_matching_element_prev
    } \
    -variable prevPbVar
Rappture::Tooltip::for .search.prevpb \
    "Go to previous match"
.search.prevpb disable

# setup next search button
set nextImg_active [image create photo -file [file join $imagesDir "green_right_arrow.png"]]
set nextImg_disabled [image create photo -file [file join $imagesDir "grey30_right_arrow.png"]]
Rappture::PushButton .search.nextpb \
    -onimage $nextImg_active \
    -offimage $nextImg_active \
    -disabledimage $nextImg_disabled \
    -command {
        after 250 {set nextPbVar 0}
        .env.cblist hilight_matching_element_next
    } \
    -variable nextPbVar
Rappture::Tooltip::for .search.nextpb \
    "Go to next match"
.search.nextpb disable

# setup editors filter
set editorImg [image create photo -file [file join $imagesDir "text-editor.png"]]
Rappture::PushButton .search.editorspb \
    -onimage $editorImg \
    -offimage $editorImg \
    -command {togglePushButtons editorPbVar EDITORS} \
    -variable editorPbVar
Rappture::Tooltip::for .search.editorspb \
    "Toggle to see editors"
lappend pbvlist editorPbVar

# setup development tools filter
set devtoolsImg [image create photo -file [file join $imagesDir "wrench.png"]]
Rappture::PushButton .search.devtoolspb \
    -onimage $devtoolsImg \
    -offimage $devtoolsImg \
    -command {togglePushButtons devPbVar DEVEL} \
    -variable devPbVar
Rappture::Tooltip::for .search.devtoolspb \
    "Toggle to see development tools"
lappend pbvlist devPbVar

# setup math / science tools filter
set mathImg [image create photo -file [file join $imagesDir "stock_insert-math-object.png"]]
Rappture::PushButton .search.mathpb \
    -onimage $mathImg \
    -offimage $mathImg \
    -command {togglePushButtons mathVar MATHSCI} \
    -variable mathVar
Rappture::Tooltip::for .search.mathpb \
    "Toggle to see math and science tools"
lappend pbvlist mathVar

frame .env -borderwidth 0

## create the details canvas
frame .details
Rappture::Scroller .details.scroller \
    -xscrollmode auto \
    -yscrollmode auto \
    -xscrollside bottom \
    -yscrollside right

Rappture::HTMLviewer .details.scroller.html -background #cfd9ed
.details.scroller contents .details.scroller.html

# create the checkbutton list
set onImage [image create photo -file [file join $imagesDir "cbon.gif"]]
set offImage [image create photo -file [file join $imagesDir "cboff.gif"]]
#cblist_create .env.cblist {.details.scroller.html load} \
    $onImage $offImage "#ffffff" "#efefef" "#cfd9ed"
Rappture::Checkboxlist .env.cblist \
    -command {.details.scroller.html load} \
    -onimage $onImage \
    -offimage $offImage \
    -bgcolors [list "#ffffff" "#efefef"] \
    -mouseovercolor "#cfd9ed" \
    -yscrollside left

# FIXME: should look at environment variables for active envs as well.
set activeEnvsFile [file join $env(HOME) ".environ"]
set activeEnvs ""

# FIXME: this check should be in parse_active_envs
if {![file exists $activeEnvsFile]} {
    # touch the .environ file if it doesn't exist
    # so we don't raise error when we try to activate
    # a new environment.
    # you see this in new workspaces
    close [open $activeEnvsFile a]
}

if {[file isfile $activeEnvsFile]} {
    set activeEnvs [parse_active_envs $activeEnvsFile]
}

set status ""
array set icons [list \
    MATHSCI $mathImg \
    DEVEL   $devtoolsImg \
    EDITORS $editorImg \
]

# conflictDB is a global var populated by parse_env_dir
foreach {fname conflict desc help ver tags} [eval parse_env_dir $env(ENVIRON_CONFIG_DIRS)] {
    set fnidx [lsearch $conflictDB($conflict) $fname]
    set conflictList [lreplace $conflictDB($conflict) $fnidx $fnidx]
    set detail [formatHTML $fname $ver $desc $help $conflictList]
    set searchName [file tail $fname]

    if {[lsearch $activeEnvs $searchName] == -1} {
        set status "off"
    } else {
        set status "on"
    }

    lappend tags $searchName

    set imageCategory [lindex $tags 0]

    set elementIcon ""
    if {[info exists icons($imageCategory)]} {
        set elementIcon $icons($imageCategory)
    }

   .env.cblist add_element $searchName $status $searchName $detail \
        onChoose onUnChoose $elementIcon $tags
}

.env.cblist show_tagged

# i'm not sure how to get the KeyPress-Up KeyPress-Down,
# KeyPress-Home, KeyPress-End, KeyPress-Prior, KeyPress-Next
# events down into the widget. they have to go through the
# cblist's scroller and down into cblist's canvas

pack .search -side bottom -fill x -expand 0
grid .search.e -in .search -row 0 -column 2 -columnspan 2 -sticky ew
grid .search.prevpb -in .search -row 0 -column 4 -sticky ew
grid .search.nextpb -in .search -row 0 -column 5 -sticky ew
grid .search.filtersl -in .search -row 0 -column 9 -columnspan 2 -sticky ew
grid .search.mathpb -in .search -row 0 -column 11 -sticky ew
grid .search.editorspb -in .search -row 0 -column 12 -sticky ew
grid .search.devtoolspb -in .search -row 0 -column 13 -sticky ew
pack .env -side left -fill both
pack .env.cblist -side left -fill both
pack .details -side left -fill both -expand 1
pack .details.scroller -side left -fill both -expand 1
pack .details.scroller.html -side left -fill both -expand 1

wm geometry . 500x320
wm deiconify .

update
.details.scroller.html load [introHTML]
