/**
 * @package     hubzero-vncproxy
 * @file		vncproxy.c
 * @author      Nicholas J. Kisseberth <nkissebe@purdue.edu>
 * @copyright   Copyright (c) 2010-2012 HUBzero Foundation, LLC.
 * @license     http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
 *
 * Copyright (c) 2010-2012 HUBzero Foundation, LLC.
 *
 * This file is part of: The HUBzero(R) Platform for Scientific Collaboration
 *
 * The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
 * software: you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * HUBzero is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * HUBzero is a registered trademark of HUBzero Foundation, LLC.
 */

/*
 * @TODO:
 *
 * Remove dependency on APR libraries, use libmysql directly
 * Don't use STDOUT file descriptor 
 *     redirect stdout to /dev/null
 *     replace STDOUT_FILENO with socket opened earlier
 * Use new heartbeat method (PID/starttime/boottime)
 * Remove portbase, record port in display table directly
 * Read DBD_DRIVER and DBD_PARAMS from configuration file
 * Make sure we are checking for errors consistently
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <apr_strings.h>
#include <apu.h>
#include <apr_dbd.h>
#include <getopt.h>
#include <libgen.h> 
#include <syslog.h>
#include <time.h>
#include <fcntl.h>
#include <netdb.h>
#include <unistd.h>
#include <sys/param.h>

int log_priority = LOG_WARNING;

#define MLOG_EMERG   __FILE__,__LINE__,LOG_EMERG
#define MLOG_ALERT   __FILE__,__LINE__,LOG_ALERT
#define MLOG_CRIT    __FILE__,__LINE__,LOG_CRIT
#define MLOG_ERR     __FILE__,__LINE__,LOG_ERR
#define MLOG_WARNING __FILE__,__LINE__,LOG_WARNING
#define MLOG_NOTICE  __FILE__,__LINE__,LOG_NOTICE
#define MLOG_INFO    __FILE__,__LINE__,LOG_INFO
#define MLOG_DEBUG   __FILE__,__LINE__,LOG_DEBUG

void
logmsg(char *file, int line, int priority, char *msg, ...)
{
	static char *priorities[8] = {"[emerg]", "[alert]", "[crit]", "[error]", "[warn]", "[notice]", "[info]", "[debug]" };
	va_list ap;
	char buffer[1500] = {0, 0};
	char pidstr[8];
	time_t t;
	struct tm tm_result;
	int len;
	static int syslog_priority = LOG_WARNING;

	if ((priority >= 0) && (priority < sizeof (priorities)) && (priority <= log_priority)) {

		if (time(&t) == -1) {
			t = 0;
		}

		buffer[0] = '[';

		if ((localtime_r(&t, &tm_result) == NULL) || (asctime_r(&tm_result, buffer + 1) == NULL)) {
			strcpy(buffer + 1, "Thu Jan 01 00:00:00 1970");
		}

		va_start(ap,msg);
		len = snprintf(buffer + 25, sizeof (buffer) - 25, "] %s %s(%d): [%d] ", priorities[priority], file, line, getpid());
			vsnprintf(buffer + 25 + len, sizeof (buffer) - 25 - len, msg, ap);
		va_end(ap);

		strcat(buffer, "\n");
		fprintf(stderr, buffer);
		fflush(stderr);
	}

	if ((syslog_priority >= 0) && (syslog_priority < sizeof (priorities)) && (priority <= syslog_priority)) {
	
		va_start(ap, msg);
		vsyslog(priority, msg, ap);
		va_end(ap);
	}
}

volatile sig_atomic_t gSignal = 0;

void
do_proxy_vnc_sighandler(int sig)
{
		gSignal = sig;
}

int
open_database(const apr_dbd_driver_t *driver, apr_pool_t *pool, char *dbd_params, apr_dbd_t **handle)
{
	int delay = 1;
	int tries = 1;

	while (apr_dbd_open(driver, pool, dbd_params, handle) != APR_SUCCESS) {
		if (tries > 6) {
			logmsg(MLOG_WARNING, "Unable to open database connection. Tried %d times. Aborting", tries);
			*handle = NULL;
			return -1;
		}

		logmsg(MLOG_WARNING, "Unable to open database connection. Retrying in %ds", delay);

		sleep(delay);

		delay *= 2;

		if (delay > 8) {
			delay = 8;
		}

		tries++;
	}

	return 0;
}

int
main(int argc, const char * const argv[])
{
	char progname[PATH_MAX] = {0};
	apr_status_t rv = 0;
	apr_pool_t *pool = NULL;
	int c = 0;
	int portbase = 4000;
	int bufsize = 8192;
	const char *token = NULL;
	char remoteip[INET6_ADDRSTRLEN] = { 0 };
	int sink = 0;
	char *dbd_driver = NULL;
	char *dbd_params = NULL;
	const apr_dbd_driver_t *driver = NULL;
	apr_dbd_t *handle = NULL;
	const char *etoken = NULL;
	const char *eremoteip = NULL;
	char query[1024] = { 0 };
	apr_dbd_results_t *res = NULL;
	apr_dbd_row_t *row = NULL;
	const char *value = NULL;
	int sessnum = 0;
	char username[33];
	char hostname[41];
	int dispnum = 0;
	int timeout = 0;
	int port = 0;
	struct timeval tv = { 0, 0 };
	double starttime = 0.0;
	int nrows = 0;
	int viewid = 0;
    struct addrinfo hints;
	struct addrinfo *hostaddr = NULL;
	struct addrinfo *tmpaddr = NULL;
	int fd = -1;
	int interval = 0;
	char *buffer1 = NULL;
    int  left1 = 0;
	char *buf1 = NULL;
	char *buffer2 = NULL;
    int  left2 = 0;
    char *buf2 = NULL;
    int nfds = 0;
	int stdin_flags = 0;
	int stdout_flags = 0;
	fd_set rfds, wfds, efds;
    int  len1 = 0;
    int  len2 = 0;
	int count = 0;
	double endtime = 0.0;
	sigset_t sigmask, orig_sigmask;
	int heartbeat = 1;
	time_t esttime = 0;
	time_t alarmtime = 0;
	struct timespec tout;
	int drain = 0;
    int quiet = 0;
	int verbose = 0;
	int debug = 0;
	struct sockaddr_storage sas;
	socklen_t saslength = sizeof(sas);
	long client[2] = { 0, 0 };
	long server[2] = { 0, 0 };
    int useIPv6=0;
    int useIPv4=0;
    int preferIPv4=0;
    int preferIPv6=0;
    int tryIPv4=0;
    int tryIPv6=0;

	rv = apr_app_initialize(&argc, &argv, NULL);

	esttime = time(NULL);

	strncpy(progname, basename((char *) argv[0]), sizeof (progname));

	openlog(progname, LOG_PID, LOG_USER);

	if (rv != APR_SUCCESS) {
		logmsg(MLOG_ERR, "Unable to initialize application");
		exit(1);
	}

	rv = atexit(apr_terminate);

	if (rv != 0) {
		logmsg(MLOG_ERR, "Failed to register apr_terminate() with atexit()");
		apr_terminate();
		exit(2);
	}

	rv = apr_pool_create(&pool, NULL);

	if (rv != APR_SUCCESS) {
		logmsg(MLOG_ERR, "Unable to create memory pool");
		exit(3);
	}

	rv = apr_dbd_init(pool);

	if (rv != APR_SUCCESS) {
		logmsg(MLOG_ERR, "Unable to initialize APR DBD environment");
		exit(4);
	}

	while (1) { 
		static struct option long_options[] = {
			{ "version", no_argument, 0, 'r' },
			{ "help", no_argument, 0, '?' },
			{ "verbose", no_argument, 0, 'v' },
			{ "quiet", no_argument, 0, 'q' },
			{ "portbase", required_argument, 0, 'p' },
			{ "buffersize", required_argument, 0, 'b' },
			{ "sslsink", no_argument, 0, 's' },
			{ "heartbeat", no_argument, 0, 'h' },
			{ "debug", no_argument, 0, 'd' },
			{ "useIPv4", no_argument, 0, '4' },
			{ "useIPv6", no_argument, 0, '6' },
			{ "preferIPv4", no_argument, 0, 'y' },
			{ "preferIPv6", no_argument, 0, 'z' },
			{ 0, 0, 0, 0 }
		};

		int option_index = 0;

		c = getopt_long(argc, (char **) argv, "r?vqp:b:shd46yz", long_options, &option_index);

		if (c == -1) 
			break;

		switch (c) {
			case 'r': fprintf(stdout, "vncproxy 2.2.3\n");
					  fprintf(stdout, "Copyright (c) 2010-2013 Purdue University\n");
					  fprintf(stdout, "This is free software: you can redistribute it and/or modify it under the terms\n");
					  fprintf(stdout, "of the GNU Lesser General Public License version 3 or later.\n");
					  fprintf(stdout, "<http://gnu.org/licenses/gpl.html>\n"); 
					  fprintf(stdout, "\n");
				  	  fprintf(stdout, "There is NO WARRANTY, to the extent permitted by law.\n");
					  fprintf(stdout, "\n");
					  fprintf(stdout, "Written by Nicholas J. Kisseberth.\n");
					  exit(0);
					  break;
			case '?': fprintf(stdout, "Usage: vncproxy [OPTION]... TOKEN\n");
					  fprintf(stdout, "Helper program for mod_vncproxy Apache module to connect\n");
				 	  fprintf(stdout, "and forward an Apache client socket to a HUBzero VNC server.\n");
					  fprintf(stdout, "\n");
					  fprintf(stdout, "TOKEN is the viewtoken from the HUBzero viewperm table to use to lookup\n");
					  fprintf(stdout, "which VNC server and display to connect to.\n");
					  fprintf(stdout, "\n");
					  fprintf(stdout, "Mandatory arguments to long options are mandatory for short options too.\n");
					  fprintf(stdout, "  -v, --verbose          print more information about connection to stderr\n");
					  fprintf(stdout, "  -q, --quiet            print no information about connection to stderr\n");
					  fprintf(stdout, "  -p, --portbase=PORT    set base VNC port to use (default 4000)\n");
					  fprintf(stdout, "  -b, --buffersize=SIZE  set data buffer size to use (default 8192)\n");  
					  fprintf(stdout, "  -s, --sink             throw away data from client until ssl packet\n");
					  fprintf(stdout, "  -h, --heartbeat        maintain heartbeat field in view table\n");
					  fprintf(stdout, "  -d, --debug            print debugging information to stderr\n");
					  fprintf(stdout, "\n");
					  fprintf(stdout, "If none of the following options are selected, host resolution is done\n");
					  fprintf(stdout, "in accordance with the current operating system defaults (typically \n");
					  fprintf(stdout, "something similar to --preferIPv6)\n");
					  fprintf(stdout, "  --useIPv4,    -4       lookup remote hosts using only IPv4\n");
					  fprintf(stdout, "  --useIPv6,    -6       lookup remote hosts using only IPv6\n");
					  fprintf(stdout, "  --preferIPv4, -y       lookup remote hosts using IPv4 first\n");
					  fprintf(stdout, "  --preferIPv6, -z       lookup remote hosts using IPv6 first\n");
					  fprintf(stdout, "\n");
					  fprintf(stdout, "  --help                 display this help and exit\n");
					  fprintf(stdout, "  --version              output version information and exit\n");
					  fprintf(stdout, "\n");
					  exit(0);
			case 'p': portbase = atoi(optarg); break;
			case 'b': bufsize = atoi(optarg); break;
			case 's': sink = 1; break;
			case 'h': heartbeat = 1; break;
			case 'v': verbose = 1; break;
			case 'q': quiet = 1; break;
			case 'd': debug = 1; break;
			case '4': useIPv4 = 1; break;
			case '6': useIPv6 = 1; break;
			case 'y': preferIPv4 = 1; break;
			case 'z': preferIPv6 = 1; break;
			default: break;
		}
	}

	if (optind < argc) {
		token = argv[optind++];
	}

	if (token == NULL || *token == 0 || optind < argc) {
		fprintf(stderr, "vncproxy: invalid option or token\n");
		fprintf(stderr, "Try `vncproxy --help` for more information.\n");
		exit(5);
	}

	if (quiet) {
		log_priority = LOG_WARNING;
	}
	else if (verbose && !debug) {
			log_priority = LOG_NOTICE;
	}
	else if	(debug && !verbose) {
			log_priority = LOG_INFO;
	}
	else if (debug && verbose) {
			log_priority = LOG_DEBUG;
	}
	else {
			log_priority = LOG_NOTICE;
	}

    if (useIPv4 + useIPv6 + preferIPv4 + preferIPv6 > 1) {
    	fprintf(stderr, "vncproxy: conflicting options\n");
    	fprintf(stderr, "Use only one of --useIPv4, --useIPv6, --preferIPv4, --preferIPv6\n");
    	exit(6);
    }

	sigemptyset(&sigmask);
	sigaddset(&sigmask, SIGHUP);
	sigaddset(&sigmask, SIGTERM);
	sigaddset(&sigmask, SIGINT);
	sigaddset(&sigmask, SIGQUIT);

	if (signal(SIGHUP, do_proxy_vnc_sighandler) == SIG_ERR) {
		logmsg(MLOG_ERR, "Failed to assign signal handler to SIGHUP");
		exit(7);
	}

	if (signal(SIGTERM, do_proxy_vnc_sighandler) == SIG_ERR) {
		logmsg(MLOG_ERR, "Failed to assign signal handler to SIGTERM");
		exit(8);
	}

	if (signal(SIGINT, do_proxy_vnc_sighandler) == SIG_ERR) {
		logmsg(MLOG_ERR, "Failed to assign signal handler to SIGINT");
		exit(9);
	}

	if (signal(SIGQUIT, do_proxy_vnc_sighandler) == SIG_ERR) {
		logmsg(MLOG_ERR, "Failed to assign signal handler to SIGINT");
		exit(10);
	}

	logmsg(MLOG_INFO, "Starting vncproxy");

	dbd_driver = getenv("DBD_DRIVER");

	if ((dbd_driver == NULL) || (*dbd_driver == 0)) {
		dbd_driver = "mysql";
	}

	dbd_driver = apr_pstrdup(pool, dbd_driver);

	if (dbd_driver == NULL) {
		logmsg(MLOG_ERR, "Unable to allocate memory for DBD_DRIVER variable");
		exit(11);
	}

	dbd_params = getenv("DBD_PARAMS");

	if ((dbd_params == NULL) || (*dbd_params == 0)) {
		dbd_params = "";
	}

	dbd_params = apr_pstrdup(pool, dbd_params);

	if (dbd_params == NULL) {
		logmsg(MLOG_ERR, "Unable to allocate memory for DBD_PARAMS variable");
		exit(12);
	}

	rv = apr_dbd_get_driver(pool, dbd_driver, &driver);

	if (rv != APR_SUCCESS) {
		logmsg(MLOG_ERR, "Unable to load APR_DBD driver [%s]", dbd_driver);
		exit(13);
	}

	// free(dbd_driver);

	rv = open_database(driver, pool, dbd_params, &handle);

	if (rv != 0) {
		logmsg(MLOG_ERR, "Unable to open database to lookup token");
		exit(14);
	}

	etoken = apr_dbd_escape(driver, pool, token, handle);

	if (etoken == NULL) {
		logmsg(MLOG_ERR, "Unable to build query safe token string");
		apr_dbd_close(driver, handle);
		exit(15);
	}

	getpeername(STDIN_FILENO, (struct sockaddr *) &sas, &saslength);

	if (sas.ss_family == AF_INET) {
		inet_ntop(AF_INET, &((struct sockaddr_in *)&sas)->sin_addr, remoteip, sizeof(remoteip));
	} else if (sas.ss_family == AF_INET6) {
		inet_ntop(AF_INET6, &((struct sockaddr_in6 *)&sas)->sin6_addr, remoteip, sizeof(remoteip));

		if (strncmp(remoteip,"::ffff:",7) == 0) {
			memmove(remoteip,remoteip+7,strlen(remoteip+7)+1);
		}
	}
	else {
		strncpy(remoteip,"<unknown address>",sizeof(remoteip));
	}

	remoteip[sizeof(remoteip)-1] = 0;

	eremoteip = apr_dbd_escape(driver, pool, remoteip, handle);

	if (eremoteip == NULL) {
		logmsg(MLOG_ERR, "Unable to build query safe remoteip string");
		apr_dbd_close(driver, handle);
		exit(16);
	}

	rv = snprintf(query, sizeof (query), "SELECT viewperm.sessnum,viewuser,display.hostname,session.dispnum,"
		" session.timeout,portbase"
		" FROM viewperm"
		" JOIN display ON viewperm.sessnum = display.sessnum"
		" JOIN session ON session.sessnum = display.sessnum"
		" JOIN host ON display.hostname = host.hostname"
		" WHERE viewperm.viewtoken='%s' LIMIT 1;", etoken);

	if (rv < 0 || rv >= sizeof (query)) {
		logmsg(MLOG_ERR, "Failed to build query string");
		apr_dbd_close(driver, handle);
		exit(17);
	}

	rv = apr_dbd_select(driver, pool, handle, &res, query, 0);

	logmsg(MLOG_DEBUG, query);

	row = NULL;

	if ((rv != APR_SUCCESS) || (apr_dbd_get_row(driver, pool, res, &row, -1) != 0)) {
		logmsg(MLOG_WARNING, "No database entry found for token [%s]", token);
		apr_dbd_close(driver, handle);
		exit(18);
	}

	value = apr_dbd_get_entry(driver, row, 0);

	if (value == NULL) {
		logmsg(MLOG_ERR, "Unable to read sessnum column from result row");
		apr_dbd_close(driver, handle);
		exit(19);
	}

	sessnum = atoi(value);

	if (sessnum <= 0) {
		logmsg(MLOG_ERR, "Invalid sessnum column read from result row");
		apr_dbd_close(driver, handle);
		exit(20);
	}

	value = apr_dbd_get_entry(driver, row, 1);

	if (value == NULL) {
		logmsg(MLOG_ERR, "Unable to read viewuser column from result row");
		apr_dbd_close(driver, handle);
		exit(21);
	}

	strncpy(username, value, sizeof (username));

	value = apr_dbd_get_entry(driver, row, 2);

	if (value == NULL) {
		logmsg(MLOG_ERR, "Unable to read hostname column from result row");
		apr_dbd_close(driver, handle);
		exit(22);
	}

	strncpy(hostname, value, sizeof (hostname));

	value = apr_dbd_get_entry(driver, row, 3);

	if (value == NULL) {
		logmsg(MLOG_ERR, "Unable to read dispnum column from result row");
		apr_dbd_close(driver, handle);
		exit(23);
	}

	dispnum = atoi(value);

	if (dispnum <= 0) {
		logmsg(MLOG_ERR, "Invalid dispnum column read from result row");
		apr_dbd_close(driver, handle);
		exit(24);
	}

	value = apr_dbd_get_entry(driver, row, 4);

	if (value == NULL) {
		logmsg(MLOG_ERR, "Unable to read timeout column from result row");
		apr_dbd_close(driver, handle);
		exit(25);
	}

	timeout = atoi(value);

	if (timeout <= 0) {
		logmsg(MLOG_ERR, "Invalid timeout column read from result row");
		apr_dbd_close(driver, handle);
		exit(26);
	}

	port = dispnum + portbase;

	logmsg(MLOG_NOTICE, "Map %s@%s for %s to %s:%d", username, remoteip, token, hostname, port);

	rv = apr_dbd_get_row(driver, pool, res, &row, -1); // clears connection

	if (rv != -1) {
		logmsg(MLOG_ERR, "Database transaction not finished when expected.");
		apr_dbd_close(driver, handle);
		exit(27);
	}

	rv = gettimeofday(&tv, NULL);

	if (rv == 0) {
		starttime = tv.tv_sec + (tv.tv_usec / 1000000.0);
	} else {
		starttime = 0;
	}

	rv = snprintf(query, sizeof (query), "UPDATE session SET accesstime=NOW() WHERE sessnum = '%d';", sessnum);

	if (rv < 0 || rv >= sizeof (query)) {
		logmsg(MLOG_ERR, "Failed to build query string");
		apr_dbd_close(driver, handle);
		exit(28);
	}

	rv = apr_dbd_query(driver, handle, &nrows, query);

	logmsg(MLOG_DEBUG, "%s", query);

	if (rv != APR_SUCCESS) {
		logmsg(MLOG_ERR, "Failed to update session accesstime");
		apr_dbd_close(driver, handle);
		exit(29);
	}

	memset(&hints, 0, sizeof(hints));

	hints.ai_family = PF_UNSPEC;
	hints.ai_socktype = SOCK_STREAM;
	hints.ai_flags |= AI_CANONNAME;

	rv = getaddrinfo(hostname, NULL, &hints, &hostaddr);

	if (rv != 0) {
		logmsg(MLOG_ERR, "Unable to get host lookup information for %s", hostname);
		exit(30);
	}

    tmpaddr = hostaddr;

    if (useIPv6 || preferIPv6) {
    	tryIPv4 = 0;
    	tryIPv6 = 1;
    } 
    else if (useIPv4 || preferIPv4) {
    	tryIPv4 = 1;
    	tryIPv6 = 0;
    }
    else {
    	tryIPv4 = 1;
    	tryIPv6 = 1;
    }

	while(tmpaddr) {

		if (((tmpaddr->ai_family == AF_INET6) && tryIPv6) || ((tmpaddr->ai_family == AF_INET) && tryIPv4))
		{
    		if ((fd = socket(tmpaddr->ai_family, SOCK_STREAM, 0)) < 0) {
				logmsg(MLOG_ERR,"socket(AF=%d, SOCK_STREAM:=%d, 0) %s [%d]", tmpaddr->ai_family,SOCK_STREAM,strerror(errno), errno);
    		}
			else {		
				((struct sockaddr_in *) tmpaddr->ai_addr)->sin_port = htons(port);

				do {
					errno = 0;
					rv = connect(fd, tmpaddr->ai_addr, tmpaddr->ai_addrlen);
				}
				while (rv != 0 && errno == EINTR);

				logmsg(MLOG_NOTICE,"connect(%d, AF=%d, %s:%d, %d): %s [%d]", fd, tmpaddr->ai_family, hostname, port, tmpaddr->ai_addrlen, strerror(errno), errno);

	 			if (rv == 0)
 					break;
		
		        shutdown(fd, SHUT_RDWR);
        		close(fd);
    		}
   		}

    	tmpaddr = tmpaddr->ai_next;

    	if (tmpaddr == NULL) {
    		if (preferIPv4 && tryIPv4) {
    			tryIPv4 = 0;
    			tryIPv6 = 1;
    			tmpaddr = hostaddr;
    		}
    		else if (preferIPv6 && tryIPv6) {
    			tryIPv4 = 1;
    			tryIPv6 = 1;
    			tmpaddr = hostaddr;
    		}
    		else
    		{
    			tryIPv4 = 0;
    			tryIPv6 = 0;
    		}
    	}
	}

	freeaddrinfo(hostaddr);

	if (rv != 0) {
		logmsg(MLOG_ERR,"Failed to connect to host %s.", hostname);
		exit(31);
	}

	dup2(fd,1);
	close(fd);

	if (sink) {
		logmsg(MLOG_DEBUG, "Sinking input until possible SSL handshake arrives");

		while (1) {
			rv = read(STDIN_FILENO, &c, 1);

			if (rv == 0) {
				break;
			}

			if (c == 16) {
				write(STDOUT_FILENO, &c, 1);
				logmsg(MLOG_DEBUG, "Possible start of SSL handshake found");
				break;
			}

			continue;
		}

		if (rv == -1) {
			logmsg(MLOG_ERR, "Sinking until SSL handshake detected failed");
			exit(32);
		}
	}

	rv = snprintf(query, sizeof (query), "INSERT INTO view(sessnum,username,remoteip,start,heartbeat) "
		" VALUE (%d,'%s','%s',now(),now());", sessnum, username, eremoteip);

	if (rv < 0 || rv >= sizeof (query)) {
		logmsg(MLOG_ERR, "Failed to build query string");
		apr_dbd_close(driver, handle);
		exit(33);
	}

	rv = apr_dbd_query(driver, handle, &nrows, query);

	logmsg(MLOG_DEBUG, "%s", query);

	if (rv != APR_SUCCESS) {
		logmsg(MLOG_ERR, "Failed to insert view record");
		apr_dbd_close(driver, handle);
		exit(34);
	}

	rv = snprintf(query, sizeof (query), "SELECT LAST_INSERT_ID();");

	if (rv < 0 || rv >= sizeof (query)) {
		logmsg(MLOG_ERR, "Failed to build query string");
		apr_dbd_close(driver, handle);
		exit(35);
	}

	rv = apr_dbd_select(driver, pool, handle, &res, query, 0);

	logmsg(MLOG_DEBUG, "%s", query);

	if (rv != APR_SUCCESS) {
		logmsg(MLOG_ERR, "Failed to query last insert id");
		apr_dbd_close(driver, handle);
		exit(36);
	}

	row = NULL;

	if (apr_dbd_get_row(driver, pool, res, &row, -1) != APR_SUCCESS) {
		logmsg(MLOG_ERR, "Fail to retrieve last insert id");
		apr_dbd_close(driver, handle);
		exit(37);
	}

	value = apr_dbd_get_entry(driver, row, 0);

	if (value == NULL) {
		logmsg(MLOG_ERR, "Unable to read insert id column from result row");
		apr_dbd_close(driver, handle);
		exit(38);
	}

	viewid = atoi(value);

	if (viewid <= 0) {
		logmsg(MLOG_ERR, "Invalid insert id column read from result row");
		apr_dbd_close(driver, handle);
		exit(39);
	}

	logmsg(MLOG_NOTICE, "View %d (%s@%s) started", viewid, username, remoteip);

	rv = apr_dbd_get_row(driver, pool, res, &row, -1); // clears connection

	if (rv != -1) {
		logmsg(MLOG_ERR, "Database transaction not finished when expected");
		apr_dbd_close(driver, handle);
		exit(40);
	}

	rv = apr_dbd_close(driver, handle);

	if (rv != APR_SUCCESS) {
		logmsg(MLOG_ERR, "Unable to close database connection");
		apr_dbd_close(driver, handle);
		exit(41);
	}

	interval = (int) (timeout * 0.40);

	alarmtime = esttime + interval;

	if (bufsize <= 0)
		bufsize = 8192;

	buffer1 = malloc(bufsize);

    if (buffer1 == NULL) {
		logmsg(MLOG_ERR, "Failed to allocate memory for buffer1");
		exit(42);
	}

	left1 = bufsize;
	buf1 = buffer1;

	buffer2 = malloc(bufsize);

    if (buffer2 == NULL) {
		logmsg(MLOG_ERR, "Failed to allocate memory for buffer2");
		free(buffer1);
		exit(43);
	}

	left2 = bufsize;
	buf2 = buffer2;

	nfds = MAX(STDIN_FILENO, STDOUT_FILENO) + 1;

	stdin_flags = fcntl(STDIN_FILENO, F_GETFD);
	stdout_flags = fcntl(STDOUT_FILENO, F_GETFD);
	
	stdin_flags &= ~(O_RDONLY | O_WRONLY | O_APPEND);
	stdout_flags &= ~(O_RDONLY | O_WRONLY | O_APPEND);

	stdin_flags |= O_RDWR | O_NONBLOCK;
	stdout_flags |= O_RDWR | O_NONBLOCK;

	fcntl(STDIN_FILENO, F_SETFD, stdin_flags);
	fcntl(STDOUT_FILENO, F_SETFD, stdout_flags);

	sigprocmask(SIG_BLOCK, &sigmask, &orig_sigmask);

	tout.tv_sec = 1;
	tout.tv_nsec = 0;

	while(1) {
	
		sigprocmask(SIG_BLOCK, &sigmask, NULL);

		if (gSignal) {
			logmsg(MLOG_DEBUG,"caught terminal signal");
			break;
		}

		// we track esttime as an estimated current time, it should be verified before acting.
		// it is of dubious use, but the idea is to limit the number of calls to time()
		// inside the inner transfer loop

		if (heartbeat && (esttime >= alarmtime)) {

			esttime = time(NULL);

			if (esttime >= alarmtime) {
				logmsg(MLOG_INFO, "Refreshing heartbeat for view %d", viewid);

				rv = snprintf(query, sizeof (query), "UPDATE view SET heartbeat=NOW() WHERE viewid='%d' LIMIT 1;", viewid);

				if (rv <= 0 || rv >= sizeof (query)) {
					logmsg(MLOG_ERR, "Failed to build view heartbeat update query string");
				}
				else if (open_database(driver, pool, dbd_params, &handle) != 0) {
					logmsg(MLOG_ERR, "Unable to open database to update heartbeat of view [%d]", viewid);
				}
				else {
					logmsg(MLOG_DEBUG, "%s", query);

					rv = apr_dbd_query(driver, handle, &nrows, query);

					if (rv != APR_SUCCESS) {
						logmsg(MLOG_ERR, "Failed to update view heartbeat");
					}

					apr_dbd_close(driver, handle);

					if (rv != APR_SUCCESS) {
						logmsg(MLOG_ERR, "Unable to close database connection after heartbeat update");
					}
				}

				alarmtime = esttime + interval;
			}
		}

		FD_ZERO(&rfds);
		FD_ZERO(&wfds);
		FD_ZERO(&efds);
		
		if ((left1 > 0) && (!drain)) { // room in read buffer for STDIN_FILENO, ready to read STDIN_FILENO
			logmsg(MLOG_DEBUG, "ready to read %d from %d", left1,STDIN_FILENO);
			FD_SET(STDIN_FILENO, &rfds);
		}

		if ((left2 > 0) && (!drain)) { // room in read buffer for STDOUT_FILENO, ready to read STDOUT_FILENO
			logmsg(MLOG_DEBUG, "ready to read %d from %d", left2, STDOUT_FILENO);
			FD_SET(STDOUT_FILENO, &rfds);
		}

		if (len1 > 0) { // data in read buffer for STDIN_FILENO, ready to write to STDOUT_FILENO
			logmsg(MLOG_DEBUG, "ready to write %d to %d", len1,STDOUT_FILENO);
			FD_SET(STDOUT_FILENO, &wfds);
		}

		if (len2 > 0) { // data in read buffer for STDOUT_FILENO, ready to write to STDIN_FILENO
			logmsg(MLOG_DEBUG, "ready to write %d to %d", len2,STDIN_FILENO);
			FD_SET(STDIN_FILENO, &wfds);
		}

		if (!drain) {
			FD_SET(STDOUT_FILENO, &efds); // always on look out for OOB data
			FD_SET(STDIN_FILENO, &efds); // always on look out for OOB data
		}

		logmsg(MLOG_DEBUG, "pselect() waiting %d s", tout.tv_sec);

		tout.tv_nsec = 0;

		rv = pselect(nfds, &rfds, &wfds, &efds, &tout, &orig_sigmask);

		tout.tv_sec = 1;

		esttime++;

		if (rv == 0) {
			esttime = time(NULL);

			if (alarmtime > esttime) {
				tout.tv_sec = alarmtime - esttime;
			}

			continue;
		}


		if (rv < 0 && errno == EINTR)
			continue;

		if (rv < 0) {
			logmsg(MLOG_ERR,"pselect() failed: %d", errno);
			break;
		}

		if (FD_ISSET(STDIN_FILENO, &efds)) { // we pass OOB data from STDIN_FILENO to STDOUT_FILENO one byte at a time
			count = recv(STDIN_FILENO, &c, 1, MSG_OOB);

			logmsg(MLOG_DEBUG,"OOB recv(STDIN_FILENO=%d) = %d/%x",STDIN_FILENO,count,c);

			if (count > 0) {
				client[0] += count;
				do {
					count = send(STDOUT_FILENO, &c, 1, MSG_OOB);
				}
				while (count == -1 && (errno == EINTR) || (errno == EAGAIN));

				if (count > 0) {
					server[1] += count;
				}

				logmsg(MLOG_DEBUG,"OOB send(STDOUT_FILENO=%d,%x) = %d",STDOUT_FILENO,c,count);
			}
			else if ((count < 0) && (errno == EINTR)) {
				continue;
			}
			else if ((count < 0) && (errno != EAGAIN)) {
				break;
			}
		}

		if (FD_ISSET(STDOUT_FILENO, &efds)) { // we pass OOB data from STDOUT_FILENO to STDIN_FILENO one byte at a time
			count = recv(STDOUT_FILENO, &c, 1, MSG_OOB);

			logmsg(MLOG_DEBUG,"OOB recv(STDOUT_FILENO=%d) = %d/%x",STDOUT_FILENO,count,c);

			if (count > 0) {
				server[0] += count;
				do {
					count = send(STDIN_FILENO, &c, 1, MSG_OOB);
				}
				while (count == -1 && (errno == EINTR) || (errno == EAGAIN));

				if (count > 0) {
					client[1] += count;
				}

				logmsg(MLOG_DEBUG,"OOB send(STDIN_FILENO=%d,%x) = %d",STDIN_FILENO,c,count);
			}
			else if ((count < 0) && (errno == EINTR)) {
				continue;
			}
			else if ((count < 0) && (errno != EAGAIN)) {
				break;
			}
		}

		if (FD_ISSET(STDIN_FILENO, &rfds)) { // read available data from STDIN_FILENO into remaining space in buffer1

			if (left1 > 0) {
				count = read(STDIN_FILENO, buf1+len1, left1);

				logmsg(MLOG_DEBUG,"read(%d,%p,%d) = %d", STDIN_FILENO, buf1+len1, left1, count);

				if (count > 0) {
					client[0] += count;
					left1 -= count;
					len1 += count;
				}
				else if (count < 0) {

					if (errno == EINTR) {
						continue;
					}

					if (errno != EAGAIN) {
						if ((errno == ETIMEDOUT) || (errno == EPIPE) || (errno == EHOSTUNREACH) || (errno == ECONNRESET) ||
							(errno == ENETUNREACH) || (errno == ECONNREFUSED)) {
							logmsg(MLOG_NOTICE, "read(%d,%p,%d): %s [%d]", STDIN_FILENO, buf1+len1, left1, strerror(errno), errno);
						}
						else {
							logmsg(MLOG_ERR, "read(%d,%p,%d): %s [%d]", STDIN_FILENO, buf1+len1, left1, strerror(errno), errno);
						}
						drain = 1;
					}
				}
				else if (count == 0) {
					logmsg(MLOG_INFO, "read(%d,%p,%d): EOF", STDIN_FILENO, buf1+len1, left1);
					drain = 1;
				}
			}
		}

		if (FD_ISSET(STDOUT_FILENO, &wfds)) { // write as much data as possible from buffer1 to STDOUT_FILENO

			if (len1 > 0) {
				count = write(STDOUT_FILENO, buf1, len1);

				logmsg(MLOG_DEBUG,"write(%d,%p,%d) = %d", STDOUT_FILENO, buf1, len1, count);

				if (count > 0) {
					server[1] += count;
					buf1 += count;
					len1 -= count;

					if (len1 == 0) {
						buf1 = buffer1;
						left1 = bufsize;
					}
				}
				else if (count < 0) {

					if (errno == EINTR) { 
						continue;
					}

					if (errno != EAGAIN) {
						if ((errno == ETIMEDOUT) || (errno == EPIPE) || (errno == EHOSTUNREACH) || (errno == ECONNRESET) ||
							(errno == ENETUNREACH) || (errno == ECONNREFUSED)) {
							logmsg(MLOG_NOTICE, "write(%d,%p,%d): %s [%d]", STDOUT_FILENO, buf1, len1, strerror(errno), errno);
						}
						else {
							logmsg(MLOG_ERR, "write(%d,%p,%d): %s [%d]", STDOUT_FILENO, buf1, len1, strerror(errno), errno);
						}
						drain = 1;
						buf1 = buffer1;
						len1 = 0;
						left1 - bufsize;
					}
				}
				else if (count == 0) {
					logmsg(MLOG_DEBUG, "write(%d,%p,%d): No error, but nothing written", STDOUT_FILENO, buf1, len1);
				}
			}
		}

		if (FD_ISSET(STDOUT_FILENO, &rfds)) { // read available data from STDOUT_FILENO into remaining space in buffer2

			if (left2 > 0) {
				count = read(STDOUT_FILENO, buf2+len2, left2);

				logmsg(MLOG_DEBUG,"read(%d,%p,%d) = %d", STDOUT_FILENO, buf2+len2, left2, count);

				if (count > 0) {
					server[0] += count;
					left2 -= count;
					len2 += count;
				}
				else if (count < 0) {
	
					if (errno == EINTR) {
						continue;
					}

					if (errno != EAGAIN) {
						if ((errno == ETIMEDOUT) || (errno == EPIPE) || (errno == EHOSTUNREACH) || (errno == ECONNRESET) ||
							(errno == ENETUNREACH) || (errno == ECONNREFUSED)) {
							logmsg(MLOG_NOTICE, "read(%d,%p,%d): %s [%d]", STDOUT_FILENO, buf2+len2, left2, strerror(errno), errno);
						}
						else {
							logmsg(MLOG_ERR, "read(%d,%p,%d): %s [%d]", STDOUT_FILENO, buf2+len2, left2, strerror(errno), errno);
						}
						drain = 1;
					}

				}
				else if (count == 0) {
					logmsg(MLOG_INFO, "read(%d,%p,%d): EOF", STDOUT_FILENO, buf2+len2, left2);
					drain = 1;
				}
			}
		}

		if (FD_ISSET(STDIN_FILENO, &wfds)) { // write as much data as possible from buffer2 to STDIN_FILENO
			if (len2 > 0) {
				count = write(STDIN_FILENO, buf2, len2);

				logmsg(MLOG_DEBUG,"write(%d,%p,%d) = %d", STDIN_FILENO, buf2, len2, count);

				if (count > 0) {
					client[1] += count;
					buf2 += count;
					len2 -= count;

					if (len2 == 0) {
						buf2 = buffer2;
						left2 = bufsize;
					}
				}
				else if (count < 0) {

					if (errno == EINTR) {
						continue;
					}

					if (errno != EAGAIN) {
						if ((errno == ETIMEDOUT) || (errno == EPIPE) || (errno == EHOSTUNREACH) || (errno == ECONNRESET) ||
							(errno == ENETUNREACH) || (errno == ECONNREFUSED)) {
							logmsg(MLOG_NOTICE, "write(%d,%p,%d): %s [%d]", STDIN_FILENO, buf2, len2, strerror(errno), errno);
						}
						else {
							logmsg(MLOG_ERR, "write(%d,%p,%d): %s [%d]", STDIN_FILENO, buf2, len2, strerror(errno), errno);
						}
						drain = 1;
						buf2 = buffer2;
						len2 = 0;
						left2 - bufsize;
					}
				}
				else if (count == 0) {
					logmsg(MLOG_DEBUG, "write(%d,%p,%d): No error, but nothing written", STDIN_FILENO, buf2, len2);
				}
			}
		}

		if (drain && (len1 == 0) && (len2 == 0)) {
			break;
		}
	}

	shutdown(STDOUT_FILENO, SHUT_RDWR);
	shutdown(STDIN_FILENO, SHUT_RDWR);
	close(STDIN_FILENO);
	close(STDOUT_FILENO);
	free(buffer1);
	free(buffer2);

	rv = gettimeofday(&tv, NULL);

	if (rv == 0) {
		endtime = tv.tv_sec + (tv.tv_usec / 1000000.0);
	} else {
		endtime = 0;
	}

	if (starttime == 0) {
		starttime = endtime;
	}

	if (endtime < starttime) {
		endtime = starttime;
	}

	logmsg(MLOG_NOTICE, "View %d (%s@%s) ended after %lf seconds. ctx[%d/%d] stx[%d/%d]",
		viewid, username, remoteip, endtime - starttime, server[1], client[0], client[1], server[0]);
	
	if (open_database(driver, pool, dbd_params, &handle) != 0) {
		logmsg(MLOG_ERR, "Unable to open database to record end of view");
		exit(44);
	}

	rv = snprintf(query, sizeof (query), "INSERT INTO viewlog(sessnum,username,remoteip,time,duration) "
		"SELECT sessnum, username, remoteip, start, %lf "
		"FROM view WHERE viewid='%d'", (double) (endtime - starttime), viewid);

	if (rv < 0 || rv >= sizeof (query)) {
		logmsg(MLOG_ERR, "Failed to build viewlog insert query string");
		exit(45);
	}

	logmsg(MLOG_DEBUG, "%s", query);

	rv = apr_dbd_query(driver, handle, &nrows, query);

	if (rv != APR_SUCCESS) {
		logmsg(MLOG_ERR, "Failed to insert viewlog record");
		exit(46);
	}

	snprintf(query, sizeof (query), "UPDATE session JOIN view ON session.sessnum=view.sessnum "
		"SET session.accesstime=now() "
		"WHERE viewid='%d'", viewid);

	if (rv < 0 || rv >= sizeof (query)) {
		logmsg(MLOG_ERR, "Failed to build session update accesstime query string");
		exit(47);
	}

	logmsg(MLOG_DEBUG, "%s", query);

	rv = apr_dbd_query(driver, handle, &nrows, query);

	if (rv != APR_SUCCESS) {
		logmsg(MLOG_ERR, "Failed to update session accesstime record");
		exit(48);
	}

	rv = snprintf(query, sizeof (query), "DELETE FROM view WHERE viewid='%d'", viewid);

	if (rv < 0 || rv >= sizeof (query)) {
		logmsg(MLOG_ERR, "Failed to build view deletion query string");
		exit(49);
	}

	logmsg(MLOG_DEBUG, "%s", query);

	rv = apr_dbd_query(driver, handle, &nrows, query);

	if (rv != APR_SUCCESS) {
		logmsg(MLOG_ERR, "Failed to delete view record");
		exit(50);
	}

	rv = apr_dbd_close(driver, handle);

	if (rv != APR_SUCCESS) {
		logmsg(MLOG_ERR, "Unable to close database connection");
		exit(51);
	}

	logmsg(MLOG_INFO, "Exiting vncproxy normally");
	exit(0);
}
