/*
 * @package      hubzero-telequotad
 * @file         telequotad.c
 * @author       Rick Kennell <kennell@purdue.edu>
 * @author       Nicholas J. Kisseberth <nkissebe@purdue.edu>
 * @copyright    Copyright (c) 2005-2014 HUBzero Foundation, LLC.
 * @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
 *
 * Copyright (c) 2005-2014 HUBzero Foundation, LLC.
 *
 * This file is part of: The HUBzero(R) Platform for Scientific Collaboration
 *
 * The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
 * software: you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * HUBzero is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * HUBzero is a registered trademark of HUBzero Foundation, LLC.
 */

#define _GNU_SOURCE

#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <stdint.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <stdarg.h>
#include <errno.h>

#include <syslog.h>

#include <sys/types.h>
#include <sys/param.h>
#if 0
#include <linux/quota.h>
#define CONFFILE "/etc/telequotad.conf"
#else
#include <sys/quota.h>
#define CONFFILE "/etc/telequotad.conf"
#endif

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

#include <pwd.h>
#include <sys/stat.h>


struct mem_dqblk {
  uint64_t dqb_bhardlimit;
  uint64_t dqb_bsoftlimit;
  uint64_t dqb_curspace;
  uint64_t dqb_ihardlimit;
  uint64_t dqb_isoftlimit;
  uint64_t dqb_curinodes;
  uint64_t dqb_btime;
  uint64_t dqb_itime;
  uint32_t dqb_valid;
};

struct devent {
  int num;
  char *name;
  char *device;
  struct devent *next; 
};
static struct devent *device_cache = NULL;


static int debug_flag = 0;
static char config_file[] = CONFFILE;
static int64_t default_hardspace = 0;
static int64_t default_softspace = 0;
static int64_t default_hardinode = 0;
static int64_t default_softinode = 0;
static char purgecmd[MAXPATHLEN] = "";
static int listen_port = 0;
static int sock4 = 0;
static int sock6 = 0;

struct allowed_addr {
  struct sockaddr_storage addr;
  struct allowed_addr *next;
};

struct allowed_addr *allowed_list = NULL;


static char option_user[100] = "";
static char option_degree[100] = "";
static int64_t option_softspace = 0;
static int64_t option_hardspace = 0;
static int64_t option_softfiles = 0;
static int64_t option_hardfiles = 0;

static char result_qstat[100] = "";
static int64_t result_softspace = 0;
static int64_t result_hardspace = 0;
static int64_t result_curspace = 0;
static int64_t result_softfiles = 0;
static int64_t result_hardfiles = 0;
static int64_t result_curfiles = 0;
static int result_spacetime = 0;
static int result_filestime = 0;
static int result_timeremaining = 0;

void parse_args(int argc, char *argv[]);
void parse_config();
void parse_line(char *line, int lineno);
void parse_allow(char **, int lineno);
void parse_default(char **, int lineno);
void parse_purgecmd(char **, int lineno);
void parse_listen(char **, int lineno);

void print_config();

void setup_listener();
void handle_connections();
void service_connection(int fd);
char *parse_command(char *line, int fd);

char *parse_getquota(char **pp);
char *parse_setquota(char **pp);
char *parse_purge(char **pp, int fd);

char *do_getquota(char *device, int uid);
char *do_setquota(char *device, int uid);

char *get_device(int devnum);
void  scan_devices(void);

char orig_command[1000] = "";
void printcmd(void);

int main(int argc, char *argv[])
{
  parse_args(argc, argv);

  if (!debug_flag) {
    openlog("telequotad", LOG_PID, LOG_DAEMON);
    daemon(0,0);
  }

  parse_config();
  print_config();
  scan_devices();
  setup_listener();

  handle_connections();
}

void logmsg(int prio, const char *format, ...)
{
  va_list ap;
  va_start(ap, format);
  if (debug_flag) {
    if (prio == LOG_ERR) {
      fprintf(stderr,"ERROR: ");
    } else if (prio = LOG_INFO) {
      fprintf(stderr,"INFO: ");
    }
    vfprintf(stderr,format,ap);
  } else {
    vsyslog(prio,format,ap);
  }
  va_end(ap);
}

void printcmd(void)
{
  logmsg(LOG_ERR,"Original command was: '%s'", orig_command);
}

void posixerror(const char *string)
{
  logmsg(LOG_ERR, "%s: %s\n", string, strerror(errno));
}

void parse_args(int argc, char *argv[])
{
  int c;

  while(1) {
    c = getopt(argc, argv, "d");
    if (c == -1)
      break;

    switch(c) {
      case 'd':
        debug_flag++;
        break;

      default:
        fprintf(stderr,"Unknown argument: '%s'", argv[optind]);
        exit(1);
        break;
    }
  }

  if (optind < argc) {
    fprintf(stderr, "Unrecognized argument: '%s'\n", argv[optind]);
    exit(1);
  }
}

void parse_config()
{
  char line[1000] = "";
  int lineno = 1;
  FILE *fp = NULL;

  fp = fopen(config_file,"r");
  if (fp == NULL) {
    logmsg(LOG_ERR,"Unable to open %s\n", config_file);
    exit(1);
  }

  while(fgets(line, sizeof(line), fp) != NULL) {
    parse_line(line,lineno++);
  }
  fclose(fp);
}

void whitespace(char **p)
{
  while(isspace(**p))
    (*p)++;
}

int eatstr(char **pp, char *str)
{
  whitespace(pp);
  int len = strlen(str);
  if (strncmp(*pp,str,len) == 0) {
    *pp += len;
    return 1;
  } else {
    return 0;
  }
}

int eatsym(char **pp, char *sym)
{
  whitespace(pp);
  int len = strlen(sym);
  if (strncmp(*pp,sym,len) == 0 && !isalnum((*pp)[len]) ) {
    *pp += len;
    return 1;
  } else {
    return 0;
  }
}

void parse_line(char *line, int lineno)
{
  char *p = line;
  if (eatstr(&p,"#")) {
    return;
  }
  if (eatsym(&p, "allow")) {
    parse_allow(&p,lineno);
  } else if (eatsym(&p, "default")) {
    parse_default(&p, lineno);
  } else if (eatsym(&p, "purgecmd")) {
    parse_purgecmd(&p, lineno);
  } else if (eatsym(&p, "listen")) {
    parse_listen(&p, lineno);
  } else if (*p == '\n') {
    return;
  } else if (*p == '\0') {
    return;
  }
}

int eatword(char **pp, char *word, int dstlen)
{
  whitespace(pp);
  char *start = NULL;
  while(!isspace(**pp) && (**pp!=',') && isprint(**pp)) {
    if (start == NULL)
      start = *pp;
    (*pp)++;
  }
  if (start != NULL) {
    int srclen = *pp - start;
    if (srclen >= dstlen) {
      srclen = dstlen - 1;
    }
    strncpy(word, start, srclen);
    word[srclen] = '\0';
    return 1;
  }
  return 0;
}

int eatnum(char **pp, int64_t *result)
{
  whitespace(pp);
  int64_t i = 0;
  while(isdigit(**pp)) {
    i *= 10L;
    i += **pp - '0';
    (*pp)++;
  }
  int64_t numer = 0;
  int64_t denom = 1.0;
  double frac = 0.0;
  if (**pp == '.') {
    (*pp)++;
    while(isdigit(**pp)) {
      numer *= 10;
      denom *= 10;
      numer += **pp - '0';
      (*pp)++;
    }
    frac = numer;
    frac /= denom;
  }

  if (**pp == 'g' || **pp == 'G') {
    (*pp)++;
    int64_t g = 1024L * 1024L * 1024L;
    *result = i*g + frac*g;
    return 1;
  }

  if (**pp == 'm' || **pp == 'M') {
    (*pp)++;
    int64_t m = 1024L * 1024L;
    *result = i*m + frac*m;
    return 1;
  }

  if (**pp == 'k' || **pp == 'K') {
    (*pp)++;
    int64_t k = 1024L;
    *result = i*k + frac*k;
    return 1;
  }

  *result = i;

  return 1;
}

void allow(struct sockaddr *addr)
{
  char straddr[INET6_ADDRSTRLEN];
  struct sockaddr_in6 *sin6;
  struct sockaddr_in *sin;
  int len = 0;

  sin = (struct sockaddr_in *) addr; 
  sin6 = (struct sockaddr_in6 *) addr;
    
  if (addr->sa_family == AF_INET) {
    inet_ntop(AF_INET, &sin->sin_addr, straddr, sizeof(straddr));
    len = sizeof(struct sockaddr_in);
  }
  else if (addr->sa_family == AF_INET6) {
    inet_ntop(AF_INET6, &sin6->sin6_addr, straddr, sizeof(straddr));
    len = sizeof(struct sockaddr_in6);
  }
  else {
      return;
  }

  struct allowed_addr *p;
  for(p=allowed_list; p; p=p->next) {
    if (memcmp(&p->addr,addr,len) == 0) {
      logmsg(LOG_INFO,"Duplicate allowed address: %s\n", straddr);
      return;
    }
  }

  logmsg(LOG_INFO,"Allowing %s\n", straddr);
  struct allowed_addr *next = calloc(sizeof(struct allowed_addr),1);
  next->next = allowed_list;
  memcpy(&next->addr, addr, len);

  allowed_list = next;
}

void parse_allow(char **pp, int lineno)
{
  struct addrinfo hints;
  struct addrinfo *hostaddr = NULL;
  struct addrinfo *tmpaddr = NULL;
  struct in6_addr server6addr;
  struct in_addr server4addr;
  int rv = 0;
  char name[100];
  eatword(pp,name,sizeof(name));
  if (name == NULL) {
    logmsg(LOG_ERR,"Error in config line %d\n", lineno);
    exit(1);
  }

  memset(&hints, 0, sizeof(hints));

  hints.ai_family = PF_UNSPEC;
  hints.ai_socktype = SOCK_STREAM;
  hints.ai_flags = AI_NUMERICSERV;
  
  rv = inet_pton(AF_INET, name, &server6addr);

  if (rv == 1) {
    hints.ai_family = AF_INET;
    hints.ai_flags |= AI_NUMERICHOST;
  }
  else {
    rv = inet_pton(AF_INET6, name, &server4addr);

    if (rv == 1) {
      hints.ai_family = AF_INET6;
      hints.ai_flags |= AI_NUMERICHOST;
    }
  }

  rv = getaddrinfo(name, NULL, &hints, &hostaddr);

  if (rv != 0) {
    logmsg(LOG_ERR, "Unable to get host lookup information for %s", name);
    exit(1);
  }

  for(tmpaddr = hostaddr; tmpaddr; tmpaddr = tmpaddr->ai_next) {
    if ( (tmpaddr->ai_family != AF_INET6) && (tmpaddr->ai_family != AF_INET) ) {
      continue;
    }
 
    allow(tmpaddr->ai_addr);
  }

  freeaddrinfo(hostaddr);
}

void parse_default(char **pp, int lineno)
{
  if (eatsym(pp,"hardspace")) {
    int64_t i;
    if (!eatnum(pp,&i)) {
      logmsg(LOG_ERR,"Bad number in line %d\n", lineno);
      exit(1);
    }
    default_hardspace = i;
  }

  else if (eatsym(pp,"softspace")) {
    int64_t i;
    if (!eatnum(pp,&i)) {
      logmsg(LOG_ERR,"Bad number in line %d\n", lineno);
      exit(1);
    }
    default_softspace = i;
  }

  else if (eatsym(pp,"hardinode")) {
    int64_t i;
    if (!eatnum(pp,&i)) {
      logmsg(LOG_ERR,"Bad number in line %d\n", lineno);
      exit(1);
    }
    default_hardinode = i;
  }

  else if (eatsym(pp,"softinode")) {
    int64_t i;
    if (!eatnum(pp,&i)) {
      logmsg(LOG_ERR,"Bad number in line %d\n", lineno);
      exit(1);
    }
    default_softinode = i;
  }

  else {
    logmsg(LOG_ERR,"Syntax error in line %d\n", lineno);
    exit(1);
  }
}

void parse_purgecmd(char **pp, int lineno)
{
  char cmd[MAXPATHLEN];
  eatword(pp,cmd,sizeof(cmd));
  strncpy(purgecmd,cmd,MAXPATHLEN);
}

void parse_listen(char **pp, int lineno)
{
  int64_t port;
  if (!eatnum(pp, &port)) {
    logmsg(LOG_ERR,"Syntax error in line %d\n", lineno);
    exit(1);
  }
  listen_port = port;
}


void print_config()
{
  logmsg(LOG_INFO,"default hardspace = %lld\n", default_hardspace);
  logmsg(LOG_INFO,"default softspace = %lld\n", default_softspace);
  logmsg(LOG_INFO,"default hardinode = %lld\n", default_hardinode);
  logmsg(LOG_INFO,"default softinode = %lld\n", default_softinode);
  logmsg(LOG_INFO,"purgecmd = \"%s\"\n", purgecmd);
}


void setup_listener()
{
  sock6 = socket(PF_INET6, SOCK_STREAM, IPPROTO_IP);

  sock4 = socket(PF_INET, SOCK_STREAM, IPPROTO_IP);

  if (sock4 < 0 && sock6 < 0) {
    posixerror("socket");
    exit(1);    
  }

  if (sock4 >= 0) {
    int value = 1;
    if (setsockopt(sock4, SOL_SOCKET, SO_REUSEADDR, &value, sizeof(value)) != 0) {
      posixerror("setsockopt SO_REUSEADDR");
      exit(1);
    }

    struct linger lin = { 0, 0 };
    if (setsockopt(sock4, SOL_SOCKET, SO_LINGER, &lin, sizeof(lin)) != 0) {
      posixerror("setsockopt SO_LINGER");
      exit(1);
    }

    struct sockaddr_in addr;
    addr.sin_port = htons(listen_port);
    addr.sin_addr.s_addr = INADDR_ANY;
    addr.sin_family = AF_INET;
    if (bind(sock4, &addr, sizeof(addr))) {
      posixerror("bind");
      exit(1);
    }

    if (listen(sock4,20) != 0) {
      posixerror("listen");
      exit(1);
    }
  }

  if (sock6 >= 0) {
    int value = 1;
    if (setsockopt(sock6, SOL_SOCKET, SO_REUSEADDR, &value, sizeof(value)) != 0) {
      posixerror("setsockopt SO_REUSEADDR");
      exit(1);
    }

    struct linger lin = { 0, 0 };
    if (setsockopt(sock6, SOL_SOCKET, SO_LINGER, &lin, sizeof(lin)) != 0) {
      posixerror("setsockopt SO_LINGER");
      exit(1);
    }

    value = 1;
    if (setsockopt(sock6, IPPROTO_IPV6, IPV6_V6ONLY, &value, sizeof(value)) != 0) {
      posixerror("setsockopt IPV6_V6ONLY");
      exit(1);
    }

    struct sockaddr_in6 addr;
    addr.sin6_port = htons(listen_port);
    addr.sin6_addr = in6addr_any;
    addr.sin6_family = AF_INET6;
    if (bind(sock6, &addr, sizeof(addr))) {
      posixerror("bind");
      exit(1);
    }

    if (listen(sock6,20) != 0) {
      posixerror("listen");
      exit(1);
    }
  }
}


void handle_connections()
{
  fd_set active_fd_set, read_fd_set;

  FD_ZERO(&active_fd_set);

  if (sock4 >= 0) {
    FD_SET(sock4, &active_fd_set);
  }

  if (sock6 >= 0) {
    FD_SET(sock6, &active_fd_set);
  }

  while(1) {
    int fd = -1;
    int found = 0;
    read_fd_set = active_fd_set;
    char straddr[INET6_ADDRSTRLEN];

    if (select(FD_SETSIZE, &read_fd_set, NULL, NULL, NULL) < 0) {
      posixerror("select");
      exit(1);
    }

    if (FD_ISSET(sock4,&read_fd_set)) {
      struct sockaddr_in addr;
      struct sockaddr_in addr2;
      int len = sizeof(addr);

      fd = accept(sock4, &addr, &len);
    
      if (fd < 0) {
        posixerror("accept");
        continue;
      }
        
      strcpy(straddr,inet_ntoa(addr.sin_addr));
      
      //logmsg(LOG_INFO,"Accepted connection from %s\n", straddr);
      found = 0;
      struct allowed_addr *p;
      for(p=allowed_list; p; p=p->next) {
        addr2 = *(struct sockaddr_in *)&p->addr;

        if (memcmp(&addr2.sin_addr, &addr.sin_addr, sizeof(addr.sin_addr)) == 0) {
          //logmsg(LOG_INFO,"Found the allowed address\n");
          found=1;
          break;
        }
      }
    }
    else if (FD_ISSET(sock6,&read_fd_set)) {
      struct sockaddr_in6 addr;
      struct sockaddr_in6 addr2;
      int len = sizeof(addr);
      fd = accept(sock6, &addr, &len);
      if (fd < 0) {
        posixerror("accept");
        continue;
      }
            
      inet_ntop(AF_INET6, &addr.sin6_addr, straddr, sizeof(straddr));

      //logmsg(LOG_INFO,"Accepted connection from %s\n",straddr);
      found = 0;
      struct allowed_addr *p;
      for(p=allowed_list; p; p=p->next) {
        addr2 = *(struct sockaddr_in6 *)&p->addr;

        if (memcmp(&addr2.sin6_addr, &addr.sin6_addr, sizeof(addr.sin6_addr)) == 0) {
          //logmsg(LOG_INFO,"Found the allowed address\n");
          found=1;
          break;
        }
      }
    }
    else {
      continue;
    }

    if (!found) {
      logmsg(LOG_INFO,"Address %s is not on the allowed list.\n", straddr);
      close(fd);
      continue;
    }

    pid_t pid = fork();
    if (pid < 0) {
      posixerror("fork");
    } else if (pid != 0) {
      int status;
      int pstatus;
      close(fd);
      status = waitpid(pid, &pstatus, 0);
    } else {
      pid = fork();
      if (pid < 0) {
        posixerror("second fork");
        exit(1);
      }

      if (pid == 0) {
        service_connection(fd);
      }
      exit(0);
    }
  }
}

void service_connection(int fd)
{

  char buffer[1000] = "";
  int offset=0;
  int status;

  while(1) {
    status = read(fd, &buffer[offset], sizeof(buffer)-offset);
    if (status < 0) {
      write(fd, "Error reading.\n", strlen("Error reading.\n"));
      close(fd);
    }
    buffer[offset+status] = '\0';
    offset += status;
    if (strchr(buffer, '\n') != NULL) {
      break;
    }
  }

  buffer[sizeof(buffer)-1] = '\0';
  strncpy(orig_command, buffer, sizeof(orig_command));
  char *repl = parse_command(buffer,fd);
  if (repl == NULL) {
    repl = "Error: Unrecognized command.\n";
  }
  status = write(fd, repl, strlen(repl));
  close(fd);
}


char *parse_command(char *line, int fd)
{
  char *p = line;
  char **pp = &p;

  if (eatsym(pp,"getquota")) {
    return parse_getquota(pp);
  }
  if (eatsym(pp,"setquota")) {
    return parse_setquota(pp);
  }
  if (eatsym(pp,"purge")) {
    return parse_purge(pp,fd);
  }
}

int parse_option(char **pp)
{
  if (eatsym(pp,"user")) {
    if (eatstr(pp,"=")) {
      if (eatword(pp, option_user, sizeof(option_user))) {
        return 1;
      }
    }
  }

  if (eatsym(pp,"degree")) {
    if (eatstr(pp,"=")) {
      if (eatword(pp, option_degree, sizeof(option_degree))) {
        return 1;
      }
    }
  }

  if (eatsym(pp,"hardspace")) {
    if (eatstr(pp,"=")) {
      int64_t val;
      if (eatnum(pp, &val)) {
        option_hardspace = val;
        return 1;
      }
    }
  }

  if (eatsym(pp,"softspace")) {
    if (eatstr(pp,"=")) {
      int64_t val;
      if (eatnum(pp, &val)) {
        option_softspace = val;
        return 1;
      }
    }
  }

  logmsg(LOG_ERR,"Unrecognized option: %s\n", *pp);
  printcmd();

  return 0;
}

void parse_optionlist(char **pp)
{
  if (!parse_option(pp)) {
    return;
  }
  while(eatstr(pp,",")) {
    parse_option(pp);
  }
}

char *get_device(int devnum)
{
  struct devent *p;
  for(p=device_cache; p; p=p->next) {
    if (p->num == devnum) {
      //logmsg(LOG_INFO,"For devnum %04x name=%s device=%s\n", devnum, p->name, p->device);
      return p->device;
    }
  }

  // Not found.  Put a null entry in the cache so we don't search again.
  {
    //struct devent *next = calloc(sizeof(struct devent),1);
    //next->num = devnum;
    //next->name = NULL;
    //next->next = device_cache;
    //device_cache = next;
    logmsg(LOG_INFO,"Could not find directory for device 0x%x.", devnum);
  }

  return NULL;
}

void scan_devices(void)
{
  // Figure device name translations from /etc/mtab...
  FILE *fp = fopen("/etc/mtab", "r");
  if (fp == NULL) {
    logmsg(LOG_ERR,"Unable to open /etc/mtab to collect device numbers.");
    return;
  }

  char line[1000];
  while(fgets(line,sizeof(line),fp) != NULL) {
    char *ptr = line;
    char **pp = &ptr;
    char dev[100];
    char dir[100];
    eatword(pp, dev, sizeof(dev));
    eatword(pp, dir, sizeof(dir));

    if (strncmp(dev,"/dev/",5) != 0)
      continue;

    struct stat sb;
    int status = stat(dir,&sb);
    if (status != 0) {
      continue;
    }

    struct devent *next = calloc(sizeof(struct devent),1);
    next->num = sb.st_dev;
    next->name = strdup(dir);
    next->device = strdup(dev);
    next->next = device_cache;
    device_cache = next;
    logmsg(LOG_INFO,"Device %s %s (%04x)\n", dev, dir, next->num);
  }
  fclose(fp);
}

char *parse_getquota(char **pp)
{
  strcpy(option_user, "");

  parse_optionlist(pp);

  if (strlen(option_user) == 0) {
    return "Error: User not specified.\n";
  }

  struct passwd *pwd = getpwnam(option_user);
  if (pwd == NULL) {
    return "Error: Unknown user.\n";
  }

  struct stat sb;
  int status = stat(pwd->pw_dir, &sb);
  if (status != 0) {
    status = mkdir(pwd->pw_dir, 0700);
    if (status != 0) {
      return "Error: Home directory cannot be created.\n";
    }
    status = chown(pwd->pw_dir, pwd->pw_uid, pwd->pw_gid);
    if (status != 0) {
      return "Error: Unable to set home directory permissions.\n";
    }
    status = stat(pwd->pw_dir, &sb);
    if (status != 0) {
      return "Error: Unknown problem with home directory.\n";
    }
  }

  char *device = get_device(sb.st_dev);
  if (device == NULL) {
    return "Device not found.\n";
  }

  char *error = do_getquota(device, pwd->pw_uid);
  if (error)
    return error;

  static char repl[1000];
  sprintf(repl,
          "status=%s,softspace=%lld,hardspace=%lld,space=%lld,files=%lld,"
	  "remaining=%d\n",
          result_qstat, result_softspace, result_hardspace, result_curspace,
          result_curfiles, result_timeremaining);
  return repl;
}

char *parse_purge(char **pp, int fd)
{
  int status;

  strcpy(option_user, "");
  strcpy(option_degree, "");
  option_softspace = -1;
  option_hardspace = -1;

  parse_optionlist(pp);

  if (strlen(option_user) == 0) {
    return "Error: User not specified.\n";
  }

  if (strlen(purgecmd) == 0) {
    return "Error: Don't know how to purge.\n";
  }

  struct passwd *pwd = getpwnam(option_user);
  if (pwd == NULL) {
    return "Error: Unknown user.\n";
  }

  struct stat sb;
  status = stat(pwd->pw_dir, &sb);
  if (status != 0) {
    return "Error: User directory does not exist.\n";
  }

  pid_t pid = fork();
  if (pid == -1) {
    return "Error: Out of processes.\n";
  } else if (pid > 0) {
    int pstatus = 0;
    int status = waitpid(pid, &pstatus, 0);
    if (pstatus != 0) {
      return "Error: Purge command indicated failure.\n";
    }
    return "Success.\n";
  }

  dup2(fd,0);
  dup2(fd,1);
  dup2(fd,2);

  status = chdir(pwd->pw_dir);
  if (status != 0) {
    fprintf(stderr,"Error: Unable to access %s\n", pwd->pw_dir);
    exit(1);
  }

  status = setregid(pwd->pw_gid, pwd->pw_gid);
  if (status != 0) {
    fprintf(stderr,"User does not have group permission.\n");
    exit(1);
  }

  status = setreuid(pwd->pw_uid, pwd->pw_uid);
  if (status != 0) {
    fprintf(stderr,"User does not have permission.\n");
    exit(1);
  }

  clearenv();
  setenv("USER",option_user,1);
  setenv("PWD",pwd->pw_dir,1);
  setenv("HOME",pwd->pw_dir,1);
  setenv("LOGNAME",option_user,1);

  if (option_degree[0] == '\0') {
    execlp(purgecmd, purgecmd, NULL);
  } else {
    char degree_param[200];
    sprintf(degree_param, "degree=%s", option_degree);
    execlp(purgecmd, purgecmd, degree_param, NULL);
  }
  fprintf(stderr,"Error: Unable to execute purge command.\n");
  exit(1);
}

char *parse_setquota(char **pp)
{
  strcpy(option_user, "");
  option_softspace = -1;
  option_hardspace = -1;

  parse_optionlist(pp);

  if (strlen(option_user) == 0) {
    return "Error: User not specified.\n";
  }

  struct passwd *pwd = getpwnam(option_user);
  if (pwd == NULL) {
    return "Error: Unknown user.\n";
  }

  struct stat sb;
  int status = stat(pwd->pw_dir, &sb);
  if (status != 0) {
    return "Error: User directory does not exist.\n";
  }

  char *device = get_device(sb.st_dev);
  if (device == NULL) {
    return "Device not found.\n";
  }

  char *error = do_getquota(device, pwd->pw_uid);
  if (error)
    return error;

  if (option_softspace >= 0) {
    logmsg(LOG_INFO,"Setting softspace to %lld for '%s'\n", option_softspace, option_user);
    result_softspace = option_softspace;
  }

  if (option_hardspace >= 0) {
    logmsg(LOG_INFO,"Setting hardspace to %lld for '%s'\n", option_hardspace, option_user);
    result_hardspace = option_hardspace;
  }

  error = do_setquota(device, pwd->pw_uid);
  if (error)
    return error;

  error = do_getquota(device, pwd->pw_uid);
  if (error)
    return error;

  static char repl[1000];
  sprintf(repl,
          "status=%s,softspace=%lld,hardspace=%lld,space=%lld,files=%lld\n",
          result_qstat, result_softspace, result_hardspace, result_curspace,
          result_curfiles);
  return repl;
}


char *do_getquota(char *device, int uid)
{
  int status;
  struct mem_dqblk dq = { 0 };

  status = quotactl(QCMD(Q_GETQUOTA,USRQUOTA), device, uid, (caddr_t)&dq);
  if (status < 0) {
    posixerror("quotactl");
    logmsg(LOG_ERR,"device=%s\n", device);
    return "Error: Unable to get quota.\n";
  }

  if (!(dq.dqb_valid & QIF_BLIMITS)) {
    dq.dqb_bhardlimit = 0;
    dq.dqb_bsoftlimit = 0;
  }

  if (!(dq.dqb_valid & QIF_SPACE)) {
    dq.dqb_curspace = 0;
  }

  if (!(dq.dqb_valid & QIF_ILIMITS)) {
    dq.dqb_ihardlimit = 0;
    dq.dqb_isoftlimit = 0;
  }

  if (!(dq.dqb_valid & QIF_INODES)) {
    dq.dqb_curinodes = 0;
  }

  if (!(dq.dqb_valid & QIF_BTIME)) {
    dq.dqb_btime = 0;
  }

  if (!(dq.dqb_valid & QIF_ITIME)) {
    dq.dqb_itime = 0;
  }


  result_softspace = dq.dqb_bsoftlimit * 1024L;
  result_hardspace = dq.dqb_bhardlimit * 1024L;
  result_curspace = dq.dqb_curspace;

  result_softfiles = dq.dqb_isoftlimit;
  result_hardfiles = dq.dqb_ihardlimit;
  result_curfiles = dq.dqb_curinodes;

  result_spacetime = dq.dqb_btime;
  result_filestime = dq.dqb_itime;

  strcpy(result_qstat,"good");

  if ((result_softspace != 0) && (result_curspace >= result_softspace)) {
    strcpy(result_qstat, "over");
    result_timeremaining = result_spacetime - time(0);
  }

  if ((result_hardspace != 0) && (result_curspace >= result_hardspace)) {
    strcpy(result_qstat, "max");
    result_timeremaining = 0;
  }

  if ((result_spacetime != 0) && (time(0) > result_spacetime)) {
    strcpy(result_qstat, "expired");
    result_timeremaining = 0;
  }

  if ((result_filestime != 0) && (time(0) > result_filestime)) {
    strcpy(result_qstat, "expired");
    result_timeremaining = 0;
  }

  return NULL;
}


char *do_setquota(char *device, int uid)
{
  int status;
  struct mem_dqblk dq = { 0 };

  dq.dqb_bsoftlimit = result_softspace / 1024;
  dq.dqb_bhardlimit = result_hardspace / 1024;
  dq.dqb_isoftlimit = result_softfiles;
  dq.dqb_ihardlimit = result_hardfiles;
  dq.dqb_btime = result_spacetime;
  dq.dqb_itime = result_filestime;
  dq.dqb_valid = QIF_LIMITS | QIF_TIMES;

  status = quotactl(QCMD(Q_SETQUOTA,USRQUOTA), device, uid, (caddr_t)&dq);
  if (status < 0) {
    posixerror("quotactl");
    logmsg(LOG_ERR,"device=%s\n", device);
    return "Error: Unable to set quota.\n";
  }

  return NULL;
}

