#!/usr/bin/env python
#
# @package      hubzero-submit-server
# @file         submit-server.py
# @author       Steven Clark <clarks@purdue.edu>
# @author       Rick Kennell <kennell@purdue.edu>
# @copyright    Copyright (c) 2006-2014 HUBzero Foundation, LLC.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2006-2014 HUBzero Foundation, LLC.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#

import os
import sys
import logging

from hubzero.submit.LogMessage   import getLogIDMessage as getLogMessage
from hubzero.submit.SubmitServer import SubmitServer

LOGDIRECTORY      = os.path.join(os.sep,'var','log','submit')
HUBLOGFILE        = 'submit-server.log'
APPLICATIONLOGGER = logging.getLogger('')

PIDFILE = os.path.join(os.sep,'var','run','submit-server.pid')

CONFIGURATIONDIRECTORY = os.path.join(os.sep,'etc','submit')
CONFIGURATIONFILE      = 'submit-server.conf'
CONFIGFILEPATH         = os.path.join(CONFIGURATIONDIRECTORY,CONFIGURATIONFILE)

DISTRIBUTORDIRECTORY = os.path.join(os.sep,'opt','submit')
DISTRIBUTORSCRIPT    = 'distributor'
JOBSTATUSDIRECTORY   = os.path.join(os.sep,'opt','submit')
JOBSTATUSSCRIPT      = 'jobStatus.py'
JOBKILLDIRECTORY     = os.path.join(os.sep,'opt','submit')
JOBKILLSCRIPT        = 'jobKill.py'

def daemonize(fdLogFile):
   if fdLogFile != sys.stdout.fileno():
      devnull = open("/dev/null",'r')
      os.dup2(devnull.fileno(),sys.stdin.fileno())
      os.dup2(fdLogFile,sys.stdout.fileno())
      os.dup2(fdLogFile,sys.stderr.fileno())

   if os.fork():
      os._exit(os.EX_OK)
   os.setsid()
   os.chdir("/")
   if os.fork():
      os._exit(os.EX_OK)


def openLogger(logDirectory,
               hubLogFile):
   class EmptyFilter(logging.Filter):
      """
      This is a filter which rejects empty messages

      """

      def filter(self,record):
         if record.getMessage() == "":
            emptyRecord = True
         else:
            emptyRecord = False

         return(not emptyRecord)

   APPLICATIONLOGGER.setLevel(logging.DEBUG)

   hubLogPath = os.path.join(logDirectory,hubLogFile)
   logHandler = logging.FileHandler(hubLogPath)
   fdLogFile = logHandler.stream.fileno()

   emptyFilter = EmptyFilter()
   logHandler.addFilter(emptyFilter)

   logFormatter = logging.Formatter('%(asctime)s %(message)s','%s [%a %b %d %H:%M:%S %Y]')
   logHandler.setFormatter(logFormatter)
   APPLICATIONLOGGER.addHandler(logHandler)

   APPLICATIONLOGGER.log(logging.INFO,getLogMessage(80*'#'))
   APPLICATIONLOGGER.log(logging.INFO,getLogMessage("Backgrounding process."))

   return(fdLogFile)


def writePidFile(pidFile):
   try:
      fpPid = open(pidFile,'w')
      fpPid.write("%d\n" % (os.getpid()))
      fpPid.close()
   except:
      APPLICATIONLOGGER.log(logging.ERROR,getLogMessage("Unable to write pid (%d) to %s" % (os.getpid(),pidFile)))
      os._exit(os.EX_CANTCREAT)


def server(configFilePath,
           distributorDirectory,
           distributorScript,
           jobStatusDirectory,jobStatusScript,
           jobKillDirectory,jobKillScript):
   submitServer = SubmitServer(configFilePath,
                               distributorDirectory,distributorScript,
                               jobStatusDirectory,jobStatusScript,
                               jobKillDirectory,jobKillScript)
   if submitServer.configure():
      if submitServer.setupMySQL():
         if submitServer.setupClientListeners():
            submitServer.run()
         else:
            APPLICATIONLOGGER.log(logging.ERROR,getLogMessage("Shutting down - no listeners configured."))
            sys.exit(1)
      else:
         APPLICATIONLOGGER.log(logging.ERROR,getLogMessage("Shutting down - could not configure MySQL database."))
         sys.exit(1)
   else:
      APPLICATIONLOGGER.log(logging.ERROR,getLogMessage("Shutting down - server configuration failed."))
      sys.exit(1)


if __name__ == '__main__':
   fdLogFile = openLogger(LOGDIRECTORY,HUBLOGFILE)

   daemonize(fdLogFile)

   writePidFile(PIDFILE)

   server(CONFIGFILEPATH,
          DISTRIBUTORDIRECTORY,DISTRIBUTORSCRIPT,
          JOBSTATUSDIRECTORY,JOBSTATUSSCRIPT,
          JOBKILLDIRECTORY,JOBKILLSCRIPT)

   sys.exit(0)


