# @package      hubzero-submit-server
# @file         SubmitServer.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2012-2014 HUBzero Foundation, LLC.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2012-2014 HUBzero Foundation, LLC.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#

__version__ = '2.7.1'

import os
import sys
import signal
import select
import socket
import re
import pwd
import subprocess
import ldap
import traceback
import math
import shutil
import time
import string
import tempfile
import logging

from hubzero.submit.LogMessage        import getLogIDMessage as getLogMessage, logSetJobId
from hubzero.submit.CommandParser     import CommandParser
from hubzero.submit.BoundConnection   import BoundConnection
from hubzero.submit.UnboundConnection import UnboundConnection
from hubzero.submit.MySQLDatabase     import MySQLDatabase

class SubmitServer:
   def __init__(self,
                configFilePath,
                distributorDirectory,
                distributorScript,
                jobStatusDirectory,
                jobStatusScript,
                jobKillDirectory,
                jobKillScript):
      self.logger         = logging.getLogger(__name__)
      self.configData     = {}
      self.configFilePath = configFilePath

      self.mySQLDatabase    = None
      self.clientListener   = None
      self.attachListener   = None
      self.serverConnection = None
      self.commandParser    = None

      self.clientData                         = {}
      self.clientData['version']              = ""
      self.clientData['doubleDashTerminator'] = False
      self.clientData['userName']             = ""
      self.clientData['password']             = ""
      self.clientData['sessionId']            = 0
      self.clientData['sessionToken']         = ""
      self.clientData['args']                 = []
      self.clientData['envs']                 = {}
      self.clientData['umask']                = 027
      self.clientData['workDirectory']        = ""
      self.clientData['workDirectoryInode']   = None
      self.clientData['workDirectoryDevice']  = None
      self.clientData['isClientTTY']          = True
      self.clientData['pegasusVersion']       = None
      self.clientData['reportMetrics']        = False
      self.clientData['isParametric']         = False
      self.clientData['localExecution']       = False
      self.clientData['inputFileMapping']     = {}

      self.serverData                         = {}
      self.serverData['hostFQDN']             = socket.getfqdn()
      self.serverData['version']              = __version__
      self.serverData['operationMode']        = 0
      self.serverData['distributorDirectory'] = distributorDirectory
      self.serverData['distributorScript']    = distributorScript
      self.serverData['jobStatusDirectory']   = jobStatusDirectory
      self.serverData['jobStatusScript']      = jobStatusScript
      self.serverData['jobKillDirectory']     = jobKillDirectory
      self.serverData['jobKillScript']        = jobKillScript
      self.serverData['pegasusExists']        = False
      self.serverData['pegasusVersion']       = None
      self.serverData['pegasusHome']          = None
      self.serverData['pegasusPythonPath']    = None
      self.serverData['type']                 = 'Listener'
      self.serverData['exitCode']             = 2
      self.serverData['uid']                  = None
      self.serverData['gid']                  = None
      self.serverData['gids']                 = None
      self.serverData['homeDirectory']        = None
      self.serverData['email']                = None
      self.serverData['authAttempts']         = 0
      self.serverData['authz']                = 0
      self.serverData['createdSessions']      = []
      self.serverData['session']              = 0
      self.serverData['jobToken']             = ""
      self.serverData['localJobId']           = "00000000"
      self.serverData['jobId']                = 0
      self.serverData['superJobId']           = 0
      self.serverData['runName']              = ""
      self.serverData['args']                 = []
      self.serverData['envs']                 = {}
      self.serverData['event']                = "simulation"
      self.serverData['ncpus']                = 1
      self.serverData['venue']                = "any"
      self.serverData['metricsRecorded']      = False
      self.serverData['metrics']              = ""
      self.serverData['waitForLowLoad']       = False
      self.serverData['detach']               = False
      self.serverData['detached']             = False
      self.serverData['socketPath']           = ""
      self.serverData['attachId']             = 0L
      self.serverData['attachJobId']          = "00000000"
      self.serverData['attached']             = False
# Assume that a db entry exists.  That's where the jobid came from.
# If we've improperly assumed it exists, then finalizeJob() will
# indicate failure when a job is aborted prematurely.
      self.serverData['readyToExit']          = False
      self.serverData['dbJobEntryExists']     = False
      self.serverData['mysqlAuthenticated']   = False
      self.serverData['ldapAuthenticated']    = False
      self.serverData['sessionLimit']         = 0
      self.serverData['timeoutInterval']      = 60
      self.serverData['doHeartbeat']          = True
      self.serverData['jobScanInterval']      = 15
      self.serverData['disconnectTime']       = 0
      self.serverData['disconnectMax']        = 2*60.
      self.serverData['selectTimeout']        = 4*60.
      self.serverData['childPid']             = None
      self.serverData['workDirectory']        = ""
      self.serverData['workDirectoryShared']  = True
      self.serverData['clientTransferPath']   = ""
      self.serverData['inputObjects']         = {}
      self.serverData['outputObjects']        = {}
      self.serverData['importObjects']        = {}
      self.serverData['exportObjects']        = {}
      self.serverData['closePendingObjects']  = []
      self.serverData['childHasExited']       = False
      self.serverData['childKillPending']     = False
      self.serverData['jobScanPath']          = ""
      self.serverData['mountPoints']          = {}
      dfCommand = ['df','--portability','--all']
      child = subprocess.Popen(dfCommand,
                               stdout=subprocess.PIPE,
                               stderr=subprocess.PIPE,
                               close_fds=True)
      dfStdOutput,dfStdError = child.communicate()
      dfExitStatus = child.returncode
      if dfExitStatus == 0:
         mounts = ''.join(dfStdOutput).strip().split('\n')[1:]
         for mount in mounts:
            device     = mount.split()[0]
            mountPoint = mount.split()[-1]
            if ':' in device:
               self.serverData['mountPoints'][mountPoint] = {'device':device}
            else:
               self.serverData['mountPoints'][mountPoint] = {'device':':'.join([self.serverData['hostFQDN'],device])}
      else:
         self.logger.log(logging.ERROR,getLogMessage("df command failed: %s\n" % (dfStdError)))

      signal.signal(signal.SIGINT,self.handleSignal)
      signal.signal(signal.SIGHUP,self.handleSignal)
      signal.signal(signal.SIGQUIT,self.handleSignal)
      signal.signal(signal.SIGABRT,self.handleSignal)
      signal.signal(signal.SIGTERM,self.handleSignal)
      signal.signal(signal.SIGCHLD,self.handleChild)


   def exit(self,
            exitCode=0):
      if   self.serverData['type'] == 'JobExecuter':
         if self.serverData['readyToExit']:
            self.killChild()
            self.unsetupWorkDirectory()
            if self.serverData['socketPath']:
               if self.serverData['socketPath'].startswith('file://'):
                  socketPath = self.serverData['socketPath'][len('file://'):]
               else:
                  socketPath = self.serverData['socketPath']
               if os.path.exists(socketPath):
                  os.remove(socketPath)
               self.serverData['socketPath'] = ""
            if self.serverData['clientTransferPath']:
               if os.path.isdir(self.serverData['clientTransferPath']):
                  shutil.rmtree(self.serverData['clientTransferPath'],True)
            exitCode = self.serverData['exitCode']
            self.logger.log(logging.INFO,getLogMessage("Server(%s) exiting(%d)." % (self.serverData['type'],exitCode)))
            sys.exit(exitCode)
         else:
            self.finalizeJob()
            self.finalizeCreatedSessions()
            if self.serverData['exitCode'] == 2:
               self.serverData['exitCode'] = exitCode
            self.clientListener.postMessage('serverExit %d\n' % (self.serverData['exitCode']))
            self.serverData['readyToExit'] = True
      elif self.serverData['type'] == 'Proxy':
         if self.serverData['readyToExit']:
            self.logger.log(logging.INFO,getLogMessage("Server(%s) exiting(%d)." % (self.serverData['type'],exitCode)))
            sys.exit(exitCode)
         else:
            self.clientListener.postMessage('serverExit %d\n' % (exitCode))
            self.serverData['readyToExit'] = True
      elif self.serverData['type'] == 'Listener':
         self.logger.log(logging.INFO,getLogMessage("Server(%s) exiting(%d)." % (self.serverData['type'],exitCode)))
         sys.exit(exitCode)


   def killChild(self):
      if self.serverData['childPid']:
         self.serverData['childKillPending'] = True
         try:
            os.kill(-self.serverData['childPid'],signal.SIGINT)
            for retry in range(0,12):
               time.sleep(10)
               os.kill(self.serverData['childPid'],0)
               if not self.serverData['childKillPending']:
                  break
            if self.serverData['childKillPending']:
               os.kill(-self.serverData['childPid'],signal.SIGTERM)
               for retry in range(0,6):
                  time.sleep(10)
                  os.kill(self.serverData['childPid'],0)
                  if not self.serverData['childKillPending']:
                     break
            if self.serverData['childKillPending']:
               os.kill(-self.serverData['childPid'],signal.SIGKILL)
         except:
            pass


   def stop(self,
            exitCode=0):
      self.serverData['readyToExit'] = True
      self.serverData['exitCode']    = exitCode
      self.killChild()
      self.clientListener.postMessage('wait\n')
      self.logger.log(logging.INFO,getLogMessage("Server stopping."))


   def handleSignal(self,
                    signalNumber,
                    frame):
      self.logger.log(logging.ERROR,getLogMessage("Server(%d) was terminated by a signal %d." % (os.getpid(),signalNumber)))
      exitCode = (1 << 7) + signalNumber
      self.serverData['exitCode'] = exitCode
      self.exit(exitCode)


   def handleChild(self,
                   signalNumber,
                   frame):
      self.serverData['childKillPending'] = False


   def scheduleHeartbeat(self):
      signal.signal(signal.SIGALRM,self.heartbeat)
      signal.alarm(self.configData['heartbeatInterval'])


   def heartbeat(self,
                 signalNumber,
                 frame):
      if self.mySQLDatabase.connect(self.configData['mysqlMiddlewareDB']):
         sqlCommand = """UPDATE job SET heartbeat=now()
                          WHERE jobid='%d'
                      """ % (self.serverData['jobId'])
         result = self.mySQLDatabase.update(sqlCommand)
         if len(result) != 0:
            self.logger.log(logging.ERROR,getLogMessage("ERROR: Heartbeat: %s" % (result)))

         self.mySQLDatabase.disconnect()

      if self.clientListener.isConnected() or self.serverData['detached']:
         self.serverData['disconnectTime'] = 0
         self.scheduleHeartbeat()
      else:
         self.serverData['disconnectTime'] = time.time() - self.clientListener.getConnectionCheckedTime()
#        log("Server has been disconnected for %f seconds." % (self.serverData['disconnectTime']))
         if self.serverData['disconnectTime'] >= self.serverData['disconnectMax']:
            message = "disconnect time (%f) >= disconnect max (%f)" % (self.serverData['disconnectTime'],
                                                                       self.serverData['disconnectMax'])
            self.logger.log(logging.ERROR,getLogMessage(message))
            if self.serverData['childPid']:
               self.killChild()
            else:
               self.exit(1)
         else:
            self.scheduleHeartbeat()


   def clientHeartbeat(self):
      if self.mySQLDatabase.connect(self.configData['mysqlMiddlewareDB']):
         sqlCommand = """UPDATE job SET heartbeat=now()
                          WHERE jobid='%d'
                      """ % (self.serverData['jobId'])
         result = self.mySQLDatabase.update(sqlCommand)
         if len(result) != 0:
            self.logger.log(logging.ERROR,getLogMessage("ERROR: clientHeartbeat: %s" % (result)))

         self.mySQLDatabase.disconnect()
         self.logger.log(logging.INFO,getLogMessage("ClientHeartbeat applied"))


   def scheduleTimeout(self,
                       interval=0):
      if interval == 0:
         interval = self.serverData['timeoutInterval']

      if self.serverData['authAttempts'] > 3:
         self.handleTimeout(0,0)
      signal.signal(signal.SIGALRM,self.handleTimeout)
      interval = int(math.ceil(interval))
      signal.alarm(interval)
      self.logger.log(logging.INFO,getLogMessage("Server will time out in %d seconds." % (interval)))


   def handleTimeout(self,
                     signalNumber,
                     frame):
      if   self.serverData['authz'] == 0:
         message = "User '%s' failed to authenticate (%d) and timed out." % (self.clientData['userName'],
                                                                             self.serverData['authAttempts'])
         self.logger.log(logging.ERROR,getLogMessage(message))
         self.exit(1)
      elif self.serverData['waitForLowLoad']:
         self.checkLoadAndLaunch()
      else:
         self.logger.log(logging.ERROR,getLogMessage("Unknown reason for timeout."))
         self.exit(1)


   def scheduleJobScan(self):
      signal.signal(signal.SIGALRM,self.handleJobScan)
      signal.alarm(self.serverData['jobScanInterval'])


   def handleJobScan(self,
                     signalNumber,
                     frame):
      if self.clientData['isParametric'] and self.clientData['isClientTTY']:
         if self.serverData['jobScanPath'] == "":
            jobScanPath = os.path.join(self.serverData['workDirectory'],
                                       self.serverData['runName'],
                                       'remoteParameterCombinations.csv')
            self.serverData['jobScanPath'] = jobScanPath
         if os.path.exists(self.serverData['jobScanPath']):
            fpJobScan = open(self.serverData['jobScanPath'],'r')
            if fpJobScan:
               jobScanText = fpJobScan.readlines()
               fpJobScan.close()
               self.clientListener.postMessageBySize('jobScanUpdate',''.join(jobScanText))

         if self.clientListener.isConnected():
            self.serverData['disconnectTime'] = 0
            self.scheduleJobScan()
         else:
            self.serverData['disconnectTime'] = time.time() - self.clientListener.getConnectionCheckedTime()
#           log("Server has been disconnected for %f seconds." % (self.serverData['disconnectTime']))
            if self.serverData['disconnectTime'] >= self.serverData['disconnectMax']:
               message = "disconnect time (%d) >= disconnect max (%d)" % (self.serverData['disconnectTime'],
                                                                          self.serverData['disconnectMax'])
               self.logger.log(logging.ERROR,getLogMessage(message))
               if self.serverData['childPid']:
                  self.killChild()
               else:
                  self.exit(1)
            else:
               self.scheduleJobScan()


   def cancelTimeout(self):
      alarmFunction = signal.getsignal(signal.SIGALRM)
      if not alarmFunction in [None,signal.SIG_IGN,signal.SIG_DFL]:
         if cmp(alarmFunction,self.handleTimeout) == 0:
            signal.alarm(0)


   def configure(self):
      sectionPattern  = re.compile('(\s*\[)([^\s]*)(]\s*)')
      keyValuePattern = re.compile('( *)(\w*)( *= *)(.*[^\s$])( *)')
      commentPattern  = re.compile('\s*#.*')
      inServerSection = False

      fpConfig = open(self.configFilePath,'r')
      if fpConfig:
         eof = False
         while not eof:
            record = fpConfig.readline()
            if record != "":
               record = commentPattern.sub("",record)
               if   sectionPattern.match(record):
                  sectionName = sectionPattern.match(record).group(2)
                  inServerSection = (sectionName == 'server')
                  if inServerSection:
                     self.configData = {'listenURIs':[],
                                        'submitSSLcert':"/etc/submit/submit_server.crt",
                                        'submitSSLkey':"/etc/submit/submit_server.key",
                                        'submitSSLCA':"/etc/submit/submit_server_ca.crt",
                                        'mysqlHost':"",
                                        'mysqlUser':"",
                                        'mysqlPassword':"",
                                        'mysqlCA':"",
                                        'mysqlMiddlewareDB':"",
                                        'mysqlUserDB':"",
                                        'ldapHosts':[],
                                        'ldapBaseDN':"",
                                        'ldapUserDN':"",
                                        'loadLimit':510,
                                        'loadHalflife':3600,
                                        'loadHorizon':86400,
                                        'heartbeatInterval':3600,
                                        'useSetup':'/etc/environ.sh',
                                        'emailFrom':""
                                       }
               elif inServerSection:
                  if keyValuePattern.match(record):
                     key,value = keyValuePattern.match(record).group(2,4)
                     if key in self.configData:
                        if   isinstance(self.configData[key],list):
                           self.configData[key] = [e.strip() for e in value.split(',')]
                        elif isinstance(self.configData[key],bool):
                           self.configData[key] = bool(value.lower() == 'true')
                        elif isinstance(self.configData[key],float):
                           self.configData[key] = float(value)
                        elif isinstance(self.configData[key],int):
                           self.configData[key] = int(value)
                        else:
                           self.configData[key] = value
                     else:
                        self.logger.log(logging.WARNING,getLogMessage("Undefined key = value pair %s = %s" % (key,value)))
            else:
               eof = True
         fpConfig.close()

      configured = True
      if len(self.configData['listenURIs']) == 0:
         self.logger.log(logging.ERROR,getLogMessage("listenURIs missing from %s" % (self.configFilePath)))
         configured = False

      if self.configData['mysqlHost'] == "" or \
         self.configData['mysqlUser'] == "" or \
         self.configData['mysqlPassword'] == "" or \
         self.configData['mysqlMiddlewareDB'] == "":
         self.logger.log(logging.ERROR,getLogMessage("MySQL information missing from %s" % (self.configFilePath)))
         configured = False
      else:
         if self.configData['mysqlUserDB'] == "":
            self.configData['mysqlUserDB'] = self.configData['mysqlMiddlewareDB']

      if len(self.configData['ldapHosts']) == 0 or \
         self.configData['ldapBaseDN'] == "" or \
         self.configData['ldapUserDN'] == "":
         self.logger.log(logging.ERROR,getLogMessage("LDAP information missing from %s" % (self.configFilePath)))
         configured = False

      return(configured)


   def setupMySQL(self):
      setup = False
      self.mySQLDatabase = MySQLDatabase(mysqlHost=self.configData['mysqlHost'],
                                         mysqlUser=self.configData['mysqlUser'],
                                         mysqlPassword=self.configData['mysqlPassword'],
                                         mysqlCA=self.configData['mysqlCA'])
      if self.mySQLDatabase.connect(self.configData['mysqlMiddlewareDB']):
         self.mySQLDatabase.disconnect()
         if self.configData['mysqlUserDB'] != self.configData['mysqlMiddlewareDB']:
            if self.mySQLDatabase.connect(self.configData['mysqlUserDB']):
               self.mySQLDatabase.disconnect()
               setup = True
         else:
            setup = True

      return(setup)


   def setupClientListeners(self):
      self.clientListener = BoundConnection(self.configData['listenURIs'],
                                            submitSSLcert=self.configData['submitSSLcert'],
                                            submitSSLkey=self.configData['submitSSLkey'],
                                            submitSSLCA=self.configData['submitSSLCA'])

      return(self.clientListener.isListening())


   def setupAttachListener(self):
      if not self.serverData['socketPath']:
         self.serverData['socketPath'] = 'file://' + os.path.join(os.sep,'tmp','submit_%s' % (self.serverData['localJobId']))
         self.attachListener = BoundConnection([self.serverData['socketPath']])

      return(self.attachListener.isListening())


   def setupServerConnection(self):
      self.serverData['socketPath'] = 'file://' + os.path.join(os.sep,'tmp','submit_%s' % (self.serverData['attachJobId']))
      self.serverConnection = UnboundConnection(UnboundConnection.TLSREQUIREMENTNONE,
                                                listenURIs=[self.serverData['socketPath']],
                                                maximumConnectionPasses=1)
      isConnected = self.serverConnection.isConnected()

      return(isConnected)


   def showHelpUsage(self,
                     operationMode):
      if operationMode & self.commandParser.OPERATIONMODEHELPUSAGE:
         if operationMode & (self.commandParser.OPERATIONMODEHELPTOOLS | \
                             self.commandParser.OPERATIONMODEHELPVENUES | \
                             self.commandParser.OPERATIONMODEHELPMANAGERS | \
                             self.commandParser.OPERATIONMODEVERSIONDISTRIBUTOR):
            showHelp = False
         else:
            showHelp = True
      else:
         showHelp = False

      return(showHelp)


   def parseCommandArguments(self):
      argumentsOK = False
      continueExecution = False
      self.commandParser = CommandParser(self.clientData['doubleDashTerminator'])
      self.commandParser.parseArguments(self.clientData['args'][1:])

      if self.commandParser.validateArguments():
         self.serverData['operationMode'] = self.commandParser.getOperationMode()
         if self.serverData['operationMode'] & self.commandParser.OPERATIONMODEVERSIONSERVER:
            message = "Submit server version: %s\n" % (self.serverData['version'])
            self.clientListener.postMessageBySize('writeStdout',message)

         if   self.showHelpUsage(self.serverData['operationMode']):
            self.commandParser.showUsage()
            argumentsOK = True
         elif self.serverData['operationMode'] & (self.commandParser.OPERATIONMODERUNSTATUS | \
                                                  self.commandParser.OPERATIONMODERUNKILL | \
                                                  self.commandParser.OPERATIONMODERUNDISTRIBUTOR):
            argumentsOK = True
            continueExecution = True
         elif self.serverData['operationMode'] & self.commandParser.OPERATIONMODERUNSERVER:
            argumentsOK = True
         elif self.serverData['operationMode'] & self.commandParser.OPERATIONMODERUNPROXY:
            self.serverData['type']        = 'Proxy'
            self.serverData['attachId']    = self.commandParser.getOption('attachId')
            self.serverData['attachJobId'] = "%08d" % (self.serverData['attachId'])
            argumentsOK = True
            continueExecution = True
         elif self.serverData['operationMode'] & self.commandParser.OPERATIONMODERUNDISTRIBUTORID:
            exitCode = self.commandParser.setSweepParameters()
            if exitCode == 0:
               exitCode = self.commandParser.setCSVDataParameters()
               if exitCode == 0:
                  if self.commandParser.getParameterCombinationCount() > 0:
                     self.clientData['isParametric'] = True

                  self.clientData['localExecution'] = self.commandParser.getOption('localExecution')
                  self.clientData['reportMetrics']  = self.commandParser.getOption('metrics')
                  self.serverData['doHeartbeat']    = self.commandParser.getOption('doHeartbeat')
                  self.serverData['detach']         = self.commandParser.getOption('detach')
                  if self.commandParser.getOption('runName'):
                     self.serverData['runName']     = self.commandParser.getOption('runName')
                  argumentsOK = True
                  continueExecution = True

      return(argumentsOK,continueExecution)


   def ldapAuthenticate(self,
                        username,
                        password):
      if username and password:
         # Check the following note for more info on ldaps...
         # http://sourceforge.net/mailarchive/forum.php?forum_id=4346&max_rows=25&style=flat&viewmonth=200508

         for ldapHost in self.configData['ldapHosts']:
            try:
               if   ldapHost.startswith('ldaps://'):
                  ldap.set_option(ldap.OPT_X_TLS_REQUIRE_CERT,ldap.OPT_X_TLS_ALLOW)
                  ldapObject = ldap.initialize(ldapHost)
               elif ldapHost.startswith('ldap://'):
                  ldapObject = ldap.initialize(ldapHost)
               else:
                  self.logger.log(logging.ERROR,getLogMessage("Invalid ldapHost declaration.  Must be ldaps:// or ldap://"))
                  continue
            except ldap.LDAPError,msg:
               self.logger.log(logging.ERROR,getLogMessage("%s: %s" % (ldapHost,msg)))
               continue

            try:
               status = ldapObject.simple_bind(self.configData['ldapUserDN'] % (username),password)
               if status != 1:
                  self.logger.log(logging.ERROR,getLogMessage("Unable to bind to LDAP server as user '%s'" % (username)))
                  continue
            except ldap.LDAPError,msg:
               self.logger.log(logging.ERROR,getLogMessage("%s: %s" % (ldapHost,msg)))
               continue

            self.serverData['ldapAuthenticated'] = True
            ldapObject.unbind_s()
            break

      return(self.serverData['ldapAuthenticated'])


   def isValidUsername(self):
      validUsername = False
      username = self.clientData['userName']
      try:
         (login,pw,uid,gid,name,homedir,shell) = pwd.getpwnam(username)
         self.serverData['uid']           = uid
         self.serverData['gid']           = gid
         self.serverData['homeDirectory'] = homedir
         validUsername = True
      except:
         self.logger.log(logging.ERROR,getLogMessage("Unable to get info for user '%s'" % (username)))

      return(validUsername)


   def getUserGroups(self):
      userInGroups = False
      self.serverData['gids'] = []
      username = self.clientData['userName']
      try:
         idCommand = ['/usr/bin/id',username,'--groups']
         child = subprocess.Popen(idCommand,
                                  stdout=subprocess.PIPE,
                                  stderr=subprocess.PIPE,
                                  close_fds=True)
         idStdOutput,idStdError = child.communicate()
         idExitStatus = child.returncode
         if idExitStatus == 0:
            idStdOutput = str(idStdOutput).strip()
            for gid in idStdOutput.split():
               self.serverData['gids'].append(int(gid))
            userInGroups = True
         else:
            self.logger.log(logging.ERROR,getLogMessage("Unable to get gidlist for user '%s'" % (username)))
            if idStdOutput:
               self.logger.log(logging.ERROR,getLogMessage(idStdOutput))
            if idStdError:
               self.logger.log(logging.ERROR,getLogMessage(idStdError))
      except:
         # Not fatal...
         self.logger.log(logging.ERROR,getLogMessage("Unable to get gidlist for user '%s'" % (username)))
         self.logger.log(logging.ERROR,getLogMessage(traceback.format_exc()))

      return(userInGroups)


   @staticmethod
   def __generateToken(requestedTokenLength,
                       characterSet=""):
      """Generate a cleartext token using urandom, with even probability of each character.
      this is done by discarding random bytes that don't fit into an even multiple of the number
      of allowed characters. Based on maxwell middleware funcction.
      """
      if not characterSet:
         characterSet = string.letters + string.digits
      nCharacterSet = len(characterSet)
      # get numbers in a batch, it's faster than one by one
      randomCharacters = os.urandom(requestedTokenLength*2)
      tokenLength      = 0 # how many random characters we've generated so far
      iRandomCharacter = 0 # index to next unused random number in array
      token = ""
      maximumCharacterOrdinal = (256 / nCharacterSet) * nCharacterSet
      while tokenLength < requestedTokenLength:
         # reject results that would skew the freq. distribution of characters
         if ord(randomCharacters[iRandomCharacter]) < maximumCharacterOrdinal:
            token += characterSet[ord(randomCharacters[iRandomCharacter]) % nCharacterSet]
            tokenLength += 1
         iRandomCharacter += 1
         if iRandomCharacter >= requestedTokenLength*2:
            # we've run out of random numbers, get some more
            randomCharacters = os.urandom(requestedTokenLength*2)
            iRandomCharacter = 0

      return(token)


   def signon(self):
      authorized = False
      if self.isValidUsername() and self.getUserGroups():
         try:
            username = self.clientData['userName']
         except:
            username = ""
         try:
            password = self.clientData['password']
         except:
            password = ""
         try:
            sessionToken = self.clientData['sessionToken']
         except:
            sessionToken = ""
         jobToken = self.serverData['jobToken']

         if   username and jobToken:
            if self.mySQLDatabase.connect(self.configData['mysqlMiddlewareDB']):
               sqlCommand = """SELECT sessnum FROM job
                               WHERE username='%s' AND jobtoken='%s'
                            """ % (username,jobToken)
               result = self.mySQLDatabase.select(sqlCommand)
               if len(result) != 0:
                  row = result[0]
                  self.serverData['session'] = int(row[0])
                  authorized = True
                  self.serverData['mysqlAuthenticated'] = True
               else:
                  message = "User %s session %d token not matched." % (username,self.serverData['session'])
                  self.logger.log(logging.ERROR,getLogMessage(message))
               self.mySQLDatabase.disconnect()
         elif username and sessionToken:
            self.logger.log(logging.INFO,getLogMessage("session = %d" % (self.clientData['sessionId'])))
            if self.mySQLDatabase.connect(self.configData['mysqlMiddlewareDB']):
               sqlCommand = """SELECT sessnum FROM session
                               WHERE username='%s' AND sesstoken='%s'
                            """ % (username,sessionToken)
               result = self.mySQLDatabase.select(sqlCommand)
               if len(result) != 0:
                  row = result[0]
                  self.serverData['session'] = int(row[0])
                  authorized = True
                  self.serverData['mysqlAuthenticated'] = True
               self.mySQLDatabase.disconnect()
         elif username and password:
            authorized = self.ldapAuthenticate(username,password)
            if authorized:
               remoteIP = self.clientListener.getRemoteIP()
               appname = "submit"
               if self.mySQLDatabase.connect(self.configData['mysqlUserDB']):
                  sqlCommand = """SELECT jobsAllowed FROM jos_xprofiles WHERE username="%s"
                               """ % (username)
                  result = self.mySQLDatabase.select(sqlCommand)
                  if len(result) == 1:
                     row = result[0]
                     self.serverData['sessionLimit'] = int(row[0])

                     if self.mySQLDatabase.connect(self.configData['mysqlMiddlewareDB']):
                        sqlCommand = """SELECT COUNT(*) FROM session WHERE username='%s' AND dispnum!=0
                                     """ % (username)
                        result = self.mySQLDatabase.select(sqlCommand)
                        if len(result) == 1:
                           row = result[0]
                           sessionCount = int(row[0])
                           if sessionCount >= self.serverData['sessionLimit']:
                              message = "User %s cannot exceed session limit of %d" % (username,self.serverData['sessionLimit'])
                              self.logger.log(logging.ERROR,getLogMessage(message))
                              self.clientListener.postMessageBySize('writeStderr',"Session limit of %d reached.\n" %
                                                                                                  (self.serverData['sessionLimit']))
                              authorized = False
                           else:
                              sessionToken = self.__generateToken(32)
                              timeout = 60*60*24
                              sqlCommand = """INSERT INTO
       session(username,remoteip,exechost,dispnum,start,accesstime,appname,sessname,  sesstoken,timeout,venue)
        VALUES(    '%s',    '%s',    '%s',      0,now(),     now(),   '%s',    '%s',       '%s',     %f, '%s')
        """ % (username,remoteIP,"submit",                         appname,appname,sessionToken,timeout,self.serverData['hostFQDN'])
                              result = self.mySQLDatabase.insert(sqlCommand)
                              if result != "":
                                 message = "ERROR: Unable to create session record for '%s'" % (username)
                                 self.logger.log(logging.ERROR,getLogMessage(message))
                                 self.logger.log(logging.ERROR,getLogMessage("Error was: %s" % (result)))
                                 self.clientListener.postMessageBySize('message',"Internal database problem.")
                                 authorized = False
                              else:
                                 sqlCommand = """SELECT sessnum,sesstoken
                                                   FROM session
                                                  WHERE sessnum=LAST_INSERT_ID()"""
                                 result = self.mySQLDatabase.select(sqlCommand)
                                 row = result[0]
                                 self.serverData['session'] = int(row[0])
                                 self.logger.log(logging.INFO,getLogMessage("Created session %d" % (self.serverData['session'])))

                                 self.serverData['createdSessions'].append(self.serverData['session'])

                        self.mySQLDatabase.disconnect()
                     else:
                        authorized = False
               else:
                  authorized = False
            else:
               self.logger.log(logging.ERROR,getLogMessage("LDAP authentication failed for user '%s'" % (username)))

         if authorized:
            if self.mySQLDatabase.connect(self.configData['mysqlUserDB']):
               sqlCommand = """SELECT email FROM jos_xprofiles WHERE username="%s"
                            """ % (username)
               result = self.mySQLDatabase.select(sqlCommand)
               if len(result) == 1:
                  row = result[0]
                  email = row[0]
                  self.serverData['email'] = email
               self.mySQLDatabase.disconnect()

      return(authorized)


   def getInputObjects(self):
      clientListeningSockets,clientReader = self.clientListener.getInputObjects()
      if self.attachListener:
         attachListeningSocket,attachReader = self.attachListener.getInputObjects()
      else:
         attachListeningSocket = []
         attachReader          = []
      if self.serverConnection:
         serverReader = self.serverConnection.getInputObject()
      else:
         serverReader = []

      return(clientListeningSockets,
             clientReader,
             attachListeningSocket,
             attachReader,
             serverReader,
             self.serverData['inputObjects'].keys(),
             self.serverData['exportObjects'].keys())


   def getOutputObjects(self):
      clientWriter = self.clientListener.getOutputObjects()
      if self.attachListener:
         attachWriter = self.attachListener.getOutputObjects()
      else:
         attachWriter = []
      if self.serverConnection:
         serverWriter = self.serverConnection.getOutputObject()
      else:
         serverWriter = []

      importObjects = []
      for importObject in self.serverData['importObjects']:
         if self.serverData['importObjects'][importObject]['buffer'] != "":
            importObjects.append(importObject)

      outputObjects = []
      for outputObject in self.serverData['outputObjects']:
         if self.serverData['outputObjects'][outputObject]['buffer'] != "":
            outputObjects.append(outputObject)

      return(clientWriter,
             attachWriter,
             serverWriter,
             importObjects,
             outputObjects)


   def readFile(self,
                fileObject):
      if   fileObject in self.serverData['inputObjects']:
         inputFile = self.serverData['inputObjects'][fileObject]
         if   inputFile == '#STDOUT#' or inputFile == '#STDERR#':
            try:
               text = os.read(fileObject.fileno(),1024)
               if text == "":
                  self.closeFile(fileObject)
               else:
                  if inputFile == '#STDOUT#':
                     self.clientListener.postMessageBySize('writeStdout',text)
                  else:
                     self.clientListener.postMessageBySize('writeStderr',text)
            except:
               self.exit(1)
         elif inputFile == '#METRICS#':
            try:
               text = os.read(fileObject.fileno(),1024)
               if text == "":
                  self.closeFile(fileObject)
               else:
                  self.serverData['metrics'] += text
            except:
               self.exit(1)
         else:
            try:
               text = os.read(fileObject.fileno(),1024)
               if text == "":
                  self.clientListener.postMessage('close %s\n' % (inputFile))
                  self.closeFile(fileObject)
               else:
                  self.clientListener.postMessageBySize('write %s' % (inputFile),text)
            except:
               self.exit(1)
      elif fileObject in self.serverData['exportObjects']:
         inputFile = self.serverData['exportObjects'][fileObject]
         try:
            text = os.read(fileObject.fileno(),1024)
            if text == "":
               self.clientListener.postMessage('close %s\n' % (inputFile))
               self.closeFile(fileObject)
            else:
               self.clientListener.postMessageBySize('write %s' % (inputFile),text)
         except:
            self.exit(1)


   def writeFile(self,
                 fileObject):
      if   fileObject in self.serverData['importObjects']:
         writeBuffer = self.serverData['importObjects'][fileObject]['buffer']
         try:
            textLength = os.write(fileObject.fileno(),writeBuffer)
            self.serverData['importObjects'][fileObject]['buffer'] = writeBuffer[textLength:]
         except:
            self.exit(1)
      elif fileObject in self.serverData['outputObjects']:
         writeBuffer = self.serverData['outputObjects'][fileObject]['buffer']
         try:
            textLength = os.write(fileObject.fileno(),writeBuffer)
            self.serverData['outputObjects'][fileObject]['buffer'] = writeBuffer[textLength:]
         except:
            self.exit(1)


   def closeFile(self,
                 fileObject):
      if fileObject in self.serverData['inputObjects']:
         del self.serverData['inputObjects'][fileObject]

      if fileObject in self.serverData['exportObjects']:
         exportFile = self.serverData['exportObjects'][fileObject]
         exportPath = os.path.join(os.sep,"tmp",exportFile)
         if os.path.exists(exportPath):
            os.remove(exportPath)
         del self.serverData['exportObjects'][fileObject]

      transferTarPath = ""
      if fileObject in self.serverData['importObjects']:
         transferTarPath = self.serverData['importObjects'][fileObject]['path']
         del self.serverData['importObjects'][fileObject]
         if len(self.serverData['importObjects']) == 0:
            self.clientListener.postMessage('exportFilesComplete\n')

      if fileObject in self.serverData['outputObjects']:
         del self.serverData['outputObjects'][fileObject]

      try:
         fileObject.close()
         if transferTarPath:
            if   transferTarPath.endswith('.tar.gz'):
               transferPath = transferTarPath[:-7]
            elif transferTarPath.endswith('.tar'):
               transferPath = transferTarPath[:-4]
            else:
               transferPath = os.path.join(self.serverData['workDirectory'],"%s_transfer" % (self.serverData['jobId']))
            if not os.path.isdir(transferPath):
               os.makedirs(transferPath)

            tarCommand = ['tar','xv','-C',transferPath,'-f',transferTarPath]
            self.logger.log(logging.INFO,getLogMessage("command: %s" % (tarCommand)))
            child = subprocess.Popen(tarCommand,
                                     stdout=subprocess.PIPE,
                                     stderr=subprocess.PIPE,
                                     close_fds=True)
            tarStdOutput,tarStdError = child.communicate()
            tarExitStatus = child.returncode
            if tarExitStatus == 0:
               os.remove(transferTarPath)
               self.serverData['clientTransferPath'] = transferPath
            else:
               self.logger.log(logging.ERROR,getLogMessage("Failed to extract transfer tarfile %s" % (transferTarPath)))
               if tarStdOutput:
                  self.logger.log(logging.ERROR,getLogMessage(tarStdOutput))
               if tarStdError:
                  self.logger.log(logging.ERROR,getLogMessage(tarStdError))
      except:
         self.logger.log(logging.ERROR,getLogMessage(traceback.format_exc()))


   def checkClosePendingObjects(self):
      markedForDeletion = []
      for fileObject in self.serverData['closePendingObjects']:
         if   fileObject in self.serverData['importObjects']:
            if self.serverData['importObjects'][fileObject]['buffer'] == "":
               self.closeFile(fileObject)
               markedForDeletion.append(fileObject)
         elif fileObject in self.serverData['outputObjects']:
            if self.serverData['outputObjects'][fileObject]['buffer'] == "":
               self.closeFile(fileObject)
               markedForDeletion.append(fileObject)
      for fileObject in markedForDeletion:
         self.serverData['closePendingObjects'].remove(fileObject)
      del markedForDeletion


   def checkLoad(self):
      loadOK = False
      if self.mySQLDatabase.connect(self.configData['mysqlMiddlewareDB']):
         #
         # Check that the load is low enough
         #
         sqlCommand = """SELECT SUM( POW(.5, timestampdiff(second,start,now())/%d) )
                           FROM job WHERE username='%s' AND timestampdiff(second,start,now()) < %d
                      """ % (self.configData['loadHalflife'],self.clientData['userName'],self.configData['loadHorizon'])
         result = self.mySQLDatabase.select(sqlCommand)
         if len(result) != 1:
            self.logger.log(logging.ERROR,getLogMessage("Error retrieving load for user %s" % (self.clientData['userName'])))
            self.scheduleTimeout()
         else:
            row = result[0]
            try:
               load = float(row[0])
            except TypeError:
               # If there are no entries for this user, result will be NULL
               load = 0.

            if load < self.configData['loadLimit']:
               message = "Cumulative job load is %.2f.  (Max: %.2f)" % (load,self.configData['loadLimit'])
               self.logger.log(logging.INFO,getLogMessage(message))
               loadOK = True
            else:
               message = "User %s cannot exceed load limit %d." % (self.clientData['userName'],self.configData['loadLimit'])
               self.logger.log(logging.ERROR,getLogMessage(message))
               if not self.serverData['waitForLowLoad']:
                  msg = "You cannot exceed your active job limit of %f. (cur: %f)" % (self.configData['loadLimit'],load)
                  self.clientListener.postMessageBySize('message',msg)
                  self.clientListener.postMessage('jobId 0\n')
               else:
                  loadRatio = load/self.configData['loadLimit']
                  waitPeriod = math.ceil(self.configData['loadHalflife'] * math.log(loadRatio) / math.log(2))
                  msg = "Cumulative job load is %.2f.  (Max: %.2f)  Sleeping %d seconds." % (load,self.configData['loadLimit'],
                                                                                             waitPeriod)
                  self.logger.log(logging.INFO,getLogMessage(msg))
                  self.clientListener.postMessageBySize('message',msg)
                  self.scheduleTimeout(waitPeriod)

         self.mySQLDatabase.disconnect()

      return(loadOK)


   def checkLoadAndLaunch(self):
      if self.checkLoad():
         self.insertJob()
         if not self.serverData['runName']:
            self.serverData['runName'] = self.serverData['localJobId']
         self.logger.log(logging.INFO,getLogMessage("Args are:" + str(self.clientData['args'])))
         if self.clientData['localExecution']:
            self.launchJob()
         else:
            self.clientListener.postMessage('serverReadyForInputMapping\n')


   def createDetachedSession(self):
      sessionOK = False
      if self.mySQLDatabase.connect(self.configData['mysqlMiddlewareDB']):
         sessionToken = self.__generateToken(32)
         timeout = 60*60*24
         sqlCommand = """INSERT INTO
                              session(username,remoteip,exechost,dispnum,start,accesstime,appname,sessname,sesstoken,timeout,venue)
                         SELECT       username,remoteip,    '%s',      0,now(),     now(),appname,sessname,     '%s',     %f, '%s'
                           FROM session WHERE sessnum=%d
                      """ % ("submit",sessionToken,timeout,self.serverData['hostFQDN'],self.serverData['session'])
         result = self.mySQLDatabase.insert(sqlCommand)
         if result != "":
            message = "ERROR: Unable to create detached session record for '%s'\nError was: %s" % (self.clientData['userName'],
                                                                                                   result)
            self.clientListener.postMessageBySize('writeStderr',message + '\n')
            self.logger.log(logging.ERROR,getLogMessage(message))
         else:
            sqlCommand = """SELECT sessnum,sesstoken
                              FROM session
                             WHERE sessnum=LAST_INSERT_ID()"""
            result = self.mySQLDatabase.select(sqlCommand)
            row = result[0]
            self.serverData['session'] = int(row[0])
            self.logger.log(logging.INFO,getLogMessage("Created detached session %d" % (self.serverData['session'])))

            self.serverData['createdSessions'].append(self.serverData['session'])
            sessionOK = True

         self.mySQLDatabase.disconnect()

      return(sessionOK)


   def updateJobSession(self):
      jobSessionUpdated = True
      if self.mySQLDatabase.connect(self.configData['mysqlMiddlewareDB']):
         sqlCommand = """UPDATE job SET sessnum=%d
                          WHERE jobid='%d'
                      """ % (self.serverData['session'],self.serverData['jobId'])
         result = self.mySQLDatabase.update(sqlCommand)
         if len(result) != 0:
            self.logger.log(logging.ERROR,getLogMessage("ERROR: updateJobSession: %s" % (result)))
            jobSessionUpdated = False

         self.mySQLDatabase.disconnect()

      return(jobSessionUpdated)


   def insertJob(self):
      if self.serverData['operationMode'] & self.commandParser.OPERATIONMODERUNDISTRIBUTORID:
         if self.mySQLDatabase.connect(self.configData['mysqlMiddlewareDB']):
            #
            # Check that the superjob is a job that we own.
            #
            if self.serverData['superJobId'] != 0:
               sqlCommand = """SELECT jobid FROM job
                                WHERE jobid=%d AND sessnum=%d
                            """ % (self.serverData['superJobId'],self.serverData['session'])
               result = self.mySQLDatabase.select(sqlCommand)
               if len(result) != 1:
                  message = "ERROR: Trying to claim superjob of %d." % (self.serverData['superJobId'])
                  self.logger.log(logging.ERROR,getLogMessage(message))
                  self.serverData['superJobId'] = 0
            self.serverData['jobToken'] = self.__generateToken(32)
            #
            # Insert the job into the job table.
            #
            if self.serverData['doHeartbeat']:
               active = 1
            else:
               active = 2
            sqlCommand = """INSERT INTO job(sessnum,superjob,username,event,ncpus,venue,start,heartbeat,active,jobtoken)
                                 VALUES(         %d,      %d,    '%s', '%s',   %d, '%s',now(),    now(),    %d,    '%s')
                         """ % (self.serverData['session'],self.serverData['superJobId'],self.clientData['userName'],
                                self.serverData['event'],self.serverData['ncpus'],self.serverData['venue'],
                                active,self.serverData['jobToken'])
            result = self.mySQLDatabase.insert(sqlCommand)
            if result != "":
               self.logger.log(logging.ERROR,getLogMessage("ERROR: Unable to insert job."))
               self.logger.log(logging.ERROR,getLogMessage("Error was: %s" % (result)))
            else:
               sqlCommand = """SELECT LAST_INSERT_ID()"""
               result = self.mySQLDatabase.select(sqlCommand)
               row = result[0]
               self.serverData['jobId'] = int(row[0])
               self.logger.log(logging.INFO,getLogMessage("Assigning jobId = %d:" % (self.serverData['jobId'])))
               logSetJobId(self.serverData['jobId'])
               self.serverData['localJobId'] = "%08d" % (self.serverData['jobId'])
               self.serverData['dbJobEntryExists'] = True

               sqlCommand = """UPDATE session SET accesstime=now()
                                WHERE sessnum=%d
                            """ % (self.serverData['session'])
               result = self.mySQLDatabase.update(sqlCommand)
               if result != "":
                  message = "ERROR: Unable to update session accesstime for jobid %d." % (self.serverData['jobId'])
                  self.logger.log(logging.ERROR,getLogMessage(message))
                  self.logger.log(logging.ERROR,getLogMessage("Error was: %s" % (result)))

            self.mySQLDatabase.disconnect()


   def updateJob(self):
      if self.serverData['operationMode'] & self.commandParser.OPERATIONMODERUNDISTRIBUTORID:
         if self.mySQLDatabase.connect(self.configData['mysqlMiddlewareDB']):
            sqlCommand = """UPDATE job SET event='%s',ncpus=%d
                             WHERE jobid=%d
                         """ % (self.serverData['event'],self.serverData['ncpus'],self.serverData['jobId'])
            result = self.mySQLDatabase.update(sqlCommand)
            if result != "":
               message = "ERROR: Unable to update job fields for jobid %d." % (self.serverData['jobId'])
               self.logger.log(logging.ERROR,getLogMessage(message))
               self.logger.log(logging.ERROR,getLogMessage("Error was: %s" % (result)))

            self.mySQLDatabase.disconnect()


   def invokeLocal(self):
      pegasusRCPath    = os.path.join(self.serverData['distributorDirectory'],'pegasus_local.rc')
      pegasusSitesPath = os.path.join(self.serverData['distributorDirectory'],'sites_local.xml')
      if os.path.exists(pegasusRCPath) and os.path.exists(pegasusSitesPath):
         fpPegasus = open(pegasusRCPath,'r')
         if fpPegasus:
            pegasusRCText = fpPegasus.readlines()
            fpPegasus.close()
            self.clientListener.postMessageBySize('pegasusRC',''.join(pegasusRCText))
         fpPegasus = open(pegasusSitesPath,'r')
         if fpPegasus:
            pegasusSitesText = fpPegasus.readlines()
            fpPegasus.close()
            self.clientListener.postMessageBySize('pegasusSites',''.join(pegasusSitesText))
      else:
         if self.clientData['isParametric']:
            msg = "Pegasus local configuration files are missing"
            self.logger.log(logging.ERROR,getLogMessage(msg))
            self.clientListener.postMessageBySize('writeStderr',msg + '\n')

      self.clientListener.postMessage('serverReadyForIO\n')
      self.scheduleHeartbeat()


   def __isClientFileShared(self,
                            clientFile,
                            clientInode,
                            clientMountDevice):
      clientFileShared = True
      if os.path.exists(clientFile):
         inode = os.lstat(clientFile).st_ino
         if inode != clientInode:
            if clientMountDevice and len(self.serverData['mountPoints']) > 0:
               mountDevice = None
               mountPoint = clientFile
               while mountPoint:
                  if os.path.ismount(mountPoint):
                     mountDevice = self.serverData['mountPoints'][mountPoint]['device']
                     break
                  else:
                     if mountPoint != os.sep:
                        mountPoint = os.path.dirname(mountPoint)
                     else:
                        break
               if mountDevice != clientMountDevice:
                  clientFileShared = False
            else:
               clientFileShared = False
      else:
         clientFileShared = False

      return(clientFileShared)


   def setupWorkDirectory(self):
      if self.__isClientFileShared(self.clientData['workDirectory'],
                                   self.clientData['workDirectoryInode'],
                                   self.clientData['workDirectoryDevice']):
         self.serverData['workDirectory'] = self.clientData['workDirectory']
      else:
         sessionDirectory = os.path.join(self.serverData['homeDirectory'],'data','sessions',str(self.serverData['session']))
         if not os.path.isdir(sessionDirectory):
            os.makedirs(sessionDirectory)
         self.serverData['workDirectory'] = tempfile.mkdtemp(prefix='run_',dir=sessionDirectory)
         self.serverData['workDirectoryShared'] = False

      os.chdir(self.serverData['workDirectory'])


   def unsetupWorkDirectory(self):
      if not self.serverData['workDirectoryShared']:
         os.chdir(os.path.join(os.sep,'tmp'))
         if os.path.isdir(self.serverData['workDirectory']):
            shutil.rmtree(self.serverData['workDirectory'],True)


   def filterEnvironment(self):
      self.serverData['envs'] = {}
      if self.serverData['operationMode'] & self.commandParser.OPERATIONMODERUNDISTRIBUTORID:
         for env in self.clientData['envs']:
            if not env[0].isalpha():
               continue
            if env == "SUBMIT_JOB":
               self.serverData['envs'][env] = self.clientData['envs'][env]
               continue
            if env == "SUBMIT_APPLICATION_REVISION":
               self.serverData['envs'][env] = self.clientData['envs'][env]
               continue
            if env == "SUBMITVENUES":
               self.serverData['envs'][env] = self.clientData['envs'][env]
               continue
            if env.endswith("_CHOICE"):
               self.serverData['envs'][env] = self.clientData['envs'][env]
               continue

         if self.clientData['isClientTTY']:
            self.serverData['envs']['SUBMIT_ISCLIENTTTY'] = '1'

         if self.clientData['workDirectory']:
            self.serverData['envs']['CLIENT_WORK_DIRECTORY'] = self.clientData['workDirectory']
         if self.serverData['clientTransferPath']:
            self.serverData['envs']['CLIENT_TRANSFER_PATH'] = self.serverData['clientTransferPath']
         if self.serverData['pegasusVersion']:
            self.serverData['envs']['PEGASUS_VERSION'] = self.serverData['pegasusVersion']
         if self.serverData['pegasusHome']:
            self.serverData['envs']['PEGASUS_HOME'] = self.serverData['pegasusHome']
         if self.serverData['pegasusPythonPath']:
            self.serverData['envs']['PYTHONPATH'] = self.serverData['pegasusPythonPath']
         if self.configData['useSetup']:
            self.serverData['envs']['USE_SETUP_SCRIPT'] = self.configData['useSetup']
         if self.serverData['email']:
            self.serverData['envs']['USER_EMAIL'] = self.serverData['email']
         if self.configData['emailFrom']:
            self.serverData['envs']['HUB_EMAIL_FROM'] = self.configData['emailFrom']

      self.serverData['envs']['DOUBLE_DASH_TERMINATOR'] = str(self.clientData['doubleDashTerminator'])
      self.serverData['envs']['HOME']                   = self.serverData['homeDirectory']
      self.serverData['envs']['USER']                   = self.clientData['userName']
      self.serverData['envs']['LOGNAME']                = self.clientData['userName']
      self.serverData['envs']['SESSION']                = str(self.serverData['session'])
      self.serverData['envs']['PATH']                   = "/usr/bin:/bin"


   def mapArguments(self):
      for arg in self.clientData['args']:
         self.serverData['args'].append(arg)

      if   self.serverData['operationMode'] & self.commandParser.OPERATIONMODERUNSTATUS:
         jobStatusPath = os.path.join(self.serverData['jobStatusDirectory'],
                                      self.serverData['jobStatusScript'])
         self.serverData['args'][0] = jobStatusPath
      elif self.serverData['operationMode'] & self.commandParser.OPERATIONMODERUNKILL:
         jobKillPath = os.path.join(self.serverData['jobKillDirectory'],
                                    self.serverData['jobKillScript'])
         self.serverData['args'][0] = jobKillPath
      else:
         distributorPath = os.path.join(self.serverData['distributorDirectory'],
                                        self.serverData['distributorScript'])
         self.serverData['args'][0] = distributorPath
         self.serverData['args'].insert(1, "--jobid")
         self.serverData['args'].insert(2, str(self.serverData['jobId']))


   def invokeRunStatus(self):
      stdinPipe  = os.pipe()
      stdoutPipe = os.pipe()
      stderrPipe = os.pipe()
      self.serverData['childPid'] = os.fork()
      if self.serverData['childPid'] == 0:
         os.setsid()
         os.dup2(stdinPipe[0],0)
         os.dup2(stdoutPipe[1],1)
         os.dup2(stderrPipe[1],2)
         os.close(stdinPipe[1])
         os.close(stdoutPipe[0])
         os.close(stderrPipe[0])

         try:
            os.execve(self.serverData['args'][0],self.serverData['args'],self.serverData['envs'])
         except OSError,err:
            self.logger.log(logging.ERROR,getLogMessage("invokeRunStatus: exec %s failed" % (self.serverData['args'][0])))
            os.write(2,"Cannot invoke runstatus.\n")
            os.write(2,err.args[1]+'\n')
            sys.exit(err.args[0])
      else:
         fpStdinPipe  = os.fdopen(stdinPipe[1],'w',0)
         fpStdoutPipe = os.fdopen(stdoutPipe[0],'r',0)
         fpStderrPipe = os.fdopen(stderrPipe[0],'r',0)
         os.close(stdinPipe[0])
         os.close(stdoutPipe[1])
         os.close(stderrPipe[1])
         self.serverData['outputObjects'][fpStdinPipe]           = {}
         self.serverData['outputObjects'][fpStdinPipe]['path']   = '#STDIN#'
         self.serverData['outputObjects'][fpStdinPipe]['buffer'] = ""
         self.serverData['inputObjects'][fpStdoutPipe] = '#STDOUT#'
         self.serverData['inputObjects'][fpStderrPipe] = '#STDERR#'

         # At this point, the I/O is ready to receive things from the client.
         # Inform the client.
         self.clientListener.postMessage('serverReadyForIO\n')


   def invokeRunKill(self):
      stdinPipe  = os.pipe()
      stdoutPipe = os.pipe()
      stderrPipe = os.pipe()
      self.serverData['childPid'] = os.fork()
      if self.serverData['childPid'] == 0:
         os.setsid()
         os.dup2(stdinPipe[0],0)
         os.dup2(stdoutPipe[1],1)
         os.dup2(stderrPipe[1],2)
         os.close(stdinPipe[1])
         os.close(stdoutPipe[0])
         os.close(stderrPipe[0])

         try:
            os.execve(self.serverData['args'][0],self.serverData['args'],self.serverData['envs'])
         except OSError,err:
            os.write(2,"Cannot invoke runkill.\n")
            os.write(2,err.args[1]+'\n')
            sys.exit(err.args[0])
      else:
         fpStdinPipe  = os.fdopen(stdinPipe[1],'w',0)
         fpStdoutPipe = os.fdopen(stdoutPipe[0],'r',0)
         fpStderrPipe = os.fdopen(stderrPipe[0],'r',0)
         os.close(stdinPipe[0])
         os.close(stdoutPipe[1])
         os.close(stderrPipe[1])
         self.serverData['outputObjects'][fpStdinPipe]           = {}
         self.serverData['outputObjects'][fpStdinPipe]['path']   = '#STDIN#'
         self.serverData['outputObjects'][fpStdinPipe]['buffer'] = ""
         self.serverData['inputObjects'][fpStdoutPipe] = '#STDOUT#'
         self.serverData['inputObjects'][fpStderrPipe] = '#STDERR#'

         # At this point, the I/O is ready to receive things from the client.
         # Inform the client.
         self.clientListener.postMessage('serverReadyForIO\n')


   def invokeDistributor(self):
      if not self.serverData['workDirectoryShared']:
         try:
            fpMarker = open('.__fileTimeMarkerSubmit','w')
            fpMarker.close()
            time.sleep(1)
         except:
            pass
      stdinPipe  = os.pipe()
      stdoutPipe = os.pipe()
      stderrPipe = os.pipe()
      metricPipe = os.pipe()
      self.serverData['childPid'] = os.fork()
      if self.serverData['childPid'] == 0:
         os.setsid()
         os.dup2(stdinPipe[0],0)
         os.dup2(stdoutPipe[1],1)
         os.dup2(stderrPipe[1],2)
         os.dup2(metricPipe[1],3)
         os.close(stdinPipe[1])
         os.close(stdoutPipe[0])
         os.close(stderrPipe[0])
         os.close(metricPipe[0])
         os.close(stdinPipe[0])
         os.close(stdoutPipe[1])
         os.close(stderrPipe[1])
         os.close(metricPipe[1])
         self.clientListener.closeConnection()

         try:
            os.execve(self.serverData['args'][0],self.serverData['args'],self.serverData['envs'])
         except OSError,err:
            os.write(2,"Cannot invoke distributor.\n")
            os.write(2,err.args[1]+'\n')
            sys.exit(err.args[0])
      else:
         fpStdinPipe  = os.fdopen(stdinPipe[1],'w',0)
         fpStdoutPipe = os.fdopen(stdoutPipe[0],'r',0)
         fpStderrPipe = os.fdopen(stderrPipe[0],'r',0)
         fpMetricPipe = os.fdopen(metricPipe[0],'r',0)
         os.close(stdinPipe[0])
         os.close(stdoutPipe[1])
         os.close(stderrPipe[1])
         os.close(metricPipe[1])
         self.serverData['outputObjects'][fpStdinPipe]           = {}
         self.serverData['outputObjects'][fpStdinPipe]['path']   = '#STDIN#'
         self.serverData['outputObjects'][fpStdinPipe]['buffer'] = ""
         self.serverData['inputObjects'][fpStdoutPipe] = '#STDOUT#'
         self.serverData['inputObjects'][fpStderrPipe] = '#STDERR#'
         self.serverData['inputObjects'][fpMetricPipe] = '#METRICS#'

         # At this point, the I/O is ready to receive things from the client.
         # Inform the client.
         self.clientListener.postMessage('serverReadyForIO\n')

         if self.clientData['isParametric'] and self.clientData['isClientTTY'] and not self.serverData['detach']:
            self.scheduleJobScan()
         else:
            self.scheduleHeartbeat()


   def startCommand(self):
      self.scheduleHeartbeat()

      self.setUserGroupIds()
      self.filterEnvironment()
      self.mapArguments()
      os.umask(self.clientData['umask'])

      if   self.serverData['operationMode'] & self.commandParser.OPERATIONMODERUNSTATUS:
         self.invokeRunStatus()
      elif self.serverData['operationMode'] & self.commandParser.OPERATIONMODERUNKILL:
         self.invokeRunKill()
      else:
         self.invokeDistributor()
         self.serverData['detach'] = False


   def launchJob(self):
      self.updateJob()
      if self.serverData['operationMode'] & self.commandParser.OPERATIONMODERUNDISTRIBUTORID:
         self.clientListener.postMessage('jobId %d\n' % (self.serverData['jobId']))
         self.clientListener.postMessage('jobToken %s\n' % (self.serverData['jobToken']))
         self.clientListener.postMessage('event %s\n' % (self.serverData['event']))

      if self.clientData['localExecution']:
         self.clientListener.postMessage('operationMode %d\n' % (self.serverData['operationMode']))
         if self.serverData['doHeartbeat']:
            self.clientListener.postMessage('heartbeatInterval %d\n' % (self.configData['heartbeatInterval']))
         self.invokeLocal()
      else:
         self.startCommand()


   def updateJobMetrics(self,
                        metricsRecord):
      if not self.serverData['metricsRecorded']:
         if self.mySQLDatabase.connect(self.configData['mysqlMiddlewareDB']):
            status   = 0
            waittime = 0.
            cputime  = 0.
            realtime = 0.
            ncpus    = 0
            program  = self.serverData['event']

            metricsRecord = metricsRecord.strip()
            metrics = metricsRecord.split()
            for metric in metrics:
               try:
                  key,value = metric.split('=')
                  if   key == 'status':
                     status = int(value)
                     self.serverData['exitCode'] = status
                  elif key == 'cputime':
                     cputime = float(value)
                  elif key == 'realtime':
                     realtime = float(value)
                  elif key == 'waittime':
                     waittime = float(value)
                  elif key == 'ncpus':
                     ncpus = int(value)
                  elif key == 'venue':
                     self.serverData['venue'] = value
                  elif key == 'event':
                     program = value
                  else:
                     self.logger.log(logging.ERROR,getLogMessage("Unknown status item: '%s'" % (metric)))
               except:
                  self.logger.log(logging.ERROR,getLogMessage("Erroneous status item: '%s'" % (metric)))

            metricsMessage = " venue=%s status=%d cpu=%f real=%f wait=%f ncpus=%d" % (self.serverData['venue'],
                                                                                      status,cputime,realtime,waittime,ncpus)
            self.logger.log(logging.INFO,getLogMessage("Job Status:" + metricsMessage))
            if self.clientData['reportMetrics'] and self.serverData['venue'] != "":
               self.logger.log(logging.INFO,getLogMessage("Sending requested metrics."))
               msg = "=SUBMIT-METRICS=>"
               msg += " job=%s" % (self.serverData['localJobId'])
               msg += metricsMessage + '\n'
               self.clientListener.postMessageBySize('writeStderr',msg)

            if waittime > 0:
               sqlCommand = """INSERT INTO joblog(sessnum,  job,superjob,event,start,walltime,venue)
                                    SELECT        sessnum,jobid,superjob, '%s',start,      %f, '%s'
                                      FROM job WHERE jobid=%d
                            """ % ("[waiting]",waittime,self.serverData['venue'],self.serverData['jobId'])
               result = self.mySQLDatabase.insert(sqlCommand)
               if result != "":
                  self.logger.log(logging.ERROR,getLogMessage("ERROR: Unable to create wait time record. (%f)" % (waittime)))
                  self.logger.log(logging.ERROR,getLogMessage("Error was: %s" % (result)))

            sqlCommand = """INSERT INTO joblog(sessnum,  job,superjob,event,start,walltime,cputime,ncpus,status,venue)
                                 SELECT        sessnum,jobid,superjob, '%s',start,      %f,     %f,   %d,    %d, '%s'
                                   FROM job WHERE jobid=%d
                         """ % (program,realtime,cputime,ncpus,status,self.serverData['venue'],self.serverData['jobId'])
            result = self.mySQLDatabase.insert(sqlCommand)
            if result != "":
               self.logger.log(logging.ERROR,getLogMessage("ERROR: Unable to copy job %d to joblog" % (self.serverData['jobId'])))
               self.logger.log(logging.ERROR,getLogMessage("Error was: %s" % (result)))

            self.serverData['metricsRecorded'] = True

            self.mySQLDatabase.disconnect()


   def finalizeJob(self):
      if self.serverData['dbJobEntryExists']:
         if self.serverData['jobId'] > 0:
            if self.mySQLDatabase.connect(self.configData['mysqlMiddlewareDB']):
               sqlCommand = """SELECT sessnum FROM job
                                WHERE jobid=%d
                            """ % (self.serverData['jobId'])
               result = self.mySQLDatabase.select(sqlCommand)
               if len(result) == 0:
                  message = "ERROR: Unable to find session for job %d." % (self.serverData['jobId'])
                  self.logger.log(logging.ERROR,getLogMessage(message))
               else:
                  row = result[0]
                  session = row[0]

                  sqlCommand = """UPDATE session SET accesstime=now()
                                   WHERE sessnum=%d
                               """ % (session)
                  result = self.mySQLDatabase.update(sqlCommand)
                  if result != "":
                     self.logger.log(logging.ERROR,getLogMessage("ERROR: Unable to update session accesstime for job."))
                     self.logger.log(logging.ERROR,getLogMessage("Error was: %s" % result))

               sqlCommand = """UPDATE job SET active=0
                                WHERE jobid=%d
                            """ % (self.serverData['jobId'])
               result = self.mySQLDatabase.update(sqlCommand)
               if result != "":
                  self.logger.log(logging.ERROR,getLogMessage("Unable to deactivate job."))
                  self.logger.log(logging.ERROR,getLogMessage("Error was: " + result))

               #
               # Clear inactive jobs beyond the load horizon.
               #
               sqlCommand = """DELETE FROM job
                                WHERE active=0 AND timestampdiff(second,job.start,now()) > %d
                            """ % (self.configData['loadHorizon'])
               result = self.mySQLDatabase.delete(sqlCommand)
               if result != "":
                  self.logger.log(logging.ERROR,getLogMessage("Unable to clear old inactive jobs."))
                  self.logger.log(logging.ERROR,getLogMessage("Error was: " + result))

               #
               # Mark as noheartbeat jobs that have start time = heartbeat time.
               #
               sqlCommand = """UPDATE job SET active=2
                                WHERE active=1 AND timestampdiff(second,job.start,now()) > %d AND
                                                   timestampdiff(second,job.start,job.heartbeat) < 2
                            """ % (self.configData['heartbeatInterval']*2)
               result = self.mySQLDatabase.update(sqlCommand)
               if result != "":
                  self.logger.log(logging.ERROR,getLogMessage("Unable to set noHeartbeat job entries."))
                  self.logger.log(logging.ERROR,getLogMessage("Error was: " + result))

               #
               # Mark inactive any jobs that have not had a recent heartbeat.
               #
               sqlCommand = """UPDATE job SET active=0
                                WHERE active=1 AND timestampdiff(second,job.heartbeat,now()) > %d
                            """ % (self.configData['heartbeatInterval']*3)
               result = self.mySQLDatabase.update(sqlCommand)
               if result != "":
                  self.logger.log(logging.ERROR,getLogMessage("Unable to deactivate moribund job entries."))
                  self.logger.log(logging.ERROR,getLogMessage("Error was: " + result))

               #
               # Mark inactive any jobs that have no heartbeat and have been moved to joblog.
               #
               sqlCommand = """UPDATE job INNER JOIN joblog ON jobid=job SET job.active=0
                                WHERE job.active=2 AND timestampdiff(second,job.start,now()) > %d
                            """ % (self.configData['loadHorizon'])
               result = self.mySQLDatabase.update(sqlCommand)
               if result != "":
                  self.logger.log(logging.ERROR,getLogMessage("Unable to deactivate recorded non-heartbeat job entries."))
                  self.logger.log(logging.ERROR,getLogMessage("Error was: " + result))

               self.mySQLDatabase.disconnect()

         self.serverData['dbJobEntryExists'] = False


   def finalizeCreatedSessions(self):
      if len(self.serverData['createdSessions']) > 0:
         if self.mySQLDatabase.connect(self.configData['mysqlMiddlewareDB']):
            for createdSession in reversed(self.serverData['createdSessions']):
               status   = 0
               sqlCommand = """INSERT INTO
    sessionlog(sessnum,username,remoteip,exechost,dispnum,start,appname,status,                                      walltime,venue)
SELECT         sessnum,username,remoteip,exechost,dispnum,start,appname,    %d,time_to_sec(timediff(now(),start)) AS walltime,venue
                              FROM session WHERE sessnum=%d
                            """ % (status,createdSession)
               result = self.mySQLDatabase.insert(sqlCommand)
               if result != "":
                  message = "ERROR: Unable to create sessionlog record for '%s'\nError was: %s" % (self.clientData['userName'],
                                                                                                   result)
                  self.clientListener.postMessageBySize('writeStderr',message + '\n')
                  self.logger.log(logging.ERROR,getLogMessage(message))
               else:
                  sqlCommand = """DELETE FROM session
                                   WHERE sessnum=%d
                               """ % (createdSession)
                  result = self.mySQLDatabase.delete(sqlCommand)
                  if result != "":
                     self.logger.log(logging.ERROR,getLogMessage("Unable to delete creted session %d" % (createdSession)))
                     self.logger.log(logging.ERROR,getLogMessage("Error was: " + result))

            self.mySQLDatabase.disconnect()


   def setUserGroupIds(self):
      if os.getuid() == 0:
         os.setregid(self.serverData['gid'],self.serverData['gid'])
         os.setgroups(self.serverData['gids'])
         os.setreuid(self.serverData['uid'],self.serverData['uid'])


   def attach(self):
# Proxy
      if self.setupServerConnection():
         self.logger.log(logging.INFO,getLogMessage("Attach to %d" % (self.serverData['attachId'])))
         self.serverConnection.postMessage('attached\n')
      else:
         message = "Unable to attach to job %s,\nPerhaps it has already completed." % (self.serverData['attachJobId'])
         self.clientListener.postMessageBySize('writeStderr',message + '\n')
         self.logger.log(logging.ERROR,getLogMessage(message))
         self.exit(1)


   def processAttachRequests(self):
# JobExecuter
      if self.attachListener:
         message = self.attachListener.pullMessage(0)
         while message:
            args = message.split()
            if args[0] != 'null':
               self.logger.log(logging.INFO,getLogMessage("attach request = %s" % (args[0])))

            if args[0] == 'attached':
               if not self.serverData['childHasExited']:
                  self.attachListener.postMessage('attached\n')
                  attachChannel,fromAttachBuffer,toAttachBuffer = self.attachListener.setChannelAndBuffers(None,"","")
                  self.clientListener.closeConnection()
                  clientChannel,fromClientBuffer,toClientBuffer = self.clientListener.setChannelAndBuffers(attachChannel,
                                                                                                           fromAttachBuffer,
                                                                                                           toAttachBuffer)
                  self.clientListener.pushMessage(fromClientBuffer)
                  self.clientListener.postMessage('attached\n')
                  self.clientListener.postMessage('runName %s %d\n' % (self.serverData['runName'],self.serverData['jobId']))
                  self.clientListener.postMessage(toClientBuffer)
                  self.serverData['detached'] = False
               else:
                  self.attachListener.postMessage('attachFailed\n')
            else:
               self.logger.log(logging.ERROR,getLogMessage("Discarded attach message: %s" % (message)))

            message = self.attachListener.pullMessage(0)


   def processServerRequests(self):
# Proxy
      if self.serverConnection:
         message = self.serverConnection.pullMessage(0)
         while message:
            args = message.split()
            if args[0] != 'null':
               self.logger.log(logging.INFO,getLogMessage("server request = %s" % (args[0])))

            if   args[0] == 'attached':
               self.serverData['attached'] = True
               break
            elif args[0] == 'attachFailed':
               message = "Unable to attach to job %s,\nPerhaps it is in the final stages of completion." % \
                                                                                  (self.serverData['attachJobId'])
               self.clientListener.postMessageBySize('writeStderr',message + '\n')
               self.logger.log(logging.ERROR,getLogMessage(message))
               self.exit(1)
            else:
               self.logger.log(logging.ERROR,getLogMessage("Discarded server message: %s" % (message)))

            message = self.serverConnection.pullMessage(0)


   def processClientRequests(self):
      message = self.clientListener.pullMessage(0)
      while message:
         args = message.split()
#        if args[0] != 'null':
#           self.logger.log(logging.DEBUG,getLogMessage("client request = %s" % (args[0])))

         if   args[0] == 'userName':
            self.clientData['userName'] = args[1]
         elif args[0] == 'password':
            self.clientData['password'] = args[1]
            self.serverData['authAttempts'] += 1
         elif args[0] == 'sessionId':
            self.clientData['sessionId'] = int(args[1])
         elif args[0] == 'sessionToken':
            self.clientData['sessionToken'] = args[1]
            self.serverData['authAttempts'] += 1
         elif args[0] == 'signon':
            self.serverData['authz'] = self.signon()
            self.clientListener.postMessage('authz %d\n' % (self.serverData['authz']))
            if self.serverData['authz'] != 0:
               self.setUserGroupIds()
               self.clientListener.postMessageBySize('message',"Congratulations - you have successfully authenticated.")
               if self.serverData['jobId'] == 0:
                  self.setupWorkDirectory()
                  argumentsOK,continueExecution = self.parseCommandArguments()
                  if argumentsOK:
                     if continueExecution:
                        if self.serverData['operationMode'] & self.commandParser.OPERATIONMODERUNPROXY:
                           self.attach()
                           self.cancelTimeout()
                        else:
                           self.scheduleHeartbeat()
                     else:
                        self.exit(0)
                  else:
                     self.logger.log(logging.ERROR,getLogMessage("Command line argument parsing failed"))
                     self.clientListener.postMessageBySize('writeStderr',"Command line argument parsing failed\n")
                     self.exit(1)
               else:
                  self.serverData['dbJobEntryExists'] = True
            else:
               self.scheduleTimeout()
         elif args[0] == 'clientVersion':
            self.clientData['version'] = args[1]
            self.clientListener.postMessage('serverVersion %s\n' % (self.serverData['version']))
         elif args[0] == 'doubleDashTerminator':
            self.clientData['doubleDashTerminator'] = bool(int(args[1]))
         elif args[0] == 'arg':
            argLength = int(args[1])
            if argLength > 0:
               argText = self.clientListener.pullMessage(argLength)
               if len(argText) > 0:
                  self.clientData['args'].append(argText)
               else:
                  self.clientListener.pushMessage(message + '\n')
                  break
         elif args[0] == 'var':
            envNameLength = int(args[1])
            envValueLength = int(args[2])
            if envNameLength+envValueLength > 0:
               envText = self.clientListener.pullMessage(envNameLength+envValueLength)
               if len(envText) > 0:
                  envName  = envText[0:envNameLength]
                  envValue = envText[envNameLength:]
                  self.clientData['envs'][envName] = envValue
               else:
                  self.clientListener.pushMessage(message + '\n')
                  break
         elif args[0] == 'inputFile':
            clientInputInode      = long(args[1])
            clientInputPathLength = int(args[2])
            if clientInputPathLength > 0:
               clientInputPath = self.clientListener.pullMessage(clientInputPathLength)
               if len(clientInputPath) > 0:
                  mapInputFile = not self.__isClientFileShared(clientInputPath,clientInputInode,None)
                  self.clientData['inputFileMapping'][clientInputPath] = mapInputFile
               else:
                  self.clientListener.pushMessage(message + '\n')
                  break
         elif args[0] == 'inputFileWithMount':
            clientInputInode        = long(args[1])
            clientInputPathLength   = int(args[2])
            clientMountDeviceLength = int(args[3])
            if clientInputPathLength+clientMountDeviceLength > 0:
               inputFileDeviceText = self.clientListener.pullMessage(clientInputPathLength+clientMountDeviceLength)
               if len(inputFileDeviceText) > 0:
                  clientInputPath   = inputFileDeviceText[0:clientInputPathLength]
                  clientMountDevice = inputFileDeviceText[clientInputPathLength:]
                  mapInputFile = not self.__isClientFileShared(clientInputPath,clientInputInode,clientMountDevice)
                  self.clientData['inputFileMapping'][clientInputPath] = mapInputFile
               else:
                  self.clientListener.pushMessage(message + '\n')
                  break
         elif args[0] == 'inputFileInodesSent':
            fileMappingRequired = False
            for clientInputPath,mapInputFile in self.clientData['inputFileMapping'].items():
               if mapInputFile:
                  self.clientListener.postMessage('addExportFile %s\n' % (clientInputPath))
                  fileMappingRequired = True
            if fileMappingRequired:
               transferTarFile = "%s_transfer.tar" % (self.serverData['jobId'])
               transferTarPath = os.path.join(os.sep,"tmp",transferTarFile)
               fp = open(transferTarPath,'w')
               if fp:
                  self.serverData['importObjects'][fp]           = {}
                  self.serverData['importObjects'][fp]['path']   = transferTarPath
                  self.serverData['importObjects'][fp]['buffer'] = ""
                  self.clientListener.postMessage('exportFiles %s\n' % (transferTarPath))
               else:
                  self.logger.log(logging.ERROR,getLogMessage("input file open failed: %s" % (transferTarPath)))
            else:
               self.clientListener.postMessage('exportFilesComplete\n')
         elif args[0] == 'read':
            clientInputPath = args[1]
            textLength = int(args[2])
            if textLength > 0:
               text = self.clientListener.pullMessage(textLength)
               if len(text) > 0:
                  found = False
                  if not found:
                     for importObject in self.serverData['importObjects']:
                        if self.serverData['importObjects'][importObject]['path'] == clientInputPath:
                           self.serverData['importObjects'][importObject]['buffer'] += text
                           found = True
                           break
                  if not found:
                     for outputObject in self.serverData['outputObjects']:
                        if self.serverData['outputObjects'][outputObject]['path'] == clientInputPath:
                           self.serverData['outputObjects'][outputObject]['buffer'] += text
                           found = True
                           break
                  if not found:
                     self.logger.log(logging.ERROR,getLogMessage("File object not found to match %s" % (clientInputPath)))
               else:
                  self.clientListener.pushMessage(message + '\n')
                  break
         elif args[0] == 'close':
            clientInputPath  = args[1]
            found = False
            for importObject in self.serverData['importObjects']:
               if self.serverData['importObjects'][importObject]['path'] == clientInputPath:
                  self.serverData['closePendingObjects'].append(importObject)
                  found = True
                  break
            if not found:
               for outputObject in self.serverData['outputObjects']:
                  if self.serverData['outputObjects'][outputObject]['path'] == clientInputPath:
                     self.serverData['closePendingObjects'].append(outputObject)
                     found = True
                     break
         elif args[0] == 'importFilesComplete':
            self.exit()
         elif args[0] == 'umask':
            self.clientData['umask'] = int(args[1],0)
         elif args[0] == 'pwd':
            clientWorkDirectoryInode      = long(args[1])
            clientWorkDirectoryPathLength = int(args[2])
            if clientWorkDirectoryPathLength > 0:
               clientWorkDirectoryPath = self.clientListener.pullMessage(clientWorkDirectoryPathLength)
               if len(clientWorkDirectoryPath) > 0:
                  self.clientData['workDirectory']      = clientWorkDirectoryPath
                  self.clientData['workDirectoryInode'] = clientWorkDirectoryInode
               else:
                  self.clientListener.pushMessage(message + '\n')
                  break
         elif args[0] == 'pwdWithMount':
            clientWorkDirectoryInode        = long(args[1])
            clientWorkDirectoryPathLength   = int(args[2])
            clientMountDeviceLength = int(args[3])
            if clientWorkDirectoryPathLength+clientMountDeviceLength > 0:
               inputFileDeviceText = self.clientListener.pullMessage(clientWorkDirectoryPathLength+clientMountDeviceLength)
               if len(inputFileDeviceText) > 0:
                  clientWorkDirectoryPath = inputFileDeviceText[0:clientWorkDirectoryPathLength]
                  clientMountDevice       = inputFileDeviceText[clientWorkDirectoryPathLength:]
                  self.clientData['workDirectory']       = clientWorkDirectoryPath
                  self.clientData['workDirectoryInode']  = clientWorkDirectoryInode
                  self.clientData['workDirectoryDevice'] = clientMountDevice
               else:
                  self.clientListener.pushMessage(message + '\n')
                  break
         elif args[0] == 'isClientTTY':
            self.clientData['isClientTTY'] = bool(int(args[1]))
#           log("isClientTTY(%s) = %s" % (args[1],self.clientData['isClientTTY']))
         elif args[0] == 'pegasusVersion':
            self.clientData['pegasusVersion'] = args[1]
            pegasusCommand = ". %s\n" % (self.configData['useSetup']) + \
                             "use -e -r pegasus-%s\n" % (self.clientData['pegasusVersion']) + \
                             "pegasus-version\n" + \
                             "pegasus-config --bin\n" + \
                             "pegasus-config --python"
            child = subprocess.Popen(pegasusCommand,shell=True,
                                     stdout=subprocess.PIPE,
                                     stderr=subprocess.PIPE,
                                     close_fds=True)
            pegasusStdOutput,pegasusStdError = child.communicate()
            pegasusExitStatus = child.returncode
            if pegasusExitStatus == 0:
               try:
                  pegasusVersion,pegasusBin,pegasusPythonPath = ''.join(pegasusStdOutput).strip().split()
                  self.serverData['pegasusVersion']    = pegasusVersion
                  self.serverData['pegasusHome']       = os.path.dirname(pegasusBin)
                  self.serverData['pegasusPythonPath'] = pegasusPythonPath
                  self.serverData['pegasusExists']     = True
               except:
                  pass
            else:
               self.logger.log(logging.ERROR,getLogMessage(pegasusStdOutput))
               self.logger.log(logging.ERROR,getLogMessage(pegasusStdError))
         elif args[0] == 'startLocal':
            self.checkLoadAndLaunch()
         elif args[0] == 'setupRemote':
            self.checkLoadAndLaunch()
         elif args[0] == 'startRemote':
            self.launchJob()
         elif args[0] == 'localWait':
            self.stop()
         elif args[0] == 'detachSignal':
            if not self.serverData['detach']:
               if self.createDetachedSession():
                  if self.updateJobSession():
                     self.serverData['detach'] = True
                     self.clientListener.postMessage('readyToDetach\n')
            else:
               self.clientListener.postMessage('readyToDetach\n')
         elif args[0] == 'detach':
            self.clientListener.closeConnection()
            if self.setupAttachListener():
               self.serverData['detached'] = True
            else:
               message = "Unable to detach job %d" % (self.serverData['jobId'])
               self.logger.log(logging.ERROR,getLogMessage(message))
         elif args[0] == 'jobId':
            self.serverData['jobId'] = int(args[1])
            self.logger.log(logging.INFO,getLogMessage("Assigning client jobId = %d:" % (self.serverData['jobId'])))
            logSetJobId(self.serverData['jobId'])
            self.serverData['localJobId'] = "%08d" % (self.serverData['jobId'])
         elif args[0] == 'jobToken':
            self.serverData['jobToken'] = args[1]
         elif args[0] == 'event':
            self.serverData['event'] = args[1]
         elif args[0] == 'operationMode':
            self.serverData['operationMode'] = int(args[1])
         elif args[0] == 'localMetrics':
            instanceIndex = int(args[1])
            status        = int(args[2])
            cpuTime       = float(args[3])
            realTime      = float(args[4])
            self.updateJobMetrics("venue=%d:local status=%d cputime=%f realtime=%f" % (instanceIndex,status,cpuTime,realTime))
            if instanceIndex > 0:
               self.serverData['metricsRecorded'] = False
         elif args[0] == 'clientHeartbeat':
            self.clientHeartbeat()
            self.stop()
         elif args[0] == 'localExit':
            status   = int(args[1])
            cpuTime  = float(args[2])
            realTime = float(args[3])
            venue = ""
            self.updateJobMetrics("venue=%s status=%d cputime=%f realtime=%f" % (venue,status,cpuTime,realTime))
            self.finalizeJob()
            self.finalizeCreatedSessions()
            self.clientListener.postMessage('exit %d\n' % (status))
            self.serverData['readyToExit'] = True
         elif args[0] == 'clientSignal':
            signalNumber = int(args[1])
            if self.serverData['childPid']:
               self.killChild()
            else:
               self.handleSignal(signalNumber,None)
         elif args[0] == 'clientExit':
            exitCode = int(args[1])
            self.exit(exitCode)
         elif args[0] == 'exit':
            self.exit()
         elif args[0] == 'null':
            pass
         else:
            self.logger.log(logging.ERROR,getLogMessage("Discarded client message: %s" % (message)))

         message = self.clientListener.pullMessage(0)


   def run(self):
      while True:
         if self.serverData['type'] == 'Proxy' and self.serverData['attached']:
            proxyMessage = self.clientListener.pullMessage(-1024)
            if proxyMessage:
#              log("client->server(%s)" % (proxyMessage))
               self.serverConnection.postMessage(proxyMessage)
            proxyMessage = self.serverConnection.pullMessage(-1024)
            if proxyMessage:
#              log("server->client(%s)" % (proxyMessage))
               self.clientListener.postMessage(proxyMessage)

         try:
            clientListenerSockets,clientReader, \
                                  attachListenerSocket,attachReader, \
                                  serverReader, \
                                  inputObjects,exportObjects  = self.getInputObjects()
            clientWriter, \
                                  attachWriter, \
                                  serverWriter, \
                                  importObjects,outputObjects = self.getOutputObjects()
            if self.serverData['childHasExited']:
               selectTimeout = 0.1
            else:
               selectTimeout = self.serverData['selectTimeout']

            readers = clientListenerSockets + clientReader + \
                      attachListenerSocket + attachReader + \
                      serverReader + \
                      inputObjects + exportObjects
            writers = clientWriter + \
                      attachWriter + \
                      serverWriter + \
                      importObjects + outputObjects
            readyReaders,readyWriters,readyExceptions = select.select(readers,writers,[],selectTimeout)
         except select.error,err:
            readyReaders = []
            readyWriters = []

         for readyReader in readyReaders:
            if   readyReader in clientListenerSockets:
               if self.clientListener.acceptConnection(readyReader):
                  # Do a double-fork to dissociate from the listening server.
                  if os.fork() != 0:
                     self.clientListener.closeConnection()  # Close the client socket in the listening server
                     os.wait()                              # Wait for the intermediate child to exit
                  else:
                     if os.fork() != 0:
                        sys.exit(0) # This is the intermediate child.  Exit.
                     else:
                        # This is the real child.
                        os.setsid()
                        self.serverData['type'] = 'JobExecuter'
                        self.scheduleTimeout()
                        if self.clientListener.handshake(readyReader):
                           self.clientListener.closeListeningConnections()
                        else:
                           self.logger.log(logging.ERROR,getLogMessage("Client connection handshake failed."))
                           self.exit(1)
            elif readyReader in clientReader:
               self.clientListener.receiveMessage()
#              self.clientListener.postMessage('null\n')
            elif readyReader in attachListenerSocket:
               if self.attachListener.acceptConnection(readyReader):
                  if not self.attachListener.handshake(readyReader):
                     self.logger.log(logging.ERROR,getLogMessage("Attach connection handshake failed."))
               else:
                  self.logger.log(logging.ERROR,getLogMessage("Attach connection failed."))
            elif readyReader in attachReader:
               self.attachListener.receiveMessage()
            elif readyReader in serverReader:
               self.serverConnection.receiveMessage()
            else:
               self.readFile(readyReader)

         if self.serverData['type'] == 'JobExecuter':
            self.processClientRequests()
            self.processAttachRequests()
         if self.serverData['type'] == 'Proxy' and not self.serverData['attached']:
            self.processServerRequests()

         for readyWriter in readyWriters:
            if   readyWriter in clientWriter:
               self.clientListener.sendMessage()
            elif readyWriter in attachWriter:
               self.attachListener.sendMessage()
            elif readyWriter in serverWriter:
               self.serverConnection.sendMessage()
            elif readyWriter in importObjects:
               self.writeFile(readyWriter)
            elif readyWriter in outputObjects:
               self.writeFile(readyWriter)
         self.checkClosePendingObjects()

         if self.serverData['readyToExit']:
            if   not self.clientListener.isMessagePending():
               self.exit()
            elif not self.clientListener.isConnected():
               self.exit()

         if self.serverData['childHasExited']:
            if not self.clientListener.isConnected():
               self.exit()

         if self.serverData['type'] == 'Proxy':
            if not self.clientListener.isConnected() and not self.serverConnection.isConnected():
               self.exit()
            if self.serverData['attached'] and not self.serverConnection.isConnected():
               self.serverData['attached'] = False
               self.exit()

         if self.serverData['type'] == 'JobExecuter':
            if not (self.clientListener.isConnected() or self.serverData['detached']):
               if self.serverData['childPid']:
                  self.killChild()

         if self.serverData['childPid']:
            pid = 0
            try:
               (pid,exitCode) = os.waitpid(self.serverData['childPid'],os.WNOHANG)
               if exitCode != 0:
                  if   os.WIFSIGNALED(exitCode):
                     exitCode = (1 << 7) | os.WTERMSIG(exitCode)
                  else:
                     if os.WIFEXITED(exitCode):
                        exitCode = os.WEXITSTATUS(exitCode)
            except:
               try:
                  os.kill(self.serverData['childPid'],0)
               except:
                  self.logger.log(logging.INFO,getLogMessage("Child has exited."))
                  pid = self.serverData['childPid']
                  exitCode = 1
            if pid != 0:
               self.serverData['childHasExited'] = True
               self.serverData['childPid'] = 0

               while len(self.serverData['inputObjects']) > 0:
                  inputObjects = self.serverData['inputObjects'].keys()
                  for inputObject in inputObjects:
                     self.readFile(inputObject)

               if self.serverData['operationMode'] & self.commandParser.OPERATIONMODERUNDISTRIBUTORID:
                  metricsRecords = self.serverData['metrics'].split('\n')
                  metricsParsed = False
                  for metricsRecord in metricsRecords:
                     if metricsRecord != "":
                        self.serverData['metricsRecorded'] = False
                        self.updateJobMetrics(metricsRecord)
                        metricsParsed = True
                  if not metricsParsed:
                     self.updateJobMetrics("status=%d" % (self.serverData['exitCode']))
               else:
                  self.serverData['exitCode'] = exitCode

               if os.path.exists(self.serverData['jobScanPath']):
                  fpJobScan = open(self.serverData['jobScanPath'],'r')
                  if fpJobScan:
                     jobScanText = fpJobScan.readlines()
                     fpJobScan.close()
                     self.clientListener.postMessageBySize('jobScanUpdate',''.join(jobScanText))
                  os.remove(self.serverData['jobScanPath'])

               if self.serverData['operationMode'] & self.commandParser.OPERATIONMODERUNDISTRIBUTORID:
                  if self.serverData['workDirectoryShared']:
                     self.clientListener.postMessage('importFile None\n')
                  else:
                     importMessageSent = False
                     try:
                        transferTarFile = "%s_output.tar" % (self.serverData['jobId'])
                        transferTarPath = os.path.join(os.sep,"tmp",transferTarFile)
                        findCommand = ['find','.','-newer','.__fileTimeMarkerSubmit','-not','-name','.','-print0']
                        tarCommand = ['xargs','--null','tar','cvf',transferTarPath,'--no-recursion']
#                       log("command: %s | %s" % (findCommand,tarCommand))
                        findChild = subprocess.Popen(findCommand,
                                                     stdout=subprocess.PIPE,
                                                     stderr=subprocess.PIPE,
                                                     close_fds=True)
                        tarChild = subprocess.Popen(tarCommand,
                                                    stdin=findChild.stdout,
                                                    stdout=subprocess.PIPE,
                                                    stderr=subprocess.PIPE,
                                                    close_fds=True)
                        tarStdOutput,tarStdError = tarChild.communicate()
                        tarExitStatus = tarChild.returncode
                        if tarExitStatus != 0:
                           self.logger.log(logging.ERROR,getLogMessage("Failed to write transfer tarfile %s" % (transferTarPath)))
                           if tarStdOutput:
                              self.logger.log(logging.ERROR,getLogMessage(tarStdOutput))
                           if tarStdError:
                              self.logger.log(logging.ERROR,getLogMessage(tarStdError))
                           exitCode = 1
                        else:
#                          if tarStdOutput:
#                             log(tarStdOutput)
                           fp = open(transferTarPath,'r')
                           if fp:
                              self.serverData['exportObjects'][fp] = transferTarFile
                              self.clientListener.postMessage('importFile %s\n' % (transferTarFile))
                              importMessageSent = True
                           else:
                              self.logger.log(logging.ERROR,getLogMessage("input file open failed: %s" % (transferTarPath)))
                     except OSError,err:
                        self.logger.log(logging.ERROR,getLogMessage("Failed to create transfer tarfile %s" % (transferTarPath)))
                        self.logger.log(logging.ERROR,getLogMessage(err.args[1]))
                     if not importMessageSent:
                        self.clientListener.postMessage('importFile None\n')
               else:
                  self.clientListener.postMessage('importFile None\n')


