#!/usr/bin/env python
#
# @package      hubzero-submit-monitors
# @file         monitorTunnelR.py
# @copyright    Copyright (c) 2004-2020 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2004-2020 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#
# ----------------------------------------------------------------------
#  monitorTunnelR.py
#
#  script which reports monitor status for all ssh tunnels
#
import os
import sys
import logging

from hubzero.submit.DaemonsInfo         import DaemonsInfo
from hubzero.submit.RemoteTunnelMonitor import RemoteTunnelMonitor

CONFIGURATIONDIRECTORY   = os.path.join(os.sep,'etc','submit')
DAEMONSCONFIGURATIONFILE = 'daemons.conf'

APPLICATIONLOGGER = logging.getLogger('') 

def openLogger():
   class EmptyFilter(logging.Filter):
      """
      This is a filter which rejects empty messages

      """

      def filter(self,record):
         if record.getMessage() == "":
            emptyRecord = True
         else:
            emptyRecord = False

         return(not emptyRecord)

   APPLICATIONLOGGER.setLevel(logging.DEBUG)

   logHandler = logging.StreamHandler()

   emptyFilter = EmptyFilter()
   logHandler.addFilter(emptyFilter)

   logFormatter = logging.Formatter('%(asctime)s %(message)s','[%a %b %d %H:%M:%S %Y]')
   logHandler.setFormatter(logFormatter)
   APPLICATIONLOGGER.addHandler(logHandler)


def monitorTunnelR(listenURI):
   """Report ssh tunnel use"""
   if len(sys.argv) > 1:
      tunnelName = sys.argv[1]
   else:
      tunnelName = ""

   remoteTunnelMonitor = RemoteTunnelMonitor(listenURI)

   report = remoteTunnelMonitor.reportTunnelUse(tunnelName)

   try:
      print("\n%-30s  %s" % ("Tunnel","useCount"))
      tunnels = report['usageReport']
      for tunnel in tunnels:
         print("%-30s  %d" % (tunnel,tunnels[tunnel]))
   except IOError:
      pass


if __name__ == '__main__':
   openLogger()

   configFilePath = os.path.join(CONFIGURATIONDIRECTORY,DAEMONSCONFIGURATIONFILE)
   daemonsInfo    = DaemonsInfo(configFilePath)
   listenURI      = daemonsInfo.getDaemonListenURI('tunnelMonitor','tcp')

   monitorTunnelR(listenURI)


