# @package      hubzero-submit-monitors
# @file         MonitorsInfo.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2012-2015 HUBzero Foundation, LLC.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2012-2015 HUBzero Foundation, LLC.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#
import sys
import os.path
import pwd
import re
import socket
import select
import subprocess
import shlex
import traceback
import logging
from errno import EPIPE

from hubzero.submit.LogMessage import getLogPIDMessage as getLogMessage

class MonitorsInfo:
   def __init__(self,
                monitorsPath):
      self.logger     = logging.getLogger(__name__)
      self.monitors   = {}
      self.childPid   = 0
      self.bufferSize = 4096

      monitorPattern  = re.compile('(\s*\[)([^\s]*)(]\s*)')
      keyValuePattern = re.compile('( *)(\w*)( *= *)(.*[^\s$])( *)')
      commentPattern  = re.compile('\s*#.*')
      monitorName     = ""

      if os.path.exists(monitorsPath):
         try:
            fpInfo = open(monitorsPath,'r')
            try:
               eof = False
               while not eof:
                  record = fpInfo.readline()
                  if record != "":
                     record = commentPattern.sub("",record)
                     if   monitorPattern.match(record):
                        monitorName = monitorPattern.match(record).group(2)
                        self.monitors[monitorName] = {'venue':'',
                                                      'venuePort':22,
                                                      'identityManager':'',
                                                      'gsiHost':socket.gethostname(),
                                                      'venueMechanism':'ssh',
                                                      'tunnelDesignator':'',
                                                      'remoteUser':'',
                                                      'x509ProxyPath':'',
                                                      'remoteBinDirectory':'',
                                                      'remoteMonitorCommand':'',
                                                      'state':'enabled'
                                                     }
                     elif keyValuePattern.match(record):
                        key,value = keyValuePattern.match(record).group(2,4)
                        if key in self.monitors[monitorName]:
                           if   isinstance(self.monitors[monitorName][key],list):
                              self.monitors[monitorName][key] = [e.strip() for e in value.split(',')]
                           elif isinstance(self.monitors[monitorName][key],bool):
                              self.monitors[monitorName][key] = bool(value.lower() == 'true')
                           elif isinstance(self.monitors[monitorName][key],float):
                              self.monitors[monitorName][key] = float(value)
                           elif isinstance(self.monitors[monitorName][key],int):
                              self.monitors[monitorName][key] = int(value)
                           elif isinstance(self.monitors[monitorName][key],dict):
                              try:
                                 sampleKey   = self.monitors[monitorName][key].keys()[0]
                                 sampleValue = self.monitors[monitorName][key][sampleKey]
                              except:
                                 sampleKey   = "key"
                                 sampleValue = "value"
                              self.monitors[monitorName][key] = {}
                              for e in value.split(','):
                                 dictKey,dictValue = e.split(':')
                                 if isinstance(sampleKey,int):
                                    dictKey = int(dictKey)
                                 if   isinstance(sampleValue,int):
                                    dictValue = int(dictValue)
                                 elif isinstance(sampleValue,float):
                                    dictValue = float(dictValue)
                                 elif isinstance(sampleValue,bool):
                                    dictValue = bool(dictValue.lower() == 'true')
                                 self.monitors[monitorName][key][dictKey] = dictValue
                           else:
                              self.monitors[monitorName][key] = value
                        else:
                           message = "Undefined key = value pair %s = %s for site %s" % (key,value,monitorName)
                           self.logger.log(logging.WARNING,getLogMessage(message))
                  else:
                     eof = True
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("Monitors configuration file %s could not be read" % (monitorsPath)))
            finally:
               fpInfo.close()
         except (IOError,OSError):
            self.logger.log(logging.ERROR,getLogMessage("Monitors configuration file %s could not be opened" % (monitorsPath)))
      else:
         self.logger.log(logging.ERROR,getLogMessage("Monitors configuration file %s is missing" % (monitorsPath)))

      markedForDeletion = []
      for monitorName in self.monitors:
         if self.monitors[monitorName]['state'] != 'enabled':
            markedForDeletion.append(monitorName)
      for monitorName in markedForDeletion:
         del self.monitors[monitorName]
      del markedForDeletion


   def __writeToStdout(self,
                       message):
      try:
         sys.stdout.write(message)
         sys.stdout.flush()
      except IOError,err:
         if not err[0] in [EPIPE]:
            self.logger.log(logging.ERROR,getLogMessage("Can't write to stdout: %s" % (message)))


   def __writeToStderr(self,
                       message):
      try:
         sys.stderr.write(message)
         sys.stderr.flush()
      except IOError,err:
         if not err[0] in [EPIPE]:
            self.logger.log(logging.ERROR,getLogMessage("Can't write to stderr: %s" % (message)))


   def getMonitorInfo(self,
                      monitorDesignator):
      monitorInfo = {}
      monitorName = ""
      if monitorDesignator in self.monitors:
         monitorName = monitorDesignator
      else:
         for monitorInfoName in self.monitors:
            if re.match(monitorInfoName,monitorDesignator):
               self.monitors[monitorDesignator] = {}
               venueTemplate                                            = self.monitors[monitorInfoName]['venue']
               self.monitors[monitorDesignator]['venue']                = venueTemplate.replace('@@{name}',monitorDesignator)
               self.monitors[monitorDesignator]['venuePort']            = self.monitors[monitorInfoName]['venuePort']
               self.monitors[monitorDesignator]['venueMechanism']       = self.monitors[monitorInfoName]['venueMechanism']
               self.monitors[monitorDesignator]['tunnelDesignator']     = self.monitors[monitorInfoName]['tunnelDesignator']
               self.monitors[monitorDesignator]['remoteUser']           = self.monitors[monitorInfoName]['remoteUser']
               self.monitors[monitorDesignator]['identityManager']      = self.monitors[monitorInfoName]['identityManager']
               self.monitors[monitorDesignator]['x509ProxyPath']        = self.monitors[monitorInfoName]['x509ProxyPath']
               self.monitors[monitorDesignator]['remoteMonitorCommand'] = self.monitors[monitorInfoName]['remoteMonitorCommand']
               self.monitors[monitorDesignator]['state']                = self.monitors[monitorInfoName]['state']
               monitorName = monitorDesignator
               break

      if monitorName:
         monitorInfo['monitorName'] = monitorName
         for key in self.monitors[monitorName]:
            monitorInfo[key] = self.monitors[monitorName][key]
      else:
         message = "Could not retrieve MonitorInfo for %s.\nPerhaps the Site Monitor needs to be reloaded!" % (monitorDesignator)
         self.logger.log(logging.ERROR,getLogMessage(message))

      return(monitorInfo)


   def __executeCommand(self,
                        command,
                        streamOutput=False):
      commandArgs = shlex.split(command)
      child = subprocess.Popen(commandArgs,bufsize=self.bufferSize,
                               stdout=subprocess.PIPE,
                               stderr=subprocess.PIPE,
                               close_fds=True)
      self.childPid = child.pid
      childout      = child.stdout
      childoutFd    = childout.fileno()
      childerr      = child.stderr
      childerrFd    = childerr.fileno()

      outEOF = False
      errEOF = False

      outData = []
      errData = []

      while True:
         toCheck = []
         if not outEOF:
            toCheck.append(childoutFd)
         if not errEOF:
            toCheck.append(childerrFd)
         try:
            ready = select.select(toCheck,[],[]) # wait for input
         except select.error,err:
            ready = {}
            ready[0] = []
         if childoutFd in ready[0]:
            outChunk = os.read(childoutFd,self.bufferSize)
            if outChunk == '':
               outEOF = True
            outData.append(outChunk)
            if streamOutput:
               self.__writeToStdout(outChunk)

         if childerrFd in ready[0]:
            errChunk = os.read(childerrFd,self.bufferSize)
            if errChunk == '':
               errEOF = True
            errData.append(errChunk)
            if streamOutput:
               self.__writeToStderr(errChunk)

         if outEOF and errEOF:
            break

      pid,err = os.waitpid(self.childPid,0)
      self.childPid = 0
      if err != 0:
         if   os.WIFSIGNALED(err):
            self.logger.log(logging.ERROR,getLogMessage("%s failed w/ signal %d" % (command,os.WTERMSIG(err))))
         else:
            if os.WIFEXITED(err):
               err = os.WEXITSTATUS(err)
            self.logger.log(logging.ERROR,getLogMessage("%s failed w/ exit code %d" % (command,err)))
         if not streamOutput:
            self.logger.log(logging.ERROR,getLogMessage("%s" % ("".join(errData))))

      return(err,"".join(outData),"".join(errData))


   def getSSHCommandX(self,
                      monitorNameInput,
                      remoteTunnelMonitor):
      sshCommand       = ""
      tunnelDesignator = ""

      monitorName = ""
      if monitorNameInput in self.monitors:
         monitorName = monitorNameInput
      else:
         for monitorInfoName in self.monitors:
            if re.match(monitorInfoName,monitorNameInput):
               self.monitors[monitorNameInput] = {}
               venueTemplate                                           = self.monitors[monitorInfoName]['venue']
               self.monitors[monitorNameInput]['venue']                = venueTemplate.replace('@@{name}',monitorNameInput)
               self.monitors[monitorNameInput]['venuePort']            = self.monitors[monitorInfoName]['venuePort']
               self.monitors[monitorNameInput]['venueMechanism']       = self.monitors[monitorInfoName]['venueMechanism']
               self.monitors[monitorNameInput]['tunnelDesignator']     = self.monitors[monitorInfoName]['tunnelDesignator']
               self.monitors[monitorNameInput]['remoteUser']           = self.monitors[monitorInfoName]['remoteUser']
               self.monitors[monitorNameInput]['identityManager']      = self.monitors[monitorInfoName]['identityManager']
               self.monitors[monitorNameInput]['x509ProxyPath']        = self.monitors[monitorInfoName]['x509ProxyPath']
               self.monitors[monitorNameInput]['remoteMonitorCommand'] = self.monitors[monitorInfoName]['remoteMonitorCommand']
               self.monitors[monitorNameInput]['state']                = self.monitors[monitorInfoName]['state']
               monitorName = monitorNameInput
               break

      if monitorName:
         try:
            if   self.monitors[monitorName]['venueMechanism'] == 'ssh':
               venue                = self.monitors[monitorName]['venue']
               venuePort            = self.monitors[monitorName]['venuePort']
               tunnelDesignator     = self.monitors[monitorName]['tunnelDesignator']
               user                 = self.monitors[monitorName]['remoteUser']
               remoteMonitorCommand = self.monitors[monitorName]['remoteMonitorCommand']

               sshBaseCommand = "ssh -T -x -a -i " + sshIdentity
               if tunnelDesignator == "":
                  sshCommand = sshBaseCommand + " " + user + "@" + venue + " '" + remoteMonitorCommand + " " + monitorName + "'"
               else:
                  tunnelAddress,tunnelPort = remoteTunnelMonitor.getTunnelAddressPort(tunnelDesignator)
                  sshCommand = sshBaseCommand + " -p " + tunnelPort + " " + user + "@" + tunnelAddress + \
                                                                      " '" + remoteMonitorCommand + " " + monitorName + "'"
            elif self.monitors[monitorName]['venueMechanism'] == 'gsissh':
               venue                = self.monitors[monitorName]['venue']
               venuePort            = self.monitors[monitorName]['venuePort']
               x509ProxyPath        = self.monitors[monitorName]['x509ProxyPath']
               remoteMonitorCommand = self.monitors[monitorName]['remoteMonitorCommand']

               sshBaseCommand = "gsissh -T -x -a "
               sshCommand = sshBaseCommand + " " + venue + " '" + remoteMonitorCommand + " " + monitorName + "'"
         except:
            self.logger.log(logging.ERROR,getLogMessage("Build sshCommand failed for %s" % (monitorNameInput)))
            self.logger.log(logging.ERROR,getLogMessage(traceback.format_exc()))
      else:
         message = "Could not build sshCommand for %s.\nPerhaps the Site Monitor needs to be reloaded!" % (monitorNameInput)
         self.logger.log(logging.ERROR,getLogMessage(message))

      return(sshCommand,tunnelDesignator)


