# @package      hubzero-submit-monitors
# @file         JobMonitor.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2012-2015 HUBzero Foundation, LLC.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2012-2015 HUBzero Foundation, LLC.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#
import os
import re
import pwd
import math
import sqlite3
import time
import signal
import traceback
import json
import logging

from hubzero.submit.LogMessage       import getLogIDMessage as getLogMessage
from hubzero.submit.BoundConnections import BoundConnections
from hubzero.submit.JobInfo          import JobInfo
from hubzero.submit.InfosInfo        import InfosInfo
from hubzero.submit.SitesInfo        import SitesInfo
from hubzero.submit.AggregatorsInfo  import AggregatorsInfo

class JobMonitor(BoundConnections):
   def __init__(self,
                configurationDirectory,
                infosConfigurationFile,
                listenURI,
                activeJobDBPath="monitorJob.db",
                activeJobDumpPath="monitorJob.sql"):
      BoundConnections.__init__(self,listenURI,logConnection=False)

      self.logger = logging.getLogger(__name__)

      self.configurationDirectory = configurationDirectory
      self.infosConfigurationFile = infosConfigurationFile

      self.infosInfo       = None
      self.sitesInfo       = None
      self.aggregatorsInfo = None
      self.settingInfo     = False

      self.pendingJobSites    = []
      self.pendingJobPostings = {}
      self.terminating        = False
      self.channels           = {}

      self.activeJobDBPath    = activeJobDBPath
      self.activeJobDumpPath  = activeJobDumpPath
      self.dbConnection       = None
      try:
         self.dbConnection = sqlite3.connect(self.activeJobDBPath)
         self.dbConnection.row_factory = sqlite3.Row
         nActiveJobCount = self.getActiveJobCount()
         self.logger.log(logging.INFO,getLogMessage("%d jobs loaded from DB file" % (nActiveJobCount)))
      except:
         self.logger.log(logging.ERROR,getLogMessage(traceback.format_exc()))

      self.signalJobSiteMonitors(signal.SIGTERM)
      time.sleep(5)
      if self.countRunningJobSiteMonitors() > 0:
         self.signalJobSiteMonitors(signal.SIGKILL)
      self.purgeActiveJobSites()


   def setInfo(self):
      errorInSetInfo = False
      if not self.settingInfo:
         self.settingInfo = True

         configFilePath = os.path.join(self.configurationDirectory,self.infosConfigurationFile)
         if self.infosInfo:
            del self.infosInfo
         self.infosInfo = InfosInfo(configFilePath)

         if self.sitesInfo:
            del self.sitesInfo
         self.sitesInfo = SitesInfo(self.infosInfo.getInfoPath('sites'),
                                    allowedVenueMechanisms=['*'],
                                    pegasusVersion='*')
         enabledSites      = self.sitesInfo.getEnabledSites()
         expandedSiteNames = self.sitesInfo.getExpandedSiteNames(enabledSites)

         if self.aggregatorsInfo:
            del self.aggregatorsInfo
         self.aggregatorsInfo = AggregatorsInfo(self.infosInfo.getInfoPath('aggregators'))

         for expandedSiteName in expandedSiteNames:
            aggregatorName = self.aggregatorsInfo.getSiteAggregator(expandedSiteName)
            if not aggregatorName:
               self.logger.log(logging.ERROR,getLogMessage("Site %s is not included in any aggregator" % (expandedSiteName)))
               errorInSetInfo = True

         del self.sitesInfo
         self.sitesInfo = None
         del expandedSiteNames
         del enabledSites

         self.settingInfo = False

      return(errorInSetInfo)


   def terminate(self):
      if not self.terminating:
         self.signalJobSiteMonitors(signal.SIGTERM)
         self.closeListeningConnection()
         self.terminating = True


   def dumpActiveJobs(self):
      if self.dbConnection:
         try:
            fpDumpFile = open(self.activeJobDumpPath,'w')
            try:
               for line in self.dbConnection.iterdump():
                   fpDumpFile.write('%s\n' % (line))
            except (IOError,OSError):
               pass
            finally:
               fpDumpFile.close()
         except (IOError,OSError):
            pass


   def close(self):
      if self.dbConnection:
         self.dbConnection.close()
      self.closeListeningConnection()


   def addActiveJobSite(self,
                        jobSite,
                        channel,
                        pid):
      self.channels[jobSite] = channel
      now = time.time()
      cursor = self.dbConnection.cursor()
      cursor.execute("INSERT INTO activeJobSites (jobSite,pid,timeUpdated) \
                           VALUES(?,?,?)",(jobSite,pid,now))
      self.dbConnection.commit()


   def updateActiveJobSite(self,
                           jobSite):
      now = time.time()
      cursor = self.dbConnection.cursor()
      cursor.execute("UPDATE activeJobSites SET timeUpdated=? WHERE(jobSite=?)",(now,jobSite))
      self.dbConnection.commit()


   def getActiveJobSites(self):
      activeJobSites = []
      cursor = self.dbConnection.cursor()
      cursor.execute("SELECT jobSite FROM activeJobSites")
      rows = cursor.fetchall()
      for row in rows:
         activeJobSites.append(row['jobSite'])

      return(activeJobSites)


   def getActiveJobSiteChannel(self,
                               jobSite):
      try:
         channel = self.channels[jobSite]
      except ValueError:
         channel = None

      return(channel)


   def getActiveJobSitePID(self,
                           jobSite):
      pid = 0
      cursor = self.dbConnection.cursor()
      cursor.execute("SELECT * FROM activeJobSites WHERE(jobSite=?)",(jobSite,))
      row = cursor.fetchone()
      if row:
         pid = row['pid']

      return(pid)


   def getActiveJobSiteTimeUpdated(self,
                                   jobSite):
      timeUpdated = 0
      cursor = self.dbConnection.cursor()
      cursor.execute("SELECT * FROM activeJobSites WHERE(jobSite=?)",(jobSite,))
      row = cursor.fetchone()
      if row:
         timeUpdated = row['timeUpdated']

      return(timeUpdated)


   def deleteActiveJobSite(self,
                           jobSite):
      cursor = self.dbConnection.cursor()
      cursor.execute("DELETE FROM activeJobSites WHERE(jobSite=?)",(jobSite,))
      self.dbConnection.commit()


   def purgeActiveJobSites(self):
      cursor = self.dbConnection.cursor()
      cursor.execute("DELETE FROM activeJobSites")
      self.dbConnection.commit()


   def isJobSiteActive(self,
                       jobSite):
      jobSiteIsActive = False
      cursor = self.dbConnection.cursor()
      cursor.execute("SELECT * FROM activeJobSites WHERE(jobSite=?)",(jobSite,))
      row = cursor.fetchone()
      if row:
         jobSiteIsActive = True

      return(jobSiteIsActive)


   def getAggregatorsWithRegisteredJob(self):
      aggregatorsWithRegisteredJob = []
      cursor = self.dbConnection.cursor()
      cursor.execute("SELECT DISTINCT aggregator FROM registeredJobs")
      rows = cursor.fetchall()
      for row in rows:
         aggregatorsWithRegisteredJob.append(row['siteName'])

      return(aggregatorsWithRegisteredJob)


   def getSitesWithRegisteredJob(self):
      sitesWithRegisteredJob = []
      cursor = self.dbConnection.cursor()
      cursor.execute("SELECT DISTINCT siteName FROM registeredJobs")
      rows = cursor.fetchall()
      for row in rows:
         sitesWithRegisteredJob.append(row['siteName'])

      return(sitesWithRegisteredJob)


   def getUsersWithRegisteredJob(self):
      usersWithRegisteredJob = []
      cursor = self.dbConnection.cursor()
      cursor.execute("SELECT DISTINCT hubUserId FROM registeredJobs \
                                               WHERE state=?",(self.JOBREGISTRATIONSTATEHELD,))
      rows = cursor.fetchall()
      for row in rows:
         usersWithRegisteredJob.append(row['hubUserId'])

      return(usersWithRegisteredJob)


   def getAggregatorUserRegisteredJob(self):
      aggregatorUserRegisteredJobs = {}
      cursor = self.dbConnection.cursor()
      cursor.execute("SELECT DISTINCT aggregator,hubUserId FROM registeredJobs \
                                                          WHERE state=?",(self.JOBREGISTRATIONSTATEHELD,))
      rows = cursor.fetchall()
      for row in rows:
         if not row['aggregator'] in aggregatorUserRegisteredJobs:
            aggregatorUserRegisteredJobs[row['aggregator']] = []
         aggregatorUserRegisteredJobs[row['aggregator']].append(row['hubUserId'])

      return(aggregatorUserRegisteredJobs)


   def getAggregatorRegisteredJobsForUser(self,
                                          aggregator,
                                          hubUserId):
      jobsInfo = []
      cursor = self.dbConnection.cursor()
      cursor.execute("SELECT localJobId,instanceId,siteName FROM registeredJobs \
                                                           WHERE(aggregator=? AND hubUserId=? AND state=?) \
                                                        ORDER BY timeRegistered", \
                                                     (aggregator,hubUserId,self.JOBREGISTRATIONSTATEHELD))
      rows = cursor.fetchall()
      for row in rows:
         jobInfo = JobInfo(localJobId=row['localJobId'],
                           instanceId=row['instanceId'],
                           siteName=row['siteName'],
                           aggregator=aggregator)
         jobsInfo.append(dict(jobInfo))

      return(jobsInfo)


   def getRegisteredJob(self,
                        localJobId,
                        instanceId):
      jobInfo = None
      cursor = self.dbConnection.cursor()
      cursor.execute("SELECT * FROM registeredJobs WHERE(localJobId=? AND instanceId=?)",(localJobId,instanceId))
      row = cursor.fetchone()
      if row:
         jobInfo = JobInfo(distributorPid=row['distributorPid'],
                           localJobId=row['localJobId'],
                           instanceId=row['instanceId'],
                           hubUserId=row['hubUserId'],
                           siteName=row['siteName'],
                           aggregator=row['aggregator'],
                           timeRecorded=row['timeRegistered'])

      return(jobInfo)


   JOBREGISTRATIONSTATEHELD     = 1 << 0
   JOBREGISTRATIONSTATERELEASED = 1 << 1


   def addRegisteredJob(self,
                        jobInfo):
      cursor = self.dbConnection.cursor()
      cursor.execute("INSERT INTO registeredJobs (aggregator,state,localJobId,instanceId, \
                                                  distributorPid,hubUserId,siteName,timeRegistered) \
                           VALUES(?,?,?,?,?,?,?,?)",(jobInfo['aggregator'],
                                                     self.JOBREGISTRATIONSTATEHELD,
                                                     jobInfo['localJobId'],
                                                     jobInfo['instanceId'],
                                                     jobInfo['distributorPid'],
   				                                      jobInfo['hubUserId'],
   				                                      jobInfo['siteName'],
   				                                      jobInfo['timeRecorded']))
      self.dbConnection.commit()


   def deleteRegisteredJob(self,
                           localJobId,
                           instanceId):
      cursor = self.dbConnection.cursor()
      cursor.execute("DELETE FROM registeredJobs WHERE(localJobId=? AND instanceId=?)",(localJobId,instanceId))
      self.dbConnection.commit()
      nDeletedRegisteredJobs = cursor.rowcount

      return(nDeletedRegisteredJobs)


   def getRegisteredJobCountAtAggregator(self,
                                         aggregator):
      cursor = self.dbConnection.cursor()
      cursor.execute("SELECT COUNT(*) FROM registeredJobs WHERE(aggregator=?)",(aggregator,))
      row = cursor.fetchone()
      nRegisteredJobCount = row[0]

      return(nRegisteredJobCount)


   def getRegisteredJobCountAtSite(self,
                                   siteName):
      cursor = self.dbConnection.cursor()
      cursor.execute("SELECT COUNT(*) FROM registeredJobs WHERE(siteName=?)",(siteName,))
      row = cursor.fetchone()
      nRegisteredJobCount = row[0]

      return(nRegisteredJobCount)


   def getSitesWithActiveJob(self):
      sitesWithActiveJob = []
      cursor = self.dbConnection.cursor()
      cursor.execute("SELECT DISTINCT siteDesignator FROM activeJobs")
      rows = cursor.fetchall()
      for row in rows:
         sitesWithActiveJob.append(row['siteDesignator'])

      return(sitesWithActiveJob)


   def getActiveJobSiteQueues(self,
                              jobSite):
      activeJobSiteQueues = []
      cursor = self.dbConnection.cursor()
      cursor.execute("SELECT DISTINCT jobQueue FROM activeJobs WHERE(siteDesignator=?)",(jobSite,))
      rows = cursor.fetchall()
      for row in rows:
         activeJobSiteQueues.append(row['jobQueue'])

      return(activeJobSiteQueues)


   def getActiveJobsInQueue(self,
                            jobSite,
                            jobQueue):
      jobsInfo = []
      cursor = self.dbConnection.cursor()
      cursor.execute("SELECT * FROM activeJobs WHERE(siteDesignator=? AND jobQueue=?)",(jobSite,jobQueue))
      rows = cursor.fetchall()
      for row in rows:
         jobInfo = JobInfo(aggregator=row['aggregator'],
                           siteName=row['siteName'],
                           siteDesignator=row['siteDesignator'],
                           remoteJobId=row['remoteJobId'],
                           distributorPid=row['distributorPid'],
                           runName=row['runName'],
                           nCores=row['nCores'],
                           localJobId=row['localJobId'],
                           instanceId=row['instanceId'],
                           jobStatus=row['jobStatus'],
                           jobStage=row['jobStage'],
                           jobQueue=row['jobQueue'],
                           hubUserId=row['hubUserId'],
                           destination=row['destination'],
                           executionHost=row['executionHost'],
                           timeRecorded=row['timeRecorded'])
         jobsInfo.append(dict(jobInfo))

      return(jobsInfo)


   def getActiveSiteDesignators(self):
      activeSiteDesignators = []
      cursor = self.dbConnection.cursor()
      cursor.execute("SELECT siteDesignator FROM activeJobs \
                                           WHERE(jobStatus!='Dr' AND jobStatus!='D')")
      rows = cursor.fetchall()
      for row in rows:
         if not row['siteDesignator'] in activeSiteDesignators:
            activeSiteDesignators.append(row['siteDesignator'])

      return(activeSiteDesignators)


   def getActiveJobsForUser(self,
                            hubUserId):
      jobsInfo = []
      cursor = self.dbConnection.cursor()
      cursor.execute("SELECT * FROM activeJobs \
                               WHERE(hubUserId=? AND jobStatus!='Dr' AND jobStatus!='D')",(hubUserId,))
      rows = cursor.fetchall()
      for row in rows:
         jobInfo = JobInfo(aggregator=row['aggregator'],
                           siteName=row['siteName'],
                           siteDesignator=row['siteDesignator'],
                           remoteJobId=row['remoteJobId'],
                           distributorPid=row['distributorPid'],
                           runName=row['runName'],
                           nCores=row['nCores'],
                           localJobId=row['localJobId'],
                           instanceId=row['instanceId'],
                           jobStatus=row['jobStatus'],
                           jobStage=row['jobStage'],
                           jobQueue=row['jobQueue'],
                           hubUserId=row['hubUserId'],
                           destination=row['destination'],
                           executionHost=row['executionHost'],
                           timeRecorded=row['timeRecorded'])
         jobsInfo.append(dict(jobInfo))

      return(jobsInfo)


   def getActiveJobPidForUser(self,
                              hubUserId,
                              localJobId):
      activeJobPid = None
      jobsInfo = []
      cursor = self.dbConnection.cursor()
      cursor.execute("SELECT * FROM activeJobs \
                               WHERE(hubUserId=? AND localJobId=? AND \
                                     jobStatus!='N' AND jobStatus!='Dr' AND jobStatus!='D') \
                               ORDER BY localJobId,instanceId",(hubUserId,localJobId))
      rows = cursor.fetchall()
      for row in rows:
         jobInfo = JobInfo(siteDesignator=row['siteDesignator'],
                           remoteJobId=row['remoteJobId'],
                           distributorPid=row['distributorPid'],
                           localJobId=row['localJobId'],
                           instanceId=row['instanceId'])
         jobsInfo.append(jobInfo)

      if len(jobsInfo) > 0:
         for jobInfo in jobsInfo:
            if jobInfo['distributorPid'] > 0:
               activeJobPid = jobInfo['distributorPid']
               break

      del jobsInfo

      return(activeJobPid)


   def getSitesActiveJobCount(self):
      sitesActiveJobCount = {}
      cursor = self.dbConnection.cursor()
      cursor.execute("SELECT siteName,COUNT(*) AS nJobs \
                        FROM activeJobs \
                       WHERE(jobStatus!='Dr' AND jobStatus!='D' AND instanceId!=0) \
                    GROUP BY siteName")
      rows = cursor.fetchall()
      for row in rows:
         sitesActiveJobCount[row['siteName']] = row['nJobs']

      return(sitesActiveJobCount)


   def getAggregatorsActiveJobCount(self):
      aggregatorsActiveJobCount = {}
      cursor = self.dbConnection.cursor()
      cursor.execute("SELECT aggregator,COUNT(*) AS nJobs \
                        FROM activeJobs \
                       WHERE(jobStatus!='Dr' AND jobStatus!='D' AND instanceId!=0) \
                    GROUP BY aggregator")
      rows = cursor.fetchall()
      for row in rows:
         aggregatorsActiveJobCount[row['aggregator']] = row['nJobs']

      return(aggregatorsActiveJobCount)


   def getUsersWithActiveJobWithIdentity(self,
                                         identityName):
      usersWithActiveJobWithIdentity = []
      cursor = self.dbConnection.cursor()
      cursor.execute("SELECT DISTINCT activeJobIdentities.hubUserId \
                                 FROM activeJobIdentities JOIN activeJobs \
                                   ON (activeJobIdentities.localJobId=activeJobs.localJobId AND \
                                       activeJobIdentities.instanceId=activeJobs.instanceId) \
                                WHERE (activeJobs.jobStatus!='Dr' AND \
                                       activeJobs.jobStatus!='D' AND \
                                       activeJobs.instanceId!=0 AND \
                                       activeJobIdentities.identityName=?)",(identityName,))
      rows = cursor.fetchall()
      for row in rows:
         usersWithActiveJobWithIdentity.append(row['hubUserId'])

      return(usersWithActiveJobWithIdentity)


   def getActiveJob(self,
                    siteDesignator,
                    remoteJobId):
      jobInfo = None
      cursor = self.dbConnection.cursor()
      cursor.execute("SELECT * FROM activeJobs WHERE(siteDesignator=? AND remoteJobId=?)",(siteDesignator,remoteJobId))
      row = cursor.fetchone()
      if row:
         jobInfo = JobInfo(aggregator=row['aggregator'],
                           siteName=row['siteName'],
                           siteDesignator=siteDesignator,
                           remoteJobId=remoteJobId,
                           distributorPid=row['distributorPid'],
                           runName=row['runName'],
                           nCores=row['nCores'],
                           localJobId=row['localJobId'],
                           instanceId=row['instanceId'],
                           jobStatus=row['jobStatus'],
                           jobStage=row['jobStage'],
                           jobQueue=row['jobQueue'],
                           hubUserId=row['hubUserId'],
                           destination=row['destination'],
                           executionHost=row['executionHost'],
                           timeRecorded=row['timeRecorded'])

      return(jobInfo)


   def addActiveJob(self,
                    jobInfo):
      cursor = self.dbConnection.cursor()
      cursor.execute("INSERT INTO activeJobs (aggregator,localJobId,instanceId,distributorPid,hubUserId,siteName, \
                                              nCores,runName,siteDesignator,remoteJobId,jobStatus,jobStage,jobQueue,destination, \
                                              executionHost,timeRecorded) \
                           VALUES(?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)",(jobInfo['aggregator'],
                                                                     jobInfo['localJobId'],
                                                                     jobInfo['instanceId'],
                                                                     jobInfo['distributorPid'],
   				                                                      jobInfo['hubUserId'],
   				                                                      jobInfo['siteName'],
                                                                     jobInfo['nCores'],
                                                                     jobInfo['runName'],
                                                                     jobInfo['siteDesignator'],
   				                                                      jobInfo['remoteJobId'],
   				                                                      jobInfo['jobStatus'],
   				                                                      jobInfo['jobStage'],
   				                                                      jobInfo['jobQueue'],
   				                                                      jobInfo['destination'],
   				                                                      jobInfo['executionHost'],
   				                                                      jobInfo['timeRecorded']))
      for identityName in jobInfo['identityNames']:
         cursor.execute("INSERT INTO activeJobIdentities (localJobId,instanceId,hubUserId,identityName) \
                              VALUES(?,?,?,?)",(jobInfo['localJobId'],
                                                jobInfo['instanceId'],
   				                                 jobInfo['hubUserId'],
   				                                 identityName))
      self.dbConnection.commit()


   def updateActiveJob(self,
                       jobInfo):
      timeRecorded = time.time()
      cursor = self.dbConnection.cursor()
      cursor.execute("UPDATE activeJobs SET jobStatus=?,jobStage=?,jobQueue=?,executionHost=?,timeRecorded=? \
                                      WHERE(siteDesignator=? AND remoteJobId=?)",(jobInfo['jobStatus'],jobInfo['jobStage'], \
                                                                                  jobInfo['jobQueue'],jobInfo['executionHost'], \
                                                                                  timeRecorded, \
                                                                                  jobInfo['siteDesignator'],jobInfo['remoteJobId']))
      self.dbConnection.commit()


   def purgeActiveJobs(self,
                       purgeJobStatus,
                       cutOffAge=0.):
      cutOffTime = time.time()-cutOffAge
      cursor = self.dbConnection.cursor()
      cursor.execute("DELETE FROM activeJobs WHERE(jobStatus=? AND timeRecorded<=?)",(purgeJobStatus,cutOffTime))
      self.dbConnection.commit()
      nPurgeActiveJobs = cursor.rowcount

      cursor.execute("DELETE FROM activeJobIdentities \
                            WHERE(localJobId || instanceId) \
                               IN(SELECT activeJobIdentities.localJobId || activeJobIdentities.instanceId \
                                    FROM activeJobIdentities \
                               LEFT JOIN activeJobs ON (activeJobIdentities.localJobId=activeJobs.localJobId AND \
                                                        activeJobIdentities.instanceId=activeJobs.instanceId) \
                                   WHERE activeJobs.localJobId IS NULL)")
      self.dbConnection.commit()

      return(nPurgeActiveJobs)


   def getFileTail(self,
                   siteDesignator,
                   remoteJobId,
                   fileName):
      tailText = ""
      cursor = self.dbConnection.cursor()
      cursor.execute("SELECT * FROM fileTailings \
                              WHERE(siteDesignator=? AND remoteJobId=? AND fileName=?)",(siteDesignator,remoteJobId,fileName))
      row = cursor.fetchone()
      if row:
         tailText = str(row['tailText'])

      return(tailText)


   def getFileTails(self,
                    siteDesignator,
                    remoteJobId):
      tailTexts = {}
      cursor = self.dbConnection.cursor()
      cursor.execute("SELECT * FROM fileTailings \
                              WHERE(siteDesignator=? AND remoteJobId=?)",(siteDesignator,remoteJobId))
      rows = cursor.fetchall()
      for row in rows:
         fileName    = row['fileName']
         tailText    = str(row['tailText'])
         timeUpdated = row['timeUpdated']
         tailTexts[fileName] = {'tailText':tailText,
                                'timeUpdated':timeUpdated}

      return(tailTexts)


   def addFileTail(self,
                   siteDesignator,
                   remoteJobId,
                   fileName,
                   tailText):
      timeUpdated = time.time()
      cursor = self.dbConnection.cursor()
      cursor.execute("INSERT INTO fileTailings (siteDesignator,remoteJobId,fileName,timeUpdated,tailText) \
                           VALUES(?,?,?,?,?)",(siteDesignator,
                                               remoteJobId,
                                               fileName,
                                               timeUpdated,
                                               buffer(tailText)))
      self.dbConnection.commit()


   def updateFileTail(self,
                      siteDesignator,
                      remoteJobId,
                      fileName,
                      tailText):
      timeUpdated = time.time()
      cursor = self.dbConnection.cursor()
      cursor.execute("UPDATE fileTailings SET timeUpdated=?,tailText=? \
                                        WHERE(siteDesignator=? AND remoteJobId=? AND fileName=?)",(timeUpdated,buffer(tailText), \
                                                                                                   siteDesignator,remoteJobId, \
                                                                                                   fileName))
      self.dbConnection.commit()


   def deleteFileTail(self,
                      siteDesignator,
                      remoteJobId,
                      fileName):
      cursor = self.dbConnection.cursor()
      cursor.execute("DELETE FROM fileTailings \
                            WHERE(siteDesignator=? AND remoteJobId=? AND fileName=?)",(siteDesignator,remoteJobId,fileName))
      self.dbConnection.commit()


   def deleteFileTails(self,
                       siteDesignator,
                       remoteJobId):
      cursor = self.dbConnection.cursor()
      cursor.execute("DELETE FROM fileTailings \
                            WHERE(siteDesignator=? AND remoteJobId=?)",(siteDesignator,remoteJobId))
      self.dbConnection.commit()


   def purgeFileTails(self):
      cursor = self.dbConnection.cursor()
      cursor.execute("DELETE FROM fileTailings \
                            WHERE(siteDesignator || remoteJobId) \
                               IN(SELECT fileTailings.siteDesignator || fileTailings.remoteJobId \
                                    FROM fileTailings \
                               LEFT JOIN activeJobs ON (fileTailings.siteDesignator=activeJobs.siteDesignator AND \
                                                        fileTailings.remoteJobId=activeJobs.remoteJobId) \
                                   WHERE activeJobs.siteDesignator IS NULL)")
      self.dbConnection.commit()
      nPurgeFileTails = cursor.rowcount

      return(nPurgeFileTails)


   def updateWFJobs(self,
                    siteDesignator):
      jobsInfo = []
      cursor = self.dbConnection.cursor()
      cursor.execute("SELECT * FROM activeJobs WHERE(siteDesignator=? AND jobStatus='WF')",(siteDesignator,))
      rows = cursor.fetchall()
      for row in rows:
         jobInfo = JobInfo(siteDesignator=row['siteDesignator'],
                           remoteJobId=row['remoteJobId'],
                           localJobId=row['localJobId'],
                           instanceId=row['instanceId'],
                           jobStatus=row['jobStatus'],
                           jobStage=row['jobStage'],
                           jobQueue=row['jobQueue'])
         jobsInfo.append(jobInfo)

      commit = False
      for jobInfo in jobsInfo:
         cursor.execute("SELECT * FROM activeJobs WHERE(localJobId=? AND instanceId=0)",(jobInfo['localJobId'],))
         row = cursor.fetchone()
         if row:
            if row['jobStatus'] == 'D' or row['jobStatus'] == 'Dr':
               jobInfo['jobStatus'] = row['jobStatus']
         else:
            jobInfo['jobStatus'] = 'Dr'
         if jobInfo['jobStatus'] != 'WF':
            self.updateActiveJob(jobInfo)
            commit = True

      if commit:
         self.dbConnection.commit()

      del jobsInfo


   def getActiveJobCount(self):
      cursor = self.dbConnection.cursor()
      cursor.execute("SELECT COUNT(*) FROM activeJobs")
      row = cursor.fetchone()
      nActiveJobCount = row[0]

      return(nActiveJobCount)


   def addUserActivity(self,
                       siteName,
                       hubUserId,
                       activityScore):
      timeUpdated = time.time()
      cursor = self.dbConnection.cursor()
      cursor.execute("INSERT INTO userActivityScores (siteName,hubUserId,activityScore,timeUpdated) \
                           VALUES(?,?,?,?)",(siteName,hubUserId,activityScore,timeUpdated))
      self.dbConnection.commit()


   def getUserActivity(self,
                       aggregator,
                       hubUserId):
      activityScore = 0.5
      timeUpdated   = -1.

      cursor = self.dbConnection.cursor()
      cursor.execute("SELECT activityScore,timeUpdated FROM userActivityScores \
                                                      WHERE(aggregator=? AND hubUserId=?)",(aggregator,hubUserId))
      row = cursor.fetchone()
      if row:
         activityScore = row['activityScore']
         timeUpdated   = row['timeUpdated']

      return(activityScore,timeUpdated)


   def getUsersActivity(self,
                        hubUserIds):
      usersActivity = {}
      cursor = self.dbConnection.cursor()
      if hubUserIds == '*':
         cursor.execute("SELECT aggregator,hubUserId,activityScore FROM userActivityScores")
      elif len(hubUserIds) == 1:
         cursor.execute("SELECT aggregator,hubUserId,activityScore FROM userActivityScores WHERE(hubUserId=?)",(hubUserIds[0],))
      else:
         cursor.execute("SELECT aggregator,hubUserId,activityScore FROM userActivityScores")

      rows = cursor.fetchall()
      for row in rows:
         if hubUserIds == '*' or row['hubUserId'] in hubUserIds:
            if not row['aggregator'] in usersActivity:
               usersActivity[row['aggregator']] = {}
            usersActivity[row['aggregator']][row['hubUserId']] = row['activityScore']

      return(usersActivity)


   def getUsersPriority(self,
                        hubUserIds):
      usersPriority = {}

      usersActivity = self.getUsersActivity('*')
      for aggregator in usersActivity:
         usersPriority[aggregator] = {}
         sumPriorities = 0.
         if hubUserIds == '*':
            for hubUserId in usersActivity[aggregator]:
               submitterClass = self.getUserSubmitterClass(hubUserId)
               priorityBoosters = self.aggregatorsInfo.getAggregatorKeyValue(aggregator,'priorityBoosters')
               if submitterClass in priorityBoosters:
                  priorityBooster = float(priorityBoosters[submitterClass])
               else:
                  priorityBooster = 1.
               usersPriority[aggregator][hubUserId] = priorityBooster/usersActivity[aggregator][hubUserId]
               sumPriorities += usersPriority[aggregator][hubUserId]
         else:
            for hubUserId in hubUserIds:
               submitterClass = self.getUserSubmitterClass(hubUserId)
               priorityBoosters = self.aggregatorsInfo.getAggregatorKeyValue(aggregator,'priorityBoosters')
               if submitterClass in priorityBoosters:
                  priorityBooster = float(priorityBoosters[submitterClass])
               else:
                  priorityBooster = 1.
               if hubUserId in usersActivity[aggregator]:
                  usersPriority[aggregator][hubUserId] = priorityBooster/usersActivity[aggregator][hubUserId]
               else:
                  usersPriority[aggregator][hubUserId] = priorityBooster/0.5
               sumPriorities += usersPriority[aggregator][hubUserId]

         for hubUserId in usersPriority[aggregator]:
            usersPriority[aggregator][hubUserId] /= sumPriorities

      return(usersPriority)


   def updateUserActivity(self,
                          aggregator,
                          hubUserId,
                          activityScore,
                          timeUpdated):
      cursor = self.dbConnection.cursor()
      cursor.execute("UPDATE userActivityScores SET activityScore=?,timeUpdated=? \
                                              WHERE(aggregator=? AND hubUserId=?)",(activityScore,timeUpdated,aggregator,hubUserId))
      self.dbConnection.commit()


   def updateUserActivityScores(self):
      now = time.time()
      currentActivityScores = {}

      cursor = self.dbConnection.cursor()
      cursor.execute("SELECT aggregator,hubUserId,SUM(nCores) AS activity FROM activeJobs \
                                                               WHERE(instanceId != 0 AND jobStatus != 'D' AND jobStatus != 'Dr') \
                                                               GROUP BY aggregator,hubUserId")
      rows = cursor.fetchall()
      for row in rows:
         if not row['aggregator'] in currentActivityScores:
            currentActivityScores[row['aggregator']] = {}
         currentActivityScores[row['aggregator']][row['hubUserId']] = row['activity']

      updatedActivityScores = {}
      cursor.execute("SELECT * FROM userActivityScores")
      rows = cursor.fetchall()
      for row in rows:
         aggregator    = row['aggregator']
         hubUserId     = row['hubUserId']
         activityScore = row['activityScore']
         timeUpdated   = row['timeUpdated']

         currentActivityScore = 0.
         if aggregator in currentActivityScores:
            if hubUserId in currentActivityScores[aggregator]:
               currentActivityScore = currentActivityScores[aggregator][hubUserId]

         beta = math.pow(0.5,(now-timeUpdated)/86400.)
         activity = beta*activityScore + (1.-beta)*currentActivityScore
         if not aggregator in updatedActivityScores:
            updatedActivityScores[aggregator] = {}
         updatedActivityScores[aggregator][hubUserId] = max(activity,0.5)

      if len(updatedActivityScores) > 0:
         cursor = self.dbConnection.cursor()
         for aggregator in updatedActivityScores:
            for hubUserId in updatedActivityScores[aggregator]:
               cursor.execute("UPDATE userActivityScores SET activityScore=?,timeUpdated=? \
                                                       WHERE(aggregator=? AND hubUserId=?)",
                             (updatedActivityScores[aggregator][hubUserId],now,aggregator,hubUserId))
         self.dbConnection.commit()

      updatedActivityScores = {}
      for aggregator in currentActivityScores:
         for hubUserId in currentActivityScores[aggregator]:
            activityScore,timeUpdated = self.getUserActivity(aggregator,hubUserId)
            if timeUpdated < 0.:
               if not aggregator in updatedActivityScores:
                  updatedActivityScores[aggregator] = {}
               updatedActivityScores[aggregator][hubUserId] = currentActivityScores[aggregator][hubUserId]

      if len(updatedActivityScores) > 0:
         cursor = self.dbConnection.cursor()
         for aggregator in updatedActivityScores:
            for hubUserId in updatedActivityScores[aggregator]:
               cursor.execute("INSERT INTO userActivityScores (aggregator,hubUserId,activityScore,timeUpdated) \
                                    VALUES(?,?,?,?)",(aggregator,hubUserId,updatedActivityScores[aggregator][hubUserId],now))
         self.dbConnection.commit()


   def updateRegisteredJobStates(self,
                                 jobsInfo,
                                 jobRegistrationState):
      cursor = self.dbConnection.cursor()
      for jobInfo in jobsInfo:
         cursor.execute("UPDATE registeredJobs SET state=? \
                                             WHERE(localJobId=? AND instanceId=?)",
                       (jobRegistrationState,jobInfo['localJobId'],jobInfo['instanceId']))
      self.dbConnection.commit()


   def releaseRegisteredJobs(self):
      nReleasedJobs   = 0
      releaseJobsInfo = []
# get list of users with held registered jobs
      usersWithRegisteredJob = self.getUsersWithRegisteredJob()
# get user priority for each aggregator
      usersPriority = self.getUsersPriority(usersWithRegisteredJob)
# get the number of active jobs for each aggregator
      aggregatorsActiveJobCount = self.getAggregatorsActiveJobCount()
# get users with held jobs at each aggregator
      aggregatorUserRegisteredJobs = self.getAggregatorUserRegisteredJob()
      for aggregator in aggregatorUserRegisteredJobs:
         if not aggregator in usersPriority:
            usersPriority[aggregator] = {}
            nAggregatorUserRegisteredJobs = len(aggregatorUserRegisteredJobs)
            for hubUserId in aggregatorUserRegisteredJobs[aggregator]:
               submitterClass = self.getUserSubmitterClass(hubUserId)
               priorityBoosters = self.aggregatorsInfo.getAggregatorKeyValue(aggregator,'priorityBoosters')
               try:
                  if submitterClass in priorityBoosters:
                     priorityBooster = float(priorityBoosters[submitterClass])
                  else:
                     priorityBooster = 1.
               except:
                  priorityBooster = 1.
               usersPriority[aggregator][hubUserId] = priorityBooster/nAggregatorUserRegisteredJobs

      for aggregator in aggregatorUserRegisteredJobs:
         maximumAggregatorActiveJobs = self.aggregatorsInfo.getAggregatorKeyValue(aggregator,'maximumActiveJobs')
         nAggregatorJobs = self.getRegisteredJobCountAtAggregator(aggregator)
         for hubUserId in aggregatorUserRegisteredJobs[aggregator]:
            priority = usersPriority[aggregator][hubUserId]
# get job info for registered held jobs at aggregator belonging to user
            jobsInfo = self.getAggregatorRegisteredJobsForUser(aggregator,hubUserId)
            nJobsToRelease = max(1,min(len(jobsInfo),int(round(float(nAggregatorJobs)*priority))))
            self.logger.log(logging.DEBUG,getLogMessage("nJobsToRelease: %d" % (nJobsToRelease)))
            for jobInfo in jobsInfo[:nJobsToRelease]:
               if not jobInfo['aggregator'] in aggregatorsActiveJobCount:
                  aggregatorsActiveJobCount[jobInfo['aggregator']] = 0
               if aggregatorsActiveJobCount[jobInfo['aggregator']] < maximumAggregatorActiveJobs:
                  releaseJobsInfo.append(jobInfo)
                  aggregatorsActiveJobCount[jobInfo['aggregator']] += 1

      if releaseJobsInfo:
         nReleasedJobs = len(releaseJobsInfo)
         self.updateRegisteredJobStates(releaseJobsInfo,self.JOBREGISTRATIONSTATERELEASED)

      del releaseJobsInfo

      return(nReleasedJobs)


   def isRegisteredJobReleased(self,
                               localJobId,
                               instanceId):
      registeredJobReleased = False
      cursor = self.dbConnection.cursor()
      cursor.execute("SELECT state FROM registeredJobs WHERE(localJobId=? AND instanceId=?)",(localJobId,instanceId))
      row = cursor.fetchone()
      if row:
         if row['state'] == self.JOBREGISTRATIONSTATERELEASED:
            registeredJobReleased = True

      return(registeredJobReleased)


   def addUserSubmitterClass(self,
                             hubUserId,
                             submitterClass):
      cursor = self.dbConnection.cursor()
      cursor.execute("INSERT INTO userSubmitterClasses (hubUserId,submitterClass) \
                           VALUES(?,?)",(hubUserId,submitterClass))
      self.dbConnection.commit()


   def updateUserSubmitterClass(self,
                                hubUserId,
                                submitterClass):
      cursor = self.dbConnection.cursor()
      cursor.execute("UPDATE userSubmitterClasses SET submitterClass=? WHERE(hubUserId=?)",(submitterClass,hubUserId))
      self.dbConnection.commit()


   def getUserSubmitterClass(self,
                             hubUserId):
      userSubmitterClass = -1
      cursor = self.dbConnection.cursor()
      cursor.execute("SELECT submitterClass FROM userSubmitterClasses WHERE(hubUserId=?)",(hubUserId,))

      row = cursor.fetchone()
      if row:
         userSubmitterClass = row['submitterClass']

      return(userSubmitterClass)


   def isJobSitePending(self,
                        jobSite):
      return(jobSite in self.pendingJobSites)


   def addPendingJobSite(self,
                         jobSite):
      self.pendingJobSites.append(jobSite)


   def deletePendingJobSite(self,
                            jobSite):
      self.pendingJobSites.remove(jobSite)


   def addPendingJobPosting(self,
                            siteDesignator,
                            remoteJobId,
                            jobWorkDirectory,
                            localJobId,
                            instanceId,
                            runName,
                            tailFiles):
      if not siteDesignator in self.pendingJobPostings:
         self.pendingJobPostings[siteDesignator] = {}
      self.pendingJobPostings[siteDesignator][remoteJobId] = {'jobWorkDirectory':jobWorkDirectory,
                                                              'localJobId':localJobId,
                                                              'instanceId':instanceId,
                                                              'runName':runName,
                                                              'tailFiles':tailFiles}


   def getPendingJobPostingSites(self):
      pendingJobPostingSites = self.pendingJobPostings.keys()

      return(pendingJobPostingSites)


   def postPendingJobPostings(self):
      markedForDeletion = []
      for siteDesignator in self.pendingJobPostings:
         if self.isJobSiteActive(siteDesignator):
            channel = self.getActiveJobSiteChannel(siteDesignator)
            for remoteJobId in self.pendingJobPostings[siteDesignator]:
               jsonObject = {'messageType':'newJobId',
                             'siteDesignator':siteDesignator,
                             'remoteJobId':remoteJobId,
                             'jobWorkDirectory':self.pendingJobPostings[siteDesignator][remoteJobId]['jobWorkDirectory'],
                             'localJobId':self.pendingJobPostings[siteDesignator][remoteJobId]['localJobId'],
                             'instanceId':self.pendingJobPostings[siteDesignator][remoteJobId]['instanceId'],
                             'runName':self.pendingJobPostings[siteDesignator][remoteJobId]['runName'],
                             'tailFiles':self.pendingJobPostings[siteDesignator][remoteJobId]['tailFiles']}
               self.postJsonMessage(channel,jsonObject)
            markedForDeletion.append(siteDesignator)

      for siteDesignator in markedForDeletion:
         del self.pendingJobPostings[siteDesignator]
      del markedForDeletion


   def signalJobSiteMonitors(self,
                             signalNumber):
      activeJobSites = self.getActiveJobSites()
      sitesToDelete = []
      for siteDesignator in activeJobSites:
         sitePID = self.getActiveJobSitePID(siteDesignator)
         self.logger.log(logging.INFO,getLogMessage("Send signal %d to %d - %s" % (signalNumber,sitePID,siteDesignator)))
         try:
            os.kill(sitePID,signalNumber)
         except OSError:
            sitesToDelete.append(siteDesignator)
         except:
            sitesToDelete.append(siteDesignator)
            self.logger.log(logging.ERROR,getLogMessage(traceback.format_exc()))

      for siteDesignator in sitesToDelete:
         self.deleteActiveJobSite(siteDesignator)


   def countRunningJobSiteMonitors(self):
      nRunningJobSiteMonitors = 0
      activeJobSites = self.getActiveJobSites()
      for siteDesignator in activeJobSites:
         sitePID = self.getActiveJobSitePID(siteDesignator)
         self.logger.log(logging.INFO,getLogMessage("Send signal %d to %d - %s" % (signalNumber,sitePID,siteDesignator)))
         try:
            os.kill(sitePID,signalNumber)
         except:
            pass
         else:
            nRunningJobSiteMonitors += 1

      return(nRunningJobSiteMonitors)


   def processRequests(self):
      for channel in self.activeChannels:
         message = self.pullMessage(channel,0)
         while message:
            args = message.split()
            if args[0] == 'json':
               jsonMessageLength = int(args[1])
               jsonMessage = self.pullMessage(channel,jsonMessageLength)
               if len(jsonMessage) > 0:
                  try:
                     jsonObject = json.loads(jsonMessage)
                  except ValueError:
                     self.logger.log(logging.ERROR,getLogMessage("JSON object %s could not be decoded" % (jsonMessage)))
                  else:
                     if   jsonObject['messageType'] == 'startJobSite':
                        self.logger.log(logging.DEBUG,getLogMessage(jsonMessage))
                        siteDesignator = jsonObject['siteDesignator']
                        if self.isJobSitePending(siteDesignator):
                           self.addActiveJobSite(siteDesignator,channel,jsonObject['pid'])
                           self.deletePendingJobSite(siteDesignator)
                        else:
                           self.logger.log(logging.ERROR,getLogMessage("Activated job site %s not pending" % (siteDesignator)))
                     elif jsonObject['messageType'] == 'siteUpdate':
                        siteDesignator = jsonObject['siteDesignator']
                        self.logger.log(logging.INFO,getLogMessage("%d/%d job status messages received from %s" % \
                                                                   (len(jsonObject['jobStates']), \
                                                                   jsonObject['nJobStates'],siteDesignator)))
                        for jobState in jsonObject['jobStates']:
                           try:
                              remoteJobId   = jobState.get('jobId')
                              jobStatus     = jobState.get('status')
                              jobStage      = jobState.get('stage')
                              jobQueue      = jobState.get('queue','?')
                              executionHost = jobState.get('executionHost','?')
                              tailFiles     = jobState.get('tailFiles',[])
                              jobInfo = JobInfo(siteDesignator=siteDesignator,
                                                remoteJobId=remoteJobId,
                                                jobStatus=jobStatus,
                                                jobStage=jobStage,
                                                jobQueue=jobQueue,
                                                executionHost=executionHost)
                              self.updateActiveJob(jobInfo)
                              for tailFile in tailFiles:
                                 tailText = tailFiles[tailFile]
                                 if tailText != '?':
                                    self.updateFileTail(siteDesignator,
                                                        remoteJobId,
                                                        tailFile,
                                                        tailText)
                           except:
                              pass
                        self.updateWFJobs(siteDesignator)
                        self.dumpActiveJobs()
                        self.updateActiveJobSite(siteDesignator)
                     elif jsonObject['messageType'] == 'stopJobSite':
                        self.logger.log(logging.DEBUG,getLogMessage(jsonMessage))
                        siteDesignator = jsonObject['siteDesignator']
                        self.deleteActiveJobSite(siteDesignator)
                     elif jsonObject['messageType'] == 'registerJob':
                        localJobId     = int(jsonObject['localJobId'])
                        instanceId     = int(jsonObject['instanceId'])
                        distributorPid = jsonObject['distributorPid']
                        hubUserId      = jsonObject['hubUserId']
                        siteName       = jsonObject['siteName']
                        submitterClass = jsonObject['submitterClass']
                        aggregator     = self.aggregatorsInfo.getSiteAggregator(siteName)

                        oldSubmitterClass = self.getUserSubmitterClass(hubUserId)
                        if   oldSubmitterClass < 0:
                           self.addUserSubmitterClass(hubUserId,submitterClass)
                        elif oldSubmitterClass != submitterClass:
                           self.updateUserSubmitterClass(hubUserId,submitterClass)

                        jobInfo = JobInfo(aggregator=aggregator,
                                          siteName=siteName,
                                          distributorPid=distributorPid,
                                          localJobId=localJobId,
                                          instanceId=instanceId,
                                          hubUserId=hubUserId)
                        self.addRegisteredJob(jobInfo)
                        del jobInfo

                        returnMessage = {'messageType':'jobInfo',
                                         'aggregator':aggregator,
                                         'siteName':siteName,
                                         'localJobId':localJobId,
                                         'instanceId':instanceId,
                                         'jobRegistrationState':self.JOBREGISTRATIONSTATEHELD}
                        self.postJsonMessage(channel,returnMessage)
                     elif jsonObject['messageType'] == 'deleteRegisteredJob':
                        localJobId = int(jsonObject['localJobId'])
                        instanceId = int(jsonObject['instanceId'])
                        nDeletedRegisteredJobs = self.deleteRegisteredJob(localJobId,instanceId)

                        returnMessage = {'messageType':'deletedRegisteredJobs',
                                         'localJobId':localJobId,
                                         'instanceId':instanceId,
                                         'nDeleted':nDeletedRegisteredJobs}
                        self.postJsonMessage(channel,returnMessage)
                     elif jsonObject['messageType'] == 'isRegisteredJobReleased':
                        localJobId = int(jsonObject['localJobId'])
                        instanceId = int(jsonObject['instanceId'])
                        registeredJobReleased = self.isRegisteredJobReleased(localJobId,instanceId)

                        returnMessage = {'messageType':'jobReleased',
                                         'localJobId':localJobId,
                                         'instanceId':instanceId,
                                         'jobReleased':registeredJobReleased}
                        self.postJsonMessage(channel,returnMessage)
                     elif jsonObject['messageType'] == 'postJob':
                        siteName         = jsonObject['siteName']
                        siteDesignator   = jsonObject['siteDesignator']
                        identityNames    = jsonObject['identityNames']
                        remoteJobId      = jsonObject['remoteJobId']
                        hubUserId        = jsonObject['hubUserId']
                        jobWorkDirectory = jsonObject['jobWorkDirectory']
                        localJobId       = int(jsonObject['localJobId'])
                        instanceId       = int(jsonObject['instanceId'])
                        destination      = jsonObject['destination']
                        runName          = jsonObject['runName']
                        nCores           = jsonObject['nCores']
                        distributorPid   = jsonObject['distributorPid']
                        tailFiles        = jsonObject['tailFiles']
                        aggregator       = self.aggregatorsInfo.getSiteAggregator(siteName)

                        jobInfo = self.getActiveJob(siteDesignator,remoteJobId)
                        if jobInfo:
                           jobStatus = jobInfo['jobStatus']
                           jobStage  = jobInfo['jobStage']
                        else:
                           jobStatus,jobStage,jobQueue = ('N','Job','?')
                           jobInfo = JobInfo(aggregator=aggregator,
                                             siteName=siteName,
                                             identityNames=identityNames,
                                             siteDesignator=siteDesignator,
                                             remoteJobId=remoteJobId,
                                             distributorPid=distributorPid,
                                             runName=runName,
                                             nCores=nCores,
                                             localJobId=localJobId,
                                             instanceId=instanceId,
                                             jobStatus=jobStatus,
                                             jobStage=jobStage,
                                             jobQueue=jobQueue,
                                             hubUserId=hubUserId,
                                             destination=destination)
                           self.addActiveJob(jobInfo)
                           self.deleteRegisteredJob(localJobId,instanceId)
                           del jobInfo
                           for tailFile in tailFiles:
                              self.addFileTail(siteDesignator,
                                               remoteJobId,
                                               tailFile,
                                               "")
                        self.dumpActiveJobs()

                        returnMessage = {'messageType':'jobInfo',
                                         'siteDesignator':siteDesignator,
                                         'remoteJobId':remoteJobId,
                                         'status':jobStatus,
                                         'stage':jobStage}
                        self.postJsonMessage(channel,returnMessage)
                        self.addPendingJobPosting(siteDesignator,remoteJobId,jobWorkDirectory,
                                                  jsonObject['localJobId'],jsonObject['instanceId'],runName,tailFiles)
                     elif jsonObject['messageType'] == 'queryJob':
                        siteDesignator = jsonObject['siteDesignator']
                        remoteJobId    = jsonObject['remoteJobId']
                        markReportedAsDone = False
                        jobInfo = self.getActiveJob(siteDesignator,remoteJobId)
                        if jobInfo:
                           jobStatus = jobInfo['jobStatus']
                           jobStage  = jobInfo['jobStage']
                           jobSite   = jobInfo['executionHost']
                           if   jobStatus == 'D':
                              markReportedAsDone = True
                           elif jobStatus == 'Dr':
                              jobStatus = 'D'
                           tailFiles = self.getFileTails(siteDesignator,remoteJobId)
                        else:
                           jobStatus,jobStage,jobSite,jobQueue = ('?','?','?','?')
                           tailFiles = {}

                        returnMessage = {'messageType':'jobInfo',
                                         'siteDesignator':siteDesignator,
                                         'remoteJobId':remoteJobId,
                                         'status':jobStatus,
                                         'stage':jobStage,
                                         'site':jobSite,
                                         'tailFiles':tailFiles}
                        self.postJsonMessage(channel,returnMessage)

                        if markReportedAsDone:
                           jobInfo['jobStatus'] = 'Dr'
                           self.updateActiveJob(jobInfo)
                     elif jsonObject['messageType'] == 'terminateJob':
                        siteDesignator = jsonObject['siteDesignator']
                        remoteJobId    = jsonObject['remoteJobId']
                        jobInfo = self.getActiveJob(siteDesignator,remoteJobId)
                        if jobInfo:
                           jobStatus = jobInfo['jobStatus']
                           jobStage  = jobInfo['jobStage']
                        else:
                           jobStatus,jobStage,jobQueue = ('?','?','?')

                        returnMessage = {'messageType':'jobInfo',
                                         'siteDesignator':siteDesignator,
                                         'remoteJobId':remoteJobId,
                                         'status':jobStatus,
                                         'stage':jobStage}
                        self.postJsonMessage(channel,returnMessage)

                        if jobInfo:
                           jobInfo['jobStatus'] = 'D'
                           jobInfo['jobStage']  = 'Job'
                           self.updateActiveJob(jobInfo)
                     elif jsonObject['messageType'] == 'queryJobs':
                        jobsStatus = {}
                        jobsInfo = []
                        jobs = jsonObject['jobs']
                        for job in jobs:
                           siteDesignator = job['siteDesignator']
                           remoteJobId    = job['remoteJobId']
                           jobInfo = self.getActiveJob(siteDesignator,remoteJobId)
                           if not jobInfo:
                              jobStatus,jobStage,jobSite,jobQueue = ('?','?','?','?')
                              jobInfo = JobInfo(siteDesignator=siteDesignator,
                                                remoteJobId=remoteJobId,
                                                jobStatus=jobStatus,
                                                jobStage=jobStage,
                                                executionHost=jobSite,
                                                jobQueue=jobQueue)
                           jobsInfo.append(jobInfo)
                           if not siteDesignator in jobsStatus:
                              jobsStatus[siteDesignator] = {}
   
                        for jobInfo in jobsInfo:
                           siteDesignator = jobInfo['siteDesignator']
                           remoteJobId    = jobInfo['remoteJobId']
                           jobStatus      = jobInfo['jobStatus']
                           jobStage       = jobInfo['jobStage']
                           jobSite        = jobInfo['executionHost']
                           jobQueue       = jobInfo['jobQueue']

                           markReportedAsDone = False
                           if   jobStatus == 'D':
                              markReportedAsDone = True
                           elif jobStatus == 'Dr':
                              jobStatus = 'D'
                           if jobStatus != '?':
                              tailFiles = self.getFileTails(siteDesignator,remoteJobId)
                           else:
                              tailFiles = {}

                           jobsStatus[siteDesignator][remoteJobId] = {'status':jobStatus,
                                                                      'stage':jobStage,
                                                                      'site':jobSite,
                                                                      'tailFiles':tailFiles}
                           if markReportedAsDone:
                              jobInfo['jobStatus'] = 'Dr'
                              self.updateActiveJob(jobInfo)

                        returnMessage = {'messageType':'jobsStatus',
                                         'jobsStatus':jobsStatus}
                        self.postJsonMessage(channel,returnMessage)
                     elif jsonObject['messageType'] == 'postWorkflow':
                        siteName       = jsonObject['siteName']
                        siteDesignator = jsonObject['siteDesignator']
                        identityNames  = jsonObject['identityNames']
                        remoteJobId    = jsonObject['remoteJobId']
                        hubUserId      = jsonObject['hubUserId']
                        localJobId     = int(jsonObject['localJobId'])
                        nInstances     = jsonObject['nInstances']
                        destination    = jsonObject['destination']
                        runName        = jsonObject['runName']
                        nCores         = jsonObject['nCores']
                        distributorPid = jsonObject['distributorPid']
                        tailFiles      = []
                        instanceId = 0
                        aggregator     = self.aggregatorsInfo.getSiteAggregator(siteName)

                        jobInfo = self.getActiveJob(siteDesignator,remoteJobId)
                        if jobInfo:
                           jobStatus = jobInfo['jobStatus']
                           jobStage  = jobInfo['jobStage']
                        else:
                           jobStatus,jobStage,jobQueue = ('N','DAG','?')
                           jobInfo = JobInfo(aggregator=aggregator,
                                             siteName=siteName,
                                             identityNames=identityNames,
                                             siteDesignator=siteDesignator,
                                             remoteJobId=remoteJobId,
                                             distributorPid=distributorPid,
                                             runName=runName,
                                             nCores=1,
                                             localJobId=localJobId,
                                             instanceId=instanceId,
                                             jobStatus=jobStatus,
                                             jobStage=jobStage,
                                             jobQueue=jobQueue,
                                             hubUserId=hubUserId,
                                             destination=destination)
                           self.addActiveJob(jobInfo)
                           self.deleteRegisteredJob(localJobId,instanceId)
                           del jobInfo
                        self.dumpActiveJobs()

                        returnMessage = {'messageType':'jobInfo',
                                         'siteDesignator':siteDesignator,
                                         'remoteJobId':remoteJobId,
                                         'status':jobStatus,
                                         'stage':jobStage}
                        self.postJsonMessage(channel,returnMessage)
                        jobWorkDirectory = ""
                        self.addPendingJobPosting(siteDesignator,remoteJobId,jobWorkDirectory,
                                                  jsonObject['localJobId'],'00',runName,tailFiles)

                        if '.' in remoteJobId:
                           wfRemoteJobIdBase = remoteJobId.split('.')[0]
                        else:
                           wfRemoteJobIdBase = remoteJobId
                        nInstanceIdDigits = max(2,int(math.log10(nInstances)+1))
                        wfJobStatus,wfJobStage,wfJobQueue = ('WF','Simulation','?')
                        for instance in xrange(1,nInstances+1):
                           wfRemoteJobId = wfRemoteJobIdBase + '.' + str(instance)
                           jobInfo = self.getActiveJob(siteDesignator,wfRemoteJobId)
                           if not jobInfo:
                              jobInfo = JobInfo(aggregator=aggregator,
                                                siteName=siteName,
                                                identityNames=identityNames,
                                                siteDesignator=siteDesignator,
                                                remoteJobId=wfRemoteJobId,
                                                distributorPid=distributorPid,
                                                runName=runName,
                                                nCores=nCores,
                                                localJobId=localJobId,
                                                instanceId=instance,
                                                jobStatus=wfJobStatus,
                                                jobStage=wfJobStage,
                                                jobQueue=wfJobQueue,
                                                hubUserId=hubUserId,
                                                destination=destination)
                              self.addActiveJob(jobInfo)
                              self.deleteRegisteredJob(localJobId,instance)
                              del jobInfo
                        self.dumpActiveJobs()
                     elif jsonObject['messageType'] == 'queryWorkflow':
                        siteDesignator = jsonObject['siteDesignator']
                        remoteJobId    = jsonObject['remoteJobId']
                        nInstances     = jsonObject['nInstances']
                        markReportedAsDone = False
                        jobInfo = self.getActiveJob(siteDesignator,remoteJobId)
                        if jobInfo:
                           jobStatus = jobInfo['jobStatus']
                           jobStage  = jobInfo['jobStage']
                           jobSite   = jobInfo['executionHost']
                           jobQueue  = jobInfo['jobQueue']
                           if   jobStatus == 'D':
                              markReportedAsDone = True
                           elif jobStatus == 'Dr':
                              jobStatus = 'D'
                        else:
                           jobStatus,jobStage,jobSite,jobQueue = ('?','?','?','?')
                        del jobInfo
                        wfInstances = {}
                        if '.' in remoteJobId:
                           wfRemoteJobIdBase = remoteJobId.split('.')[0]
                        else:
                           wfRemoteJobIdBase = remoteJobId
                        for instance in xrange(1,nInstances+1):
                           wfMarkReportedAsDone = False
                           wfRemoteJobId = wfRemoteJobIdBase + '.' + str(instance)
                           jobInfo = self.getActiveJob(siteDesignator,wfRemoteJobId)
                           if jobInfo:
                              wfJobStatus = jobInfo['jobStatus']
                              wfJobStage  = jobInfo['jobStage']
                              wfJobSite   = jobInfo['executionHost']
                              wfJobQueue  = jobInfo['jobQueue']
                              if   wfJobStatus == 'D':
                                 wfMarkReportedAsDone = True
                              elif wfJobStatus == 'Dr':
                                 wfJobStatus = 'D'
                           else:
                              wfJobStatus,wfJobStage,wfJobSite,wfJobQueue = ('?','?','?','?')
                           wfInstances[instance] = {}
                           wfInstances[instance]['remoteJobId']        = wfRemoteJobId
                           wfInstances[instance]['jobStatus']          = wfJobStatus
                           wfInstances[instance]['jobStage']           = wfJobStage
                           wfInstances[instance]['jobSite']            = wfJobSite
                           wfInstances[instance]['jobQueue']           = wfJobQueue
                           wfInstances[instance]['markReportedAsDone'] = wfMarkReportedAsDone
                           del jobInfo

                        instanceStates = {}
                        for instance in xrange(1,nInstances+1):
                           if wfInstances[instance]['jobStatus'] != '?':
                              instanceStates[instance] = {'status':wfInstances[instance]['jobStatus'],
                                                          'stage':wfInstances[instance]['jobStage'],
                                                          'site':wfInstances[instance]['jobSite']}

                        returnMessage = {'messageType':'workflowInfo',
                                         'siteDesignator':siteDesignator,
                                         'remoteJobId':remoteJobId,
                                         'status':jobStatus,
                                         'stage':jobStage,
                                         'site':jobSite,
                                         'instances':instanceStates}
                        self.postJsonMessage(channel,returnMessage)

                        if markReportedAsDone:
                           jobStatus = 'Dr'
                           jobInfo = JobInfo(siteDesignator=siteDesignator,
                                             remoteJobId=remoteJobId,
                                             jobStatus=jobStatus,
                                             jobStage=jobStage,
                                             jobQueue=jobQueue)
                           self.updateActiveJob(jobInfo)
                           del jobInfo
                        for instance in xrange(1,nInstances+1):
                           if wfInstances[instance]['markReportedAsDone']:
                              wfRemoteJobId = wfInstances[instance]['remoteJobId']
                              wfJobStatus   = 'Dr'
                              wfJobStage    = wfInstances[instance]['jobStage']
                              wfJobQueue    = wfInstances[instance]['jobQueue']
                              jobInfo = JobInfo(siteDesignator=siteDesignator,
                                                remoteJobId=wfRemoteJobId,
                                                jobStatus=wfJobStatus,
                                                jobStage=wfJobStage,
                                                jobQueue=wfJobQueue)
                              self.updateActiveJob(jobInfo)
                              del jobInfo
                        del instanceStates
                        del wfInstances
                     elif jsonObject['messageType'] == 'reportSites':
                        siteDesignator = jsonObject['siteDesignator']
                        remoteJobId    = jsonObject['remoteJobId']

                        if siteDesignator == "*":
                           siteDesignators = self.getSitesWithActiveJob()
                        else:
                           siteDesignators = [siteDesignator]

                        report = {'messageType':'siteJobInfo',
                                  'maxLastReportTime':0,
                                  'siteDesignators':{}}
                        maxLastReportTime = 0
                        for siteDesignator in siteDesignators:
                           report['siteDesignators'][siteDesignator] = {}
                           lastReportTime = self.getActiveJobSiteTimeUpdated(siteDesignator)
                           maxLastReportTime = max(maxLastReportTime,lastReportTime)
                           report['siteDesignators'][siteDesignator]['lastReportTime'] = lastReportTime

                           report['siteDesignators'][siteDesignator]['queues'] = {}
                           activeQueues = self.getActiveJobSiteQueues(siteDesignator)
                           for activeQueue in activeQueues:
                              report['siteDesignators'][siteDesignator]['queues'][activeQueue] = {}

                              if remoteJobId == "*":
                                 reportJobs = self.getActiveJobsInQueue(siteDesignator,activeQueue)
                                 for reportJob in reportJobs:
                                    report['siteDesignators'][siteDesignator]['queues'][activeQueue][reportJob['remoteJobId']] = \
                                                                                               {'status':reportJob['jobStatus'], \
                                                                                                'stage':reportJob['jobStage'], \
                                                                                                'site':reportJob['executionHost']}
                              else:
                                 reportJob = self.getActiveJob(siteDesignator,remoteJobId)
                                 if reportJob:
                                    if reportJob['jobQueue'] == activeQueue:
                                       report['siteDesignators'][siteDesignator]['queues'][activeQueue][reportJob['remoteJobId']] = \
                                                                                              {'status':reportJob['jobStatus'], \
                                                                                               'stage':reportJob['jobStage'], \
                                                                                               'site':reportJob['executionHost']}

                        report['maxLastReportTime'] = maxLastReportTime
                        self.postJsonMessage(channel,report)
                        del siteDesignators
                     elif jsonObject['messageType'] == 'listUserActivity':
                        hubUserIds = jsonObject['hubUserIds']
                        report = ""
                        usersActivity = self.getUsersActivity(hubUserIds)

                        report = {'messageType':'userActivityInfo',
                                  'reportTime':time.time(),
                                  'usersActivity':usersActivity}
                        self.postJsonMessage(channel,report)
                        del usersActivity
                     elif jsonObject['messageType'] == 'listUserPriority':
                        hubUserIds = jsonObject['hubUserIds']
                        report = ""
                        usersPriority = self.getUsersPriority(hubUserIds)

                        report = {'messageType':'userPriorityInfo',
                                  'reportTime':time.time(),
                                  'usersPriority':usersPriority}
                        self.postJsonMessage(channel,report)
                        del usersPriority
                     elif jsonObject['messageType'] == 'listUserJobs':
                        hubUserId = jsonObject['hubUserId']
                        reportJobs = self.getActiveJobsForUser(hubUserId)

                        report = {'messageType':'userJobInfo',
                                  'reportTime':time.time(),
                                  'userJobs':reportJobs}
                        self.postJsonMessage(channel,report)
                        del reportJobs
                     elif jsonObject['messageType'] == 'getJobPID':
                        hubUserId  = jsonObject['hubUserId']
                        localJobId = jsonObject['localJobId']

                        activeJobPid = self.getActiveJobPidForUser(hubUserId,localJobId)
                        report = {'messageType':'jobPID',
                                  'hubUserId':hubUserId,
                                  'reportTime':time.time(),
                                  'pid':activeJobPid}
                        self.postJsonMessage(channel,report)
                     elif jsonObject['messageType'] == 'queryIdentityUsers':
                        identityName = jsonObject['identityName']
                        usersWithActiveJobWithIdentity = []
                        hubUserIdsWithActiveJobWithIdentity = self.getUsersWithActiveJobWithIdentity(identityName)
                        for hubUserId in hubUserIdsWithActiveJobWithIdentity:
                           try:
                              hubUserName = pwd.getpwuid(hubUserId).pw_name
                           except:
                              self.logger.log(logging.ERROR,getLogMessage("Unable to get info for user '%d'" % (hubUserId)))
                           else:
                              usersWithActiveJobWithIdentity.append(hubUserName)
                        report = {'messageType':'identityUsers',
                                  'identityName':identityName,
                                  'identityActiveJobUsers':usersWithActiveJobWithIdentity}
                        self.postJsonMessage(channel,report)
                     else:
                        self.logger.log(logging.ERROR,getLogMessage("Discarded message type: %s" % (jsonObject['messageType'])))
               else:
                  self.pushMessage(channel,message + '\n')
                  break
            else:
               self.logger.log(logging.ERROR,getLogMessage("Discarded message: %s" % (message)))

            message = self.pullMessage(channel,0)


