#!/usr/bin/env python
#
# @package      hubzero-submit-monitors
# @file         monitorProbe.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2004-2014 HUBzero Foundation, LLC.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2004-2014 HUBzero Foundation, LLC.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#
# ----------------------------------------------------------------------
#  monitorProbe.py
#
#  script which gathers monitoring statistics from remote probe tests
#
import sys
import os
import select
import signal
import socket
import time
import logging
from errno import EINTR

from hubzero.submit.LogMessage   import getLogPIDMessage as getLogMessage
from hubzero.submit.ProbeMonitor import ProbeMonitor

PROBEMONITORHOST = ""
PROBEMONITORPORT = 5728

MONITORROOT = os.path.join(os.sep,'opt','submit')

MONITORLOGLOCATION = os.path.join(os.sep,'var','log','submit','monitors')
MONITORLOGFILENAME = "monitorProbe.log"
APPLICATIONLOGGER  = logging.getLogger('') 

DUMPFILENAME = "monitorProbe.dump"
DUMPFILEPATH = os.path.join(MONITORROOT,DUMPFILENAME)


def openLogger(logDirectory,
               hubLogFile):
   class EmptyFilter(logging.Filter):
      """
      This is a filter which rejects empty messages

      """

      def filter(self,record):
         if record.getMessage() == "":
            emptyRecord = True
         else:
            emptyRecord = False

         return(not emptyRecord)

   APPLICATIONLOGGER.setLevel(logging.DEBUG)

   hubLogPath = os.path.join(logDirectory,hubLogFile)
   logHandler = logging.FileHandler(hubLogPath)
   fdLogFile = logHandler.stream.fileno()

   emptyFilter = EmptyFilter()
   logHandler.addFilter(emptyFilter)

   logFormatter = logging.Formatter('%(asctime)s %(message)s','%s [%a %b %d %H:%M:%S %Y]')
   logHandler.setFormatter(logFormatter)
   APPLICATIONLOGGER.addHandler(logHandler)

   return(fdLogFile)


def daemonize(fdLogFile):
   if fdLogFile != sys.stdout.fileno():
      devnull = open("/dev/null",'r')
      os.dup2(devnull.fileno(),sys.stdin.fileno())
      os.dup2(fdLogFile,sys.stdout.fileno())
      os.dup2(fdLogFile,sys.stderr.fileno())

   if os.fork() != 0:
      os.wait()
      sys.exit(0)
   else:
      os.setsid()
      pid = os.fork()
      if pid != 0:
         sys.exit(0)

   time.sleep(2)


class MonitorProbe:
   def __init__(self,
                probeMonitorHost,
                probeMonitorPort,
                dumpFilePath):
      self.logger           = logging.getLogger(__name__)
      self.probeMonitorHost = probeMonitorHost
      self.probeMonitorPort = probeMonitorPort
      self.dumpFilePath     = dumpFilePath
      self.probeMonitor     = None

      signal.signal(signal.SIGINT,self.sigINT_handler)
      signal.signal(signal.SIGHUP,self.sigHUP_handler)
      signal.signal(signal.SIGQUIT,self.sigQUIT_handler)
      signal.signal(signal.SIGABRT,self.sigABRT_handler)
      signal.signal(signal.SIGTERM,self.sigTERM_handler)


   def terminate(self):
      if self.probeMonitor:
         self.probeMonitor.dumpActiveProbes()
      self.logger.log(logging.INFO,getLogMessage("******************************"))
      self.logger.log(logging.INFO,getLogMessage("* grid probe monitor stopped *"))
      self.logger.log(logging.INFO,getLogMessage("******************************"))


   def sigGEN_handler(self,
                      signalNumber,
                      frame):
      self.terminate()
      sys.exit(1)


   def sigINT_handler(self,
                      signalNumber,
                      frame):
      self.logger.log(logging.INFO,getLogMessage("Received SIGINT!"))
      self.sigGEN_handler(signalNumber,frame)


   def sigHUP_handler(self,
                      signalNumber,
                      frame):
      self.logger.log(logging.INFO,getLogMessage("Received SIGHUP!"))
      self.sigGEN_handler(signalNumber,frame)


   def sigQUIT_handler(self,
                       signalNumber,
                       frame):
      self.logger.log(logging.INFO,getLogMessage("Received SIGQUIT!"))
      self.sigGEN_handler(signalNumber,frame)


   def sigABRT_handler(self,
                       signalNumber,
                       frame):
      self.logger.log(logging.INFO,getLogMessage("Received SIGABRT!"))
      self.sigGEN_handler(signalNumber,frame)


   def sigTERM_handler(self,
                       signalNumber,
                       frame):
      self.logger.log(logging.INFO,getLogMessage("Received SIGTERM!"))
      self.sigGEN_handler(signalNumber,frame)


   def monitor(self):
      self.probeMonitor = ProbeMonitor(self.probeMonitorHost,self.probeMonitorPort,activeProbeDumpPath=self.dumpFilePath)
      if not self.probeMonitor.isBound():
         sys.exit(1)

      self.logger.log(logging.INFO,getLogMessage("******************************"))
      self.logger.log(logging.INFO,getLogMessage("* grid probe monitor started *"))
      self.logger.log(logging.INFO,getLogMessage("******************************"))

      self.probeMonitor.loadActiveProbes()

      inputDescriptors  = []
      clientConnections = []

      monitorSocketFd = self.probeMonitor.boundFileDescriptor()
      inputDescriptors.append(monitorSocketFd)

      while 1:
         try:
            readyInputFds = select.select(inputDescriptors,[],[])[0]
         except select.error,err:
            if err[0] == EINTR:
               readyInputFds = []
            else:
               for inputDescriptor in inputDescriptors:
                  if isinstance(inputDescriptor,socket.socket):
                     try:
                        os.fstat(inputDescriptor.fileno())
                     except:
                        self.logger.log(logging.INFO,getLogMessage(inputDescriptor))
                  else:
                     try:
                        os.fstat(inputDescriptor)
                     except:
                        self.logger.log(logging.INFO,getLogMessage(inputDescriptor))
               self.terminate()
               raise

         if monitorSocketFd in readyInputFds:
            channel = self.probeMonitor.acceptConnection()
            if channel:
               clientConnections.append(channel)
               inputDescriptors.append(channel)

         for channel in readyInputFds:
            if channel in clientConnections:
               channelClosed = self.probeMonitor.processRequest(channel)
               if channelClosed:
                  clientConnections.remove(channel)
                  inputDescriptors.remove(channel)


if __name__ == '__main__':

   fdLogFile = openLogger(MONITORLOGLOCATION,MONITORLOGFILENAME)
   daemonize(fdLogFile)

   __monitorProbe__ = MonitorProbe(PROBEMONITORHOST,PROBEMONITORPORT,DUMPFILEPATH)
   __monitorProbe__.monitor()


