# @package      hubzero-submit-monitors
# @file         JobMonitor.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2012-2013 HUBzero Foundation, LLC.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2012-2013 HUBzero Foundation, LLC.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#

import re
import math
import sqlite3
import time
import traceback

from hubzero.submit.LogMessage  import logID as log
from hubzero.submit.MessageCore import MessageCore
from hubzero.submit.JobInfo     import JobInfo

class JobMonitor(MessageCore):
   def __init__(self,
                host,
                port,
                repeatDelay=10,
                fixedBufferSize=128,
                activeJobDBPath="monitorJob.db",
                activeJobDumpPath="monitorJob.sql"):
      bindLabel = "%s:%d" % (host if host else "localhost",port)
      MessageCore.__init__(self,bindHost=host,bindPort=port,bindLabel=bindLabel,
                                reuseAddress=True,repeatDelay=repeatDelay)
      self.fixedBufferSize   = fixedBufferSize
      self.activeJobDBPath   = activeJobDBPath
      self.activeJobDumpPath = activeJobDumpPath
      self.dbConnection      = None
      try:
         self.dbConnection = sqlite3.connect(self.activeJobDBPath,isolation_level=None)
         self.dbConnection.row_factory = sqlite3.Row
         nActiveJobCount = self.getActiveJobCount()
         log("%d jobs loaded from DB file" % (nActiveJobCount))
         self.purgeActiveJobSites()
      except:
         log(traceback.format_exc())


   def dumpActiveJobs(self):
      if self.dbConnection:
         fpDumpFile = open(self.activeJobDumpPath,'w')
         if fpDumpFile:
            for line in self.dbConnection.iterdump():
                fpDumpFile.write('%s\n' % (line))
            fpDumpFile.close()


   def close(self):
      if self.dbConnection:
         self.dbConnection.close()
      MessageCore.close(self)


   def addActiveJobSite(self,
                        jobSite,
                        value):
      cursor = self.dbConnection.cursor()
      cursor.execute("INSERT INTO activeJobSites VALUES(?,?)",(jobSite,value))
      self.dbConnection.commit()


   def updateActiveJobSite(self,
                           jobSite,
                           value):
      cursor = self.dbConnection.cursor()
      cursor.execute("UPDATE activeJobSites SET timeUpdated=? WHERE(jobSite=?)",(value,jobSite))
      self.dbConnection.commit()


   def getActiveJobSiteTimeUpdated(self,
                                   jobSite):
      timeUpdated = 0
      cursor = self.dbConnection.cursor()
      cursor.execute("SELECT * FROM activeJobSites WHERE(jobSite=?)",(jobSite,))
      row = cursor.fetchone()
      if row:
         timeUpdated = row['timeUpdated']

      return(timeUpdated)


   def deleteActiveJobSite(self,
                           jobSite):
      cursor = self.dbConnection.cursor()
      cursor.execute("DELETE FROM activeJobSites WHERE(jobSite=?)",(jobSite,))
      self.dbConnection.commit()


   def purgeActiveJobSites(self):
      cursor = self.dbConnection.cursor()
      cursor.execute("DELETE FROM activeJobSites")
      self.dbConnection.commit()


   def isJobSiteActive(self,
                       jobSite):
      jobSiteIsActive = False
      cursor = self.dbConnection.cursor()
      cursor.execute("SELECT * FROM activeJobSites WHERE(jobSite=?)",(jobSite,))
      row = cursor.fetchone()
      if row:
         jobSiteIsActive = True

      return(jobSiteIsActive)


   def getSitesWithActiveJob(self):
      sitesWithActiveJob = []
      cursor = self.dbConnection.cursor()
      cursor.execute("SELECT DISTINCT messageSite FROM activeJobs")
      rows = cursor.fetchall()
      for row in rows:
         sitesWithActiveJob.append(row['messageSite'])

      return(sitesWithActiveJob)


   def getActiveJobSiteQueues(self,
                              jobSite):
      activeJobSiteQueues = []
      cursor = self.dbConnection.cursor()
      cursor.execute("SELECT DISTINCT jobQueue FROM activeJobs WHERE(messageSite=?)",(jobSite,))
      rows = cursor.fetchall()
      for row in rows:
         activeJobSiteQueues.append(row['jobQueue'])

      return(activeJobSiteQueues)


   def getActiveJobsInQueue(self,
                            jobSite,
                            jobQueue):
      jobsInfo = []
      cursor = self.dbConnection.cursor()
      cursor.execute("SELECT * FROM activeJobs WHERE(messageSite=? AND jobQueue=?)",(jobSite,jobQueue))
      rows = cursor.fetchall()
      for row in rows:
         jobInfo = JobInfo(row['messageSite'],
                           row['remoteJobId'],
                           distributorPid=row['distributorPid'],
                           runName=row['runName'],
                           nCores=row['nCores'],
                           localJobId=row['localJobId'],
                           instanceId=row['instanceId'],
                           jobStatus=row['jobStatus'],
                           jobStage=row['jobStage'],
                           jobQueue=row['jobQueue'],
                           hubUserId=row['hubUserId'],
                           destination=row['destination'],
                           timeRecorded=row['timeRecorded'])
         jobsInfo.append(jobInfo)

      return(jobsInfo)


   def getActiveJobsForUser(self,
                            hubUserId):
      jobsInfo = []
      cursor = self.dbConnection.cursor()
      cursor.execute("SELECT * FROM activeJobs \
                               WHERE(hubUserId=? AND jobStatus!='Dr' AND jobStatus!='D')",(hubUserId,))
      rows = cursor.fetchall()
      for row in rows:
         jobInfo = JobInfo(row['messageSite'],
                           row['remoteJobId'],
                           distributorPid=row['distributorPid'],
                           runName=row['runName'],
                           nCores=row['nCores'],
                           localJobId=row['localJobId'],
                           instanceId=row['instanceId'],
                           jobStatus=row['jobStatus'],
                           jobStage=row['jobStage'],
                           jobQueue=row['jobQueue'],
                           hubUserId=row['hubUserId'],
                           destination=row['destination'],
                           timeRecorded=row['timeRecorded'])
         jobsInfo.append(jobInfo)

      return(jobsInfo)


   def getActiveJobPidForUser(self,
                              hubUserId,
                              localJobId):
      activeJobPid = None
      jobsInfo = []
      cursor = self.dbConnection.cursor()
      cursor.execute("SELECT * FROM activeJobs \
                               WHERE(hubUserId=? AND localJobId=? AND \
                                     jobStatus!='N' AND jobStatus!='Dr' AND jobStatus!='D') \
                               ORDER BY localJobId,instanceId",(hubUserId,localJobId))
      rows = cursor.fetchall()
      for row in rows:
         jobInfo = JobInfo(row['messageSite'],
                           row['remoteJobId'],
                           distributorPid=row['distributorPid'],
                           localJobId=row['localJobId'],
                           instanceId=row['instanceId'])
         jobsInfo.append(jobInfo)

      if len(jobsInfo) > 0:
         for jobInfo in jobsInfo:
            if jobInfo['distributorPid'] > 0:
               activeJobPid = jobInfo['distributorPid']
               break

      del jobsInfo

      return(activeJobPid)


   def getActiveJob(self,
                    messageSite,
                    remoteJobId):
      jobInfo = None
      cursor = self.dbConnection.cursor()
      cursor.execute("SELECT * FROM activeJobs WHERE(messageSite=? AND remoteJobId=?)",(messageSite,remoteJobId))
      row = cursor.fetchone()
      if row:
         jobInfo = JobInfo(messageSite,
                           remoteJobId,
                           distributorPid=row['distributorPid'],
                           runName=row['runName'],
                           nCores=row['nCores'],
                           localJobId=row['localJobId'],
                           instanceId=row['instanceId'],
                           jobStatus=row['jobStatus'],
                           jobStage=row['jobStage'],
                           jobQueue=row['jobQueue'],
                           hubUserId=row['hubUserId'],
                           destination=row['destination'],
                           timeRecorded=row['timeRecorded'])

      return(jobInfo)


   def addActiveJob(self,
                    jobInfo):
      cursor = self.dbConnection.cursor()
      cursor.execute("INSERT INTO activeJobs VALUES(?,?,?,?,?,?,?,?,?,?,?,?,?)",(jobInfo['localJobId'],
                                                                                 jobInfo['instanceId'],
                                                                                 jobInfo['distributorPid'],
                                                                                 jobInfo['nCores'],
                                                                                 jobInfo['runName'],
                                                                                 jobInfo['messageSite'],
   				                                                 jobInfo['remoteJobId'],
   				                                                 jobInfo['jobStatus'],
   				                                                 jobInfo['jobStage'],
   				                                                 jobInfo['jobQueue'],
   				                                                 jobInfo['hubUserId'],
   				                                                 jobInfo['destination'],
   				                                                 jobInfo['timeRecorded']))
      self.dbConnection.commit()


   def updateActiveJob(self,
                       jobInfo):
      cursor = self.dbConnection.cursor()
      cursor.execute("UPDATE activeJobs SET jobStatus=?,jobStage=?,jobQueue=?,timeRecorded=? \
                                        WHERE(messageSite=? AND remoteJobId=?)",(jobInfo['jobStatus'],jobInfo['jobStage'], \
                                                                                 jobInfo['jobQueue'],time.time(), \
                                                                                 jobInfo['messageSite'],jobInfo['remoteJobId']))
      self.dbConnection.commit()


   def purgeActiveJobs(self,
                       purgeJobStatus,
                       cutOffAge=0.):
      cutOffTime = time.time()-cutOffAge
      cursor = self.dbConnection.cursor()
      cursor.execute("DELETE FROM activeJobs WHERE(jobStatus=? AND timeRecorded<=?)",(purgeJobStatus,cutOffTime))
      self.dbConnection.commit()
      nPurgeActiveJobs = cursor.rowcount

      return(nPurgeActiveJobs)


   def updateWFJobs(self,
                    messageSite):
      jobsInfo = []
      cursor = self.dbConnection.cursor()
      cursor.execute("SELECT * FROM activeJobs WHERE(messageSite=? AND jobStatus='WF')",(messageSite,))
      rows = cursor.fetchall()
      for row in rows:
         jobInfo = JobInfo(row['messageSite'],
                           row['remoteJobId'],
                           localJobId=row['localJobId'],
                           instanceId=row['instanceId'],
                           jobStatus=row['jobStatus'],
                           jobStage=row['jobStage'],
                           jobQueue=row['jobQueue'])
         jobsInfo.append(jobInfo)

      commit = False
      for jobInfo in jobsInfo:
         cursor.execute("SELECT * FROM activeJobs WHERE(localJobId=? AND instanceId=0)",(jobInfo['localJobId'],))
         row = cursor.fetchone()
         if row:
            if row['jobStatus'] == 'D' or row['jobStatus'] == 'Dr':
               jobInfo['jobStatus'] = row['jobStatus']
         else:
            jobInfo['jobStatus'] = 'Dr'
         if jobInfo['jobStatus'] != 'WF':
            self.updateActiveJob(jobInfo)
            commit = True

      if commit:
         self.dbConnection.commit()

      del jobsInfo


   def getActiveJobCount(self):
      cursor = self.dbConnection.cursor()
      cursor.execute("SELECT COUNT(localJobId) FROM activeJobs")
      row = cursor.fetchone()
      nActiveJobCount = row[0]

      return(nActiveJobCount)


   def addUserActivity(self,
                       hubUserId,
                       activityScore):
      timeUpdated = time.time()
      cursor = self.dbConnection.cursor()
      cursor.execute("INSERT INTO userActivityScores VALUES(?,?,?)",(hubUserId,activityScore,timeUpdated))
      self.dbConnection.commit()


   def getUserActivity(self,
                       hubUserId):
      activityScore = 0.5
      timeUpdated   = -1.

      cursor = self.dbConnection.cursor()
      cursor.execute("SELECT * FROM userActivityScores WHERE(hubUserId=?)",(hubUserId,))
      row = cursor.fetchone()
      if row:
         activityScore = row['activityScore']
         timeUpdated   = row['timeUpdated']

      return(activityScore,timeUpdated)


   def getUsersActivity(self,
                        hubUserId):
      usersActivity = {}
      cursor = self.dbConnection.cursor()
      if hubUserId == '*':
         cursor.execute("SELECT hubUserId,activityScore FROM userActivityScores")
         rows = cursor.fetchall()
         for row in rows:
            usersActivity[row['hubUserId']] = row['activityScore']
      else:
         cursor.execute("SELECT activityScore FROM userActivityScores WHERE(hubUserId=?)",(hubUserId,))
         row = cursor.fetchone()
         if row:
            activityScore = row['activityScore']
         else:
            activityScore = 0.5
         usersActivity[hubUserId] = activityScore

      return(usersActivity)


   def updateUserActivity(self,
                          hubUserId,
                          activityScore,
                          timeUpdated):
      cursor = self.dbConnection.cursor()
      cursor.execute("UPDATE userActivityScores SET activityScore=?,timeUpdated=? \
                                              WHERE(hubUserId=?)",(activityScore,timeUpdated,hubUserId))
      self.dbConnection.commit()


   def updateUserActivityScores(self):
      now = time.time()
      currentActivityScores = {}

      cursor = self.dbConnection.cursor()
      cursor.execute("SELECT hubUserId,SUM(nCores) AS activity FROM activeJobs \
                                                               WHERE(instanceId != 0 AND jobStatus != 'D' AND jobStatus != 'Dr') \
                                                               GROUP BY hubUserId;")
      rows = cursor.fetchall()
      for row in rows:
         currentActivityScores[row['hubUserId']] = row['activity']

      cursor.execute("SELECT * FROM userActivityScores")
      rows = cursor.fetchall()
      for row in rows:
         hubUserId     = row['hubUserId']
         activityScore = row['activityScore']
         timeUpdated   = row['timeUpdated']

         if hubUserId in currentActivityScores:
            beta = math.pow(0.5,(now-timeUpdated)/86400.)
            activity  = beta*activityScore + (1.-beta)*currentActivityScores[hubUserId]
            self.updateUserActivity(hubUserId,activity,now)
         else:
            beta = math.pow(0.5,(now-timeUpdated)/86400.)
            activity  = beta*activityScore
            self.updateUserActivity(hubUserId,activity,now)

      for hubUserId in currentActivityScores:
         activityScore,timeUpdated = self.getUserActivity(hubUserId)
         if timeUpdated < 0.:
            self.addUserActivity(hubUserId,currentActivityScores[hubUserId])


   def processRequest(self,
                      channel):
      channelClosed = False
      newJobSite    = ""
      newJobId      = ""

      message = self.receiveMessage(channel,self.fixedBufferSize)
      if message != "":
         if   re.match('[QSRT]:',message):
            nInstances = 0
            try:
               messageType,siteId = message.split(':')
               nWords = len(siteId.split())
               if   nWords == 2:
                  messageSite,remoteJobId = siteId.split()
               elif nWords == 9:
                  messageSite,remoteJobId,hubUserId,localJobId,instanceId,destination,runName,nCores,distributorPid = siteId.split()
                  if localJobId.startswith('WF;'):
                     localJobId = int(localJobId.split(';')[1])
                     nInstances = int(instanceId)
                     instanceId = 0
                  else:
                     localJobId = int(localJobId)
                     instanceId = int(instanceId)
               else:
                  log("Failed %s message request: %s" % (messageType,siteId))
                  messageType = ''
            except:
               log("Failed QSRT message request: %s" % (message))
               messageType = ''
         elif re.match('[W]:',message):
            try:
               messageType,siteId = message.split(':')
               nWords = len(siteId.split())
               if nWords == 3:
                  messageSite,remoteJobId,nInstances = siteId.split()
                  nInstances = int(nInstances)
               else:
                  log("Failed %s message request: %s" % (messageType,siteId))
                  messageType = ''
            except:
               log("Failed W message request: %s" % (message))
               messageType = ''
         elif re.match('[AU]:',message):
            try:
               messageType,hubUserId = message.strip().split(':')
            except:
               log("Failed AU message request: %s" % (message))
               messageType = ''
         elif re.match('[P]:',message):
            try:
               messageType,siteId = message.split(':')
               nWords = len(siteId.split())
               if nWords == 2:
                  hubUserId,localJobId =  siteId.split()
               else:
                  log("Failed %s message request: %s" % (messageType,siteId))
                  messageType = ''
            except:
               log("Failed P message request: %s" % (message))
               messageType = ''
         else:
            log("Failed message request: " + message)
            messageType = ''

         if   messageType == 'Q':                        # job Query
            markReportedAsDone = False
            jobInfo = self.getActiveJob(messageSite,remoteJobId)
            if jobInfo:
               jobStatus = jobInfo['jobStatus']
               jobStage  = jobInfo['jobStage']
               if   jobStatus == 'D':
                  markReportedAsDone = True
               elif jobStatus == 'Dr':
                  jobStatus = 'D'
            else:
               jobStatus,jobStage,jobQueue = ('?','?','?')
            if self.sendMessage(channel,jobStatus + " " + jobStage,self.fixedBufferSize) > 0:
               newJobSite = messageSite
               if markReportedAsDone:
                  jobInfo['jobStatus'] = 'Dr'
                  self.updateActiveJob(jobInfo)
         elif messageType == 'W':                        # workflow Query
            markReportedAsDone = False
            jobInfo = self.getActiveJob(messageSite,remoteJobId)
            if jobInfo:
               jobStatus = jobInfo['jobStatus']
               jobStage  = jobInfo['jobStage']
               jobQueue  = jobInfo['jobQueue']
               if   jobStatus == 'D':
                  markReportedAsDone = True
               elif jobStatus == 'Dr':
                  jobStatus = 'D'
            else:
               jobStatus,jobStage,jobQueue = ('?','?','?')
            del jobInfo
            wfInstances = {}
            if '.' in remoteJobId:
               wfRemoteJobIdBase = remoteJobId.split('.')[0]
            else:
               wfRemoteJobIdBase = remoteJobId
            for instance in xrange(1,nInstances+1):
               wfMarkReportedAsDone = False
               wfRemoteJobId = wfRemoteJobIdBase + '.' + str(instance)
               jobInfo = self.getActiveJob(messageSite,wfRemoteJobId)
               if jobInfo:
                  wfJobStatus = jobInfo['jobStatus']
                  wfJobStage  = jobInfo['jobStage']
                  wfJobQueue  = jobInfo['jobQueue']
                  if   wfJobStatus == 'D':
                     wfMarkReportedAsDone = True
                  elif wfJobStatus == 'Dr':
                     wfJobStatus = 'D'
               else:
                  wfJobStatus,wfJobStage,wfJobQueue = ('?','?','?')
               wfInstances[instance] = {}
               wfInstances[instance]['remoteJobId']        = wfRemoteJobId
               wfInstances[instance]['jobStatus']          = wfJobStatus
               wfInstances[instance]['jobStage']           = wfJobStage
               wfInstances[instance]['jobQueue']           = wfJobQueue
               wfInstances[instance]['markReportedAsDone'] = wfMarkReportedAsDone
               del jobInfo
            statusMessageDelimiter = ':'
            statusMessage = jobStatus + " " + jobStage
            for instance in xrange(1,nInstances+1):
               if wfInstances[instance]['jobStatus'] != '?':
                  statusMessage += statusMessageDelimiter + str(instance) + " " + \
                                                            wfInstances[instance]['jobStatus'] + " " + \
                                                            wfInstances[instance]['jobStage']
            if self.sendMessage(channel,str(len(statusMessage)),self.fixedBufferSize) > 0:
               if self.sendMessage(channel,statusMessage) > 0:
                  newJobSite = messageSite
                  if markReportedAsDone:
                     jobStatus = 'Dr'
                     jobInfo = JobInfo(messageSite,
                                       remoteJobId,
                                       jobStatus=jobStatus,
                                       jobStage=jobStage,
                                       jobQueue=jobQueue)
                     self.updateActiveJob(jobInfo)
                     del jobInfo
                  for instance in xrange(1,nInstances+1):
                     if wfInstances[instance]['markReportedAsDone']:
                        wfRemoteJobId = wfInstances[instance]['remoteJobId']
                        wfJobStatus   = 'Dr'
                        wfJobStage    = wfInstances[instance]['jobStage']
                        wfJobQueue    = wfInstances[instance]['jobQueue']
                        jobInfo = JobInfo(messageSite,
                                          wfRemoteJobId,
                                          jobStatus=wfJobStatus,
                                          jobStage=wfJobStage,
                                          jobQueue=wfJobQueue)
                        self.updateActiveJob(jobInfo)
                        del jobInfo
            del wfInstances
         elif messageType == 'S':                        # new job Submission
            jobInfo = self.getActiveJob(messageSite,remoteJobId)
            if jobInfo:
               jobStatus = jobInfo['jobStatus']
               jobStage  = jobInfo['jobStage']
            else:
               if nInstances > 0:
                  jobStatus,jobStage,jobQueue = ('N','DAG','?')
                  nCoresMaster = 1
               else:
                  jobStatus,jobStage,jobQueue = ('N','Job','?')
                  nCoresMaster = nCores
               jobInfo = JobInfo(messageSite,
                                 remoteJobId,
                                 distributorPid=distributorPid,
                                 runName=runName,
                                 nCores=nCoresMaster,
                                 localJobId=localJobId,
                                 instanceId=instanceId,
                                 jobStatus=jobStatus,
                                 jobStage=jobStage,
                                 jobQueue=jobQueue,
                                 hubUserId=hubUserId,
                                 destination=destination)
               self.addActiveJob(jobInfo)
               del jobInfo
            self.dumpActiveJobs()
            if self.sendMessage(channel,jobStatus + " " + jobStage,self.fixedBufferSize) > 0:
               newJobSite = messageSite
               newJobId   = remoteJobId
               if nInstances > 0:
                  if '.' in remoteJobId:
                     wfRemoteJobIdBase = remoteJobId.split('.')[0]
                  else:
                     wfRemoteJobIdBase = remoteJobId
                  nInstanceIdDigits = max(2,int(math.log10(nInstances)+1))
                  wfJobStatus,wfJobStage,wfJobQueue = ('WF','Simulation','?')
                  for instance in xrange(1,nInstances+1):
                     wfRemoteJobId = wfRemoteJobIdBase + '.' + str(instance)
                     jobInfo = self.getActiveJob(messageSite,wfRemoteJobId)
                     if not jobInfo:
                        jobInfo = JobInfo(messageSite,
                                          wfRemoteJobId,
                                          distributorPid=distributorPid,
                                          runName=runName,
                                          nCores=nCores,
                                          localJobId=localJobId,
                                          instanceId=instance,
                                          jobStatus=wfJobStatus,
                                          jobStage=wfJobStage,
                                          jobQueue=wfJobQueue,
                                          hubUserId=hubUserId,
                                          destination=destination)
                        self.addActiveJob(jobInfo)
                        del jobInfo
               self.dumpActiveJobs()
         elif messageType == 'T':                        # Terminate job
            jobInfo = self.getActiveJob(messageSite,remoteJobId)
            if jobInfo:
               jobStatus = jobInfo['jobStatus']
               jobStage  = jobInfo['jobStage']
            else:
               jobStatus,jobStage,jobQueue = ('?','?','?')
            if self.sendMessage(channel,jobStatus + " " + jobStage,self.fixedBufferSize) > 0:
               if jobInfo:
                  jobInfo['jobStatus'] = 'D'
                  jobInfo['jobStage']  = 'Job'
                  self.updateActiveJob(jobInfo)
         elif messageType == 'R':                        # Report active jobs
            report = ""
            if messageSite == "*":
               messageSites = self.getSitesWithActiveJob()
            else:
               messageSites = [messageSite]

            siteDelimiter = ''
            maxLastReportTime = 0
            for messageSite in messageSites:
               lastReportTime = self.getActiveJobSiteTimeUpdated(messageSite)
               maxLastReportTime = max(maxLastReportTime,lastReportTime)
               siteReport = messageSite + " " + str(lastReportTime)

               queueDelimiter = ' @ '
               activeJobSiteQueues = self.getActiveJobSiteQueues(messageSite)
               for activeJobSiteQueue in activeJobSiteQueues:
                  queueReport = activeJobSiteQueue

                  jobDelimiter = ' : '
                  if remoteJobId == "*":
                     reportJobs = self.getActiveJobsInQueue(messageSite,activeJobSiteQueue)
                     for reportJob in reportJobs:
                        queueReport += jobDelimiter + reportJob['remoteJobId'] + " " + \
                                                      reportJob['jobStatus'] + " " + \
                                                      reportJob['jobStage']
                  else:
                     reportJob = self.getActiveJob(messageSite,remoteJobId)
                     if reportJob:
                        if reportJob['jobQueue'] == activeJobSiteQueue:
                           queueReport += jobDelimiter + reportJob['remoteJobId'] + " " + \
                                                         reportJob['jobStatus'] + " " + \
                                                         reportJob['jobStage']

                  siteReport += queueDelimiter + queueReport

               report += siteDelimiter + siteReport
               siteDelimiter = ' | '

            reportLength = len(report)
            if self.sendMessage(channel,str(reportLength) + " " + str(maxLastReportTime),self.fixedBufferSize) > 0:
               if reportLength > 0:
                  self.sendMessage(channel,report)
            del messageSites
         elif messageType == 'A':                        # List user activity
            report = ""
            usersActivity = self.getUsersActivity(hubUserId)
            userDelimiter = ''
            for user,activity in usersActivity.items():
               report += userDelimiter + str(user) + " " + str(activity)
               userDelimiter = ':'

            reportLength = len(report)
            if self.sendMessage(channel,str(reportLength) + " " + str(time.time()),self.fixedBufferSize) > 0:
               if reportLength > 0:
                  self.sendMessage(channel,report)
            del usersActivity
         elif messageType == 'U':                        # List user jobs
            queueReport  = ""
            jobDelimiter = ''
            reportJobs = self.getActiveJobsForUser(hubUserId)
            for reportJob in reportJobs:
               attributes = [reportJob['localJobId'],reportJob['instanceId'],reportJob['runName'],
                             reportJob['jobQueue'],reportJob['destination'],
                             reportJob['jobStatus'],reportJob['jobStage']]
               queueReport += jobDelimiter + ' '.join([str(attribute) for attribute in attributes])
               jobDelimiter = ' : '

            reportLength = len(queueReport)
            if self.sendMessage(channel,str(reportLength) + " " + str(time.time()),self.fixedBufferSize) > 0:
               if reportLength > 0:
                  self.sendMessage(channel,queueReport)
         elif messageType == 'P':                        # Get job PID
            activeJobPid = self.getActiveJobPidForUser(hubUserId,localJobId)
            if activeJobPid:
               pidReport = str(activeJobPid)
            else:
               pidReport = ""
            reportLength = len(pidReport)
            if self.sendMessage(channel,str(reportLength) + " " + str(time.time()),self.fixedBufferSize) > 0:
               if reportLength > 0:
                  self.sendMessage(channel,pidReport)
      else:
         try:
            channel.close()
            channelClosed = True
         except:
            log("close channel failed")

      return(channelClosed,newJobSite,newJobId)


