# @package      hubzero-submit-distributor
# @file         RemoteBatchTAPIS.py
# @copyright    Copyright (c) 2020-2020 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2020-2020 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#
import os.path
import re
import logging
import socket

from hubzero.submit.LogMessage        import getLogJobIdMessage as getLogMessage
from hubzero.submit.ParameterTemplate import ParameterTemplate
from hubzero.submit.TimeConversion    import minTohhmmss

class RemoteBatchTAPIS:
   def __init__(self,
                session,
                instanceToken,
                wsJobId,
                localJobId,
                instanceId,
                tapisSiteInfo,
                submissionScriptsInfo,
                nNodes,
                ppn,
                wallTime):
      self.logger                = logging.getLogger(__name__)
      self.session               = session
      self.instanceToken         = instanceToken
      self.wsJobId               = wsJobId
      self.localJobId            = localJobId
      self.instanceId            = instanceId
      self.tapisAppId            = tapisSiteInfo['appId']
      self.tapisQueue            = tapisSiteInfo['queue']
      self.tapisStageSystem      = tapisSiteInfo['stageSystem']
      self.tapisArchiveSystem    = tapisSiteInfo['archiveSystem']
      self.submissionScriptsInfo = submissionScriptsInfo
      self.nNodes                = nNodes
      self.ppn                   = ppn
      self.wallTime              = minTohhmmss(wallTime)


   def __buildArchiveFile(self):
      rawSubmissionScript = self.submissionScriptsInfo.getSubmissionScript('Batch','TAPIS','archive')

      substitutions = {}
      substitutions["SESSION"]            = self.session
      substitutions["INSTANCETOKEN"]      = self.instanceToken
      substitutions["WSJOBID"]            = self.wsJobId
      substitutions["JOBID"]              = self.localJobId
      substitutions["INSTANCEID"]         = self.instanceId
      substitutions["NNODES"]             = self.nNodes
      substitutions["PPN"]                = self.ppn
      substitutions["WALLTIME"]           = self.wallTime
      substitutions["TAPISAPPID"]         = self.tapisAppId
      substitutions["TAPISQUEUE"]         = self.tapisQueue
      substitutions["TAPISARCHIVESYSTEM"] = self.tapisArchiveSystem
      substitutions["TAPISSTAGESYSTEM"]   = self.tapisStageSystem
      substitutions["SUBMITHOSTFQDN"]     = socket.getfqdn()

      template = ParameterTemplate(rawSubmissionScript)
      try:
         submissionScript = template.substitute_recur(substitutions)
      except KeyError as e:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Pattern substitution failed for @@%s\n" % (e.args[0])))
      except TypeError:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Submission script substitution failed:\n%s\n" % (rawSubmissionScript)))

      return(submissionScript)


   def __buildNoArchiveFile(self):
      rawSubmissionScript = self.submissionScriptsInfo.getSubmissionScript('Batch','TAPIS','noarchive')

      substitutions = {}
      substitutions["SESSION"]          = self.session
      substitutions["INSTANCETOKEN"]    = self.instanceToken
      substitutions["WSJOBID"]          = self.wsJobId
      substitutions["JOBID"]            = self.localJobId
      substitutions["INSTANCEID"]       = self.instanceId
      substitutions["NNODES"]           = self.nNodes
      substitutions["PPN"]              = self.ppn
      substitutions["WALLTIME"]         = self.wallTime
      substitutions["TAPISAPPID"]       = self.tapisAppId
      substitutions["TAPISQUEUE"]       = self.tapisQueue
      substitutions["TAPISSTAGESYSTEM"] = self.tapisStageSystem
      substitutions["SUBMITHOSTFQDN"]   = socket.getfqdn()

      template = ParameterTemplate(rawSubmissionScript)
      try:
         submissionScript = template.substitute_recur(substitutions)
      except KeyError as e:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Pattern substitution failed for @@%s\n" % (e.args[0])))
      except TypeError:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Submission script substitution failed:\n%s\n" % (rawSubmissionScript)))

      return(submissionScript)


   def getBatchTapisFile(self):
      batchTapisFileName = "%s_%s.tapis" % (self.localJobId,self.instanceId)
      if self.tapisArchiveSystem:
         tapisJson = self.__buildArchiveFile()
      else:
         tapisJson = self.__buildNoArchiveFile()

      return(batchTapisFileName,tapisJson)


   def getRemoteJobIdNumber(self,
                            remoteJobId):
      remoteJobIdNumber = -1
      try:
         label,jobId = remoteJobId.split(':')
         if label == 'jobId':
            remoteJobIdNumber = jobId.strip()
      except:
         pass
 
      return(remoteJobIdNumber)


