# @package      hubzero-submit-distributor
# @file         ManagersInfo.py
# @copyright    Copyright (c) 2012-2020 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2012-2020 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#
import os.path
import re
import glob
import logging

from hubzero.submit.LogMessage import getLogJobIdMessage as getLogMessage

class ManagersInfo:
   def __init__(self,
                managersPath):
      self.logger = logging.getLogger(__name__)

      self.managers = {}

      if os.path.isdir(managersPath):
         for managersInfoPath in glob.iglob(os.path.join(managersPath,'*')):
            self.readManagersInfoFile(managersInfoPath)
      else:
         for managersInfoPath in glob.iglob(managersPath):
            self.readManagersInfoFile(managersInfoPath)

      markedForDeletion = []
      for managerName in self.managers:
         if self.managers[managerName]['state'] != 'enabled':
            markedForDeletion.append(managerName)
      for managerName in markedForDeletion:
         del self.managers[managerName]
      del markedForDeletion


   def readManagersInfoFile(self,
                            managersInfoPath):
      managerPattern  = re.compile('(\s*\[)([^\s]*)(]\s*)')
      keyValuePattern = re.compile('( *)(\w*)( *= *)(.*[^\s$])( *)')
      commentPattern  = re.compile('\s*#.*')
      managerName     = ""

      if os.path.exists(managersInfoPath):
         try:
            fpInfo = open(managersInfoPath,'r')
            try:
               eof = False
               while not eof:
                  record = fpInfo.readline()
                  if record != "":
                     record = commentPattern.sub("",record)
                     if   managerPattern.match(record):
                        managerName = managerPattern.match(record).group(2)
                        self.managers[managerName] = {'computationMode':'mpi',
                                                      'preManagerCommands':[],
                                                      'managerCommand':'',
                                                      'postManagerCommands':[],
                                                      'mpiRankVariable':'',
                                                      'environment':[],
                                                      'moduleInitialize':'',
                                                      'modulesLoad':[],
                                                      'modulesUnload':[],
                                                      'state':'enabled'
                                                     }
                     elif keyValuePattern.match(record):
                        key,value = keyValuePattern.match(record).group(2,4)
                        if key in self.managers[managerName]:
                           if   isinstance(self.managers[managerName][key],list):
                              self.managers[managerName][key] = [e.strip() for e in value.split(',')]
                           elif isinstance(self.managers[managerName][key],bool):
                              self.managers[managerName][key] = bool(value.lower() == 'true')
                           elif isinstance(self.managers[managerName][key],float):
                              self.managers[managerName][key] = float(value)
                           elif isinstance(self.managers[managerName][key],int):
                              self.managers[managerName][key] = int(value)
                           elif isinstance(self.managers[managerName][key],dict):
                              try:
                                 sampleKey   = self.managers[managerName][key].keys()[0]
                                 sampleValue = self.managers[managerName][key][sampleKey]
                              except:
                                 sampleKey   = "key"
                                 sampleValue = "value"
                              self.managers[managerName][key] = {}
                              for e in value.split(','):
                                 dictKey,dictValue = e.split(':')
                                 if isinstance(sampleKey,int):
                                    dictKey = int(dictKey)
                                 if   isinstance(sampleValue,int):
                                    dictValue = int(dictValue)
                                 elif isinstance(sampleValue,float):
                                    dictValue = float(dictValue)
                                 elif isinstance(sampleValue,bool):
                                    dictValue = bool(dictValue.lower() == 'true')
                                 self.managers[managerName][key][dictKey] = dictValue
                           else:
                              self.managers[managerName][key] = value
                        else:
                           message = "Undefined key = value pair %s = %s for manager %s" % (key,value,managerName)
                           self.logger.log(logging.WARNING,getLogMessage(message))
                  else:
                     eof = True
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("Managers configuration file %s could not be read" % \
                                                                                             (managersInfoPath)))
            finally:
               fpInfo.close()
         except (IOError,OSError):
            self.logger.log(logging.ERROR,getLogMessage("Managers configuration file %s could not be opened" % \
                                                                                            (managersInfoPath)))
      else:
         self.logger.log(logging.ERROR,getLogMessage("Managers configuration file %s is missing" % \
                                                                                (managersInfoPath)))


   def getManagerNames(self):
      managerNames = []
      for managerName in self.managers:
         managerNames.append(managerName)

      return(managerNames)


   def managerExists(self,
                     managerName):
      return(managerName in self.managers)


   def getDefaultManagerInfo(self,
                             computationMode='serial'):
      managerInfo = {}
      managerInfo['managerName']         = 'default'
      managerInfo['computationMode']     = computationMode
      managerInfo['preManagerCommands']  = []
      managerInfo['managerCommand']      = ''
      managerInfo['postManagerCommands'] = []
      managerInfo['mpiRankVariable']     = ''
      managerInfo['environment']         = []
      managerInfo['moduleInitialize']    = ''
      managerInfo['modulesLoad']         = []
      managerInfo['modulesUnload']       = []

      return(managerInfo)


   def getManagerInfo(self,
                      managerName):
      managerInfo = self.getDefaultManagerInfo('mpi')
      if managerName in self.managers:
         managerInfo['managerName']            = managerName
         managerInfo['computationMode']        = self.managers[managerName]['computationMode']
         if 'preManagerCommands' in self.managers[managerName]:
            managerInfo['preManagerCommands']  = self.managers[managerName]['preManagerCommands']
         if 'managerCommand' in self.managers[managerName]:
            managerInfo['managerCommand']      = self.managers[managerName]['managerCommand']
         if 'postManagerCommands' in self.managers[managerName]:
            managerInfo['postManagerCommands'] = self.managers[managerName]['postManagerCommands']
         if 'mpiRankVariable' in self.managers[managerName]:
            managerInfo['mpiRankVariable']     = self.managers[managerName]['mpiRankVariable']
         if 'environment' in self.managers[managerName]:
            managerInfo['environment']         = self.managers[managerName]['environment']
         if 'moduleInitialize' in self.managers[managerName]:
            managerInfo['moduleInitialize']    = self.managers[managerName]['moduleInitialize']
         if 'modulesLoad' in self.managers[managerName]:
            managerInfo['modulesLoad']         = self.managers[managerName]['modulesLoad']
         if 'modulesUnload' in self.managers[managerName]:
            managerInfo['modulesUnload']       = self.managers[managerName]['modulesUnload']

      return(managerInfo)


