# @package      hubzero-submit-distributor
# @file         ToolFilesInfo.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2012-2015 HUBzero Foundation, LLC.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2017-2017 HUBzero Foundation, LLC.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#
import os.path
import re
import glob
import logging

from hubzero.submit.LogMessage import getLogJobIdMessage as getLogMessage

class ToolFilesInfo:
   def __init__(self,
                toolFilesPath):
      self.logger = logging.getLogger(__name__)

      self.toolFiles = {}

      if os.path.isdir(toolFilesPath):
         for toolFilesInfoPath in glob.iglob(os.path.join(toolFilesPath,'*')):
            self.readToolFilesInfoFile(toolFilesInfoPath)
      else:
         for toolFilesInfoPath in glob.iglob(toolFilesPath):
            self.readToolFilesInfoFile(toolFilesInfoPath)

      markedForDeletion = []
      for toolFilesName in self.toolFiles:
         if self.toolFiles[toolFilesName]['state'] != 'enabled':
            markedForDeletion.append(toolFilesName)
      for toolFilesName in markedForDeletion:
         del self.toolFiles[toolFilesName]
      del markedForDeletion


   def readToolFilesInfoFile(self,
                             toolFilesInfoPath):
      toolFilePattern = re.compile('(\s*\[)([^\s]*)(]\s*)')
      keyValuePattern = re.compile('( *)(\w*)( *= *)(.*[^\s$])( *)')
      commentPattern  = re.compile('\s*#.*')
      toolFilesName   = ""

      if os.path.exists(toolFilesInfoPath):
         try:
            fpInfo = open(toolFilesInfoPath,'r')
            try:
               eof = False
               while not eof:
                  record = fpInfo.readline()
                  if record != "":
                     record = commentPattern.sub("",record)
                     if   toolFilePattern.match(record):
                        toolFilesName = toolFilePattern.match(record).group(2)
                        self.toolFiles[toolFilesName] = {'dockerImage':'',
                                                         'vboxFile':'',
                                                         'boincAppFile':'',
                                                         'fileInfoAttributes':[],
                                                         'fileRefAttributes':[],
                                                         'appsFiles':[],
                                                         'fpopsEstimate':3600000000000.0,
                                                         'fpopsBound':86400000000000.0,
                                                         'memoryRequirement':500000000,
                                                         'diskRequirement':1073741824,
                                                         'delayBound':28800,
                                                         'minimumQuorum':1,
                                                         'targetResults':1,
                                                         'maximumErrorResults':5,
                                                         'maximumTotalResults':5,
                                                         'maximumSuccessResults':3,
                                                         'delayBound':28800,
                                                         'state':'enabled'
                                                        }
#                                                        'sizeClass':,
                     elif keyValuePattern.match(record):
                        key,value = keyValuePattern.match(record).group(2,4)
                        if key in self.toolFiles[toolFilesName]:
                           if   isinstance(self.toolFiles[toolFilesName][key],list):
                              self.toolFiles[toolFilesName][key] = [e.strip() for e in value.split(',')]
                           elif isinstance(self.toolFiles[toolFilesName][key],bool):
                              self.toolFiles[toolFilesName][key] = bool(value.lower() == 'true')
                           elif isinstance(self.toolFiles[toolFilesName][key],float):
                              self.toolFiles[toolFilesName][key] = float(value)
                           elif isinstance(self.toolFiles[toolFilesName][key],int):
                              self.toolFiles[toolFilesName][key] = int(value)
                           elif isinstance(self.toolFiles[toolFilesName][key],dict):
                              try:
                                 sampleKey   = self.toolFiles[toolFilesName][key].keys()[0]
                                 sampleValue = self.toolFiles[toolFilesName][key][sampleKey]
                              except:
                                 sampleKey   = "key"
                                 sampleValue = "value"
                              self.toolFiles[toolFilesName][key] = {}
                              for e in value.split(','):
                                 dictKey,dictValue = e.split(':')
                                 if isinstance(sampleKey,int):
                                    dictKey = int(dictKey)
                                 if   isinstance(sampleValue,int):
                                    dictValue = int(dictValue)
                                 elif isinstance(sampleValue,float):
                                    dictValue = float(dictValue)
                                 elif isinstance(sampleValue,bool):
                                    dictValue = bool(dictValue.lower() == 'true')
                                 self.toolFiles[toolFilesName][key][dictKey] = dictValue
                           else:
                              self.toolFiles[toolFilesName][key] = value
                        else:
                           message = "Undefined key = value pair %s = %s for toolFile %s" % (key,value,toolFilesName)
                           self.logger.log(logging.WARNING,getLogMessage(message))
                  else:
                     eof = True
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("ToolFiles configuration file %s could not be read" % \
                                                                                             (toolFilesInfoPath)))
            finally:
               fpInfo.close()
         except (IOError,OSError):
            self.logger.log(logging.ERROR,getLogMessage("ToolFiles configuration file %s could not be opened" % \
                                                                                            (toolFilesInfoPath)))
      else:
         self.logger.log(logging.ERROR,getLogMessage("ToolFiles configuration file %s is missing" % \
                                                                                (toolFilesInfoPath)))


   def getToolFileNames(self):
      toolFilesNames = []
      for toolFilesName in self.toolFiles:
         toolFilesNames.append(toolFilesName)

      return(toolFilesNames)


   def toolFileExists(self,
                      toolFilesName):
      return(toolFilesName in self.toolFiles)


   def getDefaultToolFilesInfo(self):
      toolFilesInfo = {}
      toolFilesInfo['toolFilesName']      = 'default'
      toolFilesInfo['dockerImage']        = ''
      toolFilesInfo['vboxFile']           = ''
      toolFilesInfo['boincAppFile']       = ''
      toolFilesInfo['fileInfoAttributes'] = []
      toolFilesInfo['fileRefAttributes']  = []
      toolFilesInfo['appsFiles']          = []

      toolFilesInfo['fpopsEstimate']         = 3600000000000.0
      toolFilesInfo['fpopsBound']            = 86400000000000.0
      toolFilesInfo['memoryRequirement']     = 500000000
      toolFilesInfo['diskRequirement']       = 1073741824
      toolFilesInfo['delayBound']            = 28800
      toolFilesInfo['minimumQuorum']         = 1
      toolFilesInfo['targetResults']         = 1
      toolFilesInfo['maximumErrorResults']   = 3
      toolFilesInfo['maximumTotalResults']   = 3
      toolFilesInfo['maximumSuccessResults'] = 3
#     toolFilesInfo['sizeClass']             =

      return(toolFilesInfo)


   def getToolFilesInfo(self,
                        toolFilesName):
      toolFilesInfo = self.getDefaultToolFilesInfo()
      if toolFilesName in self.toolFiles:
         toolFilesInfo['toolFilesName']         = toolFilesName
         if 'dockerImage' in self.toolFiles[toolFilesName]:
            toolFilesInfo['dockerImage']        = self.toolFiles[toolFilesName]['dockerImage']
         if 'vboxFile' in self.toolFiles[toolFilesName]:
            toolFilesInfo['vboxFile']           = self.toolFiles[toolFilesName]['vboxFile']
         if 'boincAppFile' in self.toolFiles[toolFilesName]:
            toolFilesInfo['boincAppFile']       = self.toolFiles[toolFilesName]['boincAppFile']
         if 'fileInfoAttributes' in self.toolFiles[toolFilesName]:
            toolFilesInfo['fileInfoAttributes'] = self.toolFiles[toolFilesName]['fileInfoAttributes']
         if 'fileRefAttributes' in self.toolFiles[toolFilesName]:
            toolFilesInfo['fileRefAttributes']  = self.toolFiles[toolFilesName]['fileRefAttributes']
         if 'appsFiles' in self.toolFiles[toolFilesName]:
            toolFilesInfo['appsFiles']          = self.toolFiles[toolFilesName]['appsFiles']

         if 'fpopsEstimate' in self.toolFiles[toolFilesName]:
            toolFilesInfo['fpopsEstimate']         = self.toolFiles[toolFilesName]['fpopsEstimate']
         if 'fpopsBound' in self.toolFiles[toolFilesName]:
            toolFilesInfo['fpopsBound']            = self.toolFiles[toolFilesName]['fpopsBound']
         if 'memoryRequirement' in self.toolFiles[toolFilesName]:
            toolFilesInfo['memoryRequirement']     = self.toolFiles[toolFilesName]['memoryRequirement']
         if 'diskRequirement' in self.toolFiles[toolFilesName]:
            toolFilesInfo['diskRequirement']       = self.toolFiles[toolFilesName]['diskRequirement']
         if 'delayBound' in self.toolFiles[toolFilesName]:
            toolFilesInfo['delayBound']            = self.toolFiles[toolFilesName]['delayBound']
         if 'minimumQuorum' in self.toolFiles[toolFilesName]:
            toolFilesInfo['minimumQuorum']         = self.toolFiles[toolFilesName]['minimumQuorum']
         if 'targetResults' in self.toolFiles[toolFilesName]:
            toolFilesInfo['targetResults']         = self.toolFiles[toolFilesName]['targetResults']
         if 'maximumErrorResults' in self.toolFiles[toolFilesName]:
            toolFilesInfo['maximumErrorResults']   = self.toolFiles[toolFilesName]['maximumErrorResults']
         if 'maximumTotalResults' in self.toolFiles[toolFilesName]:
            toolFilesInfo['maximumTotalResults']   = self.toolFiles[toolFilesName]['maximumTotalResults']
         if 'maximumSuccessResults' in self.toolFiles[toolFilesName]:
            toolFilesInfo['maximumSuccessResults'] = self.toolFiles[toolFilesName]['maximumSuccessResults']
#        if 'sizeClass' in self.toolFiles[toolFilesName]:
#           toolFilesInfo['sizeClass']             = self.toolFiles[toolFilesName]['sizeClass']

      return(toolFilesInfo)


