#!/usr/bin/env python
#
# @package      hubzero-submit-monitors
# @file         Scripts/boinc/submitBatchJobs.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2017-2018 HUBzero Foundation, LLC.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2017-2018 HUBzero Foundation, LLC.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#
# ----------------------------------------------------------------------
#  submitBatchJobs.py
#
#  script which submits multiple jobs as a single batch
#
import sys
import os

from submit_api import *

BOINCPROJECTURL    = ""
BOINCAUTHENTICATOR = ""


class SubmitBoincBatch:
   def __init__(self,
                boincProjectURL,
                boincAuthenticator,
                runName,
                localJobId,
                batchId):
      self.boincProjectURL    = boincProjectURL
      self.boincAuthenticator = boincAuthenticator
      self.runName            = runName
      self.localJobId         = localJobId
      self.batchId            = int(batchId)


   def run(self):
      exitStatus = 0
      batch = BATCH_DESC()
      batch.project       = self.boincProjectURL
      batch.authenticator = self.boincAuthenticator
      batch.app_name      = "boinc2docker"
      batch.batch_name    = '_'.join([self.localJobId,'00'])
      batch.batch_id      = self.batchId
      batch.jobs          = []

      jobDirectory = os.getcwd()
      if os.path.basename(jobDirectory) != self.localJobId and \
         os.path.basename(jobDirectory) != self.runName:
         jobDirectory = os.path.dirname(jobDirectory)
      if os.path.basename(jobDirectory) != self.localJobId and \
         os.path.basename(jobDirectory) != self.runName:
         exitStatus = 1
         sys.stderr.write("submitBatchJobs.py must be run in the job directory: %s\n" % (os.getcwd()))
      else:
         instanceIds = []
         dirFiles = os.listdir(jobDirectory)
         for dirFile in dirFiles:
            dirPath = os.path.join(jobDirectory,dirFile)
            if os.path.isdir(dirPath):
               try:
                  instanceNumber = int(dirFile)
               except:
                  pass
               else:
                  instanceIds.append(dirFile)
         instanceIds.sort()

         jobError = False
         job = JOB_DESC()

         for instanceId in instanceIds:
            job.name = '_'.join([self.localJobId,instanceId])
            job.command_line = ' '.join([self.localJobId,instanceId])

            inputTemplateFile = os.path.join(jobDirectory,instanceId,'_'.join([self.localJobId,instanceId]) + '.inputTemplate')
            try:
               fpInputTemplate = open(inputTemplateFile,'r')
               try:
                  inputTemplateText = fpInputTemplate.readlines()
               except (IOError,OSError):
                  print "%s could not be read" % (inputTemplateFile)
                  jobError = True
               else:
                  job.input_template = "".join(inputTemplateText)
               finally:
                  fpInputTemplate.close()
            except (IOError,OSError):
               print "%s could not be opened" % (inputTemplateFile)
               jobError = True


            outputTemplateFile = os.path.join(jobDirectory,instanceId,'_'.join([self.localJobId,instanceId]) + '.outputTemplate')
            try:
               fpOutputTemplate = open(outputTemplateFile,'r')
               try:
                  outputTemplateText = fpOutputTemplate.readlines()
               except (IOError,OSError):
                  print "%s could not be read" % (outputTemplateFile)
                  jobError = True
               else:
                  job.output_template = "".join(outputTemplateText)
               finally:
                  fpOutputTemplate.close()
            except (IOError,OSError):
               print "%s could not be opened" % (outputTemplateFile)
               jobError = True

#           job.rsc_fpops_est = 1e9
            batch.jobs.append(copy.copy(job))

         if jobError:
            exitStatus = 1
         else:
            submitBatchResponse = submit_batch(batch)

            if submitBatchResponse.find('error') is not None:
               exitStatus = 1
               errMessage = submitBatchResponse.find('error').find('error_msg').text
               sys.stderr.write(errMessage + '\n')

      return(exitStatus)


if __name__ == '__main__':

   if len(sys.argv) == 4:
      runName    = sys.argv[1]
      localJobId = sys.argv[2]
      batchId    = sys.argv[3]
   else:
      sys.exit(1)

   __submitBoincBatch__ = SubmitBoincBatch(BOINCPROJECTURL,BOINCAUTHENTICATOR,
                                           runName,localJobId,batchId)

   __exitStatus__ = __submitBoincBatch__.run()

   sys.exit(__exitStatus__)


