# @package      hubzero-submit-distributor
# @file         RemoteBatchPEGASUS.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2012-2015 HUBzero Foundation, LLC.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2012-2015 HUBzero Foundation, LLC.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#
import os
import re
import logging

try:
   from Pegasus.DAX3 import *
except:
   from hubzero.submit.DAX3 import *

from hubzero.submit.LogMessage        import getLogJobIdMessage as getLogMessage 
from hubzero.submit.ParameterTemplate import ParameterTemplate
from hubzero.submit.JobOutput         import JobOutput

class RemoteBatchPEGASUS:
   SUBMISSIONSCRIPTCOMMANDPREFIX = ''

   def __init__(self,
                hubUserName,
                hubUserId,
                submitterClass,
                runName,
                localJobId,
                instanceId,
                instanceDirectory,
                scratchDirectory,
                useSetup,
                pegasusVersion,
                pegasusHome,
                appScriptName,
                executable,
                arguments,
                isMultiCoreRequest,
                siteInfo,
                submissionScriptsInfo,
                managerInfo,
                x509SubmitProxy,
                sshIdentityPath,
                gridsite,
                pegasusSiteInfo,
                wallTime,
                timeHistoryLogs):
      self.logger            = logging.getLogger(__name__)
      self.jobOutput         = JobOutput()
      self.hubUserName       = hubUserName
      self.hubUserId         = hubUserId
      self.submitterClass    = submitterClass
      self.runName           = runName
      self.localJobId        = localJobId
      self.instanceId        = instanceId
      self.instanceDirectory = instanceDirectory
      self.scratchDirectory  = scratchDirectory
      self.useSetup          = useSetup
      self.pegasusVersion    = pegasusVersion
      self.pegasusHome       = pegasusHome
      self.appScriptName     = appScriptName
      self.wallTime          = wallTime
      self.timestampStart    = timeHistoryLogs['timestampStart']
      self.timestampFinish   = timeHistoryLogs['timestampFinish']
      self.timeResults       = timeHistoryLogs['timeResults']
      self.checkDaxPath      = ""

      basename = os.path.basename(executable)
      if basename.startswith('pegasus-'):
         self.daxPath             = ""
         self.executable          = executable
         userArguments = arguments.split()
         scriptArguments = []
         for userArgument in userArguments:
            if os.path.isfile(userArgument):
               absolutePath = os.path.abspath(userArgument)
               scriptArguments.append(absolutePath)
            else:
               scriptArguments.append(userArgument)
         self.arguments           = ' '.join(scriptArguments)
         self.checkDaxExecutables = True
      else:
         self.daxPath             = os.path.join(self.instanceDirectory,"%s_%s.dax" % (self.localJobId,self.instanceId))
         self.executable          = 'pegasus-plan'
         self.arguments           = "--dax %s" % (self.daxPath)
         self.checkDaxExecutables = False

      self.isMultiCoreRequest       = isMultiCoreRequest
      self.computationMode          = managerInfo['computationMode']
      self.pegasusTemplates         = siteInfo['pegasusTemplates']
      self.submissionScriptCommands = siteInfo['submissionScriptCommands']
      self.submissionScriptsInfo    = submissionScriptsInfo
      self.x509SubmitProxy          = x509SubmitProxy
      self.sshIdentityPath          = sshIdentityPath
      self.gridsite                 = gridsite
      self.pegasusSiteInfo          = pegasusSiteInfo

      self.nodeFileName = ""
      self.nodeList     = []


   def __buildSerialFile(self):
      rawSubmissionScript = self.submissionScriptsInfo.getSubmissionScript('Batch','PEGASUS','serial')

      commandsPEGASUS = ""
      if self.submissionScriptCommands:
         if self.SUBMISSIONSCRIPTCOMMANDPREFIX:
            commandSeparator = "\n%s " % self.SUBMISSIONSCRIPTCOMMANDPREFIX
            commandsPEGASUS = self.SUBMISSIONSCRIPTCOMMANDPREFIX + " " + commandSeparator.join(self.submissionScriptCommands)
         else:
            commandSeparator = "\n"
            commandsPEGASUS = commandSeparator.join(self.submissionScriptCommands)

      notRecognizedArguments = []
      discardedArguments     = []
      auxiliaryArguments     = []
      if os.path.exists(self.pegasusTemplates['rc']):
         try:
            fpRCTemplate = open(self.pegasusTemplates['rc'],'r')
            try:
               searchString = os.path.basename(self.executable) + '.arguments'
               settings = fpRCTemplate.readlines()
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("%s could not be read" % (self.pegasusTemplates['rc'])))
            else:
               for setting in settings:
#pegasus-plan.arguments = --nocleanup
                  if setting.count(searchString) > 0:
                     try:
                        parameter,value = setting.split('=')
                        auxiliaryArguments.append(value.strip())
                     except:
                        pass
            finally:
               fpRCTemplate.close()
         except (IOError,OSError):
            self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (self.pegasusTemplates['rc'])))

      userArguments = self.arguments.split()
      location = 0
      while location < len(userArguments):
         if userArguments[location].startswith('-'):
            recognizeArgument = True
            saveDaxPath       = False
            if   userArguments[location].startswith('-D'):
               keepArgument = False
               discardedArguments.append(userArguments[location])
            elif userArguments[location] == '-d' or userArguments[location] == '--dax':
               keepArgument = True
               if self.checkDaxExecutables:
                  saveDaxPath = True
            elif userArguments[location] == '-b' or userArguments[location] == '--basename':
               keepArgument = True
            elif userArguments[location] == '-c' or userArguments[location] == '--cache':
               keepArgument = False
               discardedArguments.append(userArguments[location])
            elif userArguments[location] == '-C' or userArguments[location] == '--cluster':
               keepArgument = True
            elif userArguments[location] == '--conf':
               keepArgument = False
               discardedArguments.append(userArguments[location])
            elif userArguments[location] == '--dir':
               keepArgument = False
               discardedArguments.append(userArguments[location])
            elif userArguments[location] == '-f' or userArguments[location] == '--force':
               keepArgument = True
            elif userArguments[location] == '--force-replan':
               keepArgument = True
            elif userArguments[location] == '-g' or userArguments[location] == '--group':
               keepArgument = True
            elif userArguments[location] == '-h' or userArguments[location] == '--help':
               keepArgument = False
               discardedArguments.append(userArguments[location])
            elif userArguments[location] == '--inherited-rc-files':
               keepArgument = True
            elif userArguments[location] == '-j' or userArguments[location] == '--j':
               keepArgument = True
            elif userArguments[location] == '-n' or userArguments[location] == '--nocleanup':
               keepArgument = False
            elif userArguments[location] == '--cleanup':
               keepArgument = False
               discardedArguments.append(userArguments[location])
            elif userArguments[location] == '-o' or \
                 userArguments[location] == '--output' or userArguments[location] == '--output-site':
               keepArgument = False
               discardedArguments.append(userArguments[location])
            elif userArguments[location] == '-q' or userArguments[location] == '--quiet':
               keepArgument = True
            elif userArguments[location] == '--relative-submit-dir':
               keepArgument = False
               discardedArguments.append(userArguments[location])
            elif userArguments[location] == '-s' or userArguments[location] == '--sites':
               keepArgument = False
               discardedArguments.append(userArguments[location])
            elif userArguments[location] == '-S' or userArguments[location] == '--submit':
               keepArgument = False
               discardedArguments.append(userArguments[location])
            elif userArguments[location].startswith('-v') or userArguments[location] == '--verbose':
               keepArgument = True
            elif userArguments[location] == '-V' or userArguments[location] == '--version':
               keepArgument = False
               discardedArguments.append(userArguments[location])
            else:
               keepArgument = False
               recognizeArgument = False
               notRecognizedArguments.append(userArguments[location])

            if keepArgument:
               auxiliaryArguments.append(userArguments[location])
            location += 1
            while location < len(userArguments) and not userArguments[location].startswith('-'):
               if   keepArgument:
                  auxiliaryArguments.append(userArguments[location])
                  if saveDaxPath:
                     self.checkDaxPath = userArguments[location]
               elif recognizeArgument:
                  discardedArguments.append(userArguments[location])
               else:
                  notRecognizedArguments.append(userArguments[location])
               location += 1
         else:
            location += 1

      substitutions = {}
      substitutions["USESETUP"]                 = self.useSetup
      substitutions["PEGASUSVERSION"]           = self.pegasusVersion
      substitutions["RUNNAME"]                  = self.runName
      substitutions["JOBID"]                    = self.localJobId
      substitutions["INSTANCEID"]               = self.instanceId
      substitutions["INSTANCEDIRECTORY"]        = self.instanceDirectory
      substitutions["SCRATCHDIRECTORY"]         = self.scratchDirectory
      substitutions["EXECUTABLE"]               = self.executable
      substitutions["GRIDSITE"]                 = self.gridsite
      arguments = ' '.join(auxiliaryArguments)
      substitutions["ARGUMENTS"]                = arguments.strip()
      arguments = ' '.join(discardedArguments)
      substitutions["DISCARDED"]                = arguments.strip()
      arguments = ' '.join(notRecognizedArguments)
      substitutions["NOTRECOGNIZED"]            = arguments.strip()
      substitutions["TS_START"]                 = self.timestampStart
      substitutions["TS_FINISH"]                = self.timestampFinish
      substitutions["TIME_RESULTS"]             = self.timeResults
      substitutions["HUBUSERNAME"]              = self.hubUserName
      substitutions["HUBUSERID"]                = str(self.hubUserId)
      substitutions["SUBMISSIONSCRIPTCOMMANDS"] = commandsPEGASUS

      template = ParameterTemplate(rawSubmissionScript)
      try:
         submissionScript = template.substitute_recur(substitutions)
      except KeyError,err:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Pattern substitution failed for @@%s\n" % (err[0])))
      except TypeError,err:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Submission script substitution failed:\n%s\n" % (rawSubmissionScript)))
      else:
         if self.SUBMISSIONSCRIPTCOMMANDPREFIX:
            submissionScript = re.sub("(\n)*\n%s" % (self.SUBMISSIONSCRIPTCOMMANDPREFIX),
                                           "\n%s" % (self.SUBMISSIONSCRIPTCOMMANDPREFIX),submissionScript)

      return(submissionScript)


   def __buildCatalogs(self):
      substitutions = {}
      substitutions["SUBMITTERCLASS"]       = str(self.submitterClass)
      substitutions["RUNNAME"]              = self.runName
      substitutions["JOBID"]                = self.localJobId
      substitutions["INSTANCEID"]           = self.instanceId
      substitutions["INSTANCEDIRECTORY"]    = self.instanceDirectory
      substitutions["BASESCRATCHDIRECTORY"] = os.path.basename(self.scratchDirectory)
      substitutions["SCRATCHDIRECTORY"]     = self.scratchDirectory
      if self.x509SubmitProxy != "":
         substitutions["X509SUBMITPROXY"]   = self.x509SubmitProxy
      else:
         substitutions["X509SUBMITPROXY"]   = os.path.join(os.sep,'tmp','hub-proxy.%s' % (self.hubUserName))
      substitutions["PEGASUSHOME"]          = self.pegasusHome
      substitutions["SSHPRIVATEKEYPATH"]    = self.sshIdentityPath

      for templateType in self.pegasusTemplates:
         pegasusTemplate = self.pegasusTemplates[templateType]
         if pegasusTemplate != "":
            try:
               fpTemplate = open(pegasusTemplate,'r')
               try:
                  template = ''.join(fpTemplate.readlines())
               except (IOError,OSError):
                  self.logger.log(logging.ERROR,getLogMessage("%s could not be read" % (pegasusTemplate)))
               else:
                  pegasusTemplate = ParameterTemplate(template)
                  try:
                     pegasusText = pegasusTemplate.substitute_recur(substitutions)
                  except KeyError,err:
                     pegasusText = ""
                     self.logger.log(logging.ERROR,getLogMessage("Pattern substitution failed for @@%s\n" % (err[0])))
                  except TypeError,err:
                     pegasusText = ""
                     self.logger.log(logging.ERROR,getLogMessage("Pegasus template substitution failed:\n%s\n" % (template)))

                  if   templateType == 'rc':
                     pegasusFile = "%s_%s.pegasusrc" % (self.localJobId,self.instanceId)
                  elif templateType == 'sites':
                     pegasusFile = "%s_%s_sites.xml" % (self.localJobId,self.instanceId)
                  elif templateType == 'tc':
                     pegasusFile = "%s_%s_tc.txt" % (self.localJobId,self.instanceId)
                  pegasusPath = os.path.join(self.instanceDirectory,pegasusFile)
                  try:
                     fpPegasusFile = open(pegasusPath,'w')
                     try:
                        fpPegasusFile.write(pegasusText)
                     except (IOError,OSError):
                        self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (pegasusPath)))
                     finally:
                        fpPegasusFile.close()
                  except (IOError,OSError):
                     self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (pegasusPath)))
               finally:
                  fpTemplate.close()
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (pegasusTemplate)))


   def __buildDAX(self):
      if self.daxPath != "":
         arch      = self.pegasusSiteInfo['arch']
         osFlavor  = self.pegasusSiteInfo['osFlavor'].lower()
         osVersion = self.pegasusSiteInfo['osVersion']
         osRelease = self.pegasusSiteInfo['osRelease']

         dax = ADAG("%s_%s" % (self.localJobId,self.instanceId))

         appScriptPath = "file://" + os.path.join(self.instanceDirectory,self.appScriptName)
         appScript = Executable(name=self.appScriptName,arch="%s" % (arch),
                                                        os="%s" % (osFlavor),
                                                        osrelease="%s" % (osRelease),
                                                        osversion="%s" % (osVersion),installed=False)
         appScript.addPFN(PFN(appScriptPath,"local"))
         dax.addExecutable(appScript)

         stageInTarFile = "%s_%s_input.tar.gz" % (self.localJobId,self.instanceId)
         inputTarPath = "file://" + os.path.join(self.instanceDirectory,stageInTarFile)
         inputTar = File(stageInTarFile)
         inputTar.addPFN(PFN(inputTarPath,"local"))
         dax.addFile(inputTar)

         stageOutTarFile = stageInTarFile.replace('input','output')
         outputTar = File(stageOutTarFile)

         jobId = "%s_0" % (self.runName)
         job = Job(name=self.appScriptName,id=jobId)
         job.addProfile(Profile(Namespace.HINTS, key='executionPool',value="%s" % (self.pegasusSiteInfo['name'])))
         job.addProfile(Profile(Namespace.GLOBUS,key='maxwalltime',value=self.wallTime))
         job.uses(inputTar,link=Link.INPUT,executable=False)
         job.uses(outputTar,link=Link.OUTPUT,register=True,transfer=True)
         dax.addJob(job)

         try:
            fpDAX = open(self.daxPath,'w')
            try:
               dax.writeXML(fpDAX)
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("%s could not be written" % (self.daxPath)))
            finally:
               fpDAX.close()
         except (IOError,OSError):
            self.logger.log(logging.ERROR,getLogMessage("%s could not be opened" % (self.daxPath)))


   def buildBatchScript(self):
      batchLogName = ""
      batchScriptName = "%s_%s.pegasus" % (self.localJobId,self.instanceId)
      if self.isMultiCoreRequest:
         batchScript = ""
      else:
         batchScript = self.__buildSerialFile()
      batchScriptExecutable = True

      self.__buildCatalogs()
      self.__buildDAX()

      return(batchLogName,batchScriptName,batchScript,batchScriptExecutable)


   def getBatchNodeList(self):
      return(self.nodeFileName,self.nodeList)


   def getUserDaxExecutables(self):
      daxExecutables = []
      if self.checkDaxPath:
         daxExecutables = self.jobOutput.getDaxExecutables(self.checkDaxPath)

      return(daxExecutables)


   def getRemoteJobIdNumber(self,
                            remoteJobId):
      try:
# 1 job(s) submitted to cluster 105.
         remoteJobIdNumber = re.search('cluster [0-9]+',remoteJobId).group().split()[1] + ".0"
      except:
         remoteJobIdNumber = "-1"

      return(remoteJobIdNumber)


