# @package      hubzero-submit-distributor
# @file         IdentitiesInfo.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2012-2015 HUBzero Foundation, LLC.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2012-2015 HUBzero Foundation, LLC.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#
import os
import re
import logging

from hubzero.submit.LogMessage import getLogJobIdMessage as getLogMessage

class IdentitiesInfo:
   def __init__(self,
                identitiesPath):
      self.logger = logging.getLogger(__name__)
      self.identities  = {}

      identityPattern = re.compile('(\s*\[)([^\s]*)(]\s*)')
      keyValuePattern = re.compile('( *)(\w*)( *= *)(.*[^\s$])( *)')
      commentPattern  = re.compile('\s*#.*')
      identityName    = ""

      if os.path.exists(identitiesPath):
         try:
            fpInfo = open(identitiesPath,'r')
            try:
               eof = False
               while not eof:
                  record = fpInfo.readline()
                  if record != "":
                     record = commentPattern.sub("",record)
                     if   identityPattern.match(record):
                        identityName = identityPattern.match(record).group(2)
                        self.identities[identityName] = {'identityType':"",
                                                         'certificateDirectory':"",
                                                         'certFile':"",
                                                         'keyFile':"",
                                                         'communityProxyFile':"",
                                                         'communityRefreshInterval':60,
                                                         'communityRefreshOnFailFactor':0.25,
                                                         'proxyGenerator':"",
                                                         'personalizeMethod':"",
                                                         'gridshibHome':"",
                                                         'userProxyFile':"",
                                                         'refreshMethod':"",
                                                         'condorConfig':"",
                                                         'condorQconstraint':"",
                                                         'permanentUsers':[],
                                                         'refreshInterval':60,
                                                         'refreshOnFailFactor':0.5,
                                                         'communityPrivateKeyPath':"",
                                                         'userPrivateKeyFile':"",
                                                         'jobKeyFile':"",
                                                         'md5HashTokens':[],
                                                         'state':'enabled'
                                                        }
                     elif keyValuePattern.match(record):
                        key,value = keyValuePattern.match(record).group(2,4)
                        if key in self.identities[identityName]:
                           if   isinstance(self.identities[identityName][key],list):
                              self.identities[identityName][key] = [e.strip() for e in value.split(',')]
                           elif isinstance(self.identities[identityName][key],bool):
                              self.identities[identityName][key] = bool(value.lower() == 'true')
                           elif isinstance(self.identities[identityName][key],float):
                              self.identities[identityName][key] = float(value)
                           elif isinstance(self.identities[identityName][key],int):
                              self.identities[identityName][key] = int(value)
                           elif isinstance(self.identities[identityName][key],dict):
                              try:
                                 sampleKey   = self.identities[identityName][key].keys()[0]
                                 sampleValue = self.identities[identityName][key][sampleKey]
                              except:
                                 sampleKey   = "key"
                                 sampleValue = "value"
                              self.identities[identityName][key] = {}
                              for e in value.split(','):
                                 dictKey,dictValue = e.split(':')
                                 if isinstance(sampleKey,int):
                                    dictKey = int(dictKey)
                                 if   isinstance(sampleValue,int):
                                    dictValue = int(dictValue)
                                 elif isinstance(sampleValue,float):
                                    dictValue = float(dictValue)
                                 elif isinstance(sampleValue,bool):
                                    dictValue = bool(dictValue.lower() == 'true')
                                 self.identities[identityName][key][dictKey] = dictValue
                           else:
                              self.identities[identityName][key] = value
                        else:
                           message = "Undefined key = value pair %s = %s for identity %s" % (key,value,identityName)
                           self.logger.log(logging.WARNING,getLogMessage(message))
                  else:
                     eof = True
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("Identities configuration file %s could not be read" % (identitiesPath)))
            finally:
               fpInfo.close()
         except (IOError,OSError):
            self.logger.log(logging.ERROR,getLogMessage("Identities configuration file %s could not be opened" % (identitiesPath)))
      else:
         self.logger.log(logging.ERROR,getLogMessage("Identities configuration file %s is missing" % (identitiesPath)))


   def getEnabledIdentityNames(self):
      enabledIdentityNames = []
      for identityName in self.identities:
         identity = self.identities[identityName]
         if identity['state'] == 'enabled':
            enabledIdentityNames.append(identityName)

      return(enabledIdentityNames)


   def getIdentityKeyValue(self,
                           identityName,
                           key):
      value = ""
      if identityName in self.identities:
         identity = self.identities[identityName]
         if key in identity:
            value = identity[key]

      return(value)


   def getIdentitiesWithKeyValue(self,
                                 key,
                                 value=None):
      identitiesWithKeyValue = []
      for identityName in self.identities:
         identity = self.identities[identityName]
         if identity['state'] == 'enabled':
            if key in identity:
               if   isinstance(identity[key],list):
                  if value:
                     if value in identity[key]:
                        identitiesWithKeyValue.append(identityName)
                  else:
                     identitiesWithKeyValue.append(identityName)
               elif isinstance(identity[key],dict):
                  if value:
                     if value in identity[key]:
                        identitiesWithKeyValue.append(identityName)
                  else:
                     identitiesWithKeyValue.append(identityName)
               elif identity[key] == value:
                  identitiesWithKeyValue.append(identityName)

      return(identitiesWithKeyValue)


   PROXYINFO = ['certificateDirectory','certFile','keyFile','communityProxyFile']


   def getIdentityGridProxyInfo(self,
                                identityName):
      identityGridProxyInfo = {}
      if identityName in self.identities:
         identity = self.identities[identityName]
         if identity['state'] == 'enabled':
            if identity['identityType'] == 'x509':
               if identity['proxyGenerator'] == 'grid':
                  for key in self.PROXYINFO:
                     identityGridProxyInfo[key] = identity[key]

      return(identityGridProxyInfo)


   def getIdentityVomsProxyInfo(self,
                                identityName):
      identityVomsProxyInfo = {}
      if identityName in self.identities:
         identity = self.identities[identityName]
         if identity['state'] == 'enabled':
            if identity['identityType'] == 'x509':
               if identity['proxyGenerator'] == 'voms':
                  for key in self.PROXYINFO:
                     identityVomsProxyInfo[key] = identity[key]

      return(identityVomsProxyInfo)


   REFRESHPROXYINFO = ['condorConfig','condorQconstraint','permanentUsers','personalizeMethod',
                       'certificateDirectory','communityProxyFile','gridshibHome','userProxyFile',
                       'refreshInterval','refreshOnFailFactor']


   def getIdentityRefreshProxyInfo(self,
                                   identityName):
      identityRefreshProxyInfo = {}
      if identityName in self.identities:
         identity = self.identities[identityName]
         if identity['state'] == 'enabled':
            if identity['identityType'] == 'x509':
               for key in self.REFRESHPROXYINFO:
                  identityRefreshProxyInfo[key] = identity[key]

      return(identityRefreshProxyInfo)


