# @package      hubzero-submit-distributor
# @file         RemoteInstantAppScript.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2012-2015 HUBzero Foundation, LLC.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2012-2015 HUBzero Foundation, LLC.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#
import os.path
import re
import logging

from hubzero.submit.LogMessage        import getLogJobIdMessage as getLogMessage 
from hubzero.submit.ParameterTemplate import ParameterTemplate

class RemoteInstantAppScript:
   SUBMISSIONSCRIPTCOMMANDPREFIX = ''

   def __init__(self,
                hubUserName,
                hubUserId,
                submitterClass,
                runName,
                localJobId,
                instanceId,
                isPegasus,
                workingDirectory,
                transferExecutable,
                executable,
                stdinput,
                arguments,
                useEnvironment,
                environment,
                isMultiCoreRequest,
                siteInfo,
                submissionScriptsInfo,
                managerInfo,
                timeHistoryLogs):
      self.logger                   = logging.getLogger(__name__)
      self.hubUserName              = hubUserName
      self.hubUserId                = hubUserId
      self.submitterClass           = submitterClass
      self.runName                  = runName
      self.localJobId               = localJobId
      self.instanceId               = instanceId
      self.isPegasus                = isPegasus
      self.workingDirectory         = workingDirectory
      self.stageFiles               = siteInfo['stageFiles']
      self.submissionScriptCommands = siteInfo['submissionScriptCommands']
      self.submissionScriptsInfo    = submissionScriptsInfo
      self.transferExecutable       = transferExecutable
      if transferExecutable:
         if self.stageFiles:
            self.executable         = os.path.join(".",os.path.basename(executable))
         else:
            self.executable         = executable
      else:
         if not executable.startswith('/') and not executable.startswith('$') and siteInfo['remoteApplicationRootDirectory'] != "":
            self.executable         = os.path.join(siteInfo['remoteApplicationRootDirectory'],executable)
         else:
            self.executable         = executable
      self.stdinput                 = stdinput
      self.arguments                = arguments
      self.useEnvironment           = useEnvironment
      self.environment              = environment
      self.isMultiCoreRequest       = isMultiCoreRequest
      self.computationMode          = managerInfo['computationMode']
      self.mpiRankVariable          = managerInfo['mpiRankVariable']
      self.preManagerCommands       = managerInfo['preManagerCommands']
      self.postManagerCommands      = managerInfo['postManagerCommands']
      self.timestampTransferred     = timeHistoryLogs['timestampTransferred']
      self.timestampStart           = timeHistoryLogs['timestampStart']
      self.timestampFinish          = timeHistoryLogs['timestampFinish']
      self.timeResults              = timeHistoryLogs['timeResults']


   def __buildAppSerialScript(self):
      rawSubmissionScript = self.submissionScriptsInfo.getSubmissionScript('Instant','APP','serial')

      commandsSCRIPT = ""
      if self.submissionScriptCommands:
         if self.SUBMISSIONSCRIPTCOMMANDPREFIX:
            commandSeparator = "\n%s " % self.SUBMISSIONSCRIPTCOMMANDPREFIX
            commandsSCRIPT = self.SUBMISSIONSCRIPTCOMMANDPREFIX + " " + commandSeparator.join(self.submissionScriptCommands)
         else:
            commandSeparator = "\n"
            commandsSCRIPT = commandSeparator.join(self.submissionScriptCommands)

      environmentExport = ""
      environmentVars = self.environment.split()
      for environmentVar in environmentVars:
         environmentExport += "export " + environmentVar + "\n"

      substitutions = {}
      if self.stdinput == "":
         substitutions["STDIN"]                 = '/dev/null'
      else:
         substitutions["STDIN"]                 = self.stdinput
      substitutions["USEENVIRONMENT"]           = self.useEnvironment
      substitutions["ENVIRONMENT"]              = environmentExport
      substitutions["EXECUTABLE"]               = self.executable
      substitutions["ARGUMENTS"]                = self.arguments
      substitutions["TS_TRANSFERRED"]           = self.timestampTransferred
      substitutions["TS_START"]                 = self.timestampStart
      substitutions["TS_FINISH"]                = self.timestampFinish
      substitutions["TIME_RESULTS"]             = self.timeResults
      substitutions["RUNNAME"]                  = self.runName
      substitutions["JOBID"]                    = self.localJobId
      substitutions["INSTANCEID"]               = self.instanceId
      substitutions["WORKINGDIRECTORY"]         = self.workingDirectory
      substitutions["PREMANAGERCOMMANDS"]       = "\n".join(self.preManagerCommands)
      substitutions["POSTMANAGERCOMMANDS"]      = "\n".join(self.postManagerCommands)
      substitutions["HUBUSERNAME"]              = self.hubUserName
      substitutions["HUBUSERID"]                = str(self.hubUserId)
      substitutions["SUBMISSIONSCRIPTCOMMANDS"] = commandsSCRIPT

      template = ParameterTemplate(rawSubmissionScript)
      try:
         submissionScript = template.substitute_recur(substitutions)
      except KeyError,err:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Pattern substitution failed for @@%s\n" % (err[0])))
      except TypeError,err:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Submission script substitution failed:\n%s\n" % (rawSubmissionScript)))
      else:
         if self.SUBMISSIONSCRIPTCOMMANDPREFIX:
            submissionScript = re.sub("(\n)*\n%s" % (self.SUBMISSIONSCRIPTCOMMANDPREFIX),
                                           "\n%s" % (self.SUBMISSIONSCRIPTCOMMANDPREFIX),submissionScript)

      return(submissionScript)


   def __buildAppMPIScript(self):
      environmentExport = ""
      environmentVars = self.environment.split()
      if self.mpiRankVariable == "":
         rawSubmissionScript = self.submissionScriptsInfo.getSubmissionScript('Instant','APP','mpinorank')
      else:
         rawSubmissionScript = self.submissionScriptsInfo.getSubmissionScript('Instant','APP','mpi')

      useEnvironmentExport = self.useEnvironment
      for environmentVar in environmentVars:
         environmentExport += "export " + environmentVar + "\n"

      substitutions = {}
      substitutions["RUNNAME"]         = self.runName
      substitutions["JOBID"]           = self.localJobId
      substitutions["INSTANCEID"]      = self.instanceId
      substitutions["EXECUTABLE"]      = self.executable
      if self.stdinput == "":
         substitutions["STDIN"]        = '/dev/null'
      else:
         substitutions["STDIN"]        = self.stdinput
      substitutions["ARGUMENTS"]       = self.arguments
      substitutions["USEENVIRONMENT"]  = useEnvironmentExport
      substitutions["ENVIRONMENT"]     = environmentExport
      substitutions["MPIRANKVARIABLE"] = self.mpiRankVariable
      substitutions["TIME_RESULTS"]    = self.timeResults
      substitutions["HUBUSERNAME"]     = self.hubUserName
      substitutions["HUBUSERID"]       = str(self.hubUserId)

      template = ParameterTemplate(rawSubmissionScript)
      try:
         submissionScript = template.substitute_recur(substitutions)
      except KeyError,err:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Pattern substitution failed for @@%s\n" % (err[0])))
      except TypeError,err:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Submission script substitution failed:\n%s\n" % (rawSubmissionScript)))

      return(submissionScript)


   def __buildAppMatlabPCTScript(self):
      rawSubmissionScript = self.submissionScriptsInfo.getSubmissionScript('Instant','APP','matlabpct')

      environmentExport = ""
      environmentVars = self.environment.split()
      for environmentVar in environmentVars:
         environmentExport += "export " + environmentVar + "\n"

      substitutions = {}
      if self.stdinput == "":
         substitutions["STDIN"]         = '/dev/null'
      else:
         substitutions["STDIN"]         = self.stdinput
      substitutions["USEENVIRONMENT"]   = self.useEnvironment
      substitutions["ENVIRONMENT"]      = environmentExport
      substitutions["EXECUTABLE"]       = self.executable
      substitutions["ARGUMENTS"]        = self.arguments
      substitutions["TIME_RESULTS"]     = self.timeResults
      substitutions["WORKINGDIRECTORY"] = self.workingDirectory
      substitutions["HUBUSERNAME"]      = self.hubUserName
      substitutions["HUBUSERID"]        = str(self.hubUserId)

      template = ParameterTemplate(rawSubmissionScript)
      try:
         submissionScript = template.substitute_recur(substitutions)
      except KeyError,err:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Pattern substitution failed for @@%s\n" % (err[0])))
      except TypeError,err:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Submission script substitution failed:\n%s\n" % (rawSubmissionScript)))

      return(submissionScript)


   def buildAppScript(self):
      appScriptName = "%s_%s.sh" % (self.localJobId,self.instanceId)
      if self.isMultiCoreRequest:
         if   self.computationMode == 'mpi':
            appScript = self.__buildAppMPIScript()
         elif self.computationMode == 'parallel':
            appScript = ""
         elif self.computationMode == 'matlabmpi':
            appScript = ""
         elif self.computationMode == 'matlabpct':
            appScript = self.__buildAppMatlabPCTScript()
      else:
         appScript = self.__buildAppSerialScript()
      appScriptExecutable = True

      return(appScriptName,appScript,appScriptExecutable)


