# @package      hubzero-submit-distributor
# @file         ToolsInfo.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2012-2015 HUBzero Foundation, LLC.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2012-2015 HUBzero Foundation, LLC.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#
import os.path
import re
import random
import logging

from hubzero.submit.LogMessage      import getLogJobIdMessage as getLogMessage
from hubzero.submit.GroupMembership import isGroupMember

class ToolsInfo:
   def __init__(self,
                toolsPath):
      self.logger     = logging.getLogger(__name__)
      self.tools      = {}

      toolPattern     = re.compile('(\s*\[)([^\s]*)(]\s*)')
      keyValuePattern = re.compile('( *)(\w*)( *= *)(.*[^\s$])( *)')
      commentPattern  = re.compile('\s*#.*')
      toolName        = ""

      if os.path.exists(toolsPath):
         try:
            fpInfo = open(toolsPath,'r')
            try:
               eof = False
               while not eof:
                  record = fpInfo.readline()
                  if record != "":
                     record = commentPattern.sub("",record)
                     if   toolPattern.match(record):
                        toolName = toolPattern.match(record).group(2)
                        if not toolName in self.tools:
                           self.tools[toolName] = []
                        toolGroupIndex = len(self.tools[toolName])
                        self.tools[toolName].append({'destinations':[],
                                                     'executablePath':'',
                                                     'restrictedToUsers':[],
                                                     'restrictedToGroups':[],
                                                     'environment':[],
                                                     'remoteManager':'',
                                                     'state':'enabled'
                                                    })
                     elif keyValuePattern.match(record):
                        key,value = keyValuePattern.match(record).group(2,4)
                        if key in self.tools[toolName][toolGroupIndex]:
                           if   isinstance(self.tools[toolName][toolGroupIndex][key],list):
                              self.tools[toolName][toolGroupIndex][key] = [e.strip() for e in value.split(',')]
                           elif isinstance(self.tools[toolName][toolGroupIndex][key],bool):
                              self.tools[toolName][toolGroupIndex][key] = bool(value.lower() == 'true')
                           elif isinstance(self.tools[toolName][toolGroupIndex][key],float):
                              self.tools[toolName][toolGroupIndex][key] = float(value)
                           elif isinstance(self.tools[toolName][toolGroupIndex][key],int):
                              self.tools[toolName][toolGroupIndex][key] = int(value)
                           elif isinstance(self.tools[toolName][toolGroupIndex][key],dict):
                              try:
                                 sampleKey   = self.tools[toolName][toolGroupIndex][key].keys()[0]
                                 sampleValue = self.tools[toolName][toolGroupIndex][key][sampleKey]
                              except:
                                 sampleKey   = "key"
                                 sampleValue = "value"
                              self.tools[toolName][toolGroupIndex][key] = {}
                              for e in value.split(','):
                                 dictKey,dictValue = e.split(':')
                                 if isinstance(sampleKey,int):
                                    dictKey = int(dictKey)
                                 if   isinstance(sampleValue,int):
                                    dictValue = int(dictValue)
                                 elif isinstance(sampleValue,float):
                                    dictValue = float(dictValue)
                                 elif isinstance(sampleValue,bool):
                                    dictValue = bool(dictValue.lower() == 'true')
                                 self.tools[toolName][toolGroupIndex][key][dictKey] = dictValue
                           else:
                              self.tools[toolName][toolGroupIndex][key] = value
                        else:
                           self.logger.log(logging.WARNING,getLogMessage("Undefined tool key %s %s in tool %s" % \
                                                                                            (key,value,toolName)))
                  else:
                     eof = True
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("Tools configuration file %s could not be read" % (toolsPath)))
            finally:
               fpInfo.close()
         except (IOError,OSError):
            self.logger.log(logging.ERROR,getLogMessage("Tools configuration file %s could not be opened" % (toolsPath)))
      else:
         self.logger.log(logging.ERROR,getLogMessage("Tools configuration file %s is missing" % (toolsPath)))

      for toolName in self.tools:
         markedForDeletion = []
         for toolGroupIndex in xrange(len(self.tools[toolName])):
            state = self.tools[toolName][toolGroupIndex]['state']
            if state == 'disabled':
               markedForDeletion.append(toolGroupIndex)
         markedForDeletion.reverse()
         for toolGroupIndex in markedForDeletion:
            del self.tools[toolName][toolGroupIndex]
         del markedForDeletion


   def applyUserRestriction(self,
                            userName):
      for toolName in self.tools:
         markedForDeletion = []
         for toolGroupIndex in xrange(len(self.tools[toolName])):
            restrictedToUsers = self.tools[toolName][toolGroupIndex]['restrictedToUsers']
            if len(restrictedToUsers) > 0:
               if userName in restrictedToUsers:
                  userOK = True
               else:
                  userOK = False
               if not userOK:
                  markedForDeletion.append(toolGroupIndex)
         markedForDeletion.reverse()
         for toolGroupIndex in markedForDeletion:
            del self.tools[toolName][toolGroupIndex]
         del markedForDeletion


   def applyGroupRestriction(self):
      for toolName in self.tools:
         markedForDeletion = []
         for toolGroupIndex in xrange(len(self.tools[toolName])):
            restrictedToGroups = self.tools[toolName][toolGroupIndex]['restrictedToGroups']
            if len(restrictedToGroups) > 0:
               groupOK = False
               for restrictedToGroup in restrictedToGroups:
                  if isGroupMember(restrictedToGroup):
                     groupOK = True
                     break
               if not groupOK:
                  markedForDeletion.append(toolGroupIndex)
         markedForDeletion.reverse()
         for toolGroupIndex in markedForDeletion:
            del self.tools[toolName][toolGroupIndex]
         del markedForDeletion


   def purgeDisabledSites(self,
                          enabledSites):
      for toolName in self.tools:
         toolInfo = self.tools[toolName]
         for toolGroupIndex in xrange(len(toolInfo)):
            markedForDeletion = []
            for toolDestination in toolInfo[toolGroupIndex]['destinations']:
               if not toolDestination in enabledSites:
                  markedForDeletion.append(toolDestination)
            markedForDeletion.reverse()
            for toolDestination in markedForDeletion:
               self.tools[toolName][toolGroupIndex]['destinations'].remove(toolDestination)
            del markedForDeletion
            if len(self.tools[toolName][toolGroupIndex]['destinations']) == 0:
               self.tools[toolName][toolGroupIndex]['state'] = 'sitesUnavailable'


   def purgeNonPegasusSites(self,
                            nonPegasusSites):
      for toolName in self.tools:
         toolInfo = self.tools[toolName]
         for toolGroupIndex in xrange(len(toolInfo)):
            markedForDeletion = []
            for toolDestination in toolInfo[toolGroupIndex]['destinations']:
               if toolDestination in nonPegasusSites:
                  markedForDeletion.append(toolDestination)
            markedForDeletion.reverse()
            for toolDestination in markedForDeletion:
               self.tools[toolName][toolGroupIndex]['destinations'].remove(toolDestination)
            del markedForDeletion
            if len(self.tools[toolName][toolGroupIndex]['destinations']) == 0:
               self.tools[toolName][toolGroupIndex]['state'] = 'sitesUnavailable'


   def purgePegasusSites(self,
                         pegasusSites):
      for toolName in self.tools:
         toolInfo = self.tools[toolName]
         for toolGroupIndex in xrange(len(toolInfo)):
            markedForDeletion = []
            nNonPegasusSites = 0
            for toolDestination in toolInfo[toolGroupIndex]['destinations']:
               if toolDestination in pegasusSites:
                  markedForDeletion.append(toolDestination)
               else:
                  nNonPegasusSites += 1
            if nNonPegasusSites > 0 and len(markedForDeletion) > 0:
               markedForDeletion.reverse()
               for toolDestination in markedForDeletion:
                  self.tools[toolName][toolGroupIndex]['destinations'].remove(toolDestination)
               if len(self.tools[toolName][toolGroupIndex]['destinations']) == 0:
                  self.tools[toolName][toolGroupIndex]['state'] = 'sitesUnavailable'
            del markedForDeletion


   def isExecutableTool(self,
                        executable):
      return(executable in self.tools)


   def isPermissionGranted(self,
                           executable):
      permissionGranted = False
      if len(self.tools[executable]) > 0:
         permissionGranted = True

      return(permissionGranted)


   def getDefaultToolInfo(self):
      toolInfo = {}
      toolInfo['destinations']   = []
      toolInfo['executablePath'] = ""
      toolInfo['environment']    = []
      toolInfo['remoteManager']  = ""

      return(toolInfo)


   def selectTool(self,
                  executable,
                  userDestinations,
                  remoteProbeMonitor):
      selectedToolInfo = {}
      selectedToolInfo['destinations']   = []
      selectedToolInfo['executablePath'] = ""
      selectedToolInfo['environment']    = []
      selectedToolInfo['remoteManager']  = ""

      toolInfo = self.tools[executable]
      if len(userDestinations) == 0:
         toolGroupIndexes = range(len(toolInfo))
         while len(toolGroupIndexes) > 0 and len(selectedToolInfo['destinations']) == 0:
            toolGroupIndex = random.choice(toolGroupIndexes)
            for toolDestination in toolInfo[toolGroupIndex]['destinations']:
               selectedToolInfo['destinations'].append(toolDestination)
            if remoteProbeMonitor:
               remoteProbeMonitor.purgeOfflineSites(selectedToolInfo['destinations'])
            if len(selectedToolInfo['destinations']) == 0:
               toolGroupIndexes.remove(toolGroupIndex)
      else:
         toolGroupIndex = -1
         for userDestination in userDestinations:
            for groupIndex in xrange(len(toolInfo)):
               if userDestination in toolInfo[groupIndex]['destinations']:
                  toolGroupIndex = groupIndex
                  break
            if toolGroupIndex != -1:
               break

         if toolGroupIndex != -1:
            for toolDestination in toolInfo[toolGroupIndex]['destinations']:
               if toolDestination in userDestinations:
                  selectedToolInfo['destinations'].append(toolDestination)
         else:
            self.logger.log(logging.ERROR,getLogMessage("Invalid destination(s) specified: %s" % (userDestinations)))

      if len(selectedToolInfo['destinations']) > 0:
         selectedToolInfo['executablePath']   = toolInfo[toolGroupIndex]['executablePath']
         if 'environment' in toolInfo[toolGroupIndex]:
            selectedToolInfo['environment']   = toolInfo[toolGroupIndex]['environment']
         if 'remoteManager' in toolInfo[toolGroupIndex]:
            selectedToolInfo['remoteManager'] = toolInfo[toolGroupIndex]['remoteManager']

      return(selectedToolInfo)


   def getToolNames(self):
      toolNames = []
      for toolName in self.tools:
         if self.isPermissionGranted(toolName):
            toolNames.append(toolName)

      return(toolNames)


   def getToolDestinations(self,
                           toolName):
      destinations = []

      toolInfo = self.tools[toolName]
      for toolGroupIndex in range(len(toolInfo)):
         for toolDestination in toolInfo[toolGroupIndex]['destinations']:
            if not toolDestination in destinations:
               destinations.append(toolDestination)

      return(destinations)


