# @package      hubzero-submit-distributor
# @file         RemoteRunjobAppScript.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2012-2015 HUBzero Foundation, LLC.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2012-2015 HUBzero Foundation, LLC.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#
import os.path
import re
import logging

from hubzero.submit.LogMessage        import getLogJobIdMessage as getLogMessage 
from hubzero.submit.ParameterTemplate import ParameterTemplate

class RemoteRunjobAppScript:
   def __init__(self,
                hubUserName,
                hubUserId,
                submitterClass,
                runName,
                localJobId,
                instanceId,
                isParametric,
                transferExecutable,
                executable,
                stdinput,
                arguments,
                environment,
                isMultiCoreRequest,
                siteInfo,
                submissionScriptsInfo,
                managerInfo,
                timeHistoryLogs):
      self.logger                = logging.getLogger(__name__)
      self.hubUserName           = hubUserName
      self.hubUserId             = hubUserId
      self.submitterClass        = submitterClass
      self.runName               = runName
      self.localJobId            = localJobId
      self.instanceId            = instanceId
      self.isParametric          = isParametric
      self.transferExecutable    = transferExecutable
      if transferExecutable:
         self.executable         = os.path.join(".",os.path.basename(executable))
      else:
         self.executable         = executable
      self.stdinput              = stdinput
      self.arguments             = arguments
      self.environment           = environment
      self.isMultiCoreRequest    = isMultiCoreRequest
      self.submissionScriptsInfo = submissionScriptsInfo
      self.computationMode       = managerInfo['computationMode']
      self.moduleInitialize      = managerInfo['moduleInitialize']
      self.modulesLoad           = managerInfo['modulesLoad']
      self.modulesUnload         = managerInfo['modulesUnload']
      self.timestampStart        = timeHistoryLogs['timestampStart']
      self.timestampFinish       = timeHistoryLogs['timestampFinish']
      self.timeResults           = timeHistoryLogs['timeResults']


   def __buildAppScript(self):
      rawSubmissionScript = self.submissionScriptsInfo.getSubmissionScript('Runjob','APP','serial')

      moduleInitialize = ""
      if self.moduleInitialize != "":
         moduleInitialize = "type module > /dev/null 2>&1\n" + \
                            "if [ $? != 0 ]; then\n" + \
                            "   . " + self.moduleInitialize + "\n" + \
                            "fi\n"
      modulesUnload = ""
      if self.modulesUnload != "":
         for module in self.modulesUnload:
            modulesUnload += 'module unload ' + module + '\n'
      modulesLoad = ""
      if self.modulesLoad != "":
         for module in self.modulesLoad:
            modulesLoad += 'module load ' + module + '\n'

      substitutions = {}
      substitutions["RUNNAME"]          = self.runName
      substitutions["JOBID"]            = self.localJobId
      substitutions["INSTANCEID"]       = self.instanceId
      substitutions["EXECUTABLE"]       = self.executable
      substitutions["STDIN"]            = self.stdinput
      substitutions["ARGUMENTS"]        = self.arguments
      substitutions["TS_START"]         = self.timestampStart
      substitutions["TS_FINISH"]        = self.timestampFinish
      substitutions["TIME_RESULTS"]     = self.timeResults
      substitutions["MODULEINITIALIZE"] = moduleInitialize
      substitutions["MODULESUNLOAD"]    = modulesUnload
      substitutions["MODULESLOAD"]      = modulesLoad
      substitutions["HUBUSERNAME"]      = self.hubUserName
      substitutions["HUBUSERID"]        = str(self.hubUserId)

      template = ParameterTemplate(rawSubmissionScript)
      try:
         submissionScript = template.substitute_recur(substitutions)
      except KeyError,err:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Pattern substitution failed for @@%s\n" % (err[0])))
      except TypeError,err:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Submission script substitution failed:\n%s\n" % (rawSubmissionScript)))

      return(submissionScript)


   def __buildAppMPIScript(self):
      rawSubmissionScript = self.submissionScriptsInfo.getSubmissionScript('Runjob','APP','mpi')

      environmentExport = ""
      environmentVars = self.environment.split()
      for environmentVar in environmentVars:
         environmentExport += "export " + environmentVar + "\n"

      moduleInitialize = ""
      if self.moduleInitialize != "":
         moduleInitialize = "type module > /dev/null 2>&1\n" + \
                            "if [ $? != 0 ]; then\n" + \
                            "   . " + self.moduleInitialize + "\n" + \
                            "fi"
      modulesUnload = ""
      if self.modulesUnload != "":
         for module in self.modulesUnload:
            modulesUnload += 'module unload ' + module + '\n'
      modulesLoad = ""
      if self.modulesLoad != "":
         for module in self.modulesLoad:
            modulesLoad += 'module load ' + module + '\n'

      substitutions = {}
      substitutions["RUNNAME"]          = self.runName
      substitutions["JOBID"]            = self.localJobId
      substitutions["INSTANCEID"]       = self.instanceId
      substitutions["EXECUTABLE"]       = self.executable
      substitutions["STDIN"]            = self.stdinput
      substitutions["ARGUMENTS"]        = self.arguments
      substitutions["ENVIRONMENT"]      = environmentExport
      substitutions["TS_START"]         = self.timestampStart
      substitutions["TS_FINISH"]        = self.timestampFinish
      substitutions["TIME_RESULTS"]     = self.timeResults
      substitutions["MODULEINITIALIZE"] = moduleInitialize
      substitutions["MODULESUNLOAD"]    = modulesUnload
      substitutions["MODULESLOAD"]      = modulesLoad
      substitutions["HUBUSERNAME"]      = self.hubUserName
      substitutions["HUBUSERID"]        = str(self.hubUserId)

      template = ParameterTemplate(rawSubmissionScript)
      try:
         submissionScript = template.substitute_recur(substitutions)
      except KeyError,err:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Pattern substitution failed for @@%s\n" % (err[0])))
      except TypeError,err:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Submission script substitution failed:\n%s\n" % (rawSubmissionScript)))

      return(submissionScript)


   def buildAppScript(self):
      appScriptName = "%s_%s.sh" % (self.localJobId,self.instanceId)
      if self.isMultiCoreRequest:
         if   self.computationMode == 'mpi':
            appScript = self.__buildAppMPIScript()
         elif self.computationMode == 'parallel':
            appScript = self.__buildAppScript()
         elif self.computationMode == 'matlabmpi':
            appScript = ""
      else:
         appScript = self.__buildAppScript()
      appScriptExecutable = True

      return(appScriptName,appScript,appScriptExecutable)


