# @package      hubzero-submit-distributor
# @file         RemoteBatchRUNJOB.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2012-2015 HUBzero Foundation, LLC.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2012-2015 HUBzero Foundation, LLC.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#
import os
import re
import time
import datetime
import logging

from hubzero.submit.LogMessage        import getLogJobIdMessage as getLogMessage 
from hubzero.submit.ParameterTemplate import ParameterTemplate

class RemoteBatchRUNJOB:
   def __init__(self,
                hubUserName,
                hubUserId,
                submitterClass,
                runName,
                localJobId,
                instanceId,
                instanceDirectory,
                appScriptName,
                environment,
                transferExecutable,
                executable,
                arguments,
                isMultiCoreRequest,
                siteInfo,
                submissionScriptsInfo,
                managerInfo,
                nCpus,
                ppn,
                wallTime,
                gridsite,
                x509SubmitProxy,
                disableProbeCheck,
                disableStateCheck,
                timeHistoryLogs):
      self.logger                = logging.getLogger(__name__)
      self.hubUserName           = hubUserName
      self.hubUserId             = hubUserId
      self.submitterClass        = submitterClass
      self.runName               = runName
      self.localJobId            = localJobId
      self.instanceId            = instanceId
      self.instanceDirectory     = instanceDirectory
      self.appScriptName         = appScriptName
      self.environment           = environment
      self.transferExecutable    = transferExecutable
      if transferExecutable:
         self.executable         = executable
      else:
         self.executable         = executable.replace('$','\$')
      self.arguments             = arguments
      self.isMultiCoreRequest    = isMultiCoreRequest
      self.submissionScriptsInfo = submissionScriptsInfo
      self.computationMode       = managerInfo['computationMode']
      self.nCpus                 = nCpus
      self.ppn                   = ppn
      self.wallTime              = str(int(wallTime))
      self.gridsite              = gridsite
      epoch = int(time.mktime(datetime.datetime.utcnow().timetuple()))
      self.workingDirectory      = "%s_%s_%s" % (str(epoch),localJobId,instanceId)
      self.x509SubmitProxy       = x509SubmitProxy
      self.disableProbeCheck     = disableProbeCheck
      self.disableStateCheck     = disableStateCheck
      self.timestampTransferred  = timeHistoryLogs['timestampTransferred']
      self.timestampStart        = timeHistoryLogs['timestampStart']
      self.timestampFinish       = timeHistoryLogs['timestampFinish']
      self.timeResults           = timeHistoryLogs['timeResults']
      self.gridHistory           = timeHistoryLogs['jobGridHistory']
      self.gridJobId             = timeHistoryLogs['jobGridJobId']
      self.gridResource          = timeHistoryLogs['jobGridResource']

      self.nodeFileName = ""
      self.nodeList     = []


   def __buildSerialFile(self):
      rawSubmissionScript = self.submissionScriptsInfo.getSubmissionScript('Batch','RUNJOB','serial')

      substitutions = {}
      substitutions["RUNNAME"]              = self.runName
      substitutions["JOBID"]                = self.localJobId
      substitutions["INSTANCEID"]           = self.instanceId
      substitutions["EXECUTABLE"]           = self.executable
      if not self.transferExecutable:
         substitutions["TRANSFEREXE"]       = "no"
      else:
         substitutions["TRANSFEREXE"]       = "yes"
      substitutions["EXECUTABLESHELL"]      = os.path.join(self.instanceDirectory,self.appScriptName)
      substitutions["WORKINGDIR"]           = self.workingDirectory
      substitutions["WALLTIME"]             = self.wallTime
      substitutions["ENVIRONMENT"]          = self.environment
      substitutions["GRIDSITE"]             = self.gridsite
      substitutions["ARGUMENTS"]            = self.arguments
      substitutions["TIMESTAMPTRANSFERRED"] = self.timestampTransferred
      substitutions["TIMESTAMPSTART"]       = self.timestampStart
      substitutions["TIMESTAMPFINISH"]      = self.timestampFinish
      substitutions["TIMERESULTS"]          = self.timeResults
      substitutions["GRIDRESOURCE"]         = self.gridResource
      substitutions["GRIDHISTORY"]          = self.gridHistory
      substitutions["GRIDJOBID"]            = self.gridJobId
      if self.disableStateCheck:
         substitutions["DISABLESTATECHECK"] = "yes"
      else:
         substitutions["DISABLESTATECHECK"] = ""
      if self.disableProbeCheck:
         substitutions["DISABLEPROBECHECK"] = "yes"
      else:
         substitutions["DISABLEPROBECHECK"] = ""
      substitutions["X509SUBMITPROXY"]      = self.x509SubmitProxy
      substitutions["HUBUSERNAME"]          = self.hubUserName
      substitutions["HUBUSERID"]            = str(self.hubUserId)

      template = ParameterTemplate(rawSubmissionScript)
      try:
         submissionScript = template.substitute_recur(substitutions)
      except KeyError,err:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Pattern substitution failed for @@%s\n" % (err[0])))
      except TypeError,err:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Submission script substitution failed:\n%s\n" % (rawSubmissionScript)))

      return(submissionScript)


   def __buildMPIFile(self):
      rawSubmissionScript = self.submissionScriptsInfo.getSubmissionScript('Batch','RUNJOB','mpi')

      substitutions = {}
      substitutions["RUNNAME"]              = self.runName
      substitutions["JOBID"]                = self.localJobId
      substitutions["INSTANCEID"]           = self.instanceId
      substitutions["EXECUTABLE"]           = self.executable
      if not self.transferExecutable:
         substitutions["TRANSFEREXE"]       = "no"
      else:
         substitutions["TRANSFEREXE"]       = "yes"
      substitutions["EXECUTABLESHELL"]      = os.path.join(self.instanceDirectory,self.appScriptName)
      substitutions["WORKINGDIR"]           = self.workingDirectory
      substitutions["WALLTIME"]             = self.wallTime
      substitutions["PPN"]                  = self.ppn
      substitutions["NPROCESSES"]           = str(self.nCpus)
      substitutions["ENVIRONMENT"]          = self.environment
      substitutions["GRIDSITE"]             = self.gridsite
      substitutions["ARGUMENTS"]            = self.arguments
      substitutions["TIMESTAMPTRANSFERRED"] = self.timestampTransferred
      substitutions["TIMESTAMPSTART"]       = self.timestampStart
      substitutions["TIMESTAMPFINISH"]      = self.timestampFinish
      substitutions["TIMERESULTS"]          = self.timeResults
      substitutions["GRIDRESOURCE"]         = self.gridResource
      substitutions["GRIDHISTORY"]          = self.gridHistory
      substitutions["GRIDJOBID"]            = self.gridJobId
      if self.disableStateCheck:
         substitutions["DISABLESTATECHECK"] = "yes"
      else:
         substitutions["DISABLESTATECHECK"] = ""
      if self.disableProbeCheck:
         substitutions["DISABLEPROBECHECK"] = "yes"
      else:
         substitutions["DISABLEPROBECHECK"] = ""
      substitutions["X509SUBMITPROXY"]      = self.x509SubmitProxy
      substitutions["HUBUSERNAME"]          = self.hubUserName
      substitutions["HUBUSERID"]            = str(self.hubUserId)

      template = ParameterTemplate(rawSubmissionScript)
      try:
         submissionScript = template.substitute_recur(substitutions)
      except KeyError,err:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Pattern substitution failed for @@%s\n" % (err[0])))
      except TypeError,err:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Submission script substitution failed:\n%s\n" % (rawSubmissionScript)))

      return(submissionScript)


   def __buildParallelFile(self):
      rawSubmissionScript = self.submissionScriptsInfo.getSubmissionScript('Batch','RUNJOB','parallel')

      substitutions = {}
      substitutions["RUNNAME"]              = self.runName
      substitutions["JOBID"]                = self.localJobId
      substitutions["INSTANCEID"]           = self.instanceId
      substitutions["EXECUTABLE"]           = self.executable
      if not self.transferExecutable:
         substitutions["TRANSFEREXE"]       = "no"
      else:
         substitutions["TRANSFEREXE"]       = "yes"
      substitutions["EXECUTABLESHELL"]      = os.path.join(self.instanceDirectory,self.appScriptName)
      substitutions["WORKINGDIR"]           = self.workingDirectory
      substitutions["WALLTIME"]             = self.wallTime
      substitutions["PPN"]                  = self.ppn
      substitutions["NPROCESSES"]           = str(self.nCpus)
      substitutions["ENVIRONMENT"]          = self.environment
      substitutions["GRIDSITE"]             = self.gridsite
      substitutions["ARGUMENTS"]            = self.arguments
      substitutions["TIMESTAMPTRANSFERRED"] = self.timestampTransferred
      substitutions["TIMESTAMPSTART"]       = self.timestampStart
      substitutions["TIMESTAMPFINISH"]      = self.timestampFinish
      substitutions["TIMERESULTS"]          = self.timeResults
      substitutions["GRIDRESOURCE"]         = self.gridResource
      substitutions["GRIDHISTORY"]          = self.gridHistory
      substitutions["GRIDJOBID"]            = self.gridJobId
      if self.disableStateCheck:
         substitutions["DISABLESTATECHECK"] = "yes"
      else:
         substitutions["DISABLESTATECHECK"] = ""
      if self.disableProbeCheck:
         substitutions["DISABLEPROBECHECK"] = "yes"
      else:
         substitutions["DISABLEPROBECHECK"] = ""
      substitutions["X509SUBMITPROXY"]      = self.x509SubmitProxy
      substitutions["HUBUSERNAME"]          = self.hubUserName
      substitutions["HUBUSERID"]            = str(self.hubUserId)

      template = ParameterTemplate(rawSubmissionScript)
      try:
         submissionScript = template.substitute_recur(substitutions)
      except KeyError,err:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Pattern substitution failed for @@%s\n" % (err[0])))
      except TypeError,err:
         submissionScript = ""
         self.logger.log(logging.ERROR,getLogMessage("Submission script substitution failed:\n%s\n" % (rawSubmissionScript)))

      return(submissionScript)


   def buildBatchScript(self):
      batchLogName = ""
      batchScriptName = "%s_%s.runjob" % (self.localJobId,self.instanceId)
      if self.isMultiCoreRequest:
         if   self.computationMode == 'mpi':
            batchScript = self.__buildMPIFile()
         elif self.computationMode == 'parallel':
            batchScript = self.__buildParallelFile()
         elif self.computationMode == 'matlabmpi':
            batchScript = ""
            batchScriptName = ""
      else:
         batchScript = self.__buildSerialFile()
      batchScriptExecutable = True

      return(batchLogName,batchScriptName,batchScript,batchScriptExecutable)


   def getBatchNodeList(self):
      return(self.nodeFileName,self.nodeList)


   def getRemoteJobIdNumber(self,
                            remoteJobId):
      try:
# Your job has been submitted with id 19.0
         remoteJobIdNumber = re.search('with id [0-9]+',remoteJobId).group().split()[2] + ".0"
      except:
         remoteJobIdNumber = "-1"

      return(remoteJobIdNumber)


