# @package      hubzero-submit-distributor
# @file         AggregatorsInfo.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2015-2015 HUBzero Foundation, LLC.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2015-2015 HUBzero Foundation, LLC.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#
import os
import re
import logging

from hubzero.submit.LogMessage import getLogPIDMessage as getLogMessage

class AggregatorsInfo:
   def __init__(self,
                aggregatorsPath):
      self.logger = logging.getLogger(__name__)

      self.aggregators           = {}
      self.siteToAggregatorCache = {}

      aggregatorPattern = re.compile('(\s*\[)([^\s]*)(]\s*)')
      keyValuePattern   = re.compile('( *)(\w*)( *= *)(.*[^\s$])( *)')
      commentPattern    = re.compile('\s*#.*')
      aggregatorName    = ""

      if os.path.exists(aggregatorsPath):
         try:
            fpInfo = open(aggregatorsPath,'r')
            try:
               eof = False
               while not eof:
                  record = fpInfo.readline()
                  if record != "":
                     record = commentPattern.sub("",record)
                     if   aggregatorPattern.match(record):
                        aggregatorName = aggregatorPattern.match(record).group(2)
                        self.aggregators[aggregatorName] = {'destinations':[],
                                                            'state':'enabled',
                                                            'maximumActiveJobs':1000,
                                                            'priorityBoosters':{1:1.}
                                                           }
                     elif keyValuePattern.match(record):
                        key,value = keyValuePattern.match(record).group(2,4)
                        if key in self.aggregators[aggregatorName]:
                           if   isinstance(self.aggregators[aggregatorName][key],list):
                              self.aggregators[aggregatorName][key] = [e.strip() for e in value.split(',')]
                           elif isinstance(self.aggregators[aggregatorName][key],bool):
                              self.aggregators[aggregatorName][key] = bool(value.lower() == 'true')
                           elif isinstance(self.aggregators[aggregatorName][key],float):
                              self.aggregators[aggregatorName][key] = float(value)
                           elif isinstance(self.aggregators[aggregatorName][key],int):
                              self.aggregators[aggregatorName][key] = int(value)
                           elif isinstance(self.aggregators[aggregatorName][key],dict):
                              try:
                                 sampleKey   = self.aggregators[aggregatorName][key].keys()[0]
                                 sampleValue = self.aggregators[aggregatorName][key][sampleKey]
                              except:
                                 sampleKey   = "key"
                                 sampleValue = "value"
                              self.aggregators[aggregatorName][key] = {}
                              for e in value.split(','):
                                 dictKey,dictValue = e.split(':')
                                 if isinstance(sampleKey,int):
                                    dictKey = int(dictKey)
                                 if   isinstance(sampleValue,int):
                                    dictValue = int(dictValue)
                                 elif isinstance(sampleValue,float):
                                    dictValue = float(dictValue)
                                 elif isinstance(sampleValue,bool):
                                    dictValue = bool(dictValue.lower() == 'true')
                                 self.aggregators[aggregatorName][key][dictKey] = dictValue
                           else:
                              self.aggregators[aggregatorName][key] = value
                        else:
                           message = "Undefined key = value pair %s = %s for site %s" % (key,value,aggregatorName)
                           self.logger.log(logging.WARNING,getLogMessage(message))
                  else:
                     eof = True
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("Aggregators configuration file %s could not be read" % \
                                                                                               (aggregatorsPath)))
            finally:
               fpInfo.close()
         except (IOError,OSError):
            self.logger.log(logging.ERROR,getLogMessage("Aggregators configuration file %s could not be opened" % \
                                                                                              (aggregatorsPath)))
      else:
         self.logger.log(logging.ERROR,getLogMessage("Aggregators configuration file %s is missing" % (aggregatorsPath)))


   def getSiteAggregator(self,
                         siteName):
      aggregatorName = ""
      if siteName in self.siteToAggregatorCache:
         aggregatorName = self.siteToAggregatorCache[siteName]
      else:
         for aggregator in self.aggregators:
            if self.aggregators[aggregator]['state'] == 'enabled':
               if siteName in self.aggregators[aggregator]['destinations']:
                  aggregatorName = aggregator
                  self.siteToAggregatorCache[siteName] = aggregatorName
                  break

      return(aggregatorName)


   def getAggregatorKeyValue(self,
                             aggregatorName,
                             key):
      value = ""

      if aggregatorName in self.aggregators:
         aggregator = self.aggregators[aggregatorName]
         if key in aggregator:
            value = aggregator[key]

      return(value)


