# @package      hubzero-submit-distributor
# @file         RemoteBatchRUNJOB.py
# @author       Steven Clark <clarks@purdue.edu>
# @copyright    Copyright (c) 2012-2014 HUBzero Foundation, LLC.
# @license      http://www.gnu.org/licenses/lgpl-3.0.html LGPLv3
#
# Copyright (c) 2012-2014 HUBzero Foundation, LLC.
#
# This file is part of: The HUBzero(R) Platform for Scientific Collaboration
#
# The HUBzero(R) Platform for Scientific Collaboration (HUBzero) is free
# software: you can redistribute it and/or modify it under the terms of
# the GNU Lesser General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any
# later version.
#
# HUBzero is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# HUBzero is a registered trademark of HUBzero Foundation, LLC.
#

import os
import re
import time
import datetime
import logging

class RemoteBatchRUNJOB:
   def __init__(self,
                hubUserName,
                hubUserId,
                runName,
                localJobId,
                instanceId,
                instanceDirectory,
                appScriptName,
                environment,
                transferExecutable,
                executable,
                arguments,
                isMultiCoreRequest,
                managerInfo,
                nCpus,
                ppn,
                wallTime,
                gridsite,
                x509SubmitProxy,
                disableProbeCheck,
                disableStateCheck,
                timeHistoryLogs):
      self.logger               = logging.getLogger(__name__)
      self.hubUserName          = hubUserName
      self.hubUserId            = hubUserId
      self.runName              = runName
      self.localJobId           = localJobId
      self.instanceId           = instanceId
      self.instanceDirectory    = instanceDirectory
      self.appScriptName        = appScriptName
      self.environment          = environment
      self.transferExecutable   = transferExecutable
      if transferExecutable:
         self.executable        = executable
      else:
         self.executable        = executable.replace('$','\$')
      self.arguments            = arguments
      self.isMultiCoreRequest   = isMultiCoreRequest
      self.computationMode      = managerInfo['computationMode']
      self.nCpus                = nCpus
      self.ppn                  = ppn
      self.wallTime             = str(int(wallTime))
      self.gridsite             = gridsite
      epoch = int(time.mktime(datetime.datetime.utcnow().timetuple()))
      self.workingDirectory     = "%s_%s_%s" % (str(epoch),localJobId,instanceId)
      self.x509SubmitProxy      = x509SubmitProxy
      self.disableProbeCheck    = disableProbeCheck
      self.disableStateCheck    = disableStateCheck
      self.timestampTransferred = timeHistoryLogs['timestampTransferred']
      self.timestampStart       = timeHistoryLogs['timestampStart']
      self.timestampFinish      = timeHistoryLogs['timestampFinish']
      self.timeResults          = timeHistoryLogs['timeResults']
      self.gridHistory          = timeHistoryLogs['jobGridHistory']
      self.gridJobId            = timeHistoryLogs['jobGridJobId']
      self.gridResource         = timeHistoryLogs['jobGridResource']

      self.nodeFileName = ""
      self.nodeList     = []


   def __makeSerialTemplate(self):
      return """#!/bin/sh
# RemoteBatchRUNJOB:makeSerialTemplate
#
# Set up the environment variables expected by runjob.sh
export RJ_JOB_ID=JOBID_INSTANCEID
export RJ_JOB_NAME=RUNNAME_INSTANCEID
export RJ_LOCAL_JOB_DIR=`pwd`
export RJ_INPUT_TARBALL=JOBID_INSTANCEID_input.tar.gz
export RJ_OUTPUT_TARBALL=JOBID_INSTANCEID_output.tar.gz
export RJ_JOB_WRAPPER=EXECUTABLESHELL
export RJ_EXECUTABLE=_EXECUTABLE_
export RJ_TRANSFER_EXEC=TRANSFEREXE
export RJ_WORKDIR=WORKINGDIR
export RJ_ARGUMENTS="_ARGUMENTS_"
export RJ_ENVIRONMENT="_ENVIRONMENT_"
export RJ_WALL_TIME=WALLTIME
export RJ_GRID_SITES="GRIDSITE"
if [ "${RJ_GRID_SITES}" = "" ] ; then
   export RJ_REQUIREMENTS="Platform==\\"linux\\""
fi
export RJ_MAX_QUEUE_WAITTIME=3*60*60
export RJ_DISABLE_STATE_CHECK=DISABLESTATECHECK
export RJ_DISABLE_PROBE_CHECK=DISABLEPROBECHECK
export RJ_PROXY_SOURCE=X509SUBMITPROXY
export RJ_GRID_RESOURCE=GRIDRESOURCE
export RJ_GRID_HISTORY=GRIDHISTORY
export RJ_GRID_JOB_ID=GRIDJOBID
export RJ_TIMESTAMP_TRANSFERRED=TIMESTAMPTRANSFERRED
export RJ_TIMESTAMP_START=TIMESTAMPSTART
export RJ_TIMESTAMP_FINISH=TIMESTAMPFINISH
export RJ_TIME_RESULTS=TIMERESULTS

runjob.sh
"""


   def __buildSerialFile(self):
      # setup regex's for the template
      re_runName           = re.compile("RUNNAME")
      re_jobId             = re.compile("JOBID")
      re_instanceId        = re.compile("INSTANCEID")
      re_exe               = re.compile("_EXECUTABLE_")
      re_transferexe       = re.compile("TRANSFEREXE")
      re_exesh             = re.compile("EXECUTABLESHELL")
      re_workingdir        = re.compile("WORKINGDIR")
      re_walltime          = re.compile("WALLTIME")
      re_environment       = re.compile("_ENVIRONMENT_")
      re_gridsite          = re.compile("GRIDSITE")
      re_arguments         = re.compile("_ARGUMENTS_")
      re_tsTransferred     = re.compile("TIMESTAMPTRANSFERRED")
      re_tsStart           = re.compile("TIMESTAMPSTART")
      re_tsFinish          = re.compile("TIMESTAMPFINISH")
      re_timeResults       = re.compile("TIMERESULTS")
      re_gridResource      = re.compile("GRIDRESOURCE")
      re_gridHistory       = re.compile("GRIDHISTORY")
      re_gridJobId         = re.compile("GRIDJOBID")
      re_disableStateCheck = re.compile("DISABLESTATECHECK")
      re_disableProbeCheck = re.compile("DISABLEPROBECHECK")
      re_x509SubmitProxy   = re.compile("X509SUBMITPROXY")
      re_hubUserName       = re.compile("HUBUSERNAME")
      re_hubUserId         = re.compile("HUBUSERID")

      template = self.__makeSerialTemplate()

      template = re_tsTransferred.sub(self.timestampTransferred,template)
      template = re_tsStart.sub(self.timestampStart,template)
      template = re_tsFinish.sub(self.timestampFinish,template)
      template = re_timeResults.sub(self.timeResults,template)
      template = re_gridResource.sub(self.gridResource,template)
      template = re_gridHistory.sub(self.gridHistory,template)
      template = re_gridJobId.sub(self.gridJobId,template)
      if self.disableStateCheck:
         template = re_disableStateCheck.sub("yes",template)
      else:
         template = re_disableStateCheck.sub("",template)
      if self.disableProbeCheck:
         template = re_disableProbeCheck.sub("yes",template)
      else:
         template = re_disableProbeCheck.sub("",template)
      template = re_x509SubmitProxy.sub(self.x509SubmitProxy,template)

      template = re_runName.sub(self.runName,template)
      template = re_jobId.sub(self.localJobId,template)
      template = re_instanceId.sub(self.instanceId,template)
      template = re_exe.sub(self.executable,template)
      if not self.transferExecutable:
         template = re_transferexe.sub("no",template)
      else:
         template = re_transferexe.sub("yes",template)
      template = re_exesh.sub(os.path.join(self.instanceDirectory,self.appScriptName),template)
      template = re_workingdir.sub(self.workingDirectory,template)
      template = re_walltime.sub(self.wallTime,template)

      template = re_environment.sub(self.environment,template)
      template = re_gridsite.sub(self.gridsite,template)
      template = re_arguments.sub(self.arguments,template)
      template = re_hubUserName.sub(self.hubUserName,template)
      template = re_hubUserId.sub(str(self.hubUserId),template)

      return(template)


   def __makeMPITemplate(self):
      return """#!/bin/sh
# RemoteBatchRUNJOB:makeMPITemplate
#
# Set up the environment variables expected by runjob.sh
export RJ_JOB_ID=JOBID_INSTANCEID
export RJ_JOB_NAME=RUNNAME_INSTANCEID
export RJ_LOCAL_JOB_DIR=`pwd`
export RJ_INPUT_TARBALL=JOBID_INSTANCEID_input.tar.gz
export RJ_OUTPUT_TARBALL=JOBID_INSTANCEID_output.tar.gz
export RJ_JOB_WRAPPER=EXECUTABLESHELL
export RJ_EXECUTABLE=_EXECUTABLE_
export RJ_TRANSFER_EXEC=TRANSFEREXE
export RJ_WORKDIR=WORKINGDIR
export RJ_ARGUMENTS="_ARGUMENTS_"
export RJ_ENVIRONMENT="_ENVIRONMENT_"
export RJ_WALL_TIME=WALLTIME
export RJ_MPI_CPU_COUNT=NPROCESSES
export RJ_APP_CPUS_PER_HOST=PPN
export RJ_GRID_SITES="GRIDSITE"
#if [ "${RJ_GridSite}" = "" ] ; then
#   export RJ_REQUIREMENTS=""
#else
#   export RJ_REQUIREMENTS='Name=="'${RJ_GridSite}'"'
#fi
export RJ_MAX_QUEUE_WAITTIME=3*60*60
export RJ_DISABLE_STATE_CHECK=DISABLESTATECHECK
export RJ_DISABLE_PROBE_CHECK=DISABLEPROBECHECK
export RJ_PROXY_SOURCE=X509SUBMITPROXY
export RJ_GRID_RESOURCE=GRIDRESOURCE
export RJ_GRID_HISTORY=GRIDHISTORY
export RJ_GRID_JOB_ID=GRIDJOBID
export RJ_TIMESTAMP_TRANSFERRED=TIMESTAMPTRANSFERRED
export RJ_TIMESTAMP_START=TIMESTAMPSTART
export RJ_TIMESTAMP_FINISH=TIMESTAMPFINISH
export RJ_TIME_RESULTS=TIMERESULTS

runjob.sh
"""


   def __buildMPIFile(self):
      # setup regex's for the template
      re_runName           = re.compile("RUNNAME")
      re_jobId             = re.compile("JOBID")
      re_instanceId        = re.compile("INSTANCEID")
      re_exe               = re.compile("_EXECUTABLE_")
      re_transferexe       = re.compile("TRANSFEREXE")
      re_exesh             = re.compile("EXECUTABLESHELL")
      re_workingdir        = re.compile("WORKINGDIR")
      re_walltime          = re.compile("WALLTIME")
      re_ppn               = re.compile("PPN")
      re_nprocesses        = re.compile("NPROCESSES")
      re_environment       = re.compile("_ENVIRONMENT_")
      re_gridsite          = re.compile("GRIDSITE")
      re_arguments         = re.compile("_ARGUMENTS_")
      re_tsTransferred     = re.compile("TIMESTAMPTRANSFERRED")
      re_tsStart           = re.compile("TIMESTAMPSTART")
      re_tsFinish          = re.compile("TIMESTAMPFINISH")
      re_timeResults       = re.compile("TIMERESULTS")
      re_gridResource      = re.compile("GRIDRESOURCE")
      re_gridHistory       = re.compile("GRIDHISTORY")
      re_gridJobId         = re.compile("GRIDJOBID")
      re_disableStateCheck = re.compile("DISABLESTATECHECK")
      re_disableProbeCheck = re.compile("DISABLEPROBECHECK")
      re_x509SubmitProxy   = re.compile("X509SUBMITPROXY")
      re_hubUserName       = re.compile("HUBUSERNAME")
      re_hubUserId         = re.compile("HUBUSERID")

      template = self.__makeMPITemplate()

      template = re_tsTransferred.sub(self.timestampTransferred,template)
      template = re_tsStart.sub(self.timestampStart,template)
      template = re_tsFinish.sub(self.timestampFinish,template)
      template = re_timeResults.sub(self.timeResults,template)
      template = re_gridResource.sub(self.gridResource,template)
      template = re_gridHistory.sub(self.gridHistory,template)
      template = re_gridJobId.sub(self.gridJobId,template)
      if self.disableStateCheck:
         template = re_disableStateCheck.sub("yes",template)
      else:
         template = re_disableStateCheck.sub("",template)
      if self.disableProbeCheck:
         template = re_disableProbeCheck.sub("yes",template)
      else:
         template = re_disableProbeCheck.sub("",template)
      template = re_x509SubmitProxy.sub(self.x509SubmitProxy,template)

      template = re_runName.sub(self.runName,template)
      template = re_jobId.sub(self.localJobId,template)
      template = re_instanceId.sub(self.instanceId,template)
      template = re_exe.sub(self.executable,template)
      if not self.transferExecutable:
         template = re_transferexe.sub("no",template)
      else:
         template = re_transferexe.sub("yes",template)
      template = re_exesh.sub(os.path.join(self.instanceDirectory,self.appScriptName),template)
      template = re_workingdir.sub(self.workingDirectory,template)
      template = re_walltime.sub(self.wallTime,template)

      template = re_ppn.sub(self.ppn,template)
      template = re_nprocesses.sub(str(self.nCpus),template)
      template = re_environment.sub(self.environment,template)
      template = re_gridsite.sub(self.gridsite,template)
      template = re_arguments.sub(self.arguments,template)
      template = re_hubUserName.sub(self.hubUserName,template)
      template = re_hubUserId.sub(str(self.hubUserId),template)

      return(template)


   def __makeParallelTemplate(self):
      return """#!/bin/sh
# RemoteBatchRUNJOB:makeParallelTemplate
#
# Set up the environment variables expected by runjob.sh
export RJ_JOB_ID=JOBID_INSTANCEID
export RJ_JOB_NAME=RUNNAME_INSTANCEID
export RJ_LOCAL_JOB_DIR=`pwd`
export RJ_INPUT_TARBALL=JOBID_INSTANCEID_input.tar.gz
export RJ_OUTPUT_TARBALL=JOBID_INSTANCEID_output.tar.gz
export RJ_JOB_WRAPPER=EXECUTABLESHELL
export RJ_EXECUTABLE=_EXECUTABLE_
export RJ_TRANSFER_EXEC=TRANSFEREXE
export RJ_WORKDIR=WORKINGDIR
export RJ_ARGUMENTS="_ARGUMENTS_"
export RJ_ENVIRONMENT="_ENVIRONMENT_"
export RJ_WALL_TIME=WALLTIME
export RJ_PARALLEL_CPU_COUNT=NPROCESSES
export RJ_APP_CPUS_PER_HOST=PPN
export RJ_GRID_SITES="GRIDSITE"
#if [ "${RJ_GridSite}" = "" ] ; then
#   export RJ_REQUIREMENTS=""
#else
#   export RJ_REQUIREMENTS='Name=="'${RJ_GridSite}'"'
#fi
export RJ_MAX_QUEUE_WAITTIME=3*60*60
export RJ_DISABLE_STATE_CHECK=DISABLESTATECHECK
export RJ_DISABLE_PROBE_CHECK=DISABLEPROBECHECK
export RJ_PROXY_SOURCE=X509SUBMITPROXY
export RJ_GRID_RESOURCE=GRIDRESOURCE
export RJ_GRID_HISTORY=GRIDHISTORY
export RJ_GRID_JOB_ID=GRIDJOBID
export RJ_TIMESTAMP_TRANSFERRED=TIMESTAMPTRANSFERRED
export RJ_TIMESTAMP_START=TIMESTAMPSTART
export RJ_TIMESTAMP_FINISH=TIMESTAMPFINISH
export RJ_TIME_RESULTS=TIMERESULTS

runjob.sh
"""


   def __buildParallelFile(self):
      # setup regex's for the template
      re_runName           = re.compile("RUNNAME")
      re_jobId             = re.compile("JOBID")
      re_instanceId        = re.compile("INSTANCEID")
      re_exe               = re.compile("_EXECUTABLE_")
      re_transferexe       = re.compile("TRANSFEREXE")
      re_exesh             = re.compile("EXECUTABLESHELL")
      re_workingdir        = re.compile("WORKINGDIR")
      re_walltime          = re.compile("WALLTIME")
      re_ppn               = re.compile("PPN")
      re_nprocesses        = re.compile("NPROCESSES")
      re_environment       = re.compile("_ENVIRONMENT_")
      re_gridsite          = re.compile("GRIDSITE")
      re_arguments         = re.compile("_ARGUMENTS_")
      re_tsTransferred     = re.compile("TIMESTAMPTRANSFERRED")
      re_tsStart           = re.compile("TIMESTAMPSTART")
      re_tsFinish          = re.compile("TIMESTAMPFINISH")
      re_timeResults       = re.compile("TIMERESULTS")
      re_gridResource      = re.compile("GRIDRESOURCE")
      re_gridHistory       = re.compile("GRIDHISTORY")
      re_gridJobId         = re.compile("GRIDJOBID")
      re_disableStateCheck = re.compile("DISABLESTATECHECK")
      re_disableProbeCheck = re.compile("DISABLEPROBECHECK")
      re_x509SubmitProxy   = re.compile("X509SUBMITPROXY")
      re_hubUserName       = re.compile("HUBUSERNAME")
      re_hubUserId         = re.compile("HUBUSERID")

      template = self.__makeParallelTemplate()

      template = re_tsTransferred.sub(self.timestampTransferred,template)
      template = re_tsStart.sub(self.timestampStart,template)
      template = re_tsFinish.sub(self.timestampFinish,template)
      template = re_timeResults.sub(self.timeResults,template)
      template = re_gridResource.sub(self.gridResource,template)
      template = re_gridHistory.sub(self.gridHistory,template)
      template = re_gridJobId.sub(self.gridJobId,template)
      if self.disableStateCheck:
         template = re_disableStateCheck.sub("yes",template)
      else:
         template = re_disableStateCheck.sub("",template)
      if self.disableProbeCheck:
         template = re_disableProbeCheck.sub("yes",template)
      else:
         template = re_disableProbeCheck.sub("",template)
      template = re_x509SubmitProxy.sub(self.x509SubmitProxy,template)

      template = re_runName.sub(self.runName,template)
      template = re_jobId.sub(self.localJobId,template)
      template = re_instanceId.sub(self.instanceId,template)
      template = re_exe.sub(self.executable,template)
      if not self.transferExecutable:
         template = re_transferexe.sub("no",template)
      else:
         template = re_transferexe.sub("yes",template)
      template = re_exesh.sub(os.path.join(self.instanceDirectory,self.appScriptName),template)
      template = re_workingdir.sub(self.workingDirectory,template)
      template = re_walltime.sub(self.wallTime,template)

      template = re_ppn.sub(self.ppn,template)
      template = re_nprocesses.sub(str(self.nCpus),template)
      template = re_environment.sub(self.environment,template)
      template = re_gridsite.sub(self.gridsite,template)
      template = re_arguments.sub(self.arguments,template)
      template = re_hubUserName.sub(self.hubUserName,template)
      template = re_hubUserId.sub(str(self.hubUserId),template)

      return(template)


   def buildBatchScript(self):
      batchLogName = ""
      batchScriptName = "%s_%s.runjob" % (self.localJobId,self.instanceId)
      if self.isMultiCoreRequest:
         if   self.computationMode == 'mpi':
            batchScript = self.__buildMPIFile()
         elif self.computationMode == 'parallel':
            batchScript = self.__buildParallelFile()
         elif self.computationMode == 'matlabmpi':
            batchScript = ""
            batchScriptName = ""
      else:
         batchScript = self.__buildSerialFile()

      return(batchLogName,batchScriptName,batchScript)


   def getBatchNodeList(self):
      return(self.nodeFileName,self.nodeList)


